//	Advanced System UI
//	Copyright (c) 2010-2011 Brand Huntsman <http://qzx.com/mail/>
//
//	XRR initialization and rotation code ported from advanced-backlight
//	(C) 2008 Jonas Hurrelmann <j@outpo.st>.
//	(C) 2008 Adam Harwell <aharwell@trinity.edu>.
//		Randr handling has been ported from gpe-conf
//		(C) 2002 Moray Allan <moray@sermisy.org>, Pierre TARDY <tardyp@free.fr>
//		    2003-2007 Florian Boor <florian.boor@kernelconcepts.de>.

//	DEPENDANCIES:
//		libxpm4

#define APPNAME "Advanced SystemUI"

#include "window.h"

#include <X11/Xproto.h>
#include <X11/Xatom.h>
#include <X11/xpm.h>
#include <X11/extensions/Xrandr.h>
#include <Xft.h>
#include <assert.h>

#include "main.h"
#include "text.h"
#include "config.h"
#include "services.h"
#include "hardware.h"
#include "dialog.h"

#include "images/button_toflightmode.xpm"
#include "images/button_tonormalmode.xpm"
#include "images/reboot.xpm"
#include "images/shutdown.xpm"

#include "images/icon_brightness.xpm"
#include "images/icon_volume.xpm"

////////////////////////////////////////////////////////////////////////// GLOBALS

s_window window;

#define NR_COLORS 15
#define BG_COLOR 0
#define ROTATION_COLOR 6
XColor *bg_color, *line_color, *innerline_color, *fg1_color, *fg2_color, *white_color, *rotation_color;
XColor *gray_status_color, *yellow_status_color, *green_status_color, *red_status_color;
XColor *scrollbar_color, *scroller_color, *selection_color, *secondary_selection_color;

s_image im_toflight, im_tonormal, im_reboot, im_shutdown, im_portrait, im_portrait_inv, im_landscape_inv;
s_image im_volume, im_brightness;

unsigned black_theme;

unsigned blank_window_open;
int blank_fd;

//////////////////////////////////////////////////////////////////////////

XColor white_colors[NR_COLORS], black_colors[NR_COLORS];

#define MWM_DECORATIONS (1L<<1)
struct s_mwm_hints {
	CARD32 flags;
	CARD32 functions;
	CARD32 decorations;
	INT32 input_mode;
	CARD32 status;
};

////////////////////////////////////////////////////////////////////////// XPM ICONS

static void load_masked_xpm( char **image_xpm, s_image *image){
	XpmAttributes attributes;
	XpmColorSymbol colors[1] = {{"none", "none", 0}};

	attributes.numsymbols = 1;
	attributes.colorsymbols = colors;
	attributes.closeness = 40000;
	attributes.valuemask = XpmColorSymbols | XpmCloseness;
	XpmCreatePixmapFromData(window.display, window.root_window, image_xpm, &image->black_pixmap, &image->mask, &attributes);
	image->width = attributes.width;
	image->height = attributes.height;
}

static void load_xpm( char **image_xpm, s_image *image){
	XpmAttributes attributes;
	int z, bytes_per_pixel, length;
	XImage *xi;

	attributes.closeness = 40000;
	attributes.valuemask = XpmCloseness;
	XpmCreatePixmapFromData(window.display, window.root_window, image_xpm, &image->black_pixmap, NULL, &attributes);

	// invert image for white theme
	image->white_pixmap = XCreatePixmap(window.display, window.root_window, attributes.width, attributes.height, window.depth);
	xi = XGetImage(window.display, image->black_pixmap, 0,0, attributes.width, attributes.height, AllPlanes, ZPixmap);

	bytes_per_pixel = (xi->bits_per_pixel)>>3;
	length = attributes.height * attributes.width * bytes_per_pixel;
	for(z = 0; z < length; z += bytes_per_pixel){
		unsigned inverted = 65535 - (xi->data[z] + (xi->data[z+1]<<8));
		xi->data[z] = inverted & 0xff;
		xi->data[z+1] = inverted>>8;
	}
	XPutImage(window.display, image->white_pixmap, window.gc, xi, 0,0, 0,0, attributes.width, attributes.height);
	XDestroyImage(xi);
	image->width = attributes.width;
	image->height = attributes.height;
}

////////////////////////////////////////////////////////////////////////// ROTATION ICONS

static void _rotate_into_image( Pixmap source, float angle, Pixmap destination ){
	// rotate source into destination
	{
		Picture src_picture = None;
		Picture dst_picture = None;
		XRenderPictFormat *pictformat = NULL;
		XRenderPictureAttributes pictattr;
		XTransform transform;

		pictattr.repeat = RepeatNormal;
		pictformat = XRenderFindVisualFormat(window.display, window.visual);
		src_picture = XRenderCreatePicture(window.display, source, pictformat, CPRepeat, &pictattr);
		dst_picture = XRenderCreatePicture(window.display, destination, pictformat, CPRepeat, &pictattr);
		transform.matrix[0][0] = (XFixed)(0x10000 * (angle == 0.0 ? -1.0 : 0.0));						//	cosa	 0		-1		 0
		transform.matrix[0][1] = (XFixed)(0x10000 * (angle == 1.0 ? -1.0 : (angle == 0.0 ? 0.0 : 1.0)));//	sina	-1		 0		 1
		transform.matrix[0][2] = (XFixed)(0x10000 * 0.0);
		transform.matrix[1][0] = (XFixed)(0x10000 * angle);												//	-sina	 1		 0		-1
		transform.matrix[1][1] = (XFixed)(0x10000 * (angle == 0.0 ? -1.0 : 0.0));						//	cosa	 0		-1		 0
		transform.matrix[1][2] = (XFixed)(0x10000 * 0.0);
		transform.matrix[2][0] = (XFixed)(0x10000 * 0.0);												// angle =	 1		 0		-1
		transform.matrix[2][1] = (XFixed)(0x10000 * 0.0);
		transform.matrix[2][2] = (XFixed)(0x10000 * 1.0);
		XRenderSetPictureTransform(window.display, src_picture, &transform);
		XRenderSetPictureFilter(window.display, src_picture, "fast", NULL, 0);
		XRenderComposite(window.display, PictOpSrc, src_picture, None, dst_picture, 0, 0, 0, 0, 0, 0, 64,64);

		if(src_picture != None)
			XRenderFreePicture(window.display, src_picture);
		if(dst_picture != None)
			XRenderFreePicture(window.display, dst_picture);
	}
}

static void rotate_into_image( Pixmap black_source, Pixmap white_source, float angle, s_image *image ){
	// setup destination pixmap and mask
	image->black_pixmap = XCreatePixmap(window.display, window.root_window, 64, 64, window.depth);
	image->white_pixmap = XCreatePixmap(window.display, window.root_window, 64, 64, window.depth);
	image->width = image->height = 64;

	_rotate_into_image(black_source, angle, image->black_pixmap);
	_rotate_into_image(white_source, angle, image->white_pixmap);
}

static void generate_rotation_icons( ){
	XftFont *font;
	XGlyphInfo info;
	XftResult result;
	XftPattern *parsed, *matched;
	XftDraw *surface;
	XftColor xft_color;
	Pixmap black_pixmap, white_pixmap;

	// get font
	parsed = XftNameParse("sans:medium:pixelsize=60");
	matched = XftFontMatch(window.display, window.screen, parsed, &result);
	font = XftFontOpenPattern(window.display, matched);
	XftPatternDestroy(parsed);
	XftPatternDestroy(matched);
	XftTextExtentsUtf8(window.display, font, (const FcChar8 *)"A", 1, &info);

	// black theme
		// set color
		xft_color.pixel = black_colors[ROTATION_COLOR].pixel;
		xft_color.color.red = black_colors[ROTATION_COLOR].red;
		xft_color.color.green = black_colors[ROTATION_COLOR].green;
		xft_color.color.blue = black_colors[ROTATION_COLOR].blue;
		xft_color.color.alpha = 0xffff;
		// create pixmap with 'A'
		black_pixmap = XCreatePixmap(window.display, window.root_window, 64, 64, window.depth);
			XSetForeground(window.display, window.gc, black_colors[BG_COLOR].pixel);
			XFillRectangle(window.display, black_pixmap, window.gc, 0,0, 64,64);
		surface = XftDrawCreate(window.display, black_pixmap, window.visual, window.colormap);
		XftDrawStringUtf8(surface, &xft_color, font, 32 - (info.width>>1), 32 + (info.height>>1) - (info.height - info.y), (const FcChar8 *)"A", 1);
		XftDrawDestroy(surface);

	// white theme
		// set color
		xft_color.pixel = white_colors[ROTATION_COLOR].pixel;
		xft_color.color.red = white_colors[ROTATION_COLOR].red;
		xft_color.color.green = white_colors[ROTATION_COLOR].green;
		xft_color.color.blue = white_colors[ROTATION_COLOR].blue;
		xft_color.color.alpha = 0xffff;
		// create pixmap with 'A'
		white_pixmap = XCreatePixmap(window.display, window.root_window, 64, 64, window.depth);
			XSetForeground(window.display, window.gc, white_colors[BG_COLOR].pixel);
			XFillRectangle(window.display, white_pixmap, window.gc, 0,0, 64,64);
		surface = XftDrawCreate(window.display, white_pixmap, window.visual, window.colormap);
		XftDrawStringUtf8(surface, &xft_color, font, 32 - (info.width>>1), 32 + (info.height>>1) - (info.height - info.y), (const FcChar8 *)"A", 1);
		XftDrawDestroy(surface);

	// rotate pixmap into each icon
	rotate_into_image(black_pixmap, white_pixmap,  1.0, &im_portrait); // top points left
	rotate_into_image(black_pixmap, white_pixmap, -1.0, &im_portrait_inv); // top points right
	rotate_into_image(black_pixmap, white_pixmap,  0.0, &im_landscape_inv); // top points down

	XFreePixmap(window.display, black_pixmap);
	XFreePixmap(window.display, white_pixmap);
}

////////////////////////////////////////////////////////////////////////// WINDOW INITIALIZATION

static int window_create( unsigned first_time ){
	if(first_time){
		// open display
		if((window.display = XOpenDisplay(NULL)) == 0){
			error_write("can't open display: %s", XDisplayName(NULL));
			return 0;
		}
		window.root_window = DefaultRootWindow(window.display);
		window.screen = DefaultScreen(window.display);
		window.visual = DefaultVisual(window.display, window.screen);
		window.depth = DefaultDepth(window.display, window.screen);
		window.colormap = DefaultColormap(window.display, window.screen);

		// check for rotation support (from advanced-backlight)
		int xrr_error_base, xrr_major, xrr_minor;
		if(!XRRQueryExtension(window.display, &window.xrr_event_base, &xrr_error_base) || !XRRQueryVersion(window.display, &xrr_major, &xrr_minor) || xrr_major < 1){
			window.supports_rotation = 0;
			debug_write("no rotation support (%d %d %d.%d)", window.xrr_event_base, xrr_error_base, xrr_major, xrr_minor);
		} else {
			Rotation current_xrr_orientation;
			XRRRotations(window.display, window.screen, &current_xrr_orientation);
			switch (current_xrr_orientation){
			case RR_Rotate_270: window.orientation = O_PORTRAIT_INV; break;
			case RR_Rotate_180: window.orientation = O_LANDSCAPE_INV; break;
			case RR_Rotate_90: window.orientation = O_PORTRAIT; break;
			case RR_Rotate_0: window.orientation = O_LANDSCAPE; break;
			default:
				error_write("unknown XRR orientation: %d", current_xrr_orientation);
				XSetCloseDownMode(window.display, DestroyAll);
				XCloseDisplay(window.display);
				return 0;
			}
			window.supports_rotation = 1;
			debug_write("rotation supported (%d %d %d.%d)", window.xrr_event_base, xrr_error_base, xrr_major, xrr_minor);
		}

		// get screen dimensions
		window.width = XDisplayWidth(window.display, window.screen);
		window.height = XDisplayHeight(window.display, window.screen);
	}

	// create window
	{
		XSetWindowAttributes winattr;

		window.window = XCreateWindow(window.display, window.root_window, 0, 0, window.width, window.height,
										0, CopyFromParent, CopyFromParent, CopyFromParent, 0, &winattr);
		window.gc = XCreateGC(window.display, window.window, 0, NULL);
		XSetGraphicsExposures(window.display, window.gc, False);
		XSelectInput(window.display, window.window, StructureNotifyMask
			| ButtonPressMask | ButtonMotionMask | ButtonReleaseMask
			| KeyPressMask | KeyReleaseMask
			| (use_minimal_ui ? 0 : FocusChangeMask) | ExposureMask);
		if(window.supports_rotation)
			XRRSelectInput(window.display, window.window, RRScreenChangeNotifyMask);
	}

	// set name
	{
		XTextProperty text_prop;

		text_prop.value = APPNAME;
		text_prop.encoding = XInternAtom(window.display, "STRING", False);
		text_prop.format = 8;
		text_prop.nitems = strlen(APPNAME);
		XSetWMName(window.display, window.window, &text_prop);
	}

	// set protocols -- no window decorations
	{
		struct s_mwm_hints motif_hints;
		Atom props[1];
		Atom motif_atom;

		XSetWMProtocols(window.display, window.window, props, 0);

		motif_hints.flags = MWM_DECORATIONS;
		motif_hints.decorations = 0; // disable window decorations
		motif_atom = XInternAtom(window.display, "_MOTIF_WM_HINTS", False);
		XChangeProperty(window.display, window.window, motif_atom, motif_atom, 32, PropModeReplace,
			(unsigned char *)&motif_hints, sizeof(motif_hints) / 4);
	}

	if(first_time){
		// initialize colors
		#define x_get_color(black, white) \
			XParseColor(window.display, window.colormap, black, &black_colors[i]); XAllocColor(window.display, window.colormap, &black_colors[i  ]); \
			XParseColor(window.display, window.colormap, white, &white_colors[i]); XAllocColor(window.display, window.colormap, &white_colors[i++]);
		int i = 0;
		x_get_color("#000000", "#ffffff"); // bg_color
		x_get_color("#444444", "#bbbbbb"); // line_color
		x_get_color("#333333", "#cccccc"); // innerline_color
		x_get_color("#dddddd", "#222222"); // fg1_color
		x_get_color("#aaaaaa", "#555555"); // fg2_color
		x_get_color("#ffffff", "#000000"); // white_color
		x_get_color("#888888", "#777777"); // rotation_color
		x_get_color("#777777", "#cccccc"); // gray_status_color
		x_get_color("#cccc33", "#aaaa00"); // yellow_status_color
		x_get_color("#33aa33", "#00aa00"); // green_status_color
		x_get_color("#ff2222", "#ff0000"); // red_status_color
		x_get_color("#aaaaff", "#6666dd"); // scrollbar_color
		x_get_color("#555555", "#aaaaaa"); // scroller_color
		x_get_color("#7777ff", "#0000dd"); // selection_color
		x_get_color("#fb9200", "#ee6100"); // secondary_selection_color

		// load images
		load_xpm(button_toflightmode_icon, &im_toflight);
		load_xpm(button_tonormalmode_icon, &im_tonormal);
		load_xpm(button_reboot_icon, &im_reboot);
		load_xpm(button_shutdown_icon, &im_shutdown);
		generate_rotation_icons();
		load_masked_xpm(volume_icon, &im_volume);
		load_masked_xpm(brightness_icon, &im_brightness);

		// initialize fonts
		text_init();
	}

	window.is_mapped = 0;
	window.has_focus = 0;
	return 1;
}

static double last_orientation_check;

int window_init( ){
	blank_window_open = 0;

	last_orientation_check = 0.0;

	black_theme = 0;
	window_set_theme(0); // initialize to black
	window_auto_set_theme(1); // set based on cfg_theme

	dialog_init();

	// check for rotation capabilities and create resources
	return window_create(1);
}

void window_set_theme( unsigned white_theme ){
	XColor *theme;
	int i = 0;

	if(black_theme != white_theme) return; // no change

	black_theme = (white_theme ? 0 : 1);
	hw_changes |= HW_THEME;

	theme = (black_theme ? black_colors : white_colors);

	bg_color = &theme[i++];
	line_color = &theme[i++];
	innerline_color = &theme[i++];
	fg1_color = &theme[i++];
	fg2_color = &theme[i++];
	white_color = &theme[i++];
	rotation_color = &theme[i++];

	gray_status_color = &theme[i++];
	yellow_status_color = &theme[i++];
	green_status_color = &theme[i++];
	red_status_color = &theme[i++];

	scrollbar_color = &theme[i++];
	scroller_color = &theme[i++];
	selection_color = &theme[i++];
	secondary_selection_color = &theme[i++];

	assert(i == NR_COLORS);
}

void window_auto_set_theme( unsigned initialize ){
	unsigned white_theme;

	if(cfg_theme)
		// manual theme
		white_theme = cfg_theme - 1;
	else if(initialize)
		// no ALS data yet, set theme based on white threshold value
		white_theme = (cfg_threshold_white > 0 ? 0 : 1);
	else if(black_theme && hw_sensors.light >= cfg_threshold_white)
		// ALS transition from black to white
		white_theme = 1;
	else if(!black_theme && cfg_threshold_white > 0 && hw_sensors.light < cfg_threshold_black)
		// ALS transition from white to black
		white_theme = 0;
	else return;
		// no ALS transitions, do nothing

	window_set_theme(white_theme);
}

void window_map( ){
	debug_write("window_map()");

	if(blank_window_open)
		window_close_blank();

	// map the window
	{
		XEvent e;

		XMapWindow(window.display, window.window);
		for(;;){
			usleep(50000); // sleep for 50ms
			XNextEvent(window.display, &e);
			if(e.type == MapNotify) break;
		}
	}

	////////////////////

	if(!use_minimal_ui){
		// grab keyboard (home key)
		if(XGrabKeyboard(window.display, window.window, False, GrabModeAsync, GrabModeAsync, CurrentTime) != GrabSuccess){
			// a menu is open, unmap window
			debug_write("window_map() - can't get keyboard, unmap");
			XUnmapWindow(window.display, window.window);
			x_flush();
			return;
		}
		// grab pointer to prevent another program from stealing it away from ASUI
		XGrabPointer(window.display, window.window, False, ButtonPressMask|ButtonMotionMask|ButtonReleaseMask,
						GrabModeAsync, GrabModeAsync, window.window, None, CurrentTime);
	}
	window.has_focus = 1;

	////////////////////

// TODO: svc_refresh is called by svc_init and will be called twice at startup if cfg_start_visible

	hw_refresh_cpu();
	svc_refresh();
	brightness_widget_page = volume_widget_page = battery_widget_page = flightmode_button_page = keyrepeat_button_page = screenstayslit_button_page = screenautolock_button_page = primary_widgets;
	hw_refresh(); // query all values and then reset pages to prevent further queryies unless on current page
	brightness_widget_page = volume_widget_page = battery_widget_page = flightmode_button_page = keyrepeat_button_page = screenstayslit_button_page = screenautolock_button_page = !primary_widgets;

	window.is_mapped = 1;

	draw_window(1); // force
}

void window_unmap( ){
	debug_write("window_unmap()");

	// close dialog if open -- this is needed to reset anything it did to the main loop (the process viewer refresh)
	dialog_reset();

	// inform the main loop
	main_reset();

	if(!use_minimal_ui){
		// release keyboard (home key)
		XUngrabKeyboard(window.display, CurrentTime);
		XUngrabPointer(window.display, CurrentTime);
	}
	window.has_focus = 0;

	// unmap the window
	XUnmapWindow(window.display, window.window);
	x_flush();

	window.is_mapped = 0;
}

void window_close( unsigned terminate ){
	if(window.is_mapped)
		window_unmap();

	if(terminate)
		text_free_fonts();

	XFreeGC(window.display, window.gc);
	XDestroyWindow(window.display, window.window);
	x_flush();

	if(terminate){
		XSetCloseDownMode(window.display, DestroyAll);
		XCloseDisplay(window.display);
	}
}

////////////////////////////////////////////////////////////////////////// ROTATION

void rotate_screen( e_orientation new_orientation, unsigned keep_mapped ){
	Rotation new_xrr_orientation = RR_Rotate_0, current_xrr_orientation;
	XRRScreenConfiguration *xrr_config;
	int xrr_size, needs_resize = 0;
	Window tmp_window = 0;

	if(new_orientation == window.orientation) return;

	// rotate the dpad
	{
		char *keyname[4] = { "Up", "Right", "Down", "Left" };
		KeyCode keycode[4] = { 111, 114, 116, 113 };
		int i;

		for(i = 0; i < 4; i++){
			KeySym keysym = XStringToKeysym(keyname[(new_orientation + i) % 4]);
			XChangeKeyboardMapping(window.display, keycode[i], 1, &keysym, 1);
		}
	}
	// prepare the rotation
	xrr_config = XRRGetScreenInfo(window.display, window.root_window);
	xrr_size = XRRConfigCurrentConfiguration(xrr_config, &current_xrr_orientation);
	switch(new_orientation){
	case O_LANDSCAPE:
		new_xrr_orientation = RR_Rotate_0;
		if(window.orientation == O_PORTRAIT || window.orientation == O_PORTRAIT_INV) needs_resize = 1;
		break;
	case O_PORTRAIT:
		new_xrr_orientation = RR_Rotate_90;
		if(window.orientation == O_LANDSCAPE || window.orientation == O_LANDSCAPE_INV) needs_resize = 1;
		break;
	case O_LANDSCAPE_INV:
		new_xrr_orientation = RR_Rotate_180;
		if(window.orientation == O_PORTRAIT || window.orientation == O_PORTRAIT_INV) needs_resize = 1;
		break;
	case O_PORTRAIT_INV:
		new_xrr_orientation = RR_Rotate_270;
		if(window.orientation == O_LANDSCAPE || window.orientation == O_LANDSCAPE_INV) needs_resize = 1;
		break;
	}

	// create a secondary window to mask screen resize distortion
	if(needs_resize && keep_mapped){
		unsigned size;
		GC gc;

		if(window.orientation == O_LANDSCAPE || window.orientation == O_LANDSCAPE_INV)
			size = window.width;
		else
			size = window.height;

		// create window
		{
			XSetWindowAttributes winattr;

			tmp_window = XCreateWindow(window.display, window.root_window, 0,0, size,size,
				0, CopyFromParent, CopyFromParent, CopyFromParent, 0, &winattr);
			gc = XCreateGC(window.display, tmp_window, 0, NULL);
			XSetGraphicsExposures(window.display, gc, False);
			XSelectInput(window.display, tmp_window, StructureNotifyMask);
		}

		// set protocols
		{
			struct s_mwm_hints motif_hints;
			Atom props[1];
			Atom motif_atom;

			XSetWMProtocols(window.display, tmp_window, props, 0);

			motif_hints.flags = MWM_DECORATIONS;
			motif_hints.decorations = 0; // disable window decorations
			motif_atom = XInternAtom(window.display, "_MOTIF_WM_HINTS", False);
			XChangeProperty(window.display, tmp_window, motif_atom, motif_atom, 32, PropModeReplace,
				(unsigned char *)&motif_hints, sizeof(motif_hints) / 4);

			props[0] = XInternAtom(window.display, "_NET_WM_STATE_ABOVE", True);
			XChangeProperty(window.display, tmp_window, XInternAtom(window.display, "_NET_WM_STATE", False),
				XA_ATOM, 32, PropModeReplace, (unsigned char *)&props, 1);
		}

		// map the window
		{
			XEvent e;

			XMapWindow(window.display, tmp_window);
			for(;;){
				usleep(50000); // sleep for 50ms
				XNextEvent(window.display, &e);
				if(e.type == MapNotify) break;
			}
		}

		// paint black
		{
			Window orig_window = window.window;
			GC orig_gc = window.gc;

			window.window = tmp_window;
			window.gc = gc;

			x_set_color(bg_color);
			x_fill_rectangle(0, 0, size,size);
			draw_text("rotating...", 32, white_color, 240, 240, ALIGN_CENTER, ALIGN_MIDDLE, 0);
			XFreeGC(window.display, gc);
			x_flush();

			window.window = orig_window;
			window.gc = orig_gc;
		}
	}

	// rotate screen
	XRRSetScreenConfig(window.display, xrr_config, window.root_window, xrr_size, new_xrr_orientation, CurrentTime);
	XRRFreeScreenConfigInfo(xrr_config);
/*
// TODO:
	if(is_770){
		// Force automatic update of the LCD when on 770 and rotated
		int fb, mode = (new_xrr_orientation = RR_Rotate_0 ? OMAPFB_MANUAL_UPDATE : OMAPFB_AUTO_UPDATE);
		if((fb = open(FBDEV, O_RDWR)) >= 0){
			ioctl(fb, OMAPFB_SET_UPDATE_MODE, &mode);
			close(fb);
		}
	}
*/

	window.orientation = new_orientation;

	// resize window or redraw with new rotation icons
	if(needs_resize){
		// swap screen dimensions
		unsigned width = window.width;
		window.width = window.height;
		window.height = width;

		// unmap window
		if(keep_mapped){
			XUnmapWindow(window.display, window.window);
			x_flush();
		} else window_unmap();
		// resize window
		XWindowChanges changes;
		changes.width = window.width;
		changes.height = window.height;
		XConfigureWindow(window.display, window.window, CWWidth|CWHeight, &changes);
		{
			XEvent e;

			for(;;){
				usleep(50000); // sleep for 50ms
				XNextEvent(window.display, &e);
				if(e.type == ConfigureNotify) break;
			}
		}

		if(keep_mapped){
			// map the window
			XMapWindow(window.display, window.window);
			{
				XEvent e;

				for(;;){
					usleep(50000); // sleep for 50ms
					XNextEvent(window.display, &e);
					if(e.type == MapNotify) break;
				}
			}
			draw_window(1); // force

			// grab keyboard (home key)
			XGrabKeyboard(window.display, window.window, False, GrabModeAsync, GrabModeAsync, CurrentTime);
			// grab pointer to prevent another program from stealing it away from ASUI
			XGrabPointer(window.display, window.window, False, ButtonPressMask|ButtonMotionMask|ButtonReleaseMask,
							GrabModeAsync, GrabModeAsync, window.window, None, CurrentTime);

			// destroy the secondary window
			XDestroyWindow(window.display, tmp_window);
			x_flush();
		}
	} else {
		// redraw rotation icons if visible
		hw_changes |= HW_ROTATION;
		draw_window(0);
	}
}

void update_orientation( XEvent *event ){
	XRRScreenChangeNotifyEvent *xrr_event = (XRRScreenChangeNotifyEvent *)event;
	unsigned flip = 0;

// TODO: an X event is only sent when the screen dimensions change and not when the screen orientation is merely flipped

	// did it flip or resize
	switch(window.orientation){
	case O_LANDSCAPE:		flip = (xrr_event->rotation == RR_Rotate_180 ? 1 : 0); break;
	case O_PORTRAIT:		flip = (xrr_event->rotation == RR_Rotate_270 ? 1 : 0); break;
	case O_LANDSCAPE_INV:	flip = (xrr_event->rotation == RR_Rotate_0 ? 1 : 0); break;
	case O_PORTRAIT_INV:	flip = (xrr_event->rotation == RR_Rotate_90 ? 1 : 0); break;
	}
	// get new orientation, ignore if same as current
	switch (xrr_event->rotation){
	case RR_Rotate_270: if(window.orientation == O_PORTRAIT_INV) return; window.orientation = O_PORTRAIT_INV; break;
	case RR_Rotate_180: if(window.orientation == O_LANDSCAPE_INV) return; window.orientation = O_LANDSCAPE_INV; break;
	case RR_Rotate_90: if(window.orientation == O_PORTRAIT) return; window.orientation = O_PORTRAIT; break;
	case RR_Rotate_0: if(window.orientation == O_LANDSCAPE) return; window.orientation = O_LANDSCAPE; break;
	default:
		error_write("unknown XRR orientation: %d", xrr_event->rotation);
		exit(1);
	}

	// update window orientation
	if(!flip){
//		unsigned was_mapped = window.is_mapped;

		// close the window
//		window_close(0); // keep display open

		// swap screen dimensions
		unsigned width = window.width;
		window.width = window.height;
		window.height = width;

		// unmap window
		if(window.is_mapped){
			XUnmapWindow(window.display, window.window);
			x_flush();
		}
		// resize window
		XWindowChanges changes;
		changes.width = window.width;
		changes.height = window.height;
		XConfigureWindow(window.display, window.window, CWWidth|CWHeight, &changes);
		{
			XEvent e;

			for(;;){
				usleep(50000); // sleep for 50ms
				XNextEvent(window.display, &e);
				if(e.type == ConfigureNotify) break;
			}
		}

		// open a new window
//		if(!window_create(0))
//			exit(1); // kill app if window wasn't recreated

		if(window.is_mapped){
			// map the window
			XEvent e;

			XMapWindow(window.display, window.window);
			for(;;){
				usleep(50000); // sleep for 50ms
				XNextEvent(window.display, &e);
				if(e.type == MapNotify) break;
			}
			draw_window(1); // force
		}

//		if(was_mapped)
//			window_map();
	} else if(window.is_mapped)
		draw_window(1); // force to get new rotation icons
}

int check_orientation( ){
	Rotation current_xrr_orientation;

	// only check once every 10 seconds
	if(last_orientation_check + 10 > current_time) return 0;
	last_orientation_check = current_time;

	XRRScreenConfiguration *xrr_config = XRRGetScreenInfo(window.display, window.root_window);
	XRRConfigRotations(xrr_config, &current_xrr_orientation);
	XRRFreeScreenConfigInfo(xrr_config);
	if(current_xrr_orientation == RR_Rotate_270 && window.orientation == O_PORTRAIT){
		window.orientation = O_PORTRAIT_INV;
		return 1;
	} else if(current_xrr_orientation == RR_Rotate_180 && window.orientation == O_LANDSCAPE){
		window.orientation = O_LANDSCAPE_INV;
		return 1;
	} else if(current_xrr_orientation == RR_Rotate_90 && window.orientation == O_PORTRAIT_INV){
		window.orientation = O_PORTRAIT;
		return 1;
	} else if(current_xrr_orientation == RR_Rotate_0 && window.orientation == O_LANDSCAPE_INV){
		window.orientation = O_LANDSCAPE;
		return 1;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////// BLANK WINDOW

static Display *blank_display;
static Window blank_window;
static GC blank_gc;

void window_open_blank( ){
	// this window catches the first key press or screen tap used to wake the device when not locked

	if(window.is_mapped || blank_window_open) return;

	debug_write("window_open_blank()");

	// open display
	if((blank_display = XOpenDisplay(NULL)) == 0){
		error_write("can't open blank display: %s", XDisplayName(NULL));
		return;
	}

	// create window
	{
		XSetWindowAttributes winattr;

		blank_window = XCreateWindow(blank_display, DefaultRootWindow(blank_display), 0,0, 1,1,
										0, CopyFromParent, CopyFromParent, CopyFromParent, 0, &winattr);
		blank_gc = XCreateGC(blank_display, blank_window, 0, NULL);
		XSetGraphicsExposures(blank_display, blank_gc, False);
		XSelectInput(blank_display, blank_window, StructureNotifyMask | ButtonPressMask | ButtonReleaseMask | KeyPressMask | KeyReleaseMask);
	}

	// set name
	{
		XTextProperty text_prop;

		text_prop.value = "ASUI NET";
		text_prop.encoding = XInternAtom(blank_display, "STRING", False);
		text_prop.format = 8;
		text_prop.nitems = strlen(APPNAME);
		XSetWMName(blank_display, blank_window, &text_prop);
	}

	// set protocols -- no window decorations
	{
		struct s_mwm_hints motif_hints;
		Atom props[5];
		Atom motif_atom;

		XSetWMProtocols(blank_display, blank_window, props, 0);

		motif_hints.flags = MWM_DECORATIONS;
		motif_hints.decorations = 0; // disable window decorations
		motif_atom = XInternAtom(blank_display, "_MOTIF_WM_HINTS", False);
		XChangeProperty(blank_display, blank_window, motif_atom, motif_atom, 32, PropModeReplace,
						(unsigned char *)&motif_hints, sizeof(motif_hints) / 4);
	}

	// map the window
	{
		XEvent e;

		XMapWindow(blank_display, blank_window);
		for(;;){
			usleep(50000); // sleep for 50ms
			XNextEvent(blank_display, &e);
			if(e.type == MapNotify) break;
			debug_write("    waiting for MapNotify event on blank window...");
		}
	}

	// grab keyboard and pointer to catch the wakeup tap/key
	if(XGrabPointer(blank_display, blank_window, False, ButtonPressMask|ButtonReleaseMask, GrabModeAsync, GrabModeAsync, blank_window, None, CurrentTime) != GrabSuccess){
		// a menu is open, close blank window
		debug_write("    failed to open blank window");

		XUnmapWindow(blank_display, blank_window);
		XFreeGC(blank_display, blank_gc);
		XDestroyWindow(blank_display, blank_window);

		x_flush();

		XSetCloseDownMode(blank_display, DestroyAll);
		XCloseDisplay(blank_display);
	} else {
		XGrabKeyboard(blank_display, blank_window, False, GrabModeAsync, GrabModeAsync, CurrentTime);

		x_flush();

		blank_fd = ConnectionNumber(blank_display);
		blank_window_open = 1;

		debug_write("    blank window ready");
	}
}

void window_close_blank( ){
	if(blank_window_open){
		debug_write("window_close_blank()");

		blank_fd = 0;
		XUnmapWindow(blank_display, blank_window);
		XFreeGC(blank_display, blank_gc);
		XDestroyWindow(blank_display, blank_window);

		// release keyboard and pointer
		XUngrabPointer(blank_display, CurrentTime);
		XUngrabKeyboard(blank_display, CurrentTime);

		x_flush();

		XSetCloseDownMode(blank_display, DestroyAll);
		XCloseDisplay(blank_display);

		blank_window_open = 0;
	}
}


unsigned window_blank_events( ){
	unsigned runaway_event_counter = 0;

	if(!blank_window_open) return 0;

	debug_write("any X events for blank window?");
	while(XPending(blank_display)){
		XEvent e;

		XNextEvent(blank_display, &e);
		debug_write("    event %d", e.type);

		runaway_event_counter++;
		if(runaway_event_counter > MAX_RUNAWAY_EVENTS){ debug_write("stuck in a runaway event loop (>%d), killing ASUI", MAX_RUNAWAY_EVENTS); return 1; }

		// process events
		switch(e.type){
		case ButtonPress:
			// ignore taps from the blank window
			break;
		case ButtonRelease:
			// close the blank window
			window_close_blank();
			return 0; // get out of this loop or it will segfault
		case KeyPress:
			// ignore keys from the blank window
			break;
		case KeyRelease:
			// close the blank window
			window_close_blank();
			return 0; // get out of this loop or it will segfault
		case NoExpose:			// 14
		case DestroyNotify:		// 17
		case UnmapNotify:		// 18
		case MapNotify:			// 19
		case ReparentNotify:	// 21
		case ConfigureNotify:	// 22
		case ClientMessage:		// 33
		case MappingNotify:		// 34
			// ignore
			break;
		default:
			debug_write("unknown event %d", e.type);
			break;
		}
	}

	return 0;
}
