//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <http://qzx.com/mail/>

#define ASUI_VERSION "0.5.0"

/*
/etc/init.d/hildon-desktop stop
su - user -c "maemo-summoner /usr/bin/hildon-desktop.launch"
*/

#include <hildon/hildon-defines.h>
#include <libhildondesktop/libhildondesktop.h>
#include <libhildondesktop/statusbar-item.h>

#include <log-functions.h>
#include <libosso.h>
#include <osso-log.h>

#include <dbus/dbus.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>

//////////////////////////////////////////////////////////////////////////

G_BEGIN_DECLS

typedef struct _AsuiBatteryPlugin AsuiBatteryPlugin;
typedef struct _AsuiBatteryPluginClass AsuiBatteryPluginClass;

#define ASUI_BATTERY_PLUGIN_PRIORITY 1

#define ASUI_BATTERY_PLUGIN_ICON_SIZE 40

#define ASUI_BATTERY_PLUGIN_TYPE			(asui_battery_plugin_get_type())
#define ASUI_BATTERY_PLUGIN(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPlugin))
#define ASUI_BATTERY_PLUGIN_CLASS(klass)	(G_TYPE_CHECK_CLASS_CAST((klass),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginClass))
#define IS_ASUI_BATTERY_PLUGIN(obj)			(G_TYPE_CHECK_INSTANCE_TYPE((obj),	ASUI_BATTERY_PLUGIN_TYPE))
#define IS_ASUI_BATTERY_PLUGIN_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass),	ASUI_BATTERY_PLUGIN_TYPE))
#define ASUI_BATTERY_PLUGIN_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj),	ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginClass))

struct _AsuiBatteryPlugin { StatusbarItem parent; };
struct _AsuiBatteryPluginClass { StatusbarItemClass parent_class; };
GType asui_battery_plugin_get_type(void);

typedef struct {
	osso_context_t *osso; // osso
	DBusGConnection *connection; // system bus
	GtkWidget *icon; // icon in button
	GtkWidget *button; // button in statusbar
} AsuiBatteryPluginPrivate;

G_END_DECLS

HD_DEFINE_PLUGIN(AsuiBatteryPlugin, asui_battery_plugin, STATUSBAR_TYPE_ITEM);

#define ASUI_BATTERY_PLUGIN_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE((object), ASUI_BATTERY_PLUGIN_TYPE, AsuiBatteryPluginPrivate))

//////////////////////////////////////////////////////////////////////////

static void asui_battery_plugin_finalize( GObject *object );

static void asui_battery_plugin_class_init( AsuiBatteryPluginClass *klass ){
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	object_class->finalize = asui_battery_plugin_finalize;
	g_type_class_add_private(klass, sizeof(AsuiBatteryPluginPrivate));
}

//////////////////////////////////////////////////////////////////////////

static const char *charging_notification = "Charging...";
static const char *battery_full_notification = "Battery Full";

static void display_notification( AsuiBatteryPluginPrivate *p, const char *msg ){
	// initialize dbus
	DBusGConnection *connection;
	GError *error = NULL;
	if((connection = dbus_g_bus_get(DBUS_BUS_SESSION, &error)) == NULL){
		ULOG_WARN("%s: can't connect to session method bus: %s\n", __FUNCTION__, error->message);
		g_error_free(error);
		return;
	}
	DBusConnection *session_bus = dbus_g_connection_get_connection(connection);

	DBusMessage *message = dbus_message_new_method_call(
		"org.freedesktop.Notifications", "/org/freedesktop/Notifications", "org.freedesktop.Notifications", "SystemNoteInfoprint");
	dbus_message_set_no_reply(message, TRUE);
	dbus_message_append_args(message, DBUS_TYPE_STRING, &msg, DBUS_TYPE_INVALID);
	if(dbus_connection_send(session_bus, message, NULL) != FALSE)
		dbus_connection_flush(session_bus);
	dbus_message_unref(message);
	dbus_g_connection_unref(connection);
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_set_icon( const gchar *name, AsuiBatteryPluginPrivate *p ) {
	GtkIconTheme *icon_theme;
	GdkPixbuf*pixbuf;

	icon_theme = gtk_icon_theme_get_default();
	pixbuf = (name != NULL ? gtk_icon_theme_load_icon(icon_theme, name, ASUI_BATTERY_PLUGIN_ICON_SIZE, GTK_ICON_LOOKUP_NO_SVG, NULL) : NULL);
	gtk_image_set_from_pixbuf(GTK_IMAGE(p->icon), pixbuf);

	if(pixbuf != NULL) g_object_unref(pixbuf);
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_dbus_send_method( AsuiBatteryPluginPrivate *p, const char *destination, const char *path, const char *interface, const char *method ){
	DBusConnection *system_bus = dbus_g_connection_get_connection(p->connection);
	DBusMessage *message = dbus_message_new_method_call(destination, path, interface, method);
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}

static void asui_battery_update_icon( AsuiBatteryPluginPrivate *p );

static gboolean asui_battery_reset_press( gpointer data ){
	asui_battery_update_icon((AsuiBatteryPluginPrivate *)data);
	return FALSE;
}

static gboolean asui_battery_open_asui( GtkButton *widget, GdkEventButton *button, gpointer data ){
	AsuiBatteryPluginPrivate *p = (AsuiBatteryPluginPrivate *)data;

	g_return_val_if_fail(data, FALSE);
	gtk_button_released(GTK_BUTTON(p->button));
	(void) button;

	asui_battery_dbus_send_method(p, "com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "show");

	asui_battery_set_icon("asui_power_pressed", p);
	g_timeout_add(1000, asui_battery_reset_press, (void *)p);

	return TRUE;
}

//////////////////////////////////////////////////////////////////////////

typedef enum e_battery_status { BATTERY_DRAINING, BATTERY_CHARGING, BATTERY_AC } e_battery_status;
static e_battery_status asui_battery_status;
static unsigned asui_battery_capacity, asui_battery_initialized, asui_battery_init_timer, asui_battery_charging_cycle, asui_battery_charging_timer;

static gboolean asui_battery_cycle_charging_icon( gpointer data ){
	if(asui_battery_status == BATTERY_CHARGING){
		switch(asui_battery_charging_cycle){
		case 0: asui_battery_set_icon("asui_charging33", (AsuiBatteryPluginPrivate *)data); asui_battery_charging_cycle = 1; break;
		case 1: asui_battery_set_icon("asui_charging66", (AsuiBatteryPluginPrivate *)data); asui_battery_charging_cycle = 2; break;
		case 2: asui_battery_set_icon("asui_charging100", (AsuiBatteryPluginPrivate *)data); asui_battery_charging_cycle = 0; break;
		}
		return TRUE;
	}
	asui_battery_charging_timer = 0;
	return FALSE;
}

static void asui_battery_update_icon( AsuiBatteryPluginPrivate *p ){
	switch(asui_battery_status){
	case BATTERY_DRAINING:
		if(asui_battery_capacity > 83) asui_battery_set_icon("asui_battery100", p);
		else if(asui_battery_capacity > 66) asui_battery_set_icon("asui_battery83", p);
		else if(asui_battery_capacity > 49) asui_battery_set_icon("asui_battery66", p);
		else if(asui_battery_capacity > 32) asui_battery_set_icon("asui_battery49", p);
		else if(asui_battery_capacity > 15) asui_battery_set_icon("asui_battery32", p);
		else asui_battery_set_icon("asui_battery15", p);
		break;
	case BATTERY_CHARGING:
		asui_battery_charging_cycle = 0;
		asui_battery_set_icon("asui_charging100", p);
		if(!asui_battery_charging_timer)
			asui_battery_charging_timer = g_timeout_add(1000, asui_battery_cycle_charging_icon, (void *)p);
		break;
	case BATTERY_AC:
		asui_battery_set_icon("asui_charged", p);
		break;
	}
}

static DBusHandlerResult asui_battery_signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	DBusError dbus_error;
	if(dbus_message_is_signal(message, "com.qzx.asui", "charging_status")){
		unsigned status;

		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, DBUS_TYPE_UINT32, &status, DBUS_TYPE_INVALID)){
			asui_battery_initialized &= 1;
			if(asui_battery_status != status){
				asui_battery_status = status;
				if(!asui_battery_initialized) asui_battery_update_icon((AsuiBatteryPluginPrivate *)user_data);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.qzx.asui", "battery_capacity")){
		unsigned capacity;

		dbus_error_init(&dbus_error);
		if(dbus_message_get_args(message, &dbus_error, DBUS_TYPE_UINT32, &capacity, DBUS_TYPE_INVALID)){
			asui_battery_initialized &= 2;
			if(asui_battery_capacity != capacity){
				asui_battery_capacity = capacity;
				if(!asui_battery_initialized) asui_battery_update_icon((AsuiBatteryPluginPrivate *)user_data);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_full")){
		display_notification((AsuiBatteryPluginPrivate *)user_data, battery_full_notification);
		return DBUS_HANDLER_RESULT_HANDLED;
	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_connected")){
		display_notification((AsuiBatteryPluginPrivate *)user_data, charging_notification);
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gboolean asui_battery_req_battery_status( gpointer data ){
	if(asui_battery_initialized){
		// send request every 2 seconds until initialized
		asui_battery_dbus_send_method((AsuiBatteryPluginPrivate *)data, "com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "req_battery_status");
		return TRUE;
	}
	asui_battery_init_timer = 0;
	return FALSE;
}

//////////////////////////////////////////////////////////////////////////

static void asui_battery_plugin_finalize( GObject *object ){
	AsuiBatteryPluginPrivate *p = ASUI_BATTERY_PLUGIN_GET_PRIVATE(object);

	// remove timers
	if(asui_battery_init_timer) g_source_remove(asui_battery_init_timer);
	if(asui_battery_charging_timer) g_source_remove(asui_battery_charging_timer);
	// remove dbus signal filter
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_bus_remove_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_remove_filter(signal_bus, asui_battery_signal_filter, (void *)p);
	dbus_g_connection_unref(p->connection);
	// remove osso context
	osso_deinitialize(p->osso);

	LOG_CLOSE();

	G_OBJECT_CLASS(g_type_class_peek_parent(G_OBJECT_GET_CLASS(object)))->finalize(object);
}

static void asui_battery_plugin_init( AsuiBatteryPlugin *plugin ){
	AsuiBatteryPluginPrivate *p = ASUI_BATTERY_PLUGIN_GET_PRIVATE(plugin);

	ULOG_OPEN("asui-statusbar-battery");

	g_return_if_fail(p);

	// initialize dbus
	GError *error = NULL;
	if((p->connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error)) == NULL){
		ULOG_WARN("%s: can't connect to system method bus: %s\n", __FUNCTION__, error->message);
		g_error_free(error);
		return;
	}
	DBusConnection *signal_bus = dbus_g_connection_get_connection(p->connection);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/qzx/asui',interface='com.qzx.asui'", NULL);
	dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", NULL);
	dbus_connection_flush(signal_bus);
	dbus_connection_add_filter(signal_bus, asui_battery_signal_filter, (void *)p, NULL);

	p->icon = gtk_image_new_from_pixbuf(NULL);
	asui_battery_set_icon("asui_power", p);
	p->button = gtk_toggle_button_new();
	gtk_container_add(GTK_CONTAINER(p->button), GTK_WIDGET(p->icon));
	gtk_container_add(GTK_CONTAINER(plugin), p->button);
	g_signal_connect(G_OBJECT(p->button), "button-press-event", G_CALLBACK(asui_battery_open_asui), p);

	// osso context
	p->osso = osso_initialize("com.qzx.asui_statusbar_battery", ASUI_VERSION, TRUE, NULL);
	if(!p->osso) ULOG_WARN("%s: error while initializing osso\n", __FUNCTION__);

	gtk_widget_show_all(GTK_WIDGET(plugin));

	asui_battery_status = BATTERY_DRAINING;
	asui_battery_capacity = 0;
	asui_battery_initialized = 3;
	asui_battery_charging_timer = 0;
	asui_battery_init_timer = g_timeout_add(2000, asui_battery_req_battery_status, (void *)p);
	// 2 second wait before sending out first request
}
