//	Advanced System UI
//	Copyright (c) 2011 Brand Huntsman <http://qzx.com/mail/>

// seconds between clock updates (double)
#define REFRESH_PERIOD 10.0

// seconds between triple failures (double)
#define TRIPLE_FAILURE_TIMEOUT 30.0

// seconds to clear code when idle (double)
#define CODE_CLEAR_TIMEOUT 60.0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "hardware.h"
#include "dialog.h"
#include "dsme.h"
 #include "secure.h"

// open flashlight on fullscreen button
#include "flashlight.h"
// draw function for battery status
#include "widget_battery.h"
// draw function for clock
#include "widget_clock.h"

//////////////////////////////////////////////////////////////////////////

static struct {
	int x, y, w, h;
	char *label;
	void (*callback)();
} buttons[13];
static unsigned nr_digits;
static char digits[11];
static unsigned nr_failures;
static double last_change_time, last_failure_time;

static void draw_button( unsigned button, int x, int y, int w, int h, unsigned size, char *label, XColor *color, void (*callback)() ){
	x_set_color(line_color);
	x_draw_rectangle(x+16,y+32, w,h);
	draw_text(label, size, color, x+16+(w>>1),y+32+(h>>1), ALIGN_CENTER, ALIGN_MIDDLE, 0);
	buttons[button].x = x+16;
	buttons[button].y = y+32;
	buttons[button].w = w;
	buttons[button].h = h;
	buttons[button].label = (callback == NULL ? label : NULL);
	buttons[button].callback = callback;
}

static void clear_code( ){
	nr_digits = 0;
	int d; for(d = 0; d < 11; d++) digits[d] = '\0';
}

static void backspace( ){
	if(nr_digits){
		nr_digits--;
		digits[nr_digits] = '\0';
		last_change_time = get_utime();
	}
}

static void insert_digit( unsigned button ){
	highlight_selected_area(buttons[button].x, buttons[button].y, buttons[button].w+1, buttons[button].h+1);
	digits[nr_digits] = buttons[button].label[0];
	nr_digits++;
	last_change_time = get_utime();
}

static void unlock( ){
	if(device_lock_code)
		if(nr_digits == strlen(device_lock_code))
			if(!strcmp(digits, device_lock_code)){
				mce_unlock_device();
				set_string(&device_lock_code, NULL, 0);
				clear_code();
				return;
			}
	debug_write("secure keypad unlock failed, reposition and reset keypad");
	last_failure_time = get_utime();
	nr_failures++;
	clear_code();
	draw_secure_keypad(1); // randomize position on failure
}

static void draw_keypad( unsigned force, int x, int y ){
	if(force){
		// draw separator line
		x_set_color(line_color);
		x_draw_line(0,99, window.width,99);
	}

	double current_time = get_utime();
	if(nr_failures >= 3 && current_time - last_failure_time < TRIPLE_FAILURE_TIMEOUT){
		// clear entire keypad area
		x_set_color(bg_color);
		x_fill_rectangle(0,100, window.width, window.height-100);

		char buffer[100];
		snprintf(buffer, sizeof(buffer), "Disabled for %.0f+ seconds", TRIPLE_FAILURE_TIMEOUT - (current_time - last_failure_time));
		draw_text(buffer, 36, red_status_color, window.width>>1, 100+((window.height-100)>>1), ALIGN_CENTER, ALIGN_BOTTOM, 0);
	} else {
		if(nr_failures >= 3){
			nr_failures = 0; // reset after timeout

			// clear entire keypad area
			x_set_color(bg_color);
			x_fill_rectangle(0,100, window.width, window.height-100);
		} else if(!force){
			// clear keypad area
			x_set_color(bg_color);
			x_fill_rectangle(x+16,y, 448, 380);
		}

		// clear code after idle delay
		if(nr_digits && current_time - last_change_time > CODE_CLEAR_TIMEOUT)
			clear_code();

		// digit bar
		unsigned d;
		x_set_color((device_lock_code == NULL ? red_status_color : green_status_color));
		for(d = 0; d < 10; d++){
			if(d == nr_digits && device_lock_code) x_set_color(gray_status_color);
			if(d >= nr_digits || device_lock_code == NULL)
				x_fill_ellipse(x+42+d*44-5,y+16-5, 10,10); // small gray or red dots
			else
				x_fill_ellipse(x+42+d*44-9,y+16-9, 18,18); // large green dots
		}
		// buttons
		XColor *color = (device_lock_code == NULL || nr_digits == 10 ? line_color : white_color);
		draw_button( 6, x+  0, y+0*87, 100, 71, 60, "7",		color, NULL);
		draw_button( 3, x+  0, y+1*87, 100, 71, 60, "4",		color, NULL);
		draw_button( 0, x+  0, y+2*87, 100, 71, 60, "1",		color, NULL);
		draw_button( 9, x+  0, y+3*87, 216, 71, 60, "0",		color, NULL);
		draw_button( 7, x+116, y+0*87, 100, 71, 60, "8",		color, NULL);
		draw_button( 4, x+116, y+1*87, 100, 71, 60, "5",		color, NULL);
		draw_button( 1, x+116, y+2*87, 100, 71, 60, "2",		color, NULL);
		draw_button( 8, x+232, y+0*87, 100, 71, 60, "9",		color, NULL);
		draw_button( 5, x+232, y+1*87, 100, 71, 60, "6",		color, NULL);
		draw_button( 2, x+232, y+2*87, 100, 71, 60, "3",		color, NULL);
		draw_button(10, x+232, y+3*87, 216, 71, 30, "UNLOCK",	(nr_digits ? green_status_color : line_color), unlock);
		draw_button(11, x+348, y+0*87, 100, 71, 30, "CLR",		(nr_digits ? secondary_selection_color : line_color), clear_code);
		draw_button(12, x+348, y+1*87, 100,158, 50, "<",		(nr_digits ? secondary_selection_color : line_color), backspace);
		// the indices of last three buttons are used below, DON'T CHANGE THEM!!!
	}
}

static void draw_brightness_toggle( unsigned force ){
	if(force){
		x_set_color(bg_color);
		x_fill_rectangle(200,0, 75, 99);
	}

	x_set_color(red_status_color);
	unsigned size = (unsigned)(19.0 + (30.0 * ((float)hw_screen.brightness_level / (float)hw_screen.brightness_max)));
	unsigned x = 238 - (size>>1);
	unsigned y = 50 - (size>>1);
	x_fill_ellipse(x,y, size, size);
	draw_masked_image(&im_brightness, 229+1,41+1);
}

static void click_secure_keypad( ){
	if(nr_failures >= 3) return; // disabled during triple failure timeout

	if(click.y < 100){
		if(click.type != CLICK_SHORT) return;

		if(click.x > (window.width > window.height ? 640 : 320)){
			// clock widget
			highlight_selected_area((window.width > window.height ? 640 : 320), 0, 159, 99);
			dialog_close();
			clockFullscreen();
		} else if(click.x < 200){
			// battery widget, manually change theme
			highlight_selected_area(0, 0, 199, 99);
			// if ALS or black then switch to white
			// if white then switch to black
			cfg_theme = (cfg_theme < 2 ? 2 : 1);
			conf_set_int(conf_theme, cfg_theme);
			window_set_theme(cfg_theme - 1);
			draw_secure_keypad(0);
		} else if(click.x < 275){
			// brightness toggle
			highlight_selected_area(200, 0, 75, 99);
			if(hw_screen.brightness_level >= (unsigned)(cfg_brightness_enhanced ? hw_screen.brightness_max*0.9 : 5)){ // 90% or 5
				// at high, change to low
				hw_set_brightness((cfg_brightness_enhanced ? hw_screen.brightness_max*0.1 : 1), 1); // 10% or 1
			} else if(hw_screen.brightness_level >= (unsigned)(cfg_brightness_enhanced ? hw_screen.brightness_max*0.5 : 3)){ // 50% or 3
				// at medium, change to high
				hw_set_brightness(hw_screen.brightness_max, 1); // 100%
			} else {
				// at low, change to medium
				hw_set_brightness((cfg_brightness_enhanced ? hw_screen.brightness_max*0.5 : 3), 1); // 50% or 3
			}
			draw_brightness_toggle(1);
		}
	} else {
		if(click.type != CLICK_SHORT) return;

		unsigned b;
		for(b = 0; b < 13; b++)
			if(click.x >= buttons[b].x && click.x <= buttons[b].x+buttons[b].w && click.y >= buttons[b].y && click.y <= buttons[b].y+buttons[b].h){
				if(buttons[b].callback){
					if(nr_digits){
						highlight_selected_area(buttons[b].x, buttons[b].y, buttons[b].w+1, buttons[b].h+1);
						buttons[b].callback();
						draw_secure_keypad(0);
					}
				} else if(nr_digits < 10){
					insert_digit(b);
					draw_secure_keypad(0);
				}
			}
	}
}

static void keypress_secure_keypad( ){
	if(nr_failures >= 3) return; // disabled during triple failure timeout

	if(keypress.type != KEY_SHORT) return;

	if(keypress.keycode >= 24 && keypress.keycode <= 33){ // 1234567890 (normal, shifted or FN)
		if(nr_digits < 10){
			insert_digit(keypress.keycode-24);
			draw_secure_keypad(0);
		}
	} else switch(keypress.keycode){
	case KEYCODE_ESCAPE:
		// clear all digits
		if(nr_digits){
			highlight_selected_area(buttons[11].x, buttons[11].y, buttons[11].w+1, buttons[11].h+1);
			clear_code();
			draw_secure_keypad(0);
		}
		break;
	case 22: // backspace
		// clear last digit
		if(nr_digits){
			highlight_selected_area(buttons[12].x, buttons[12].y, buttons[12].w+1, buttons[12].h+1);
			backspace();
			draw_secure_keypad(0);
		}
		break;
	case KEYCODE_ENTER:
	case KEYCODE_DPAD_CENTER:
		// unlock
		if(nr_digits){
			highlight_selected_area(buttons[10].x, buttons[10].y, buttons[10].w+1, buttons[10].h+1);
			unlock();
			draw_secure_keypad(0);
		}
		break;
	case KEYCODE_HOME:
		// open fullscreen clock
		highlight_selected_area((window.width > window.height ? 640 : 320), 0, 159, 99);
		dialog_close();
		clockFullscreen();
		break;
	case KEYCODE_FULLSCREEN:
		// open flashlight
		flashlight_open();
		break;
	default:
		// ignore all other keys
		break;
	}
}

//////////////////////////////////////////////////////////////////////////

static int random_offset;

void draw_secure_keypad( unsigned force ){
	if(force || hw_changes & HW_THEME){
		// dialog background
		x_set_color(bg_color);
		x_fill_rectangle(0, 0, window.width, window.height);
		// randomize position
		if(force) random_offset = (int)(320.0 * (double)rand()/(double)RAND_MAX);
		else force = 1;
	}

	hw_query_temperature();
	hw_query_brightness_with_delay();

	if(window.width > window.height){
		// LANDSCAPE
		draw_battery_widget(force, 0, 0);
		draw_brightness_toggle(force);
		draw_clock_widget(force, 640, 0); // only 160 wide

		draw_keypad(force, random_offset,100);
	} else {
		// PORTRAIT
		draw_battery_widget(force, 0, 0);
		draw_brightness_toggle(force);
		draw_clock_widget(force, 320, 0); // only 160 wide

		draw_keypad(force, 0,100+random_offset);
	}

	hw_changes = 0;
	x_flush();
}

//////////////////////////////////////////////////////////////////////////

static unsigned got_code;

void secureUpdate( ){
	if(got_code){
		// redraw
		debug_write("secureUpdate: redraw keypad, timer expired");
		draw_secure_keypad(0);

// TODO: digits should be cleared if the user goes idle
//		store time when each key is pressed
//		if current time > key_time + 30 seconds then clear_code()

		// reset timer
		reset_timer(&dialog_refresh_timer);
	} else if(device_lock_code == NULL){
		// code isn't ready, try again
		debug_write("secureUpdate: code not ready, loop");
		dsme_query_devicelock_code();
		set_timer(&dialog_refresh_timer, 0.5); // half second interval
	} else {
		debug_write("secureUpdate: initialize");
		got_code = 1;
		// redraw keypad to ungray buttons
		draw_secure_keypad(0);
		// enable click/key handlers and set refresh timer for clock
		current_dialog_click = click_secure_keypad;
//		current_dialog_keypress = keypress_secure_keypad
		set_timer(&dialog_refresh_timer, REFRESH_PERIOD);
	}
}

void secureShowKeypad( ){
	if(current_dialog_draw != draw_secure_keypad){
		if(current_dialog_draw) dialog_reset();
		clear_code();
		got_code = 0;
		nr_failures = 0;
		dialog_open(draw_secure_keypad, NULL, keypress_secure_keypad, NULL, DLG_REDRAW_ON); // no clicks
		dsme_query_devicelock_code();
		set_timer(&dialog_refresh_timer, 0.5); // half second interval
	}
}

void secureHideKeypad( ){
	if(current_dialog_draw == draw_secure_keypad)
		dialog_close();
}
