//	Advanced System UI
//	Copyright (c) 2010-2011 Brand Huntsman <http://qzx.com/mail/>
//
//	device model detection code ported from advanced-backlight
//	(C) 2008 Jonas Hurrelmann <j@outpo.st>.
//	(C) 2008 Adam Harwell <aharwell@trinity.edu>.

// write debug messages to stdout
//#define DEBUG_TO_STDOUT

// milliseconds
#define DOUBLE_TAP_DURATION 500

// seconds (double) and pixel deviation
#define LONG_PRESS_DURATION 1.0
#define DRAG_DEVIATION 20

// seconds (double)
// must be less than 1 second and LONG_PRESS_DURATION must be evenly divisible by BUTTON_REFRESH_PERIOD
#define BUTTON_CHECK_PERIOD 0.100

// seconds (double)
#define AUTOREPEAT_INITIAL_DELAY 0.500
// 25% decrease until same as BUTTON_CHECK_PERIOD
#define AUTOREPEAT_RATE 0.75

// seconds (double)
#define BRIGHTNESS_CHECK_PERIOD 0.100

// seconds (double)
#define MAIN_REFRESH_PERIOD 5.0

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>
#include <signal.h>

#include "window.h"
#include <X11/extensions/Xrandr.h>

#include "main.h"
#include "draw.h"
#include "text.h"
#include "config.h"
#include "services.h"
#include "hardware.h"
#include "dialog.h"
#include "dbus.h"
#include "dsme.h"
#include "secure.h"

// open flashlight on fullscreen button
#include "flashlight.h"
// draw function for minimal UI and battery refresh timer
#include "widget_battery.h"
// draw function for minimal UI
#include "widget_clock.h"

// for RBV keys and brightness mismatch timer
#include "widget_brightness.h"
// for RBV keys
#include "widget_volume.h"

////////////////////////////////////////////////////////////////////////// GLOBALS

// device model
unsigned is_770, is_800, is_810;

// use minimal interface when device is off but charging
unsigned use_minimal_ui;

// widget coordinates -- needed by click handlers
int widget_x, widget_y;

// type and location of clicks
s_click click;
s_keypress keypress;

// throttle taps that open dialogs -- throttle all UI taps
s_tap_throttler dialog_throttle, ui_throttle;

// current time, set each time main loop wakes
double current_time;

// button down callback
void (*button_callback)( );
unsigned button_autorepeat;

// drag motion event interval timer and callback
s_timer motion_timer;
unsigned (*motion_callback)( );

unsigned primary_widgets, widget_edit_mode;
s_selected_cell selected_cell;
unsigned brightness_widget_page, volume_widget_page, battery_widget_page;
unsigned flightmode_button_page, keyrepeat_button_page, screenstayslit_button_page, screenautolock_button_page;

// idle timer to autolock device
s_timer idle_timer;
unsigned is_idle;

////////////////////////////////////////////////////////////////////////// GET MICRO TIME

double get_utime( ){
	struct timeval tv;
	gettimeofday(&tv, NULL);
	return (double)tv.tv_sec + (double)tv.tv_usec / 1000000.0;
}

void set_timer( s_timer *timer, double duration ){
	if(duration < 0.0) duration = 0.0; // can't be negative
	timer->last_time = current_time;
	timer->remaining = timer->duration = duration;
}

void reset_timer( s_timer *timer ){
	timer->last_time = current_time;
	timer->remaining = timer->duration;
}

unsigned is_timer_ready( s_timer *timer ){
	if(timer->remaining == 0.0) return 1;

	timer->remaining -= (current_time - timer->last_time);
	timer->last_time = current_time;
	if(timer->remaining <= 0.0){ timer->remaining = 0.0; return 1; }

	return 0;
}

////////////////////////////////////////////////////////////////////////// AUTOREPEAT

static double ar_start_time, ar_delay;

static void button_repeat_init( ){
	ar_start_time = current_time;
	ar_delay = AUTOREPEAT_INITIAL_DELAY;
}
static unsigned is_button_repeat_ready( ){
	if(!button_autorepeat) return 1;
	if(current_time >= ar_start_time + ar_delay){
		ar_start_time += ar_delay;
		ar_delay *= AUTOREPEAT_RATE;
		if(ar_delay < BUTTON_CHECK_PERIOD) ar_delay = BUTTON_CHECK_PERIOD;
		return 1;
	}
	return 0; // not ready to repeat
}

////////////////////////////////////////////////////////////////////////// STRING ALLOCATION

void set_string( char **dst, char *src, unsigned limit ){
	char *tmp = *dst, *tmp_string;
	if(src != NULL){
		unsigned len = strlen(src);
		unsigned size = (limit && limit < len ? limit : len) + 1;
		if(*dst != NULL)
			if(!strcmp(*dst, src))
				return; // same, don't realloc
		tmp_string = (char *)malloc(size);
		debug_write("malloc in set_string()");
		if(tmp_string != NULL) snprintf(tmp_string, size, "%s", src);
		*dst = tmp_string;
	} else
		*dst = NULL;
	if(tmp) free(tmp);
	// tmp and tmp_string ensure *dst always contains NULL or a valid string
	// this is required since the main thread could attempt to display *dst while the dbus thread is modifying it here
}

////////////////////////////////////////////////////////////////////////// FILE EXISTS

unsigned file_exists( const char *file ){
	if(access(file, R_OK)) return 0;
	return 1;
}

////////////////////////////////////////////////////////////////////////// BUTTON CELLS

static void _draw_button( unsigned force, int x, int y, s_button *button ){
	if(button == NULL) return;
	button->draw(force, x, y);
}

static void _click_button( s_button *button, unsigned cell ){
	if(button == NULL) return;
	button->click();
}

////////////////////////////////////////////////////////////////////////// WIDGET CELLS

static void _draw_widget( unsigned force, int x, int y, s_widget_cell *widget_cell){
	if(widget_cell->size == FULL_WIDGET){
		if(widget_edit_mode){
			if(force){
				XColor *color;
				if(selected_cell.widget_cell == widget_cell){
					x_set_color(selection_color);
					x_fill_rectangle(x,y, 399,99);
					color = bg_color;
				} else color = fg1_color;
				draw_text((char *)widget_cell->left->name, 20, color, x+199, y+50, ALIGN_CENTER, ALIGN_MIDDLE, 0);
			}
		} else
			widget_cell->left->draw(force, x, y);
	} else if(widget_cell->size == HALF_WIDGET){
		if(widget_cell->left){
			if(widget_edit_mode){
				if(force){
					XColor *color;
					if(selected_cell.widget_cell == widget_cell && selected_cell.subcell == 0){
						x_set_color(selection_color);
						x_fill_rectangle(x,y, 199,99);
						color = bg_color;
					} else color = fg1_color;
					draw_text((char *)widget_cell->left->name, 20, color, x+99, y+50, ALIGN_CENTER, ALIGN_MIDDLE, 0);
				}
			} else
				widget_cell->left->draw(force, x, y);
		}
		if(force){
			x_set_color(line_color);
			x_draw_line(x+199, y, x+199, y+99);
		}
		if(widget_cell->right){
			if(widget_edit_mode){
				if(force){
					XColor *color;
					if(selected_cell.widget_cell == widget_cell && selected_cell.subcell == 1){
						x_set_color(selection_color);
						x_fill_rectangle(x+200,y, 199,99);
						color = bg_color;
					} else color = fg1_color;
					draw_text((char *)widget_cell->right->name, 20, color, x+299, y+50, ALIGN_CENTER, ALIGN_MIDDLE, 0);
				}
			} else
				widget_cell->right->draw(force, x+200, y);
		}
	}
}

static unsigned _move_widget( s_widget_cell *widget_cell, unsigned cell, unsigned subcell ){
	e_widget_size src_size = selected_cell.widget_cell->size, dst_size = widget_cell->size;

	if(src_size == FULL_WIDGET)
		highlight_selected_area(widget_x, widget_y, 399, 99);
	else
		highlight_selected_area(widget_x+(subcell ? 200 : 0), widget_y, 199, 99);


	if(src_size == FULL_WIDGET && (dst_size == FULL_WIDGET || dst_size == EMPTY_CELL)){
		// swap full widget with another full widget or an empty cell
//printf("full [%u] -> [%u]\n", selected_cell.cell, cell);
		s_widget *dst = widget_cell->left;
		widget_cell->left = selected_cell.widget_cell->left;
		selected_cell.widget_cell->left = dst;

		selected_cell.widget_cell->size = dst_size;
	} else if(src_size == HALF_WIDGET && (dst_size == HALF_WIDGET || dst_size == EMPTY_CELL)){
		// swap half widget with another half widget or an empty cell
//printf("half [%u:%u] -> [%u:%u]\n", selected_cell.cell, selected_cell.subcell, cell, subcell);
			s_widget *dst = (subcell ? widget_cell->right : widget_cell->left);
			s_widget *src = (selected_cell.subcell ? selected_cell.widget_cell->right : selected_cell.widget_cell->left);
			if(subcell) widget_cell->right = src; else widget_cell->left = src;
			if(selected_cell.subcell) selected_cell.widget_cell->right = dst; else selected_cell.widget_cell->left = dst;

			if(selected_cell.widget_cell->left == NULL && selected_cell.widget_cell->right == NULL)
				selected_cell.widget_cell->size = EMPTY_CELL;
	} else {
		// cells aren't the same size, swap entire cell contents
//printf("mixed [%u:%u] -> [%u:%u]\n", selected_cell.cell, selected_cell.subcell, cell, subcell);
		s_widget *dst_left = widget_cell->left;
		s_widget *dst_right = widget_cell->right;
		if(src_size == HALF_WIDGET && subcell != selected_cell.subcell){
			// swap ordering of half widgets
			widget_cell->left = selected_cell.widget_cell->right;
			widget_cell->right = selected_cell.widget_cell->left;
		} else {
			widget_cell->left = selected_cell.widget_cell->left;
			widget_cell->right = selected_cell.widget_cell->right;
		}
		selected_cell.widget_cell->left = dst_left;
		selected_cell.widget_cell->right = dst_right;

		selected_cell.widget_cell->size = dst_size;
	}
	widget_cell->size = src_size;

	// save configuration
	unsigned landscape = (window.width > window.height ? 1 : 0);
	if(widget_cell->size == FULL_WIDGET && widget_cell->left)
		config_save_widget(widget_cell->left, cell, 0, landscape);
	else if(widget_cell->size == HALF_WIDGET){
		if(widget_cell->left) config_save_widget(widget_cell->left, cell, 0, landscape);
		if(widget_cell->right) config_save_widget(widget_cell->right, cell, 1, landscape);
	}
	if(selected_cell.widget_cell != widget_cell){
		if(selected_cell.widget_cell->size == FULL_WIDGET && selected_cell.widget_cell->left)
			config_save_widget(selected_cell.widget_cell->left, selected_cell.cell, 0, landscape);
		else if(selected_cell.widget_cell->size == HALF_WIDGET){
			if(selected_cell.widget_cell->left) config_save_widget(selected_cell.widget_cell->left, selected_cell.cell, 0, landscape);
			if(selected_cell.widget_cell->right) config_save_widget(selected_cell.widget_cell->right, selected_cell.cell, 1, landscape);
		}
	}
//printf("\n");

	// clear selected cell
	selected_cell.widget_cell = NULL;
	draw_window(1);

// TODO: window isn't refreshing immediately and a usleep doesn't help

	return 1;
}

static void _click_widget( s_widget_cell *widget_cell, unsigned cell ){
	if(widget_edit_mode && selected_cell.widget_cell && click.type == CLICK_SHORT){

		unsigned subcell = (click.x >= 200 ? 1 : 0);
		if(widget_cell == selected_cell.widget_cell && (widget_cell->size == FULL_WIDGET || (widget_cell->size == HALF_WIDGET && selected_cell.subcell == subcell))){
			if(widget_cell->size == FULL_WIDGET)
				highlight_long_press(widget_x, widget_y, 399,99);
			else
				highlight_long_press(widget_x+(subcell ? 200 : 0), widget_y, 199,99);
			selected_cell.widget_cell = NULL;
			reset_long_press(); // highlight removed when window is redrawn
			draw_window(1);
		} else
			_move_widget(widget_cell, cell, subcell);

	} else if(widget_cell->size == FULL_WIDGET){

		if(widget_edit_mode){
			if(click.type == CLICK_SHORT){
				selected_cell.widget_cell = widget_cell;
				selected_cell.cell = cell;
				selected_cell.subcell = 0;
				_draw_widget(1, widget_x, widget_y, widget_cell);
			}
		} else
			widget_cell->left->click();

	} else if(widget_cell->size == HALF_WIDGET){

		if(click.x >= 200){
			if(widget_cell->right){
				if(widget_edit_mode){
					if(click.type == CLICK_SHORT){
						selected_cell.widget_cell = widget_cell;
						selected_cell.cell = cell;
						selected_cell.subcell = 1;
						_draw_widget(1, widget_x, widget_y, widget_cell);
					}
				} else {
					widget_x += 200;
					click.x -= 200;
					widget_cell->right->click();
				}
			}
		} else if(widget_cell->left){
			if(widget_edit_mode){
				if(click.type == CLICK_SHORT){
					selected_cell.widget_cell = widget_cell;
					selected_cell.cell = cell;
					selected_cell.subcell = 0;
					_draw_widget(1, widget_x, widget_y, widget_cell);
				}
			} else
				widget_cell->left->click();
		}

	}
}

////////////////////////////////////////////////////////////////////////// DEBUGGING

#ifdef DEBUG
static void draw_self_rss( unsigned force, int x, int y, unsigned landscape ){
	if(force || svc_changes & SVC_SELF){
		char buffer[20];
		x_set_color(bg_color);
		snprintf(buffer, sizeof(buffer), "%u", svc_self_size);
		if(landscape){
			x_fill_rectangle(x-48,y, 96,20);
			draw_text(buffer, 16, fg1_color, x, y, ALIGN_CENTER, ALIGN_TOP, 0);
		} else {
			x_fill_rectangle(x-96,y-10, 96,20);
			draw_text(buffer, 16, fg1_color, x, y, ALIGN_RIGHT, ALIGN_MIDDLE, 0);
		}
	}
}
#else
 #define draw_self_rss(f,x,y,o) {}/* do nothing */
#endif

#ifdef DEBUG
 #include <stdarg.h>
static FILE *debug_file;
static int debug_runlevel;
static double debug_start_time;
void debug_write( const char *format, ... ){
	char buffer[2048];
	va_list ap;

	va_start(ap, format);
	vsnprintf(buffer, sizeof(buffer), format, ap);
	va_end(ap);
	if(debug_file){
		fprintf(debug_file, "[%d] [%.6f]  %s\n", debug_runlevel, current_time-debug_start_time, buffer);
		fflush(debug_file);
	}
	#ifdef DEBUG_TO_STDOUT
	printf("[%d] [%.6f]  %s\n", debug_runlevel, current_time-debug_start_time, buffer);
	#endif
}
#endif

#ifdef TESTLOG
 #include <stdarg.h>
static FILE *test_file;
static double test_start_time;
void test_write( const char *format, ... ){
	char buffer[2048];
	va_list ap;

	va_start(ap, format);
	vsnprintf(buffer, sizeof(buffer), format, ap);
	va_end(ap);
	if(test_file){
		fprintf(test_file, "[%.6f]  %s\n", current_time-test_start_time, buffer);
		fflush(test_file);
	}
}
#endif

#ifdef TEST
void error_write( const char *format, ... ){
	char buffer[2048];
	va_list ap;

	va_start(ap, format);
	vsnprintf(buffer, sizeof(buffer), format, ap);
	va_end(ap);
	fprintf(stderr, BINNAME ": %s\n", buffer);
	#ifdef DEBUG
	debug_write("ERROR: %s", buffer);
	#endif
}
#endif

////////////////////////////////////////////////////////////////////////// DRAW WINDOW

/*
static void draw_button_guide_lines( unsigned rows ){
	if(rows == 2){
		x_set_color(line_color);
		x_draw_line(0,28, window.width,28);
		x_draw_line(0,52, window.width,52);
		x_set_color(selection_color);
		x_set_style(1, LineOnOffDash);
		x_draw_line(0,33, window.width,33);
		x_draw_line(0,57, window.width,57);
		x_reset_style();
	} else if(rows == 3){
		x_set_color(line_color);
		x_draw_line(0,20, window.width,20);
		x_draw_line(0,40, window.width,40);
		x_draw_line(0,60, window.width,60);
		x_set_color(selection_color);
		x_set_style(1, LineOnOffDash);
		x_draw_line(0,25, window.width,25);
		x_draw_line(0,45, window.width,45);
		x_draw_line(0,65, window.width,65);
		x_reset_style();
	}
	//				top			middle		bottom
	//
	//	1 row					+6
	//
	//	2 rows		-6						+18
	//
	//	3 rows		-14			+6			+26
	//
}
*/

static void _draw_selection_button( int x, int y ){
	if(selected_cell.widget_cell && ((primary_widgets && selected_cell.cell > 7) || (!primary_widgets && selected_cell.cell < 8))){
		s_widget *widget = (selected_cell.subcell == 0 ? selected_cell.widget_cell->left : selected_cell.widget_cell->right);
		x_set_color(selection_color);
		x_fill_rectangle(x,y, 80,80);
		draw_text((char *)widget->name, 12, bg_color, x+40,y+40, ALIGN_CENTER, ALIGN_MIDDLE, 76);
		draw_text((widget->size == FULL_WIDGET ? "FULL" : "HALF"), 8, fg1_color, x+75,y+5, ALIGN_RIGHT, ALIGN_TOP, 0);
	}
}

static void _draw_page_button( int x, int y ){
	draw_text((primary_widgets ? "1" : "2"), 20, selection_color, x, y+40+6, ALIGN_CENTER, ALIGN_BASELINE, 0);
}

static void _draw_exit_button( int x, int y ){
	draw_text("exit", 16, fg2_color, x+40, y+40+6, ALIGN_CENTER, ALIGN_BASELINE, 0);
}

static void draw_editmode_buttons( unsigned landscape ){
	if(landscape){ // LANDSCAPE
		_draw_selection_button(0,0);	//  1
		_draw_page_button(400,0);		// 5.5
		_draw_exit_button(720,0);		// 10
	} else { // PORTRAIT
		_draw_selection_button(400,0);	//  1
		_draw_page_button(400,400);		// 5.5
		_draw_exit_button(400,720);		// 10
	}
}

void draw_window( unsigned force ){
	if(!window.is_mapped) return;

	if(hw_changes & HW_THEME) force = 1; // color theme changed

	if(current_dialog_draw){

		// draw dialog
		current_dialog_draw(force);

	} else if(use_minimal_ui){

		if(force){
			// window background
			x_set_color(bg_color);
			x_fill_rectangle(0, 0, window.width, window.height);

			// power button message
			draw_text("press and hold power button to startup", 14, fg2_color, window.width>>1, 0, ALIGN_CENTER, ALIGN_TOP, 0);
		}

		if(window.width > window.height){
			// LANDSCAPE
			draw_battery_widget(force, 200, 90);
			draw_clock_widget(force, 200, 290);
			draw_self_rss(force, window.width>>1, 30, 1);
		} else {
			// PORTRAIT (shutting down to charging runlevel while rotated)
			draw_battery_widget(force, 40, 200);
			draw_clock_widget(force, 40, 500);
			draw_self_rss(force, window.width>>1, 30, 0);
		}

	} else {

		if(force){
			// window background
			x_set_color(bg_color);
			x_fill_rectangle(0, 0, window.width, window.height);
		}

		// widgets and buttons
		if(window.width > window.height){
			// LANDSCAPE

			// 10 horizontal buttons -- 80x80

			// 8 widgets in 2 columns -- 399x99
			// each widget has a line above it that is clickable but not drawable
			//		clicks start on the line but drawing starts below it
			// each widget has a line to the right that is clickable but not drawable (invisible for widgets in right column)

			// half widgets are 199x99 with a clickable line on the right side of each (invisible for widgets in right subcell in right column)

			if(force){
				// separators
				x_set_color(line_color);
				x_draw_line(399, 81, 399, 480);
				int y;
				for(y = 80; y < 479; y += 100)
					x_draw_line(0, y, 800, y);
			}

			if(primary_widgets){
				// PRIMARY WIDGETS (LANDSCAPE)

				// buttons
				if(widget_edit_mode){
					if(force) draw_editmode_buttons(1);
				} else {
					int x, i;
					for(x = 0, i = 0; i < 10; x += 80, i++)
						_draw_button(force, x, 0, landscape_bcells[i]);

/*
// button guide lines
x_set_color(line_color);
x_draw_line(0,40, window.width,40);
x_set_color(selection_color);
x_set_style(1, LineOnOffDash);
x_draw_line(0,27, window.width,27);
x_draw_line(0,59, window.width,59);
x_reset_style();
*/

					draw_self_rss(force, 400, 0, 1);
				}

				// widgets
				_draw_widget(force,   0,  81, &landscape_cells[0]);
				_draw_widget(force,   0, 181, &landscape_cells[1]);
				_draw_widget(force,   0, 281, &landscape_cells[2]);
				_draw_widget(force,   0, 381, &landscape_cells[3]);
				_draw_widget(force, 400,  81, &landscape_cells[4]);
				_draw_widget(force, 400, 181, &landscape_cells[5]);
				_draw_widget(force, 400, 281, &landscape_cells[6]);
				_draw_widget(force, 400, 381, &landscape_cells[7]);
			} else {
				// SECONDARY WIDGETS (LANDSCAPE)

				// buttons
				if(widget_edit_mode){
					if(force) draw_editmode_buttons(1);
				} else {
					int x, i;
					for(x = 0, i = 10; i < 20; x += 80, i++)
						_draw_button(force, x, 0, landscape_bcells[i]);
				}

				// widgets
				_draw_widget(force,   0,  81, &landscape_cells[8]);
				_draw_widget(force,   0, 181, &landscape_cells[9]);
				_draw_widget(force,   0, 281, &landscape_cells[10]);
				_draw_widget(force,   0, 381, &landscape_cells[11]);
				_draw_widget(force, 400,  81, &landscape_cells[12]);
				_draw_widget(force, 400, 181, &landscape_cells[13]);
				_draw_widget(force, 400, 281, &landscape_cells[14]);
				_draw_widget(force, 400, 381, &landscape_cells[15]);
			}

		} else {
			// PORTRAIT

			// 10 horizontal buttons -- 80x80

			// 8 widgets in 1 column -- 399x99
			// each widget has a line above it that is clickable but not drawable (invisible for top widget)
			//		clicks start on the line but drawing starts below it
			// each widget has a line to the right that is clickable but not drawable

			// half widgets are 199x99 with a clickable line on the right side of each

			if(force){
				// separators
				x_set_color(line_color);
				x_draw_line(399, 0, 399, 800);
				int y;
				for(y = 100; y < 799; y += 100)
					x_draw_line(0, y, 399, y);
			}

			if(primary_widgets){
				// PRIMARY WIDGETS (PORTRAIT)
				int y, i;

				// buttons
				if(widget_edit_mode){
					if(force) draw_editmode_buttons(0);
				} else {
					for(y = 0, i = 0; i < 10; y += 80, i++)
						_draw_button(force, 400, y, portrait_bcells[i]);

					draw_self_rss(force, 479, 400, 0);
				}

				// widgets
				for(y = 1, i = 0; i < 8; y += 100, i++)
					_draw_widget(force, 0, y, &portrait_cells[i]);
			} else {
				// SECONDARY WIDGETS (PORTRAIT)
				int y, i;

				// buttons
				if(widget_edit_mode){
					if(force) draw_editmode_buttons(0);
				} else {
					for(y = 0, i = 10; i < 20; y += 80, i++)
						_draw_button(force, 400, y, portrait_bcells[i]);
				}

				// widgets
				for(y = 1, i = 8; i < 16; y += 100, i++)
					_draw_widget(force, 0, y, &portrait_cells[i]);
			}
		}

	}

	// clear changes
	hw_changes = svc_changes = 0;

	x_flush();
}

////////////////////////////////////////////////////////////////////////// CLICKS

static void click_button( unsigned button ){
	if(widget_edit_mode){
		switch(button){
		case 0: // selected widget button

			if(click.type != CLICK_SHORT || selected_cell.widget_cell == NULL || !widget_edit_mode) return;

			highlight_long_press(widget_x, widget_y, 80,80);
			selected_cell.widget_cell = NULL;
			reset_long_press();

			// redraw window
			draw_window(1);
			break;

		case 9: // exit widget edit mode

			if(click.type != CLICK_SHORT || !widget_edit_mode) return;

			highlight_selected_area(widget_x, widget_y, 80,80);
			widget_edit_mode = 0;
			selected_cell.widget_cell = NULL;

			// redraw window
			draw_window(1);
			break;

		}
	} else {
		button += (primary_widgets ? 0 : 10);
		if(window.width > window.height)
			_click_button(landscape_bcells[button], button); // LANDSCAPE
		else
			_click_button(portrait_bcells[button], button); // PORTRAIT
	}
}

static void click_widget( unsigned widget ){
	widget += (primary_widgets ? 0 : 8);
	if(window.width > window.height)
		_click_widget(&landscape_cells[widget], widget); // LANDSCAPE
	else
		_click_widget(&portrait_cells[widget], widget); // PORTRAIT
}

////////////////////////////////////////////////////////////////////////// SCREEN PRESS

static struct {
	e_double_tap_zone zone;
	unsigned timestamp;
} double_tap_data;

int get_double_tap( e_double_tap_zone zone ){
	if(!cfg_double_tap) return 1;
	if(zone == double_tap_data.zone){
		if(double_tap_data.timestamp + DOUBLE_TAP_DURATION > click.timestamp)
			return 1; // double tap
		// not quick enough, reset for another attempt
	} // else wrong zone, prepare for double tap in this zone
	double_tap_data.zone = zone;
	double_tap_data.timestamp = click.timestamp;
	return 0;
}

int throttle_taps( s_tap_throttler *zone ){
	if(zone->timestamp){
		int difference = zone->timestamp - click.timestamp;
		if(difference < 0) difference = 0;
		if(difference > 0 && difference < 60000) return 1; // ignore tap throttling if greater than 60 seconds
		//if(click.timestamp < zone->timestamp) return 1;
	}
	zone->utime = current_time;
	zone->timestamp = click.timestamp;
	return 0;
}

static void highlight_button_bar( ){
	x_set_color(selection_color);
	if(window.width > window.height)
		x_draw_rectangle(0, 0, window.width-1, 79);
	else
		x_draw_rectangle(400, 0, 79, window.height-1);
	x_flush();
	usleep(100000); // display highlight for 100ms
	// this usleep also fixes a problem where X wouldn't update the window contents for long periods of time
}

static int screen_event( ){
	if(current_dialog_draw){

		if(current_dialog_click && click.timestamp > dialog_throttle.timestamp + DIALOG_THROTTLE_DURATION) current_dialog_click();
		// dialog_throttle shoudln't be updated, this only throttles clicks while opening the dialog

	} else if(click.timestamp > ui_throttle.timestamp){ // throttle clicks for entire UI

		if(window.width > window.height){
			// LANDSCAPE
			if(click.start_y < 80){
				// buttons
				if(click.type == CLICK_DRAG && click.start_y + click.offset_y < 80 && (click.offset_x >= 200 || click.offset_x <= -200)){
					// change page with a horizontal swipe along button bar (25% length)
					highlight_button_bar();
					primary_widgets = (primary_widgets ? 0 : 1);
					hw_refresh();
					draw_window(1);

					// throttle taps when switching pages
					throttle_taps(&ui_throttle);
					finalize_tap(ui_throttle);
				} else if(click.type == CLICK_DRAG && click.offset_x >= -100 && click.offset_x <= 100 && click.start_y + click.offset_y > 90){
					// hide the window with a vertical swipe from button bar to widgets (10 pixels into widgets, +-100 pixel horizontal deviation)
					window_unmap();
				} else {
					widget_x = (click.start_x / 80)*80;
					widget_y = 0;
					click_button(click.start_x / 80);
				}
			} else if(click.start_x < 400){
				// left widgets
				uint widget = (click.start_y-80) / 100;
				widget_x = 0;
				widget_y = widget*100+81;
				click.x = click.start_x;
				click.y = click.start_y - (80 + widget*100);
				click_widget(widget);
			} else {
				// right widgets
				uint widget = (click.start_y-80) / 100;
				widget_x = 400;
				widget_y = widget*100+81;
				click.x = click.start_x - 400;
				click.y = click.start_y - (80 + widget*100);
				click_widget(widget+4);
			}
		} else {
			// PORTRAIT
			if(click.start_x >= 400){
				// buttons
				if(click.type == CLICK_DRAG && click.start_x + click.offset_x >= 400 && click.start_y + click.offset_y >= 200){
					// change page with a vertical swipe along button bar (25% length)
					highlight_button_bar();
					primary_widgets = (primary_widgets ? 0 : 1);
					hw_refresh();
					draw_window(1);

					// throttle taps when switching pages
					throttle_taps(&ui_throttle);
					finalize_tap(ui_throttle);
				} else if(click.type == CLICK_DRAG && click.offset_y >= -100 && click.offset_y <= 100 && click.start_x + click.offset_x < 390){
					// hide the window with a horizontal swipe from button bar to widgets (10 pixels into widgets, +-100 pixel vertical deviation)
					window_unmap();
				} else {
					widget_x = 400;
					widget_y = (click.start_y / 80)*80;
					click_button(click.start_y / 80);
				}
			} else {
				// widgets
				uint widget = (click.start_y / 100);
				widget_x = 0;
				widget_y = widget*100+1;
				click.x = click.start_x;
				click.y = click.start_y - widget*100;
				click_widget(widget);
			}
		}

	}

	return 0;
}

////////////////////////////////////////////////////////////////////////// ROTATE, BRIGHTNESS, VOLUME BUTTONS

static int button_up( unsigned button_group ){
	if(button_group == cfg_buttons_rotate){
		if(widget_edit_mode) return 0; // rotation can't be changed in widget edit mode
		if(keypress.type != KEY_SHORT) return 1;

		if(window.supports_rotation){
			switch(window.orientation){
			case O_LANDSCAPE: rotate_screen(O_PORTRAIT, 1); break;
			case O_PORTRAIT: rotate_screen(O_LANDSCAPE_INV, 1); break;
			case O_LANDSCAPE_INV: rotate_screen(O_PORTRAIT_INV, 1); break;
			case O_PORTRAIT_INV: rotate_screen(O_LANDSCAPE, 1); break;
			}
		}
		return 1;
	} else if(button_group == cfg_buttons_brightness){
		if(current_dialog_draw != NULL || widget_edit_mode || brightness_widget_page != primary_widgets)
			return 0; // brightness can only be changed when slider is visible
		if(keypress.type != KEY_SHORT) return 1;

		if(hw_screen.brightness_level < hw_screen.brightness_max){
			unsigned brightness = hw_screen.brightness_level + cfg_buttons_brightness_step;
			set_brightness(brightness > hw_screen.brightness_max ? hw_screen.brightness_max : brightness);
		}
		return 1;
	} else if(button_group == cfg_buttons_volume){
		if(current_dialog_draw != NULL || widget_edit_mode || volume_widget_page != primary_widgets)
			return 0; // volume can only be changed when slider is visible
		if(keypress.type != KEY_SHORT) return 1;

		if(hw_volume < 0) hw_volume = -hw_volume; // unmute

		if(hw_volume < 100){
			unsigned volume = hw_volume + cfg_buttons_volume_step;
			set_volume(volume > 100 ? 100 : volume);
		}
		return 1;
	}
	return 0;
}

static int button_down( unsigned button_group ){
	if(button_group == cfg_buttons_rotate){
		if(widget_edit_mode) return 0; // rotation can't be changed in widget edit mode
		if(keypress.type != KEY_SHORT) return 1;

		if(window.supports_rotation){
			switch(window.orientation){
			case O_LANDSCAPE: rotate_screen(O_PORTRAIT_INV, 1); break;
			case O_PORTRAIT: rotate_screen(O_LANDSCAPE, 1); break;
			case O_LANDSCAPE_INV: rotate_screen(O_PORTRAIT, 1); break;
			case O_PORTRAIT_INV: rotate_screen(O_LANDSCAPE_INV, 1); break;
			}
		}
		return 1;
	} else if(button_group == cfg_buttons_brightness){
		if(current_dialog_draw != NULL || widget_edit_mode || brightness_widget_page != primary_widgets)
			return 0; // brightness can only be changed when slider is visible
		if(keypress.type != KEY_SHORT) return 1;

		if((cfg_brightness_allow_zero && hw_screen.brightness_level > 0) || (hw_screen.brightness_level > 1)){
			int brightness = hw_screen.brightness_level - cfg_buttons_brightness_step;
			if(brightness < 0) brightness = 0;
			set_brightness(brightness);
		}
		return 1;
	} else if(button_group == cfg_buttons_volume){
		if(current_dialog_draw != NULL || widget_edit_mode || brightness_widget_page != primary_widgets)
			return 0; // volume can only be changed when slider is visible
		if(keypress.type != KEY_SHORT) return 1;

		if(hw_volume < 0) hw_volume = -hw_volume; // unmute

		if(hw_volume > 0){
			int volume = hw_volume - cfg_buttons_volume_step;
			set_volume(volume < 0 ? 0 : (unsigned)volume);
		}
		return 1;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////// KEY HANDLER

void press_lock_button( unsigned action ){
	switch(action){
	case 0: // not bound
		break;
	case 1: // lock
		if(cfg_lock_and_close && !hw_device_autolocked) window_unmap();
		mce_lock_screen();
		break;
	case 2: // lock&blank
		if(cfg_lock_and_close && !hw_device_autolocked) window_unmap();
		mce_lock_and_blank_screen();
		break;
	case 3: // blank
		if(cfg_lock_and_close && !hw_device_autolocked) window_unmap();
		mce_blank_screen();
		break;
	case 4: // secure
		if(cfg_secure_button){
			mce_lock_device();

			if(cfg_lock_and_close){
				hw_device_autolocked = 1;
				hw_device_autolock_map_state = 0; // unmap after code is entered
			}

			press_lock_button(cfg_secure_action);
		}
		break;
	}
}

int is_arrow( e_arrow_key arrow ){
	int value = (int)arrow;
	switch(window.orientation){
	case O_LANDSCAPE: break; // no translation
	case O_PORTRAIT_INV: value--; if(value < 0) value+=4; break;
	case O_LANDSCAPE_INV: value-=2; if(value < 0) value+=4; break;
	case O_PORTRAIT: value-=3; if(value < 0) value+=4; break;
	}
	switch((e_arrow_key)value){
	case ARROW_UP:
		if(keypress.keycode == KEYCODE_DPAD_UP) return 1;
		break;
	case ARROW_DOWN:
		if(keypress.keycode == KEYCODE_DPAD_DOWN) return 1;
		break;
	case ARROW_RIGHT:
		if(keypress.keycode == KEYCODE_DPAD_RIGHT) return 1;
		break;
	case ARROW_LEFT:
		if(keypress.keycode == KEYCODE_DPAD_LEFT) return 1;
		break;
	}
	return 0;
}

int dpadlock_keypress_handler( ){
	#ifdef ARCH_armel
	if(keypress.keycode == KEYCODE_ENTER /* n800 dpad center */ || keypress.keycode == KEYCODE_DPAD_CENTER /* n810 */){
		if(keypress.type == KEY_SHORT)
			press_lock_button(cfg_buttons_dpad_short);
		else if(keypress.type == KEY_LONG)
			press_lock_button(cfg_buttons_dpad_long);
		return 1;
	}
	#endif
	return 0;
}

int brv_keypress_handler( ){
	if(is_arrow(ARROW_UP))
		return button_up(CFG_BUTTONS_UD);
	else if(is_arrow(ARROW_DOWN))
		return button_down(CFG_BUTTONS_UD);
	else if(is_arrow(ARROW_RIGHT))
		return button_up(CFG_BUTTONS_LR);
	else if(is_arrow(ARROW_LEFT))
		return button_down(CFG_BUTTONS_LR);
	else if(keypress.keycode == KEYCODE_ZOOM_PLUS)
		return button_up(CFG_BUTTONS_ZOOM);
	else if(keypress.keycode == KEYCODE_ZOOM_MINUS)
		return button_down(CFG_BUTTONS_ZOOM);
	return 0;
}

static void main_keypress_handler( ){
	switch(keypress.keycode){
	case KEYCODE_ESCAPE:
		if(keypress.type == KEY_SHORT){
			if(widget_edit_mode){
				// exit widget edit mode
				highlight_selected_area(window.width-80, (window.width > window.height ? 0 : window.height-80), 80,80); // highlight removed when window is redrawn
				widget_edit_mode = 0;
				selected_cell.widget_cell = NULL;
				draw_window(1); // refresh, force to clear highlight
			} else
				window_unmap();
		}
		break;
	case KEYCODE_MENU:
		if(keypress.type == KEY_SHORT){
			error_write("key release (menu)");

// TODO: use these to toggle focus from buttons to widgets to off
// TODO: one direction on dpad will jump from button to button or widget to widget
// TODO: other direction will move focus between cells inside widgets
// TODO: support widget edit mode

		}
		break;
	case KEYCODE_HOME:
		if(keypress.type == KEY_SHORT){
			// switch between primary and secondary widgets
			highlight_button_bar();
			primary_widgets = (primary_widgets ? 0 : 1);
			hw_refresh();
			draw_window(1);
		}
		break;
	case KEYCODE_FULLSCREEN:
		if(keypress.type == KEY_SHORT) flashlight_open();
		break;
	default:
		if(!dpadlock_keypress_handler())
			if(!brv_keypress_handler())
				error_write("key release (%d)", keypress.keycode);
	}
}

////////////////////////////////////////////////////////////////////////// EVENT LOOP

void main_reset( ){
	// return to primary widget page
	primary_widgets = 1;
	// exit widget edit mode
	widget_edit_mode = 0;
}

static int clean_up_stage, dbus_fd;
static void clean_up( ){
	#ifdef DEBUG
	if(debug_file) fclose(debug_file);
	#endif
	#ifdef TESTLOG
	if(test_file) fclose(test_file);
	#endif
	if(clean_up_stage > 0){
		dsme_disconnect();
		if(clean_up_stage > 1){
			dbus_disconnect();
			close(dbus_fd);
			if(clean_up_stage > 2){
				window_close_blank(); // close blank window, if open
				window_close(1); // free fonts and close display
			}
		}
	}
}

static void kill_handler( int nothing ){
	debug_write("service killed: cleaning up");
	test_write("service killed: cleaning up");
	clean_up();
	exit(EXIT_SUCCESS);
}

static void segfault_handler( int nothing ){
	debug_write("Segmentation Fault");
	test_write("Segmentation Fault");
	clean_up();
	exit(EXIT_FAILURE);
}

static unsigned ready_to_read( int fd ){
	struct timeval timeout;
	fd_set set;

	FD_ZERO(&set);
	FD_SET(fd, &set);
	timeout.tv_sec = timeout.tv_usec = 0;
	if(select(fd+1, &set, NULL, NULL, &timeout) > 0) return 1;
	return 0;
}

int main( int argc, char *argv[] ){
	// detect runlevel
	int runlevel = 0;
	{
		FILE *in = popen("runlevel", "r");
		if(in){
			fscanf(in, "N %d", &runlevel);
			if(runlevel == 0) fscanf(in, "%*d %d", &runlevel);
			pclose(in);
		}
	}

	#ifdef DEBUG
	// initialize debug log file
	{
		time_t t = time(NULL);
		struct tm *tm = localtime(&t);
		char buffer[40];
		snprintf(buffer, sizeof(buffer), "/root/asui-debug-%d%02d%02d-%02d%02d%02d-r%d",
					tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec, runlevel);
		debug_file = fopen(buffer, "w"); /* static global */
		debug_runlevel = runlevel; /* static global */
		if(debug_file){
			FILE *in = popen("runlevel", "r");
			char buffer[100];
			fgets(buffer, sizeof(buffer), in);
			fprintf(debug_file, "[%d] runlevel = %s\n", runlevel, buffer);
			fflush(debug_file);
		}
	}
	#endif
	#ifdef TESTLOG
	// initialize test log file
	{
		time_t t = time(NULL);
		struct tm *tm = localtime(&t);
		char buffer[40];
		snprintf(buffer, sizeof(buffer), "/root/asui-test-%d%02d%02d-%02d%02d%02d",
					tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		test_file = fopen(buffer, "w"); /* static global */
	}
	#endif

	clean_up_stage = 0;

	// initialize signal handlers
	signal(SIGINT, kill_handler);
	signal(SIGTERM, kill_handler);
	signal(SIGSEGV, segfault_handler);

/*
	if(runlevel == 2){
		// sleep until system has fully booted
		int idle_boot = 0;
		while(!idle_boot){
			FILE *in;
			if((in = fopen("/proc/uptime", "r")) != NULL){
				float uptime, idle;
				fscanf(in, "%f %f", &uptime, &idle);
				fclose(in);
				if(idle > 2.0) idle_boot = 1;
			}
			if(!idle_boot) sleep(2);
		}
	}
*/

	// time at which service was started, excluding the above sleep state
	current_time = get_utime(); /* global */
	#ifdef DEBUG
	debug_start_time = current_time; /* static global */
	#endif
	#ifdef TESTLOG
	test_start_time = current_time; /* static global */
	#endif

	////////////////////

	// get device model
	{
		FILE *fp;
		is_770 = is_800 = is_810 = 0;
		// code ported from advanced-backlight
		if((fp = fopen("/proc/component_version", "r")) != NULL){
			char version;
			fseek(fp, sizeof(char)*15, 0); // The important character is the 16th in the file
			fread(&version, sizeof(char), 1, fp);
			fclose(fp);
			switch(version){
			case '4': is_810 = 1; break;
			case '3': is_800 = 1; break;
			case '1': is_770 = 1; break;
			default:  is_810 = 1; // assume N810
			}
		} else
			is_810 = 1; // assume N810
	}

	// initialize gconf settings
	config_init();

	// use minimal UI in runlevel 5
	if(runlevel == 5){
		use_minimal_ui = 1;			// only show battery and clock widgets, centered
		cfg_start_visible = 1;		// make sure it gets mapped
		cfg_battery_idle_time = 1;	// enable idle time since battery statusbar app isn't loaded
	} else
		use_minimal_ui = 0;

	////////////////////

	// initialize dsme socket
	int dsme_fd;
	#ifdef ARCH_armel
	if(!(dsme_fd = dsme_init())){
		clean_up();
		return EXIT_FAILURE;
	}
	clean_up_stage++;
	#else
	dsme_fd = 0;
	#endif

	// initialize dbus thread
	if(!(dbus_fd = dbus_init())){
		clean_up();
		return EXIT_FAILURE;
	}
	clean_up_stage++;

	// initialize the window
	if(!window_init()){
		clean_up();
		return EXIT_FAILURE;
	}
	clean_up_stage++;

	// initialize services and hardware
	svc_init();
	hw_init();

	////////////////////

	primary_widgets = 1;
	widget_edit_mode = 0;
	if(cfg_start_visible) window_map();

	////////////////////

	int x_fd = ConnectionNumber(window.display);
	unsigned main_refresh_all = 0, infinite_timeout = (!window.is_mapped || !hw_screen.visible ? 1 : 0);
	double next_timer_duration = MAIN_REFRESH_PERIOD;

	s_timer refresh_timer, button_timer, key_timer, brightness_timer;
	set_timer(&refresh_timer, MAIN_REFRESH_PERIOD);
	set_timer(&button_timer, BUTTON_CHECK_PERIOD);				button_callback = NULL; /* global */		click.button_down = 0; /* global */
	set_timer(&key_timer, BUTTON_CHECK_PERIOD);																keypress.key_down = 0; /* global */
	set_timer(&brightness_timer, BRIGHTNESS_CHECK_PERIOD);
	/* global motion_timer, don't need to clear it */			motion_callback = NULL; /* global */		unsigned has_motion = 0;
	is_idle = 0;

	double_tap_data.zone = Z_NULL; /* static global */

	// event loop
	for(;;){
		unsigned runaway_event_counter;
		struct timeval timeout;
		int highest_fd = (x_fd > dbus_fd ? (x_fd > dsme_fd ? x_fd : dsme_fd) : (dbus_fd > dsme_fd ? dbus_fd : dsme_fd));
		fd_set set;

		FD_ZERO(&set);
		#ifdef ARCH_armel
		FD_SET(dsme_fd, &set);
		#endif
		FD_SET(dbus_fd, &set);
		FD_SET(x_fd, &set);
		if(blank_fd){ FD_SET(blank_fd, &set); highest_fd = (blank_fd > highest_fd ? blank_fd : highest_fd); }
		if(!infinite_timeout){
			if(next_timer_duration > 0.0){
				// calculate timeout for next refresh
				timeout.tv_sec = floor(next_timer_duration);
				timeout.tv_usec = floor((next_timer_duration - (float)timeout.tv_sec) * 1000000.0);
			} else {
				// no active timers, check loop every second
				timeout.tv_sec = 1;
				timeout.tv_usec = 0;
			}
		}
		debug_write("select() inf=%d time=%.6f (%u.%u)", infinite_timeout, next_timer_duration, (unsigned)timeout.tv_sec, (unsigned)timeout.tv_usec);
		select(highest_fd+1, &set, NULL, NULL, (infinite_timeout ? NULL : &timeout));
		current_time = get_utime();
		debug_write("wake from select()");

		// terminate if dbus thread returns
		if(!dbus_listening) goto exit;

		////////////////////

		// catch X events for blank window
		if(blank_fd)
			if(window_blank_events()) goto exit;

		// catch X events
		debug_write("any X events?");
		runaway_event_counter = 0;
		while(XPending(window.display)){
			XEvent e;

			XNextEvent(window.display, &e);
			debug_write("    event %d", e.type);

			runaway_event_counter++;
			if(runaway_event_counter > MAX_RUNAWAY_EVENTS){ debug_write("stuck in a runaway event loop (>%d), killing ASUI", MAX_RUNAWAY_EVENTS); goto exit; }

			// touchscreen/keypad is unlocked briefly when restarted from shutdown runlevel
			if(use_minimal_ui) click.enabled = 0;

			// process events
			switch(e.type){
			case ButtonPress:
				// e.xbutton.x, e.xbutton.y

				if(!click.enabled && hw_screen.visible && !use_minimal_ui)
					if(current_time - hw_screen.visible_time > 0.5)
						click.enabled = 1; // screen has been on for more than 500ms, allow this tap

				click.button_down = 1;

				if(click.enabled){
					click.type = CLICK_DOWN;
					click.long_wait = current_time; // long press hasn't been highlighted yet
					click.timestamp = (unsigned)e.xbutton.time;
					click.start_x = click.motion_x = click.x = e.xbutton.x;
					click.start_y = click.motion_y = click.y = e.xbutton.y;
					click.offset_x = click.offset_y = 0; // reset motion

					if(screen_event()) goto exit;

					if(button_callback && button_autorepeat)
						button_repeat_init();
				}

				reset_timer(&button_timer);

				break;
			case MotionNotify:
				// e.xmotion.x, e.xmotion.y

				if(click.enabled){
					if(click.motion_x != e.xmotion.x || click.motion_y != e.xmotion.y){
						click.type = CLICK_MOTION;
						click.offset_x = e.xmotion.x - click.start_x;
						click.offset_y = e.xmotion.y - click.start_y;
						click.motion_x = e.xmotion.x;
						click.motion_y = e.xmotion.y;

						if(motion_callback == NULL){
							if(screen_event()) goto exit;
						} else if(is_timer_ready(&motion_timer)){
							if(motion_callback()) reset_timer(&motion_timer);
							has_motion = 0;
						} else has_motion = 1;
					}
				}

				break;
			case ButtonRelease:
				// e.xbutton.x, e.xbutton.y

				if(click.enabled){

					if(e.xbutton.x > click.start_x+DRAG_DEVIATION || e.xbutton.x < click.start_x-DRAG_DEVIATION
					|| e.xbutton.y > click.start_y+DRAG_DEVIATION || e.xbutton.y < click.start_y-DRAG_DEVIATION){
						// unhighlight if CLICK_HOLD, it turned into a drag
						unhighlight_long_press();

						// drag event
						click.type = CLICK_DRAG;
						click.offset_x = e.xbutton.x - click.start_x;
						click.offset_y = e.xbutton.y - click.start_y;
					} else if(click.timestamp + (unsigned)(LONG_PRESS_DURATION*1000) < (unsigned)e.xbutton.time){
						// long press
						click.type = CLICK_LONG;
					} else {
						// short press
						click.type = CLICK_SHORT;
					}

					reset_long_press();
					if(screen_event()) goto exit;

				} else {

					// ignore the tap used to wake screen
					click.enabled = 1;

				}

				click.button_down = 0;

				break;
			case KeyPress:
				// e.xkey.keycode

				if(!click.enabled && hw_screen.visible && !use_minimal_ui)
					if(current_time - hw_screen.visible_time > 0.5)
						click.enabled = 1; // screen has been on for more than 500ms, allow this key

				keypress.key_down = 1;

				if(click.enabled){
					keypress.type = KEY_DOWN;
					keypress.timestamp = (unsigned)e.xkey.time;
					keypress.keycode = e.xkey.keycode;

					if(current_dialog_draw == NULL)
						main_keypress_handler();
					else if(current_dialog_keypress)
						current_dialog_keypress();
				}

				reset_timer(&key_timer);

				break;
			case KeyRelease:
				// e.xkey.keycode

				if(click.enabled){

					if(keypress.timestamp + (unsigned)(LONG_PRESS_DURATION*1000) < (unsigned)e.xkey.time)
						keypress.type = KEY_LONG;
					else
						keypress.type = KEY_SHORT;

					if(current_dialog_draw == NULL){
						#ifndef ARCH_armel
						if(keypress.keycode == 24) goto exit; // 'q'
						#endif
						main_keypress_handler();
					} else if(current_dialog_keypress)
						current_dialog_keypress();

				} else {

					// ignore the key used to wake screen
					click.enabled = 1;

				}

				keypress.key_down = 0;

				break;
			case FocusIn:			// 9
				if(window.is_mapped && !window.has_focus && !use_minimal_ui){
					// grab keyboard (home key) when bluetooth or wifi dialog closes
					XGrabKeyboard(window.display, window.window, False, GrabModeAsync, GrabModeAsync, CurrentTime);
					// grab pointer to prevent another program from stealing it away from ASUI
					XGrabPointer(window.display, window.window, False, ButtonPressMask|ButtonMotionMask|ButtonReleaseMask,
									GrabModeAsync, GrabModeAsync, window.window, None,CurrentTime);
					window.has_focus = 1;
				}
				break;
			case FocusOut:			// 10
				if(window.is_mapped && window.has_focus && !use_minimal_ui){
					// release keyboard (home key) when bluetooth or wifi dialog opens
					XUngrabKeyboard(window.display, CurrentTime);
					XUngrabPointer(window.display, CurrentTime);
					window.has_focus = 0;
				}
				break;
			case Expose:			// 12
				// force a window redraw for systems without save-unders
				if(e.xexpose.count == 0) draw_window(1);

// TODO: rotating from landscape <-> portrait forces a redraw and then an expose event forces another redraw

// TODO: flipping the screen does a non-force redraw to get new rotation icons and then two expose events forces TWO redraws

				break;
			case NoExpose:			// 14
			case DestroyNotify:		// 17
			case UnmapNotify:		// 18
			case MapNotify:			// 19
			case ReparentNotify:	// 21
			case ConfigureNotify:	// 22
			case ClientMessage:		// 33
			case MappingNotify:		// 34
				// ignore
				break;
			default:
				if(XRRUpdateConfiguration(&e)){
					if(e.type - window.xrr_event_base == RRScreenChangeNotify){ // 90
						update_orientation(&e);
					} else {
						debug_write("unknown RR event %d", e.type - window.xrr_event_base);
					}
				} else {
					debug_write("unknown event %d", e.type);
				}
			}
		}

		////////////////////

		// catch signals sent from dbus thread
		debug_write("any dbus messages?");
		while(ready_to_read(dbus_fd)){
			char message;

			if(read(dbus_fd, (void *)&message, 1) == 1){
				switch(message){
				case _WAKEUP_SHOW_:

					// show the window
					debug_write("    message SHOW");
					if(!window.is_mapped) window_map();
					break;

				case _WAKEUP_HIDE_:

					// hide the window
					debug_write("    message HIDE");
					if(window.is_mapped && !hw_device_locked) window_unmap();
					break;

				case _WAKEUP_ROTATE_LEFT_:

					// rotate screen left
					if(!widget_edit_mode){ // rotation can't be changed in widget edit mode
						if(window.supports_rotation){
							switch(window.orientation){
							case O_LANDSCAPE: rotate_screen(O_PORTRAIT_INV, window.is_mapped); break;
							case O_PORTRAIT: rotate_screen(O_LANDSCAPE, window.is_mapped); break;
							case O_LANDSCAPE_INV: rotate_screen(O_PORTRAIT, window.is_mapped); break;
							case O_PORTRAIT_INV: rotate_screen(O_LANDSCAPE_INV, window.is_mapped); break;
							}
						}
					}
					break;

				case _WAKEUP_ROTATE_RIGHT_:

					// rotate screen right
					if(!widget_edit_mode){ // rotation can't be changed in widget edit mode
						if(window.supports_rotation){
							switch(window.orientation){
							case O_LANDSCAPE: rotate_screen(O_PORTRAIT, window.is_mapped); break;
							case O_PORTRAIT: rotate_screen(O_LANDSCAPE_INV, window.is_mapped); break;
							case O_LANDSCAPE_INV: rotate_screen(O_PORTRAIT_INV, window.is_mapped); break;
							case O_PORTRAIT_INV: rotate_screen(O_LANDSCAPE, window.is_mapped); break;
							}
						}
					}
					break;

				case _WAKEUP_POWER_BUTTON_:

					// only map or unmap if the display has been on for atleast 1 second and is not locked
					// this way the power button can be used to wake or unlock the device and it
					// will be in the same state it was when the screen blanked or locked
					debug_write("    message POWER BUTTON");
					if(hw_screen.visible && !hw_screen.locked && current_time - hw_screen.visible_time > 1.0){
						if(hw_device_locked){
							press_lock_button(cfg_secure_power_action+1);
						} else if(window.is_mapped){
							// hide the window
							if(!use_minimal_ui && !hw_device_locked && !cfg_n800_lock_mode)
								window_unmap();
						} else
							window_map();
					}
					if(hw_screen.locked){
						// send unlock signal
						mce_unlock_screen();
					}
					break;

				case _WAKEUP_LOCK_BUTTON_:

					// ignore it -- no known reason to have it wake main loop
					// just do it in case something like lock&blank suspends the main loop
					debug_write("    message LOCK BUTTON");
					break;

				case _WAKEUP_DEVICE_LOCK_:

					// device was locked or unlocked
					// if locked and window is not mapped then map window
					if(hw_device_locked){
						debug_write("    message DEVICE LOCK");
						if(!window.is_mapped)
							window_map();
						if(current_dialog_draw != draw_fullsreen_clock && current_dialog_draw != draw_flashlight)
							secureShowKeypad();
					} else {
						debug_write("    message DEVICE UNLOCK");
						secureHideKeypad();
						if(hw_device_autolocked){
							hw_device_autolocked = 0;
							if(!hw_device_autolock_map_state) window_unmap();
						}
					}
					break;

				case _WAKEUP_REFRESH_:

					// ignore it -- used to wake main loop and refresh window
					debug_write("    message WAKEUP");
					break;

				case _WAKEUP_SHUTDOWN_:

					// shutdown signal received
					debug_write("    message SHUTDOWN");

					// detect runlevel transition (2 -> 5)
					FILE *in = popen("runlevel", "r");
					if(in){
						int runlevel = 0;
						fscanf(in, "2 %d", &runlevel);
						pclose(in);
						if(runlevel == 5){
							#ifdef DEBUG
							debug_runlevel = runlevel;
							#endif
							debug_write("entering charging runlevel (shutdown)");

							use_minimal_ui = 1; // only show battery and clock widgets, centered
							cfg_battery_idle_time = 1;  // enable idle time since battery statusbar app isn't loaded
							hw_query_battery(1); // flush
							if(hw_battery.status == BATTERY_DRAINING)
								hw_battery.status = BATTERY_CHARGING; // it should never be BATTERY_DRAINING in this mode

							if(current_dialog_draw != NULL)
								dialog_close(); // close the current dialog, most likely the shutdown dialog
							else if(!window.is_mapped)
								window_map(); // map ASUI if not already mapped
							else
								draw_window(1);
						}
					}
					break;

				default:

					// ignore unknown messages
					debug_write("    message UNKNOWN");
					break;

				}
			} else break;
		}

		////////////////////

		// read messages from dsme socket
		#ifdef ARCH_armel
		debug_write("any dsme messages?");
		while(ready_to_read(dsme_fd))
			dsme_read();
		#endif

		////////////////////

		debug_write("mapped (asui=%u, blank=%u) -- display (on=%u, on_time=%.6f)",
						window.is_mapped, blank_window_open, hw_screen.visible, current_time - hw_screen.visible_time);
		if(blank_window_open && hw_screen.visible)
			if(current_time - hw_screen.visible_time > 1.0)
				window_close_blank(); // screen has been on for more than one second, close the blank window

		if(hw_device_locked && !window.is_mapped) window_map();

		// query ambient light sensor and update theme when a dialog is open
		// dialogs that don't use refresh timers depend on dbus signals to wake the loop and won't respond to light changes until this happens
		if(current_dialog_draw != NULL && window.is_mapped && hw_screen.visible)
			hw_query_als();

		infinite_timeout = ((!window.is_mapped && !blank_window_open) || !hw_screen.visible || current_dialog_redraw == DLG_REDRAW_OFF ? 1 : 0);

		// auto refresh when display is lit, window is mapped and dialog has a refresh timer
		if(!infinite_timeout){

			debug_write("process timers");

			next_timer_duration = (blank_window_open ? 1.0 : 0.0); // poll every second if the blank window is open

			// secure dialog is always active when device is code locked (fullscreen clock and flashlight can also be active)
			if(hw_device_locked && !use_minimal_ui
			   && current_dialog_draw != draw_secure_keypad
			   && current_dialog_draw != draw_fullsreen_clock
			   && current_dialog_draw != draw_flashlight
			)
				secureShowKeypad();

			// MAIN LOOP
			if(hw_screen.visible && window.is_mapped && current_dialog_draw == NULL && !widget_edit_mode){
				if(is_timer_ready(&refresh_timer)){
					debug_write("    main refresh");

					// refresh cpu for battery widget every 5 seconds
					if(battery_widget_page == primary_widgets)
						hw_refresh_cpu(); // this needs to be polled before services and hardware otherwise frequency is always 400mhz

					// re-synchronize cycle to update everything if ASUI hasn't been updated in a while
					if(current_time - last_hw_refresh > 2*MAIN_REFRESH_PERIOD) main_refresh_all = 1;

					// refresh service and hardware values
					if(main_refresh_all){
						if(use_minimal_ui){
							// detect runlevel transition (5 -> 2)
							FILE *in = popen("runlevel", "r");
							if(in){
								int runlevel = 0;
								fscanf(in, "5 %d", &runlevel);
								pclose(in);
								if(runlevel == 2){
									debug_write("leaving charging runlevel (boot)");
									goto exit; // terminate and let DSME restart it with full interface
								}
							}

//							// continuously poll battery capacity once fully charged in case the signals don't work properly
//							if(hw_battery.status == BATTERY_AC)
//								hw_query_battery(1); // flush

							#ifdef DEBUG
							svc_refresh(); // not needed unless svc_self_size is required for memory leaks
							#endif
						} else
							svc_refresh();

						hw_refresh();
					}

					if(main_refresh_all){
						// redraw main window
						// this could be conditional on hw_changes||svc_changes but then the clock wouldn't update
						draw_window(0);
					} else if(battery_widget_page == primary_widgets){
						// only redraw battery widget
						draw_battery_widget(0, 0, 0);
						hw_changes &= ~ALL_BATTERY_FLAGS; // reset flags
						x_flush();
					}

					main_refresh_all = (main_refresh_all ? 0 : 1);
					reset_timer(&refresh_timer);
				}

				if(next_timer_duration <= 0.0 || refresh_timer.remaining < next_timer_duration)
					next_timer_duration = refresh_timer.remaining;
			}

			// DIALOG LOOP
			if(hw_screen.visible && current_dialog_draw != NULL && current_dialog_redraw == DLG_REDRAW_ON){
				if(is_timer_ready(&dialog_refresh_timer)){
					if(hw_device_locked && current_dialog_draw == draw_secure_keypad){
						secureUpdate();
						// secureUpdate will reset the timer
					} else {
						// redraw dialog
						draw_window(0);

						// dialog must reset timer
					}
				}

				if(next_timer_duration <= 0.0 || dialog_refresh_timer.remaining < next_timer_duration)
					next_timer_duration = dialog_refresh_timer.remaining;
			}

			// BUTTON DOWN
			if(click.enabled && click.button_down){
				if(is_timer_ready(&button_timer)){
					// timer is needed to periodically check for ButtonRelease
					// and to handle auto-repeat and long press highlighting

					if(window.is_mapped){
						if(button_callback){
							if(is_button_repeat_ready())
								button_callback();
						} else if(click.long_wait){
							if(click.long_wait + LONG_PRESS_DURATION < current_time && !(click.offset_x > DRAG_DEVIATION || click.offset_x < -DRAG_DEVIATION || click.offset_y > DRAG_DEVIATION || click.offset_y < -DRAG_DEVIATION)){
								// screen press is now a long press
								click.type = CLICK_HOLD;
								screen_event();

								click.long_wait = 0.0;
							} // else screen is being pressed but is still a short press
						} else if(click.offset_x > DRAG_DEVIATION || click.offset_x < -DRAG_DEVIATION || click.offset_y > DRAG_DEVIATION || click.offset_y < -DRAG_DEVIATION){
							// screen is being pressed but has exceeded the long press duration
							// location has also drifted and is can no longer be a long press
							unhighlight_long_press();
							reset_long_press();
						} else {
							// screen is being pressed but has exceeded the long press duration
							// refresh the highlight in case it was erased
							click.type = CLICK_HOLD;
							screen_event();
						}
					}

					reset_timer(&button_timer);
				}

				if(next_timer_duration <= 0.0 || button_timer.remaining < next_timer_duration)
					next_timer_duration = button_timer.remaining;
			}

			// KEY DOWN
			if(click.enabled && keypress.key_down){
				if(is_timer_ready(&key_timer)){
					// timer is needed to periodically check for KeyRelease
					// and to handle auto-repeat and long press highlighting

					// do nothing

// TODO: support KEY_HOLD and KEY_LONG and invoke handler

					reset_timer(&key_timer);
				}

				if(next_timer_duration <= 0.0 || key_timer.remaining < next_timer_duration)
					next_timer_duration = key_timer.remaining;
			}

			// BRIGHTNESS
			if(cfg_brightness_enhanced && hw_screen.brightness_mismatch && current_dialog_draw == NULL && brightness_widget_page == primary_widgets){
				if(is_timer_ready(&brightness_timer)){
					// brightness changed, continuously update on-screen value until it stabilizes

					hw_query_brightness();
					draw_brightness_widget(0, 0, 0);
					hw_changes &= ~HW_BRIGHTNESS; // reset flag
					x_flush();
					if(hw_screen.brightness_level == hw_screen.brightness_current || hw_screen.dimmed) hw_screen.brightness_mismatch = 0;

					reset_timer(&brightness_timer);
				}

				if(hw_screen.brightness_mismatch && (next_timer_duration <= 0.0 || brightness_timer.remaining < next_timer_duration))
					next_timer_duration = brightness_timer.remaining;
			}

			// DRAG MOTION
			if(motion_callback && has_motion){
				if(is_timer_ready(&motion_timer)){
					// periodically dispatch a callback when as finger moves across screen
					if(motion_callback()) reset_timer(&motion_timer);
					has_motion = 0;
				}

				if(has_motion && (next_timer_duration <= 0.0 || motion_timer.remaining < next_timer_duration))
					next_timer_duration = motion_timer.remaining;
			}

		}

		// refresh on dbus signals and method returns
			// all dialogs should change on HW_THEME
			// cpu policy dialog needs HW_CPU
			// secure keypad dialog needs HW_BATTERY and HW_TEMPERATURE
		if(hw_changes && (current_dialog_draw == NULL || hw_changes & (HW_THEME|HW_CPU|HW_BATTERY|HW_TEMPERATURE)))
			draw_window(0);

		if(is_idle){
			if(is_timer_ready(&idle_timer)){
				// device has exceeded the devicelock autolock timeout, lock device
				test_write("timeout expired, locking device...");
				mce_lock_device();
				hw_device_autolocked = 1;
				hw_device_autolock_map_state = window.is_mapped;
				is_idle = 0;
			} else if(infinite_timeout){
				next_timer_duration = idle_timer.remaining;
				infinite_timeout = 0;
			} else if(next_timer_duration <= 0.0 || idle_timer.remaining < next_timer_duration)
				next_timer_duration = idle_timer.remaining;
		}

	}

	////////////////////

exit:

	debug_write("exit: cleaning up");
	test_write("exit: cleaning up");
	clean_up(2);
	return EXIT_SUCCESS;
}
