//  Advanced System UI
//  Copyright (c) 2010-2011 Brand Huntsman <http://qzx.com/mail/>

// http://maemo.org/api_refs/4.1/dbus-1.0.2/
// http://dbus.freedesktop.org/doc/dbus/api/html/index.html
// http://dbus.freedesktop.org/doc/dbus-tutorial.html

#ifdef ARCH_armel
//#define DEBUG_SIGNALS
//#define DEBUG_UNHANDLED_SIGNALS
#endif

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#ifndef __USE_ISOC99
 #define __USE_ISOC99
#endif
#include <math.h>
#include <fcntl.h>
#include <pthread.h>

#include "window.h"
#include "main.h"
 #include "dbus.h"
#include "config.h"
#include "hardware.h"

////////////////////////////////////////////////////////////////////////// GLOBALS

DBusError dbus_error;

unsigned dbus_listening;

char dbus_wakeup_message;
int dbus_wakeup_fd;

unsigned original_screen_autolock, original_screen_stayslit;

////////////////////////////////////////////////////////////////////////// LOCAL GLOBALS

static DBusConnection *signal_bus, *system_bus, *session_bus;

static pthread_t thread;

#ifdef ARCH_armel
static const char *_notification_service_ = "org.freedesktop.Notifications";
static const char *_notification_path_ = "/org/freedesktop/Notifications";
static const char *_notification_interface_ = "org.freedesktop.Notifications";
static const char *_notification_method_ = "SystemNoteInfoprint"; // method
#endif

//////////////////////////////////////////////////////////////////////////

#ifdef ARCH_armel
static void check_dimmed_state( ){
	if(hw_screen.dimmed && hw_battery.status != BATTERY_DRAINING && conf_get_screen_stayslit_value()){
		// unset dimmed state if on charger and screen stays lit
		hw_screen.brightness_mismatch = 1;
		hw_screen.dimmed = 0;
		hw_changes |= HW_BRIGHTNESS;
		dbus_send_wakeup(_WAKEUP_REFRESH_);
	}
}

static void set_battery_status( e_battery_status status ){
	unsigned charging = (status == BATTERY_CHARGING ? 1 : 0);
	if(hw_battery.initialize){ hw_battery.initialize = 0; hw_battery.status = status; }
	else if((!charging && hw_battery.status == BATTERY_CHARGING) || (charging && hw_battery.status != BATTERY_CHARGING)){
		hw_battery.status = (charging ? BATTERY_CHARGING : BATTERY_AC);
		dbus_send_charging_status_signal();
		hw_changes |= HW_BATTERY;
		dbus_send_wakeup(_WAKEUP_REFRESH_);
		if(!charging) hw_query_battery(1); // flush
	}

	check_dimmed_state();
}

static DBusHandlerResult signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	// needs to be updated here for debug_write() and hw_query_battery()
	current_time = get_utime();

	if(dbus_message_is_method_call(message, "com.qzx.asui", "flush_calls")){

		debug_write("[DBUS] method: flush calls");

		// signal bus is now awake so any method calls sent over it can be flushed here
		dbus_connection_flush(signal_bus);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "reload_config")){

		debug_write("[DBUS] method: reload config");

		config_reload();
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "show")){

		debug_write("[DBUS] method: show");

		if(!window.is_mapped) dbus_send_wakeup(_WAKEUP_SHOW_);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "hide")){

		debug_write("[DBUS] method: hide");

		if(window.is_mapped) dbus_send_wakeup(_WAKEUP_HIDE_);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "rotate_left")){

		debug_write("[DBUS] method: rotate_left");

		dbus_send_wakeup(_WAKEUP_ROTATE_LEFT_);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "rotate_right")){

		debug_write("[DBUS] method: rotate_right");

		dbus_send_wakeup(_WAKEUP_ROTATE_RIGHT_);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_method_call(message, "com.qzx.asui", "req_battery_status")){

		debug_write("[DBUS] method: req_battery_status");

		if(!(hw_initialized & HWIS_BME_CAPACITY)){
			dbus_send_charging_status_signal();
			dbus_send_battery_capacity_signal();
		}
		return DBUS_HANDLER_RESULT_HANDLED;

//	} else if(dbus_message_is_method_call(message, "com.nokia.system_ui.request", "devlock_open")){
//
//		printf("[DBUS] method: com.nokia.system_ui.request.devlock_open\n");
//
//		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "Condition")){
		char *s1, *s2;

		debug_write("[DBUS] signal: org.freedesktop.Hal.Device.Condition");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _STRING_, &s2, _END_)){
			if(!strcmp(s1, "ButtonPressed")){
				if(!strcmp(s2, "power")){
					debug_write("[DBUS]         power button pressed (blank=%u)", blank_window_open);
					if(blank_window_open)
						window_close_blank(); // power key caught, close the blank window
					else
						dbus_send_wakeup(_WAKEUP_POWER_BUTTON_);
					return DBUS_HANDLER_RESULT_HANDLED;
				} else if(!strcmp(s2, "connection")){
					debug_write("[DBUS]         headphone connection (blank=%u)", blank_window_open);
					window_close_blank(); // this will turn the screen on, close the blank window
					hw_query_headphones();
					dbus_send_wakeup(_WAKEUP_REFRESH_);
					return DBUS_HANDLER_RESULT_HANDLED;
				}
			}
		}

	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "PropertyModified")){
		const char *signal_path = dbus_message_get_path(message) + 29; // strlen("/org/freedesktop/Hal/devices/")

		if(!strcmp(signal_path, "usb_device_0_0_musb_hdrc")){

			debug_write("[DBUS] signal: org.freedesktop.Hal.Device.PropertyModified (usb_device_0_0_musb_hdrc)");

			hw_query_usb();
			return DBUS_HANDLER_RESULT_HANDLED;

		} else if(!strcmp(signal_path, "bme")){

			debug_write("[DBUS] signal: org.freedesktop.Hal.Device.PropertyModified (bme)");

			hw_query_battery(1); // flush
			return DBUS_HANDLER_RESULT_HANDLED;

		}

	////////////////////
	// MCE
	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "tklock_mode_ind")){
		char *s1;

		debug_write("[DBUS] signal: com.nokia.mce.signal.tklock_mode_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _END_)){
			if(!strcmp(s1, "unlocked")){
				debug_write("[DBUS] signal: com.nokia.mce.signal.tklock_mode_ind (unlock)");
				if(hw_screen.locked){
					// screen was just unlocked
					notify_screen_unlocked();
				}
				hw_unlock_keypad(); // always make sure keypad is unlocked, even if ASUI doesn't think the screen is locked
				hw_screen.locked = 0;
//printf("com.nokia.mce.signal.tklock_mode_ind -- screen UNLOCKED\n");
			} else if(!strcmp(s1, "locked")){
				debug_write("[DBUS] signal: com.nokia.mce.signal.tklock_mode_ind (lock)");
				if(!hw_screen.locked && hw_screen.visible){
					// screen was just locked but is still visible
					notify_screen_locked();
				}
				hw_lock_keypad(); // always make sure keypad is locked, even if ASUI doesn't think the screen is unlocked
				hw_screen.locked = 1;
//printf("com.nokia.mce.signal.tklock_mode_ind -- screen LOCKED\n");

				window_close_blank(); // close the blank window, if open
				click.enabled = 1; // don't ignore first tap since screen can't be unlocked by tapping it
			}
			dbus_send_wakeup(_WAKEUP_LOCK_BUTTON_);
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "display_status_ind")){
		char *s1;

		debug_write("[DBUS] signal: com.nokia.mce.signal.display_status_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _END_)){
			// off, dimmed, on
			unsigned visible = (strcmp(s1, "off") ? 1 : 0);
			unsigned dimmed = (strcmp(s1, "dimmed") ? 0 : 1);

			debug_write("[DBUS] signal: com.nokia.mce.signal.display_status_ind (visible = %u)", visible);
//printf("com.nokia.mce.signal.display_status_ind -- %s\n", s1);

			if(!hw_screen.visible && visible){
				// screen was just turned on
				hw_screen.visible_time = current_time;
				hw_screen.brightness_mismatch = 1;
				if(original_screen_autolock){
					// restored scree_autolock setting
					conf_set_screen_autolock_value(original_screen_autolock-1);
					original_screen_autolock = 0;
				}
				if(original_screen_stayslit){
					// restore screen_stayslit setting
					conf_set_screen_stayslit_value(original_screen_stayslit-1);
					original_screen_stayslit = 0;
					check_dimmed_state();
				}
			} else if(hw_screen.visible && !visible){
				// screen was just turned off
				if(!hw_screen.locked){
					// screen is not locked
					if(!conf_get_screen_autolock_value()){ // screen does not autolock
						if(!window.is_mapped)
							window_open_blank(); // screen is off but not locked, ASUI is not open -- open a blank window to catch the wakeup tap/key
						click.enabled = 0; // ignore first screen tap when unblanked
					}
				}
			}
			hw_screen.visible = visible;
			// don't need to wakeup main loop for this, dbus unlock signal or X screen/key events will wake it

			if(hw_screen.dimmed != dimmed){
				if(!dimmed || !(hw_battery.status != BATTERY_DRAINING && conf_get_screen_stayslit_value())){
					// ignore dimmed signal if on charger and screen stays lit
					hw_screen.brightness_mismatch = 1;
					hw_screen.dimmed = dimmed;
					hw_changes |= HW_BRIGHTNESS;
					dbus_send_wakeup(_WAKEUP_REFRESH_);
				}
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "devicelock_mode_ind")){
		char *s1;

		debug_write("[DBUS] signal: com.nokia.mce.signal.devicelock_mode_ind");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &s1, _END_)){
			if(!strcmp(s1, "unlocked")){
//printf("com.nokia.mce.signal.devicelock_mode_ind -- code UNLOCKED\n");
				if(hw_device_locked){
					debug_write("[DBUS] signal: com.nokia.mce.signal.devicelock_mode_ind (code unlock)");
					test_write("signal: unlock device");
					hw_device_locked = 0;
					dbus_send_wakeup(_WAKEUP_DEVICE_LOCK_);
				}
			} else if(!strcmp(s1, "locked")){
//printf("com.nokia.mce.signal.devicelock_mode_ind -- code LOCKED\n");
				if(!hw_device_locked){
					debug_write("[DBUS] signal: com.nokia.mce.signal.device_mode_ind (code lock)");
					test_write("signal: lock device");
					hw_device_locked = 1;
					dbus_send_wakeup(_WAKEUP_DEVICE_LOCK_);
				}
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "system_inactivity_ind")){
		unsigned state;

		debug_write("[DBUS] signal: com.nokia.mce.signal.system_inactivity_ind");

		if(dbus_message_get_args(message, &dbus_error, _BOOL_, &state, _END_)){
			if(state && !hw_device_locked && !use_minimal_ui){
				if(!is_idle){
					if(conf_get_bool("/system/osso/dsm/locks/devicelock_autolock_enabled")){
						test_write("device is idle, start timer...");
						set_timer(&idle_timer, (double)conf_get_int("/system/osso/dsm/locks/devicelock_autolock_timeout")*60.0);
						is_idle = 1;
						dbus_send_wakeup(_WAKEUP_REFRESH_); // get out of an infinite timeout
					}
				}
			} else
				is_idle = 0;
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.mce.signal", "shutdown_ind")){

		debug_write("[DBUS] signal: com.nokia.mce.signal.shutdown_ind");

		dbus_send_wakeup(_WAKEUP_SHUTDOWN_);
		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// BME
//	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "PropertyModified")){

		// handled above

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_state_changed")){

		if(hw_initialized & HWIS_BME_CHARGER){
			// this signal is sent along with charger state in response to the status_info_req signal
			// status_info_req signal will be sent until this signal is received
			debug_write("[DBUS] signal: com.nokia.bme.signal.battery_state_changed");
			hw_initialized &= ~HWIS_BME_CHARGER;
		}

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_timeleft")){
		unsigned idle_minutes, active_minutes;

		debug_write("[DBUS] signal: com.nokia.bme.signal.battery_timeleft");

		if(dbus_message_get_args(message, &dbus_error, _UINT32_, &idle_minutes, _UINT32_, &active_minutes, _END_)){
			if(hw_battery.active_minutes != active_minutes || hw_battery.idle_minutes != idle_minutes){
				hw_battery.active_minutes = active_minutes;
				hw_battery.idle_minutes = idle_minutes;
				hw_changes |= HW_BATTERY;
				dbus_send_wakeup(_WAKEUP_REFRESH_);
				hw_initialized &= ~HWIS_BME_ACTIVE;
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_connected")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_connected");
//printf("com.nokia.bme.signal.charger_connected\n");

		if(hw_screen.locked){
			// lock&blank doesn't properly lock the screen and connecting charger will unlock it
			// without sending an unlock signal
			mce_lock_screen();
		}

		if(hw_battery.status == BATTERY_DRAINING){
			hw_battery.status = BATTERY_AC;
			dbus_send_charging_status_signal();
			hw_changes |= HW_BATTERY;
			dbus_send_wakeup(_WAKEUP_REFRESH_);
			hw_query_battery(1); // flush
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_disconnected")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_disconnected");
//printf("com.nokia.bme.signal.charger_disconnected\n");

		if(hw_battery.status != BATTERY_DRAINING){
			hw_battery.status = BATTERY_DRAINING;
			dbus_send_charging_status_signal();
			hw_changes |= HW_BATTERY;
			dbus_send_wakeup(_WAKEUP_REFRESH_);
			hw_query_battery(1); // flush
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_charging_on")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_charging_on");
//printf("com.nokia.bme.signal.charger_charging_on\n");

		set_battery_status(BATTERY_CHARGING);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "charger_charging_off")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.charger_charging_off");
//printf("com.nokia.bme.signal.charger_charging_off\n");

		set_battery_status(BATTERY_DRAINING);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.bme.signal", "battery_full")){

		debug_write("[DBUS] signal: com.nokia.bme.signal.battery_full");
//printf("com.nokia.bme.signal.battery_full\n");

		set_battery_status(BATTERY_AC);
		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// BLUEZ
	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "ModeChanged")){
		char *mode;

		debug_write("[DBUS] signal: org.bluez.Adapter.ModeChanged");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &mode, _END_)){
			unsigned enabled = (strcmp(mode, "off") ? 1 : 0);
			unsigned visible = (strcmp(mode, "discoverable") ? 0 : 1);

// TODO: connectable, discoverable or limited

			if(hw_bluetooth.enabled != enabled || hw_bluetooth.visible != visible){
				hw_bluetooth.enabled = enabled;
				hw_bluetooth.visible = visible;
				hw_changes |= HW_BLUETOOTH;
				dbus_send_wakeup(_WAKEUP_REFRESH_);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "NameChanged")){
		char *name;

		debug_write("[DBUS] signal: org.bluez.Adapter.NameChanged");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &name, _END_)){
			if(hw_bluetooth.name == NULL || strcmp(hw_bluetooth.name, name)){
				set_string(&hw_bluetooth.name, name, BLUETOOTH_NAME_LIMIT);
				hw_changes |= HW_BLUETOOTH;
				dbus_send_wakeup(_WAKEUP_REFRESH_);
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "RemoteDeviceConnected")){

		debug_write("[DBUS] signal: org.bluez.Adapter.RemoteDeviceConnected");

		hw_bluetooth.nr_devices++; hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_);
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.Adapter", "RemoteDeviceDisconnected")){

		debug_write("[DBUS] signal: org.bluez.Adapter.RemoteDeviceDisconnected");

		if(hw_bluetooth.nr_devices){ hw_bluetooth.nr_devices--; hw_changes |= HW_BLUETOOTH; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.audio.Headset", "Connected")){

		debug_write("[DBUS] signal: org.bluez.audio.Headset.org.bluez.audio.Headset");

		if(!hw_bluetooth.headset){ hw_bluetooth.headset = 1; hw_changes |= HW_VOLUME_OUTPUT; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "org.bluez.audio.Headset", "Disconnected")){

		debug_write("[DBUS] signal: org.bluez.audio.Headset.Disconnected");

		if(hw_bluetooth.headset){ hw_bluetooth.headset = 0; hw_changes |= HW_VOLUME_OUTPUT; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// ICD
	} else if(dbus_message_is_signal(message, "com.nokia.wlancond.signal", "connected")){

		debug_write("[DBUS] signal: com.nokia.wlancond.signal.connected");

		if(!hw_wifi.enabled){ hw_wifi.enabled = 1; hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.wlancond.signal", "disconnected")){

		debug_write("[DBUS] signal: com.nokia.wlancond.signal.disconnected");

		if(hw_wifi.enabled){ hw_wifi.enabled = 0; hw_changes |= HW_WIFI; dbus_send_wakeup(_WAKEUP_REFRESH_); }
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.icd", "status_changed")){
		char *name, *type, *state_string, *unknown;

		debug_write("[DBUS] signal: com.nokia.icd.status_changed");

		if(dbus_message_get_args(message, &dbus_error, _STRING_, &name, _STRING_, &type, _STRING_, &state_string, _STRING_, &unknown, _END_)){
			if(strncmp(state_string, "SCAN", 4)){
				// get connection state (and name/IP if connected)
				unsigned state, name_changed = 0;
				if(!strcmp(state_string, "CONNECTING")) state = 1;
				else if(!strcmp(state_string, "CONNECTED")){
					state = 2;
					name_changed = hw_wifi_get_name(name);
					hw_wifi_get_ipinfo(); dbus_connection_flush(signal_bus);
				} else state = 0;
				// refresh if name or state changed
				if(name_changed || hw_wifi.state != state){
					if(hw_wifi.state != state && state == 0){
						// free name/IP when disconnected
						set_string(&hw_wifi.name, NULL, WIFI_NAME_LIMIT);
						set_string(&hw_wifi.address, NULL, WIFI_ADDRESS_LIMIT);
					}
					hw_wifi.state = state;
					hw_wifi.enabled = (state ? 1 : 0);
					hw_changes |= HW_WIFI;
					dbus_send_wakeup(_WAKEUP_REFRESH_);
				}
			}
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	} else if(dbus_message_is_signal(message, "com.nokia.icd2", "state_sig") && !strcmp(dbus_message_get_signature(message), "sussuaysu")){
		char *svc_type, *svc_id, *net_type, *net_id, *error;
		unsigned svc_attr, net_attr, net_state;
		int net_len;

		debug_write("[DBUS] signal: com.nokia.icd2.state_sig sussuaysu");

		if(dbus_message_get_args(message, &dbus_error,
			_STRING_, &svc_type, _UINT32_, &svc_attr, _STRING_, &svc_id,
			_STRING_, &net_type, _UINT32_, &net_attr,
			_ARRAY_, DBUS_TYPE_BYTE, &net_id, &net_len,
			_STRING_, &error, _UINT32_, &net_state, _END_))
		{
			// get connection state (and name/IP if connected)
			e_hw_wifi_state state;
			unsigned name_changed = 0;
			if(net_state == 0 /* DISCONNECTED */) state = WIFI_DISCONNECTED;
			else if(net_state == 2 /* CONNECTED */){
				state = WIFI_CONNECTED;
				name_changed = hw_wifi_get_name(net_id);
				hw_wifi_get_ipinfo(); dbus_connection_flush(signal_bus);
			} else state = WIFI_CONNECTING;
			// refresh if name or state changed
			if(name_changed || hw_wifi.state != state){
				if(hw_wifi.state != state && state == WIFI_DISCONNECTED){
					// free name/IP when disconnected
					set_string(&hw_wifi.name, NULL, WIFI_NAME_LIMIT);
					set_string(&hw_wifi.address, NULL, WIFI_ADDRESS_LIMIT);
				}
				hw_wifi.state = state;
				hw_wifi.enabled = (state == WIFI_DISCONNECTED ? 0 : 1);
				hw_changes |= HW_WIFI;
				dbus_send_wakeup(_WAKEUP_REFRESH_);
			}
			// now get rid of this signal
			// this signal is sent in response to the state_req method used to receive the initial wifi state
			dbus_bus_remove_match(signal_bus, "path='/com/nokia/icd2',interface='com.nokia.icd2',member='state_sig'", &dbus_error);
			hw_initialized &= ~HWIS_WIFI_STATE;
		}
		return DBUS_HANDLER_RESULT_HANDLED;

	////////////////////
	// USB
//	} else if(dbus_message_is_signal(message, "org.freedesktop.Hal.Device", "PropertyModified")){

		// handled above

	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
#endif

//////////////////////////////////////////////////////////////////////////

#ifdef DEBUG_UNHANDLED_SIGNALS
static DBusHandlerResult debug_signal_filter( DBusConnection *connection, DBusMessage *message, void *user_data ){
	switch(dbus_message_get_type(message)){
	case DBUS_MESSAGE_TYPE_METHOD_CALL: printf("dbus: received message type = method_call\n"); break;
	case DBUS_MESSAGE_TYPE_METHOD_RETURN: printf("dbus: received message type = method_return\n"); break;
	case DBUS_MESSAGE_TYPE_ERROR: printf("dbus: received message type = error\n"); break;
	case DBUS_MESSAGE_TYPE_SIGNAL: printf("dbus: received message type = signal\n"); break;
	default: printf("dbus: received message type = UNKNOWN\n"); return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	printf("\tsender = %s\n", dbus_message_get_sender(message));
	printf("\tserial = 0x%X\n", dbus_message_get_serial(message)); // send this back if replying to this message
	printf("\tdestination = %s\n", dbus_message_get_destination(message));
	printf("\tpath = %s\n", dbus_message_get_path(message));
	printf("\tinterface = %s\n", dbus_message_get_interface(message));
	printf("\tmember = %s\n", dbus_message_get_member(message));
	printf("\twants reply? = %s\n", (dbus_message_get_no_reply(message) ? "no" : "yes"));
	printf("\treply serial = 0x%X\n", dbus_message_get_reply_serial(message)); // will be the serial sent to the service replying right now
	printf("\tsignature = %s\n", dbus_message_get_signature(message));

	return DBUS_HANDLER_RESULT_HANDLED;
/*
	if(dbus_message_is_signal(message, DBUS_INTERFACE_LOCAL, "Disconnected")){
		printf("dbus: \t--> disconnected\n");
		return DBUS_HANDLER_RESULT_HANDLED;
	}
*/
}
#endif

//////////////////////////////////////////////////////////////////////////

#define	PARENT_READ  threadpipe[0]
#define	CHILD_WRITE  threadpipe[1]
static int threadpipe[2] = {-1,-1};

static void *dbus_thread( void *arg ){
	debug_write("[DBUS] thread starting...");

	#ifdef ARCH_armel
		dbus_wakeup_fd = CHILD_WRITE;

		// listen for flush signals from self
		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/qzx/asui',interface='com.qzx.asui'", &dbus_error);
//		dbus_bus_add_match(signal_bus, "type='method_call',path='/com/nokia/system_ui/request',interface='com.nokia.system_ui.request'", &dbus_error);

		// power button signal
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/computer_logicaldev_input_0',"	// n810
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='power'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/computer_logicaldev_input',"		// n800
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='power'", &dbus_error);
		// headphone jack
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/platform_headphone',"
										"interface='org.freedesktop.Hal.Device',member='Condition',arg0='ButtonPressed',arg1='connection'", &dbus_error);

		// mce signals (lock button, display status and devicelock)
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/mce/signal',interface='com.nokia.mce.signal'", &dbus_error);

		// bme signals (battery status)
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/bme',interface='org.freedesktop.Hal.Device',member='PropertyModified'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/bme/signal',interface='com.nokia.bme.signal'", &dbus_error);

		// bluez signals (bluetooth status)
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='ModeChanged'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='NameChanged'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='RemoteDeviceConnected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/bluez/hci0',interface='org.bluez.Adapter',member='RemoteDeviceDisconnected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',interface='org.bluez.audio.Headset',member='Connected'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',interface='org.bluez.audio.Headset',member='Disconnected'", &dbus_error);

		// icd signals (wifi status)
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/wlancond/signal',interface='com.nokia.wlancond.signal'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/icd',interface='com.nokia.icd',member='status_changed'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/icd2',interface='com.nokia.icd2',member='state_sig'", &dbus_error);
		dbus_bus_add_match(signal_bus, "type='signal',path='/com/nokia/icd2',interface='com.nokia.icd2',member='statistics_sig'", &dbus_error);

		// usb signals
		dbus_bus_add_match(signal_bus, "type='signal',path='/org/freedesktop/Hal/devices/usb_device_0_0_musb_hdrc',"
										"interface='org.freedesktop.Hal.Device',member='PropertyModified'", &dbus_error);

		dbus_connection_add_filter(signal_bus, signal_filter, NULL, NULL);
		#ifdef DEBUG_UNHANDLED_SIGNALS
		dbus_connection_add_filter(signal_bus, debug_signal_filter, NULL, NULL);
		#endif
		debug_write("[DBUS] filters ready");
		while(dbus_connection_read_write_dispatch(signal_bus, -1));
		debug_write("[DBUS] thread terminated");

		dbus_listening = 0; // inform main loop that signal bus was closed
		dbus_connection_close(signal_bus);
		close(dbus_wakeup_fd);
	#else
		while(1) sleep(10);
	#endif
	return NULL;
}

//////////////////////////////////////////////////////////////////////////

int dbus_init( ){
	#ifdef ARCH_armel
	original_screen_autolock = original_screen_stayslit = 0;

	dbus_threads_init_default();
	dbus_error_init(&dbus_error);
	if((signal_bus = dbus_bus_get_private(DBUS_BUS_SYSTEM, &dbus_error)) == NULL){
		error_write("can't connect to system signal bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}
	if((system_bus = dbus_bus_get_private(DBUS_BUS_SYSTEM, &dbus_error)) == NULL){
		error_write("can't connect to system method bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}
	if((session_bus = dbus_bus_get_private(DBUS_BUS_SESSION, &dbus_error)) == NULL){
		error_write("can't connect to session method bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}

	if(dbus_bus_request_name(signal_bus, "com.qzx.asui", DBUS_NAME_FLAG_REPLACE_EXISTING, &dbus_error) == -1){
		error_write("dbus won't let me register as com.qzx.asui: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
	}
//	if(dbus_bus_request_name(signal_bus, "com.nokia.system_ui", DBUS_NAME_FLAG_REPLACE_EXISTING, &dbus_error) == -1){
//		error_write("dbus won't let me register as com.nokia.system_ui: %s", dbus_error.message);
//		dbus_error_free(&dbus_error);
//	}
	#endif

	if(pipe(threadpipe) < 0){
		error_write("can't open dbus thread pipe");
		return 0;
	}

	dbus_listening = 1;
	pthread_create(&thread, 0, dbus_thread, NULL);

	return PARENT_READ;
}

void dbus_disconnect( ){
	#ifdef ARCH_armel
	dbus_connection_close(system_bus);
	dbus_connection_close(session_bus);
	#endif
}

////////////////////////////////////////////////////////////////////////// SEND METHOD

void dbus_send_method( const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	#ifdef DEBUG_SIGNALS
	printf("dbus: send SYSTEM %s.%s(%s)\n", interface, method, (first_arg_type == _END_ ? "" : "..."));
	#endif

	message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}

void dbus_send_signal( const char *path, const char *interface, const char *signal, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	#ifdef DEBUG_SIGNALS
	printf("dbus: send SYSTEM SIGNAL %s.%s(%s)\n", interface, signal, (first_arg_type == _END_ ? "" : "..."));
	#endif

	message = dbus_message_new_signal(path, interface, signal);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(system_bus, message, NULL) != FALSE)
		dbus_connection_flush(system_bus);
	dbus_message_unref(message);
}

////////////////////////////////////////////////////////////////////////// SEND METHOD w/ RETURN

void dbus_recv_method( DBusPendingCallNotifyFunction callback, const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;
	DBusPendingCall *pc;

	#ifdef DEBUG_SIGNALS
	printf("dbus: recv SYSTEM %s.%s(%s)\n", interface, method, (first_arg_type == _END_ ? "" : "..."));
	#endif

	message = dbus_message_new_method_call(destination, path, interface, method);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	dbus_connection_send_with_reply(signal_bus, message, &pc, 5000); // 5 second timeout
	if(pc)
		dbus_pending_call_set_notify(pc, callback, NULL, NULL);
	dbus_message_unref(message);
}

void dbus_flush_messages( ){
	#ifdef ARCH_armel
	dbus_send_method("com.qzx.asui", "/com/qzx/asui", "com.qzx.asui", "flush_calls", _END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// SEND SESSION METHOD

void dbus_send_session_method( const char *destination, const char *path, const char *interface, const char *method, int first_arg_type, ... ){
	DBusMessage *message;
	va_list ap;

	#ifdef DEBUG_SIGNALS
	printf("dbus: send SESSION %s.%s(%s)\n", interface, method, (first_arg_type == _END_ ? "" : "..."));
	#endif

	message = dbus_message_new_method_call(destination, path, interface, method);
	dbus_message_set_no_reply(message, TRUE);
	if(first_arg_type != _END_){
		va_start(ap, first_arg_type);
		dbus_message_append_args_valist(message, first_arg_type, ap);
		va_end(ap);
	}
	if(dbus_connection_send(session_bus, message, NULL) != FALSE)
		dbus_connection_flush(session_bus);
	dbus_message_unref(message);
}

////////////////////////////////////////////////////////////////////////// NOTIFICATIONS

void notify_screen_locked( ){
	#ifdef ARCH_armel
	char *locked_message = "Touch screen and keys locked";
	dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_, _STRING_, &locked_message, _END_);
	#endif
}

void notify_screen_unlocked( ){
	#ifdef ARCH_armel
	char *unlocked_message = "Touch screen and keys active";
	dbus_send_session_method(_notification_service_, _notification_path_, _notification_interface_, _notification_method_, _STRING_, &unlocked_message, _END_);
	#endif
}

////////////////////////////////////////////////////////////////////////// STATUSBAR APPLET SIGNALS

void dbus_send_charging_status_signal( ){
	unsigned status = (unsigned)hw_battery.status;
	dbus_send_signal("/com/qzx/asui", "com.qzx.asui", "charging_status", _UINT32_, &status, _END_);
}

void dbus_send_battery_capacity_signal( ){
	unsigned capacity = (unsigned)roundf(hw_battery.capacity);
	dbus_send_signal("/com/qzx/asui", "com.qzx.asui", "battery_capacity", _UINT32_, &capacity, _END_);
}
