# -*- coding: utf-8 -*-
'''
Advanced Power Config
2008-2009(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

from locale import Locale

try: #most distros have a separate gconf
  import gconf
except: #but Maemo doesn't
  from gnome import gconf

class Config():

  settings_path = '/apps/advanced-power/settings'
  plugins_path = '/apps/advanced-power/plugins'
  translators_path = '/apps/advanced-power/translators'

  def __init__(self):
    self.config = {'settings': {}, 'plugins': {}, 'translators': {}}
    self.gconf_client = gconf.client_get_default()
    self.gconf_set = {'bool': self.gconf_client.set_bool,
                      'int': self.gconf_client.set_int,
                      'str': self.gconf_client.set_string
                     }
    self.load()

  def sort_plugins_by_pos(self):
    return sorted(self.config['plugins'].iteritems(), key=lambda(k,v): (v['position'],k))

  def save(self):
    self.save_settings()
    self.save_plugins()

  def save_settings(self):
    settings = self.config['settings']
    for key in settings:
      self.save_settings_key(key)

  def get_settings_keys(self):
    return self.config['settings'].keys()

  def get_settings_values(self):
    return self.config['settings'].values()

  def get_settings_key(self, key):
    try:
      return self.config['settings'][key]['type'], self.config['settings'][key]['value']
    except KeyError:
      return None, None

  def get_settings_value(self, key):
    try:
      return self.config['settings'][key]['value']
    except KeyError:
      return None

  def add_settings_key(self, key_name, key_type=None, key_value=None):
    if not key_name in self.config['settings']:
      self.config['settings'][key_name] = {'type': key_type, 'value': key_value}
    return self.config['settings'][key_name]

  def set_settings_value(self, key, new_value, callback=None):
    try:
      self.config['settings'][key]['value'] = new_value
    except KeyError:
      key_type = type(new_value)
      ctype = None
      if (key_type == str):
        ctype = 'str'
      elif (key_type == bool):
        ctype = 'bool'
      elif (key_type == int):
        ctype = 'int'
      self.config['settings'][key_name] = {'type': ctype, 'value': new_value}
    if callback:
      callback(new_value)

  def save_settings_key(self, key, callback=None):
    try:
      self.gconf_set[self.config['settings'][key]['type']](('%s/%s' % (self.settings_path, key)), self.config['settings'][key]['value'])
      if callback:
        callback(new_value)
    except KeyError:
      pass

  def save_plugins(self):
    plugins = self.config['plugins']
    for key in plugins:
      self.save_plugins_key(key)

  def get_plugins_keys(self):
    return self.config['plugins'].keys()

  def get_pugings_values(self):
    return self.config['plugins'].values()

  def get_plugins_key(self, key_name):
    try:
      return self.config['plugins'][key_name]['text'], self.config['plugins'][key_name]['value'], self.config['plugins'][key_name]['position']
    except KeyError:
      return key_name, None, None

  def get_plugins_value(self, key_name):
    try:
      return self.config['plugins'][key_name]['value']
    except KeyError:
      return None

  def add_plugins_key(self, key_name, text=None, value=None, position=0):
    if not key_name in self.config['plugins']:
      self.config['plugins'][key_name] = {'text': Locale.get_text(text), 'value': value, 'position': position}
    return self.config['plugins'][key_name]

  def set_plugins_key(self, key, new_text, new_value, new_position=0, callback=None):
    try:
      self.config['plugins'][key].clear()
    except KeyError:
      pass
    self.config['plugins'][key] = {'text': Locale.get_text(new_text), 'value': new_value, 'position': new_position}
    if callback:
      callback(key)

  def save_plugins_key(self, key, callback=None):
    try:
      plugins = self.config['plugins']
      self.gconf_set['bool']('%s/%s/value' % (self.plugins_path, key), plugins[key]['value'])
      self.gconf_set['int']('%s/%s/position' % (self.plugins_path, key), plugins[key]['position'])
      if callback:
        callback(key)
    except KeyError:
      pass

  def get_translators_keys(self):
    return self.config['translators'].keys()

  def get_translators_values(self):
    return self.config['translators'].values()

  def load(self):
    self.load_settings()
    self.load_plugins()
    self.load_translators()

  def load_settings(self):
    if not self.gconf_client:
      return
    self.config['settings'].clear()
    gconf_entries = self.gconf_client.all_entries(self.settings_path)
    for entry in gconf_entries:
      name = entry.get_key().replace(('%s/' % self.settings_path), '', 1)
      e_value = entry.get_value()
      e_type = e_value.type
      value = None
      ctype = None
      if (e_type == gconf.VALUE_STRING):
        ctype = 'str'
        value = e_value.get_string()
      elif (e_type == gconf.VALUE_BOOL):
        ctype = 'bool'
        value = e_value.get_bool()
      elif (e_type == gconf.VALUE_INT):
        ctype = 'int'
        value = e_value.get_int()
      self.add_settings_key(name, ctype, value)

  def load_plugins(self):
    if not self.gconf_client:
      return
    self.config['plugins'].clear()
    for path in self.gconf_client.all_dirs(self.plugins_path):
      name = path.replace('%s/' % self.plugins_path, '', 1)
      text = self.gconf_client.get_string('%s/text' % path)
      value = self.gconf_client.get_bool('%s/value' % path)
      position = self.gconf_client.get_int('%s/position' % path)
      if text and value != None and position != None:
        self.add_plugins_key(name, text, value, position)

  def load_translators(self):
    if not self.gconf_client:
      return
    self.config['translators'].clear()
    for entry in self.gconf_client.all_entries(self.translators_path):
      name = entry.get_key().replace('%s/' % self.translators_path, '', 1)
      value = entry.get_value().get_string()
      self.config['translators'][name] = {'text': Locale.get(name), 'authors': value}
