/*
 * Copyright (C) 2008 Adam Harwell, all rights reserved.
 *
 * Contact: Adam Harwell <adam.harwell@trinity.edu>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib-object.h>

#include <libhildondesktop/libhildondesktop.h>
#include <libhildondesktop/hildon-desktop-popup-window.h>
#include <libhildondesktop/hildon-status-bar-item.h>
#include <hildon/hildon.h>
#include <hildon/hildon-controlbar.h>
#include <hildon/hildon-hvolumebar.h>
#include <hildon/hildon-sound.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <dirent.h>
#include <time.h>
#include <sys/time.h>
#include <syslog.h>

#include <gconf/gconf-client.h>
#define DEFAULT_BEEP "/usr/share/sounds/ui-default_beep.wav"
#define GKEY "/apps/osso/sound"
#define GVOLUME "/apps/osso/sound/master_volume"
#define true 1
#define false 0

#include "advanced-backlight.h"

/* Utility macro which defines the plugin dynamic type bits */
HD_DEFINE_PLUGIN (AdvancedBacklightPlugin, advanced_backlight_plugin, STATUSBAR_TYPE_ITEM)

static void platform_check();
static void pressed(GtkToggleButton *button);
static int window_init();
static void update_icon();
static void set_level_dsme(GtkRange *range, gpointer user_data);
static void set_level_dsme_worker(int level);
void popup_place(HildonDesktopPopupWindow *menu, gint *x, gint *y, gpointer user_data);
static void set_volume_gconf(HildonVolumebar *localvbar, gpointer user_data);
static void mute_volume_gconf(HildonVolumebar *localvbar, gpointer user_data);
static int get_volume_gconf();
static void load_settings();
static void save_settings();
static void gconf_notify_func(GConfClient * gclient, guint cnxn_id, GConfEntry * entry, gpointer user_data);
static void configure_applet(GtkMenuItem *item, gpointer data);
static gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event, void * dummy);
static void fill_theme_list(GtkWidget* combobox);

/////////////////////////////////////////////////////////////////////////////////////////////////////
//// I'm new to GTK so I'm not sure how to pass certain things around... So they're global. Ick. ////
/////////////////////////////////////////////////////////////////////////////////////////////////////
GtkWidget* window = NULL;
int lastvalue = 100;
GtkWidget *button = NULL;
GConfClient * ggclient = NULL;
//GtkIconTheme *icon_theme;
int allow_zero = 0;
AdvancedBacklightPlugin* my_data = NULL;
char current_theme[30];
GtkWidget* cbar = NULL;
GtkWidget* vbar = NULL;
int n810 = 0;
struct timeval last_beep;
int brightness_jump = 10;
int volume_jump = 10;
int volume_beep_bool = 0;
int debug = true;

char oldfile[130]; // To keep the statusbar icon up to date, I keep track of the current icon (oldfile)
                  // so I can update only when it is different (no flickering icon).

int popstate = 0;

/////////////////////////////////////////////////////////////////////
//// This is the initialization function. It is called on entry. ////
/////////////////////////////////////////////////////////////////////
static void advanced_backlight_plugin_init(AdvancedBacklightPlugin *statusbar_plugin)
{
	my_data = statusbar_plugin;
	platform_check(); // See if we are on an n810
	ggclient = gconf_client_get_default();
	load_settings();
	gettimeofday(&last_beep,NULL);
	//icon_theme = gtk_icon_theme_get_default ();

	//FILE* myfile = fopen("/sys/devices/platform/omapfb/panel/backlight_level","r");
	//fscanf(myfile,"%d",&lastvalue); // A failed attempt to get the current backlight level. Gets only the software level...
	//fclose(myfile);

	button = gtk_button_new(); // The statusbar icon...

	gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_file("/usr/share/icons/hicolor/40x40/hildon/qgn_stat_displaybright4.png"));
	gtk_widget_show_all(button);
	gtk_container_add(GTK_CONTAINER(statusbar_plugin), button); // Shove the button into the statusbar


	if (!window_init())
		exit(EXIT_FAILURE);
	g_signal_connect(button, "button-press-event", G_CALLBACK(pressed),NULL); // On click it will call the function "pressed()".
	syslog(LOG_DEBUG, "adv-backlight initialized.");
}

static void advanced_backlight_plugin_finalize(GObject *object)
{
	AdvancedBacklightPlugin *statusbar_plugin=ADVANCED_BACKLIGHT_PLUGIN(object);

	save_settings();
	//g_source_remove(statusbar_plugin->eventID);
	G_OBJECT_CLASS(g_type_class_peek_parent(G_OBJECT_GET_CLASS(object)))->finalize(object);
	g_free(statusbar_plugin);
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight finalized.");
}

//////////////////////////////////////////////////
//// Check to see if we are on an n810 or not ////
//////////////////////////////////////////////////
static void platform_check()
{
	char version;

	FILE * fp = fopen("/proc/component_version","r");
	fseek(fp,sizeof(char)*15,0); // The important character is the 16th in the file.
	fread(&version,sizeof(char),1,fp);
	fclose(fp);

	if (version == '4') // 4 means n810, 3 means n800.
	{
		n810 = 1;
		allow_zero = 1;
		if (debug)
			syslog(LOG_DEBUG, "adv-backlight running on n810 hardware.");
	}
	else // Err on the side of caution.
	{
		n810 = 0;
		allow_zero = 0;
		if (debug)
			syslog(LOG_DEBUG, "adv-backlight running on n800 hardware.");
	}
	//return n810;
}

static void popstatedown(GtkWidget *window)
{
	popstate = 0;
}

static void popstateup(GtkWidget *window)
{
	popstate = 1;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//// Called on statusbar icon click. Pops a whole window at the moment, which is less than ideal ////
/////////////////////////////////////////////////////////////////////////////////////////////////////
static void pressed(GtkToggleButton *btn)
{
	if (popstate) // If the window is already initialized, we want to kill it.
	{
		hildon_desktop_popup_window_popdown(HILDON_DESKTOP_POPUP_WINDOW(window));
	}
	else
	{
		hildon_desktop_popup_window_popup(HILDON_DESKTOP_POPUP_WINDOW(window), popup_place, btn, 2);
	}
}

static int window_init()
{
	if (window == NULL)
	{
		window = hildon_desktop_popup_window_new(0, GTK_ORIENTATION_VERTICAL, HD_POPUP_WINDOW_DIRECTION_RIGHT_BOTTOM);
		
		//gdk_window_set_type_hint(GDK_WINDOW(window), GDK_WINDOW_TYPE_HINT_MENU);
		//gtk_window_set_resizable(GTK_WINDOW(window),0);
		//gtk_window_set_decorated(GTK_WINDOW(window),0);
		//gtk_container_set_border_width (GTK_CONTAINER (window), 20);
		//gdk_window_set_decorations(GDK_WINDOW(window),GDK_DECOR_BORDER);
		//gdk_window_set_keep_above(GDK_WINDOW(window),1);
		
		//GtkWidget *cbar = hildon_controlbar_new();
		cbar = hildon_controlbar_new();
		vbar = hildon_hvolumebar_new();

		gconf_client_add_dir(ggclient, GKEY, GCONF_CLIENT_PRELOAD_NONE, NULL);
		gconf_client_notify_add(ggclient, GVOLUME, gconf_notify_func, vbar, NULL, NULL);

		double volume_from_gconf = get_volume_gconf(ggclient);
		if (volume_from_gconf < 0.0)
			hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar), 1);
		hildon_volumebar_set_level(HILDON_VOLUMEBAR(vbar), volume_from_gconf);
		
		if (allow_zero)	// Only let n810 set brightness 0, since they have a transflexive screen
			hildon_controlbar_set_range(HILDON_CONTROLBAR (cbar), 0, 127);
		else		// An n800 user would lose their backlight entirely and be confused
			hildon_controlbar_set_range(HILDON_CONTROLBAR (cbar), 1, 127);

		hildon_controlbar_set_value(HILDON_CONTROLBAR(cbar), lastvalue); // Return the controlbar to the last specified value.
		gtk_range_set_update_policy(GTK_RANGE(cbar), GTK_UPDATE_DELAYED);

		update_icon();

		GtkWidget* vbox = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), cbar, 0,0,25);
		gtk_box_pack_start(GTK_BOX(vbox), vbar, 0,0,5);
		//gtk_widget_show(cbar);
		//gtk_widget_show(vbar);

		/* Start config button code */
		GtkWidget* hbar = gtk_hseparator_new();
		gtk_box_pack_start(GTK_BOX(vbox), hbar, 0,0,5);
		//gtk_widget_show(hbar);

		GtkWidget* configmenu = gtk_button_new_with_label("Settings");
		gtk_box_pack_start(GTK_BOX(vbox), configmenu, 0,0,5);
		//gtk_widget_show(configmenu);
		g_signal_connect(configmenu, "clicked", G_CALLBACK(configure_applet),my_data);
		/* End config button code */

		GtkWidget* hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), vbox, 0,0,25);
		//gtk_widget_show(vbox);
		//gtk_widget_show(hbox);

		GtkWidget * myframe = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(myframe), GTK_SHADOW_OUT);
		//gtk_widget_show(myframe);
		gtk_container_add(GTK_CONTAINER(myframe), hbox);

		gtk_container_add(GTK_CONTAINER(window), myframe);
		gtk_widget_show_all(myframe);
		// Event bindings:
		g_signal_connect(cbar, "value-changed", G_CALLBACK(set_level_dsme), NULL);
		g_signal_connect(window, "popdown-window", G_CALLBACK(popstatedown), NULL);
		g_signal_connect(window, "popup-window", G_CALLBACK(popstateup), NULL);
		g_signal_connect (vbar, "mute_toggled", G_CALLBACK(mute_volume_gconf), NULL);
		g_signal_connect (vbar, "level_changed", G_CALLBACK(set_volume_gconf), NULL);
		g_signal_connect(G_OBJECT(window), "key_press_event", G_CALLBACK(key_press_cb), NULL);
		
		if (debug)
			syslog(LOG_DEBUG, "adv-backlight main window set up complete.");
	}

	if (window == NULL)
		return FALSE;
	return TRUE;
}

static void do_volume_beep()
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight performing volume beep.");

	struct timeval current_time;
	gettimeofday(&current_time, NULL);
	
	long time_since_beep = (current_time.tv_sec - last_beep.tv_sec) * 1000000 + (current_time.tv_usec - last_beep.tv_usec);
	
	if (time_since_beep > 500000)
	{
		hildon_play_system_sound(DEFAULT_BEEP);
		last_beep = current_time;
	}

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight volume beep performed.");
}

static void set_volume_gconf(HildonVolumebar *localvbar, gpointer user_data)
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight setting volume.");

	if (ggclient != NULL)
		gconf_client_set_int(ggclient, GVOLUME, (int)hildon_volumebar_get_level(localvbar), NULL);
	update_icon();
	if (volume_beep_bool)
		do_volume_beep();

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight volume set.");
}

static void mute_volume_gconf(HildonVolumebar *localvbar, gpointer user_data)
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight toggling mute.");

	if (ggclient != NULL)
	{
		if (hildon_volumebar_get_mute(localvbar))
			gconf_client_set_int(ggclient, GVOLUME, -1 * (int)hildon_volumebar_get_level(localvbar), NULL);
		else if (hildon_volumebar_get_mute(localvbar) == 0)
			gconf_client_set_int(ggclient, GVOLUME, abs((int)hildon_volumebar_get_level(localvbar)), NULL);
		update_icon();
	}

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight mute toggled.");
}

static int get_volume_gconf()
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight getting volume.");

	int ret = 0;
	if (ggclient != NULL)
		ret = gconf_value_get_int(gconf_client_get(ggclient, GVOLUME, NULL));

	return ret;
}

static void load_settings() {
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight loading settings.");

	GConfValue* conf_value;
	
	//Brightness level
	conf_value = gconf_client_get(ggclient, "/apps/osso/display/brightness", NULL);
	if (conf_value)
	{
		lastvalue = gconf_value_get_int(conf_value);
		set_level_dsme_worker(lastvalue*2+1);
	}
	else
		lastvalue = 100;

	//Brightness allow zero
	conf_value = gconf_client_get(ggclient, "/apps/osso/display/allow_zero", NULL);
	if (conf_value)
		allow_zero = gconf_value_get_bool(conf_value);

	//Selected theme
	conf_value = gconf_client_get(ggclient,"/apps/osso/display/theme", NULL);
	if (conf_value)
		strncpy(current_theme,gconf_value_get_string(conf_value),30);
	else
		strncpy(current_theme,"Default",30);

	//Brightness jump
	conf_value = gconf_client_get(ggclient, "/apps/osso/display/brightness_jump", NULL);
	if (conf_value)
		brightness_jump = gconf_value_get_int(conf_value);
	else
		brightness_jump = 10;
	
	//Volume jump
	conf_value = gconf_client_get(ggclient, "/apps/osso/sound/volume_jump", NULL);
	if (conf_value)
		volume_jump = gconf_value_get_int(conf_value);
	else
		volume_jump = 20;

	//Volume beep
	conf_value = gconf_client_get(ggclient, "/apps/osso/sound/volume_beep", NULL);
	if (conf_value)
		volume_beep_bool = gconf_value_get_bool(conf_value);
	else
		volume_beep_bool = true;

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight settings loaded.");
}

static void save_settings() {
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight saving settings.");

	//Brightness level
	gconf_client_set_int(ggclient, "/apps/osso/display/brightness", lastvalue, NULL);
	
	//Brightness allow zero
	gconf_client_set_bool(ggclient, "/apps/osso/display/allow_zero", allow_zero, NULL);

	//Selected theme
	gconf_client_set_string(ggclient, "/apps/osso/display/theme", current_theme, NULL);

	//Brightness jump
	gconf_client_set_int(ggclient, "/apps/osso/display/brightness_jump", brightness_jump, NULL);

	//Volume jump
	gconf_client_set_int(ggclient, "/apps/osso/sound/volume_jump", volume_jump, NULL);

	//Volume beep
	gconf_client_set_bool(ggclient, "/apps/osso/sound/volume_beep", volume_beep_bool, NULL);

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight settings saved.");
}

static void gconf_notify_func(GConfClient * gclient, guint cnxn_id, GConfEntry * entry, gpointer vbar)
{
	double volume_from_gconf;
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight gconf notified.");

	if (vbar != NULL) {
		volume_from_gconf = get_volume_gconf();
		if (volume_from_gconf >= 0)
		{
			//if (hildon_volumebar_get_mute(vbar) == 0)
				hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar), 0);
				hildon_volumebar_set_level(HILDON_VOLUMEBAR(vbar), volume_from_gconf);
			//else
		}
		else if (hildon_volumebar_get_mute(vbar) == 0)
			hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar), 1);
	}
	//hildon_play_system_sound("/usr/share/sounds/ui-default_beep.wav");
}

static void configure_applet(GtkMenuItem *item, gpointer data) {
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight changing settings.");
	
	//AdvancedBacklightPlugin *statusbar_plugin=(AdvancedBacklightPlugin*)data;
	GtkWidget *dialog;
	GtkWidget *dialog2;
	gint response;
	gint response2;
	GtkWidget *vbox;
	GtkWidget *vbox2;
	GtkWidget *allow_zero_checkbox;
	GtkWidget *theme_caption;
	GtkWidget *brightness_spinner;
	GtkWidget *volume_spinner;
	GtkWidget *brightness_caption;
	GtkWidget *volume_caption;
	GtkWidget *choose_theme;
	GtkWidget *verifytext;
	GtkWidget *volume_beep_checkbox;

	/*
	GtkTreeModel *child_model;
	GtkTreeModel *sort_model;
	sort_model1 = gtk_tree_model_sort_new_with_model (child_model);
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (sort_model), COLUMN_1, GTK_SORT_ASCENDING);
	*/
	//char** themelist;
	
	dialog=gtk_dialog_new_with_buttons("Configure Advanced Backlight",NULL,GTK_DIALOG_MODAL,GTK_STOCK_CANCEL,GTK_RESPONSE_REJECT,GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,NULL);
	vbox=gtk_vbox_new(FALSE,0);

	allow_zero_checkbox=gtk_check_button_new_with_label("Allow zero backlight\n(N800 users beware!)");
	if (allow_zero)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(allow_zero_checkbox),TRUE);

	volume_beep_checkbox=gtk_check_button_new_with_label("Beep on volume change");
	if (volume_beep_bool)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(volume_beep_checkbox),TRUE);

	choose_theme = gtk_combo_box_new_text();
	//gtk_combo_box_set_model(GTK_COMBO_BOX(choose_theme),sort_model);
	
	//themelist = get_theme_list();
	fill_theme_list(choose_theme);

	//gtk_combo_box_append_text(GTK_COMBO_BOX(choose_theme),"Default");
	//if (g_strcasecmp(current_theme,"Default")==0)
	//	gtk_combo_box_set_active(GTK_COMBO_BOX(choose_theme),0);
	
	theme_caption = hildon_caption_new(NULL,"Choose theme",choose_theme,NULL,HILDON_CAPTION_MANDATORY);

	//brightness_spinner = gtk_spin_button_new_with_range (1,255,1);	
	brightness_spinner = hildon_number_editor_new(1,255);
	hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(brightness_spinner), brightness_jump);
	brightness_caption = hildon_caption_new(NULL,"Brightness interval",brightness_spinner,NULL,HILDON_CAPTION_MANDATORY);

	volume_spinner = hildon_number_editor_new(1,100);
	hildon_number_editor_set_value(HILDON_NUMBER_EDITOR(volume_spinner), volume_jump);
	volume_caption = hildon_caption_new(NULL,"Volume interval",volume_spinner,NULL,HILDON_CAPTION_MANDATORY);

	gtk_box_pack_start(GTK_BOX(vbox),theme_caption,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),brightness_caption,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),volume_caption,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),volume_beep_checkbox,FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),allow_zero_checkbox,FALSE,FALSE,0);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox),vbox);

	gtk_widget_show_all(dialog);
	response=gtk_dialog_run(GTK_DIALOG(dialog));

	if (response==GTK_RESPONSE_ACCEPT)
	{
		allow_zero=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(allow_zero_checkbox));
		if (!n810 && allow_zero) {
			dialog2=gtk_dialog_new_with_buttons("Warning!",NULL,GTK_DIALOG_MODAL,GTK_STOCK_CANCEL,GTK_RESPONSE_REJECT,GTK_STOCK_OK,GTK_RESPONSE_ACCEPT,NULL);
			verifytext = gtk_label_new("Are you sure you want to allow the backlight to turn off?\nIf the backlight turns off, it may be very difficult to turn it\nback on, as the screen will no longer be visible.");

			vbox2=gtk_vbox_new(FALSE,0);
			gtk_box_pack_start(GTK_BOX(vbox2),verifytext,FALSE,FALSE,0);
			gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog2)->vbox),vbox2);
			gtk_widget_show_all(dialog2);
			response2=gtk_dialog_run(GTK_DIALOG(dialog2));

			if (response2!=GTK_RESPONSE_ACCEPT)
				allow_zero=0;
		}

		if (allow_zero)
			hildon_controlbar_set_range(HILDON_CONTROLBAR (cbar), 0, 127);
		else
			hildon_controlbar_set_range(HILDON_CONTROLBAR (cbar), 1, 127);

		brightness_jump = hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(brightness_spinner));
		volume_jump = hildon_number_editor_get_value(HILDON_NUMBER_EDITOR(volume_spinner));
		volume_beep_bool=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(volume_beep_checkbox));

		strncpy(current_theme,gtk_combo_box_get_active_text(GTK_COMBO_BOX(choose_theme)),30);
		update_icon();
	}
	gtk_widget_destroy(dialog2);
	gtk_widget_destroy(dialog);
	save_settings();

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight done with settings.");
}

static void fill_theme_list(GtkWidget *combobox) 
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight filling theme list.");
	
	DIR *d;
	struct dirent *de;
	int count = 0;
	d = opendir("/usr/share/icons/hicolor/40x40/apps/adv-backlight");

	if (d)
	{
		while ((de = readdir(d)) != NULL) {
			if (g_strcasecmp(de->d_name,".")!=0 && g_strcasecmp(de->d_name,"..")!=0)
			{
				gtk_combo_box_append_text(GTK_COMBO_BOX(combobox),de->d_name);
				if (g_strcasecmp(current_theme,de->d_name)==0)
					gtk_combo_box_set_active(GTK_COMBO_BOX(combobox),count);
				count++;
			}
		}
	}
	closedir(d);

	if (debug)
		syslog(LOG_DEBUG, "adv-backlight theme list filled.");
}
/*
char** get_theme_list() {
	int count = 0;
	DIR d*;
	dirent de*;

	d = opendir("/usr/share/icons/hicolor/40x40/apps/adv-backlight");
	if (d)
	{
		while ((de = readdir(d)) != NULL) {
			count++;
		}
	}
	closedir(d);

	char* themes[count];
	char* temp;
	count=0;

	d = opendir("/usr/share/icons/hicolor/40x40/apps/adv-backlight");
	if (d)
	{
		while ((de = readdir(d)) != NULL) {
			themes[count++] = strdup(de->d_name);
		}
	}
	closedir(d);

	return themes;
}
*/

static int get_light_sensor() {
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight getting light sensor value.");

	FILE * fp;
	int value = -1;

	fp = fopen("/sys/devices/platform/i2c_omap.2/i2c-0/0-0029/lux","r");

	if (fp) {
		fscanf(fp,"%d",&value);
		fclose(fp);
	}

	return(value);
	
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight got light sensor value.");
}


/////////////////////////////////////////////////////////////////
//// Propose a statusbar icon change based on the new        ////
//// bright/vol level. Available images are 1-7 b, m/0-4 v.  ////
/////////////////////////////////////////////////////////////////
static void update_icon()
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight updating icon.");

	char newfile[130];
	/* Icon Theming Support - Coming Soon
	GdkPixbuf *pixbuf;
	GError *error = NULL;
	char iconname[26];
	*/
	int volume = get_volume_gconf();
	int brightness_num = (int)(round((double)lastvalue / 21.16) + 1);
	char volume_char;
	if (volume >= 0) {
		volume_char = (char)('0' + (int)(ceil((double)volume / 25.0)));
	}
	else {
		volume_char = 'm';
	}
	
	//if (strcmp(current_theme,"Default")==0)
		//sprintf(newfile,"/usr/share/icons/hicolor/40x40/apps/adv-backlight/adv-backlight-icon-%d.%c.png", brightness_num, volume_char);
	//else
		sprintf(newfile,"/usr/share/icons/hicolor/40x40/apps/adv-backlight/%s/adv-backlight-icon-%d.%c.png", current_theme, brightness_num, volume_char);
	//sprintf(iconname,"adv-backlight-icon-%d.%c", brightness_num, volume_char);

	if (strcmp(oldfile,newfile) != 0) { // If proposed icon is different than the current one, go ahead and change it.
		/* Icon Theming Support - Coming Soon
		pixbuf = gtk_icon_theme_load_icon (icon_theme, iconname, 40, 0, &error);

		if (!pixbuf) {
			g_warning ("Couldn't load icon: %s", error->message);
			g_error_free (error);
		}
		else {
			gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_pixbuf(pixbuf));
			g_object_unref (pixbuf);
		}
		*/
		
		gtk_button_set_image(GTK_BUTTON (button), gtk_image_new_from_file(newfile));
		strncpy(oldfile,newfile,90); // Keep track of the current icon after the change.
	}
	
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight icon updated.");
}
/////////////////////////////////////////////////////////
//// Does the actual work for setting the backlight. ////
/////////////////////////////////////////////////////////
static void set_level_dsme(GtkRange *range, gpointer user_data)
{
	if (debug)
		syslog(LOG_DEBUG, "adv-backlight setting backlight level.");

	gdouble myval = gtk_range_get_value(range); // Get the value out of the GtkRange.
	int myint = floor(myval); // We need an integer, not a double.
	lastvalue=myint; // Keep track of the last value set.
	myint = myint * 2 + 1;

	set_level_dsme_worker(myint);

	update_icon();
}

static void set_level_dsme_worker(int level) {
	char command[80] = "sudo /usr/sbin/chroot /mnt/initfs/ dsmetest -l ";

	char brightness[4];
	snprintf(brightness, 4, "%d", level);
	strncat(command, brightness, 4);
	g_spawn_command_line_async(command,NULL);
}

void popup_place(HildonDesktopPopupWindow *menu, gint *x, gint *y, gpointer user_data) {
	GtkRequisition req;
	GtkWidget *btn = GTK_WIDGET(user_data);
	gint sw;

	gtk_widget_size_request(GTK_WIDGET(menu), &req);
	sw = gdk_screen_get_width(gtk_widget_get_screen(btn));

	gdk_window_get_position(btn->window, x, y);

	*y += btn->allocation.y + btn->allocation.height + 10;
	*x += btn->allocation.x + 40;

	if (*x + req.width > sw) {
		*x -= req.width - btn->allocation.width;
	}
}

gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event, void * dummy)
{
    switch (event->keyval) {
    case GDK_Up: //Volume Up
		hildon_volumebar_set_level(HILDON_VOLUMEBAR(vbar),hildon_volumebar_get_level(HILDON_VOLUMEBAR(vbar))+volume_jump);
        return TRUE;

    case GDK_Down: //Volume Down
		hildon_volumebar_set_level(HILDON_VOLUMEBAR(vbar),hildon_volumebar_get_level(HILDON_VOLUMEBAR(vbar))-volume_jump);
        return TRUE;

    case GDK_Left: //Brightness down
		gtk_range_set_value(GTK_RANGE(cbar),gtk_range_get_value(GTK_RANGE(cbar))-brightness_jump);
        return TRUE;

    case GDK_Right: //Brightness up
		gtk_range_set_value(GTK_RANGE(cbar),gtk_range_get_value(GTK_RANGE(cbar))+brightness_jump);
        return TRUE;

    case GDK_Return: //Mute
		hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar),!hildon_volumebar_get_mute(HILDON_VOLUMEBAR(vbar)));
        return TRUE;

    case GDK_Escape: //Close
		hildon_desktop_popup_window_popdown(HILDON_DESKTOP_POPUP_WINDOW(window));
        return TRUE;
    }

    return FALSE;
}

////////////////////////////////////////////////////////////////////////////
//// This is necessary, according to the statusbar template I followed. ////
////////////////////////////////////////////////////////////////////////////
static void advanced_backlight_plugin_class_init(AdvancedBacklightPluginClass *class)
{
  GObjectClass *object_class=G_OBJECT_CLASS(class);
  object_class->finalize=advanced_backlight_plugin_finalize;
}
