/*
    advanced-backlight -- Status bar plugin for Hildon
    -----------------------------------------
    
    - Supports rotation, brightness and volume.
    - Resembles the closed source status bar style "hildon-status-bar-popup".

    (C) 2008 Jonas Hurrelmann <j@outpo.st>.
    (C) 2008 Adam Harwell <aharwell@trinity.edu>.
    
    Randr handling has been ported from gpe-conf
    (C) 2002 Moray Allan <moray@sermisy.org>, Pierre TARDY <tardyp@free.fr>
        2003-2007 Florian Boor <florian.boor@kernelconcepts.de>.

    Minor parts where derived from bt-plugin.c
    (C) 2005 Nokia Corporation, Tomas Junnonen <tomas.junnonen@nokia.com>, 
        Jari Tenhunen <jari.tenhunen@nokia.com>.
 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
    
*/

/* TODO: autoconf this */
#define XRANDR 1

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <glib/gasyncqueue.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <locale.h>
#include <hildon/hildon-hvolumebar.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon-banner.h>
#include <libhildonwm/hd-wm.h>
#include <libhildonwm/hd-wm-application.h>
#include <libhildondesktop/statusbar-item.h>
#include <libhildondesktop/libhildondesktop.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#ifdef XRANDR
#include <X11/extensions/Xrandr.h>
#endif
#include <dlfcn.h>

/* INCLUDES imported from adv-backlight */
#include <gconf/gconf-client.h>
#include <hildon/hildon-sound.h>
#include <dirent.h>
#include <time.h>
#include <hildon/hildon-controlbar.h>
#include <hildon/hildon.h>
#include <string.h>

#include "advanced-backlight.h"


#define _(text) dgettext(GETTEXT_PACKAGE, text)

#define NUM_PIXBUFS 7*5
#define HILDON_STATUS_BAR_BORDER_WIDTH 20
#define MENU_WINDOW_WIDTH 400

#define ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE(object) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((object), ADVANCED_BACKLIGHT_PLUGIN_TYPE, AdvancedBacklightPluginPrivate))

#define ICON_DIR "/usr/share/icons/hicolor/40x40/apps/adv-backlight"
#define NUM_BRIGHTNESS_ICONS 7
#define NUM_VOLUME_ICONS 6

#define GCONF_OSSO_SOUND_DIR "/apps/osso/sound"
#define GCONF_MASTER_VOLUME GCONF_OSSO_SOUND_DIR "/master_volume"

#define BRIGHTNESS_CHANGE_DELAY 300


#define GCONF_PREFIX "/apps/adv-backlight"
#define GCONF_KEY_BRIGHTNESS GCONF_PREFIX "/brightness"
#define GCONF_KEY_ALLOW_ZERO GCONF_PREFIX "/allow_zero"
#define GCONF_KEY_THEME GCONF_PREFIX "/theme"
#define GCONF_KEY_BRIGHTNESS_JUMP GCONF_PREFIX "/brightness_jump"
#define GCONF_KEY_VOLUME_JUMP GCONF_PREFIX "/volume_jump"
#define GCONF_KEY_VOLUME_BEEP GCONF_PREFIX "/volume_beep"


#define SYSFS_BRIGHTNESS "/sys/devices/platform/omapfb/panel/backlight_level"

#define DEFAULT_BEEP "/usr/share/sounds/ui-default_beep.wav"

#undef G_LOG_DOMAIN
#define G_LOG_DOMAIN "advanced-backlight"


/* Private data for the plugin */
typedef struct _AdvancedBacklightPluginPrivate AdvancedBacklightPluginPrivate;
struct _AdvancedBacklightPluginPrivate
{
  GdkPixbuf *pixbuf[NUM_BRIGHTNESS_ICONS][NUM_VOLUME_ICONS];
  GtkWidget *icon;
  GtkWidget *button;
  GtkWidget *menu;
  GtkWidget *menu_settings;
  GtkWidget *brightness_slider;
  GtkWidget *volume_slider;
  GtkWidget *eventbox;
  GtkWidget *rotation_button[4];
  GtkWidget *brightness_level_label;
  GtkWidget *volume_level_label;
  GtkWidget *volume_scale;
  GtkWidget *brightness_scale;

  gboolean in_area;
  gboolean on_border;
  gboolean on_button;
  gboolean button_released;

  gboolean is_active;

  gboolean has_rotation;
  gboolean has_dsme;
  gboolean all_done;

  GConfClient *gconf_client;
  void *dl_handle;

  gchar *current_theme;


  gint is_n810;
  gint is_770;

  gint notify_id;
  guint timer_id;
  struct timeval last_beep;
  gint brightness_jump;
  gint volume_jump;
  gboolean volume_beep;
  gboolean allow_zero;
  gint max_brightness;

  gint brightness_level;
  gint volume_level;
  gboolean mute;

  GThread *brightness_thread;
  GAsyncQueue *brightness_queue;
  GString *brightness_level_string;
  GString *brightness_level_dsme_string;
  //GThread *volume_thread;
  //GAsyncQueue *volume_queue;
  GString *volume_level_string;

};

/* DSME message blob */
struct dsme_message
{
  int len;
  int request;
  void *value;
};





/* Initialization/destruction functions */
static void advanced_backlight_plugin_class_init (AdvancedBacklightPluginClass * klass);
static void advanced_backlight_plugin_init (AdvancedBacklightPlugin * rsbp);
static void advanced_backlight_plugin_finalize (GObject * object);


static void show_menu (GtkButton * widget, AdvancedBacklightPlugin *);
static void advanced_backlight_plugin_create_menu (AdvancedBacklightPlugin *);
static void advanced_backlight_plugin_check_init_rotation (AdvancedBacklightPlugin *);
static void advanced_backlight_plugin_init_threads (AdvancedBacklightPlugin *);
static void advanced_backlight_plugin_init_dsme (AdvancedBacklightPlugin *);
static void advanced_backlight_plugin_dsme_update_brightness_value (AdvancedBacklightPluginPrivate *, gint);
static void advanced_backlight_plugin_770_update_brightness_value (AdvancedBacklightPluginPrivate *, gint);
static void advanced_backlight_plugin_update_volume_value (AdvancedBacklightPlugin *, gint);
static gpointer advanced_backlight_plugin_dsme_update_brightness_thread (gpointer);
//static gpointer advanced_backlight_plugin_update_volume_thread (gpointer);
/*static void advanced_backlight_plugin_dsme_update_brightness (AdvancedBacklightPlugin *);*/
static void advanced_backlight_plugin_configure_applet (AdvancedBacklightPlugin * vsp);
static void advanced_backlight_plugin_load_icons (AdvancedBacklightPlugin * vsp);
static gboolean volume_beep (GtkWidget * w, GdkEventButton * e, AdvancedBacklightPlugin * vsp);

static gboolean brightness_level_changed (AdvancedBacklightPlugin * vsp);
static void brightness_level_changing (GtkWidget * widget, AdvancedBacklightPlugin * vsp);

static void advanced_backlight_plugin_check_init_rotation (AdvancedBacklightPlugin * vsp);
static gint advanced_backlight_plugin_get_rotation (AdvancedBacklightPlugin * vsp);
static void advanced_backlight_plugin_set_rotation (AdvancedBacklightPlugin * vsp, gint rotation);
static void advanced_backlight_plugin_update_icons (AdvancedBacklightPlugin *);
static void advanced_backlight_plugin_load_settings (AdvancedBacklightPlugin * vsp);
static void advanced_backlight_plugin_save_settings (AdvancedBacklightPlugin * vsp);
static void advanced_backlight_plugin_fill_theme_list (AdvancedBacklightPlugin * vsp, GtkWidget * combobox);
static void advanced_backlight_plugin_update_volume_fields (AdvancedBacklightPlugin * vsp);
static int get_light_sensor (AdvancedBacklightPlugin * vsp);
static gboolean is_n810 ();
static gboolean is_770 ();

#ifdef XRANDR
/* D-Pad mapping */
static char *keyname[4] = { "Up", "Right", "Down", "Left" };
static KeyCode keycode[4] = { 111, 114, 116, 113 };
#endif

int (*dsmesock_connect) ();
void (*dsmesock_send) (int, struct dsme_message *);
void (*dsmesock_close) (int);

GtkWidget *
advanced_backlight_plugin_new (void)
{
  g_debug ("%s\n", __PRETTY_FUNCTION__);
  return g_object_new (ADVANCED_BACKLIGHT_PLUGIN_TYPE, NULL);
}


static void
advanced_backlight_plugin_class_init (AdvancedBacklightPluginClass * klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  object_class->finalize = advanced_backlight_plugin_finalize;
  g_type_class_add_private (object_class, sizeof (AdvancedBacklightPluginPrivate));
}

#if TEST_PLUGIN
/* Register static object instead of module */
GType
advanced_backlight_plugin_get_type (void)
{
  static GType item_type = 0;

  if (!item_type)
    {
      static const GTypeInfo item_info = {
        sizeof (AdvancedBacklightPluginClass),
        NULL,                   /* base_init */
        NULL,                   /* base_finalize */
        (GClassInitFunc) advanced_backlight_plugin_class_init,
        NULL,                   /* class_finalize */
        NULL,                   /* class_data */
        sizeof (AdvancedBacklightPlugin),
        0,                      /* n_preallocs */
        (GInstanceInitFunc) advanced_backlight_plugin_init,
      };
      item_type = g_type_register_static (HILDON_DESKTOP_TYPE_PANEL_ITEM, "AdvancedBacklightPlugin", &item_info, 0);
    }

  return item_type;
}
#else
HD_DEFINE_PLUGIN (AdvancedBacklightPlugin, advanced_backlight_plugin, STATUSBAR_TYPE_ITEM);
#endif


static void
advanced_backlight_plugin_init (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
#endif
  memset (priv, 0, sizeof (AdvancedBacklightPluginPrivate));

  priv->gconf_client = gconf_client_get_default ();

  advanced_backlight_plugin_init_threads (vsp);
  advanced_backlight_plugin_init_dsme (vsp);
  advanced_backlight_plugin_check_init_rotation (vsp);
  advanced_backlight_plugin_update_volume_fields (vsp);
  /* MERGED from ABL */
  priv->is_n810 = is_n810 ();
  priv->is_770 = is_770 ();

  if (priv->is_770)
    priv->max_brightness = 126;
  else
    priv->max_brightness = 127;

  gettimeofday (&priv->last_beep, NULL);
  advanced_backlight_plugin_load_settings (vsp);

#if 0
  priv->has_rotation = TRUE;    /* force enable */
#endif

  advanced_backlight_plugin_load_icons (vsp);

  /* Create status bar plugin button */
  priv->button = gtk_toggle_button_new ();
  advanced_backlight_plugin_update_icons (vsp);
  gtk_container_add (GTK_CONTAINER (priv->button), priv->icon);
  gtk_container_add (GTK_CONTAINER (vsp), priv->button);

  /* Connect signals */
  g_signal_connect (G_OBJECT (priv->button), "pressed", G_CALLBACK (show_menu), vsp);

  /* Show widgets */
  gtk_widget_show_all (priv->button);

  /* Create main panel */
  advanced_backlight_plugin_create_menu (vsp);
}

static void
advanced_backlight_plugin_finalize (GObject * object)
{
  AdvancedBacklightPlugin *vsp = ADVANCED_BACKLIGHT_PLUGIN (object);
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  int *brightness_value;
  int *volume_value;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  advanced_backlight_plugin_save_settings (vsp);
  gconf_client_notify_remove (priv->gconf_client, priv->notify_id);

  priv->all_done = TRUE;

  /* Wake up threads */
  g_string_free (priv->brightness_level_string, TRUE);
  brightness_value = g_new (int, 1);
  *brightness_value = priv->brightness_level;
  g_async_queue_push (priv->brightness_queue, brightness_value);

  g_thread_join (priv->brightness_thread);
  g_async_queue_unref (priv->brightness_queue);
  g_async_queue_unref (priv->brightness_queue); /* why twice? */

  volume_value = g_new (int, 1);
  *volume_value = priv->volume_level;
  //g_async_queue_push (priv->volume_queue, volume_value);
  //g_thread_join (priv->volume_thread);
  //g_async_queue_unref (priv->volume_queue);
  //g_async_queue_unref (priv->volume_queue);

  g_string_free (priv->volume_level_string, TRUE);


  /* Destroy local widgets */
  if (priv->menu)
    {
      gtk_widget_destroy (priv->menu);
    }

  G_OBJECT_CLASS (g_type_class_peek_parent (G_OBJECT_GET_CLASS (object)))->finalize (object);
}

static void
advanced_backlight_plugin_init_threads (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  if (!g_thread_supported ())
    g_thread_init (NULL);

  priv->all_done = FALSE;

  priv->brightness_queue = g_async_queue_new ();
  priv->brightness_thread = g_thread_create (advanced_backlight_plugin_dsme_update_brightness_thread, vsp, TRUE, NULL);

  //priv->volume_queue = g_async_queue_new ();
  //priv->volume_thread = g_thread_create (advanced_backlight_plugin_update_volume_thread, vsp, TRUE, NULL);
}

/*
    Dynamically load dsme library functions so it will build without it.
*/
static void
advanced_backlight_plugin_init_dsme (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  char *error;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  priv->has_dsme = FALSE;
  priv->dl_handle = dlopen ("libdsme.so.0.1.0", RTLD_LAZY);
  if (!priv->dl_handle)
    {
      fprintf (stderr, "%s\n", dlerror ());
      return;
    }
  dlerror ();

  *(void **) (&dsmesock_connect) = dlsym (priv->dl_handle, "dsmesock_connect");
  if ((error = dlerror ()) != NULL)
    {
      fprintf (stderr, "%s\n", error);
      return;
    }

  *(void **) (&dsmesock_send) = dlsym (priv->dl_handle, "dsmesock_send");
  if ((error = dlerror ()) != NULL)
    {
      fprintf (stderr, "%s\n", error);
      return;
    }

  *(void **) (&dsmesock_close) = dlsym (priv->dl_handle, "dsmesock_close");
  if ((error = dlerror ()) != NULL)
    {
      fprintf (stderr, "%s\n", error);
      return;
    }
  priv->has_dsme = TRUE;
}


/*
 Close the menu window and release mouse and keyboard.
 */
static void
advanced_backlight_plugin_close_window (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  gdk_keyboard_ungrab (GDK_CURRENT_TIME);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->button), FALSE);
  gtk_grab_remove (priv->menu);
  gtk_widget_hide_all (GTK_WIDGET (priv->menu));
}

/*
  Check if the pointer is within the bounds of the menu or on the status bar button.
  This will update the private data of the class.
 */
static void
advanced_backlight_plugin_update_pointer_location (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint x, y, w, h;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  w = GTK_WIDGET (priv->menu)->allocation.width;
  h = GTK_WIDGET (priv->menu)->allocation.height;

  gtk_widget_get_pointer (GTK_WIDGET (priv->menu), &x, &y);

  priv->on_button = FALSE;

  /* Pointer on menu area */
  if ((x >= 0) && (x <= w) && (y >= 0) && (y <= h))
    {
      priv->in_area = TRUE;

      if (x < HILDON_STATUS_BAR_BORDER_WIDTH || x > w - HILDON_STATUS_BAR_BORDER_WIDTH
          || y < HILDON_STATUS_BAR_BORDER_WIDTH || y > h - HILDON_STATUS_BAR_BORDER_WIDTH)
        {
          priv->on_border = TRUE;
        }
      else
        priv->on_border = FALSE;
    }
  else if (!priv->button_released)
    {
      w = GTK_WIDGET (priv->button)->allocation.width;
      h = GTK_WIDGET (priv->button)->allocation.height;

      gtk_widget_get_pointer (GTK_WIDGET (priv->button), &x, &y);

      /* Pointer on button area */
      if ((x >= 0) && (x <= w) && (y >= 0) && (y <= h))
        {
          priv->in_area = TRUE;
          priv->on_button = TRUE;
        }
      else
        priv->in_area = FALSE;
    }

  /* Pointer out of menu or button area */
  else
    priv->in_area = FALSE;

  priv->button_released = TRUE;

}

/*
 Handle key events.
 TODO: Add more event.
 */
static gboolean
key_pressed (GtkWidget * widget, GdkEventKey * event, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  switch (event->keyval)
    {
    case GDK_F5:               /* Hardware home key */
      advanced_backlight_plugin_close_window (vsp);
      return FALSE;

    case GDK_Up:               /* Volume Up */
      hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->volume_slider),
                                  hildon_volumebar_get_level (HILDON_VOLUMEBAR (priv->volume_slider)) + priv->volume_jump);
      return TRUE;

    case GDK_Down:             /* Volume Down */
      hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->volume_slider),
                                  hildon_volumebar_get_level (HILDON_VOLUMEBAR (priv->volume_slider)) - priv->volume_jump);
      return TRUE;

    case GDK_Left:             /* Brightness down */
      hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->brightness_slider),
                                  hildon_volumebar_get_level (HILDON_VOLUMEBAR (priv->brightness_slider)) -
                                  priv->brightness_jump);
      return TRUE;

    case GDK_Right:            /* Brightness up */
      hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->brightness_slider),
                                  hildon_volumebar_get_level (HILDON_VOLUMEBAR (priv->brightness_slider)) +
                                  priv->brightness_jump);
      return TRUE;

    case GDK_Return:           /* Mute */
      hildon_volumebar_set_mute (HILDON_VOLUMEBAR (priv->volume_slider),
                                 !hildon_volumebar_get_mute (HILDON_VOLUMEBAR (priv->volume_slider)));
      return TRUE;
/*
    case GDK_Escape: //Close
		hildon_desktop_popup_window_popdown(HILDON_DESKTOP_POPUP_WINDOW(window));
        return TRUE;
*/
    default:
      return FALSE;
    }
}


/*
 When leaving the eventbox, deselect the menu item
 */
static gboolean
leave_notify_event (GtkWidget * widget, GdkEventCrossing * event, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  gtk_menu_item_deselect (GTK_MENU_ITEM (priv->menu_settings));
  return TRUE;
}


/*
 When entering the eventbox, select the menu item
 */
static gboolean
enter_notify_event (GtkWidget * widget, GdkEventCrossing * event, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  gtk_menu_item_select (GTK_MENU_ITEM (priv->menu_settings));
  return TRUE;
}

/*
 Close the window when we revieve a delete.
 This will be called when the screen gets blank.
 Important note: 
 The event is only triggered, if gtk_window_set_is_temporary is set to TRUE.
 */
static void
delete_event (GtkWidget * widget, GdkEvent * event, AdvancedBacklightPlugin * vsp)
{
  g_debug ("%s\n", __PRETTY_FUNCTION__);
  advanced_backlight_plugin_close_window (vsp);
}


/*
 Activate the menu item if button was released
 */
static gboolean
button_released_on_settings (GtkWidget * widget, GdkEventButton * event, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  gtk_menu_item_activate (GTK_MENU_ITEM (priv->menu_settings));
  return TRUE;
}


/*
 When the menu is shown, we catch all clicks.
 Either close the window, if clicked outside or propagate the event.
 */
static gboolean
window_event (GtkWidget * widget, GdkEvent * event, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  advanced_backlight_plugin_update_pointer_location (vsp);

  if (priv->in_area)
    {
      if (priv->on_border || priv->on_button)
        return TRUE;
      return FALSE;
    }

  advanced_backlight_plugin_close_window (vsp);

  return TRUE;
}


/*
 Show the settings menu
 */
static void
activate_settings_item (GtkWidget * widget, AdvancedBacklightPlugin * vsp)
{
  g_debug ("%s\n", __PRETTY_FUNCTION__);
  advanced_backlight_plugin_close_window (vsp);
  advanced_backlight_plugin_configure_applet (vsp);
}



/*
 When the menu is shown, catch global mouse and keyboard events.
 On such events window_event will be called.
 */
static gboolean
map_event (GtkWidget * widget, GdkEvent * event, AdvancedBacklightPlugin * vsp)
{
  GdkCursor *cursor;
  GdkEventMask event_mask;
  GdkGrabStatus grab_status;
  GdkDisplay *display;
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  event_mask = GDK_POINTER_MOTION_MASK |
    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK;

  cursor = NULL;
  grab_status = gdk_pointer_grab (GDK_WINDOW (priv->menu->window), TRUE, event_mask, NULL, cursor, GDK_CURRENT_TIME);
  if (grab_status != GDK_GRAB_SUCCESS)
    {
      advanced_backlight_plugin_close_window (vsp);
      return TRUE;
    }

  grab_status = gdk_keyboard_grab (GDK_WINDOW (priv->menu->window), TRUE, GDK_CURRENT_TIME);
  if (grab_status != GDK_GRAB_SUCCESS)
    {
      g_debug ("Could not grab keyboard\n");
      display = gdk_drawable_get_display (GDK_DRAWABLE (priv->menu->window));
      gdk_display_pointer_ungrab (display, GDK_CURRENT_TIME);
      return TRUE;
    }

  return FALSE;
}


/*
 Update the orientation buttons to match the physical rotation.
 */
static void
advanced_backlight_plugin_update_rotate_button_orientation (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  PangoFontDescription *rot_pfd;
  gint i, rotation;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  if (!priv->has_rotation)
    return;

  rotation = advanced_backlight_plugin_get_rotation (vsp);
  g_debug ("current rotation is %d\n", rotation);

  rot_pfd = pango_font_description_copy (priv->menu->style->font_desc);
  pango_font_description_set_size (rot_pfd, 30 * PANGO_SCALE);

  for (i = 0; i < 4; i++)
    {
      gtk_widget_set_sensitive (priv->rotation_button[i], i != rotation);
      pango_font_description_set_gravity (rot_pfd, (4 - rotation + i) % 4);
      gtk_widget_modify_font (GTK_WIDGET (gtk_bin_get_child (GTK_BIN (priv->rotation_button[i]))), rot_pfd);
    }
  pango_font_description_free (rot_pfd);

}

static void
rotation_button_clicked (GtkButton * button, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint i;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  advanced_backlight_plugin_close_window (vsp);
  for (i = 0; i < 4; i++)
    {
      if (button == GTK_BUTTON (priv->rotation_button[i]))
        {
          advanced_backlight_plugin_set_rotation (vsp, i);
          g_debug ("rotate %d\n", i);
        }
    }

}

static gboolean
brightness_level_changed (AdvancedBacklightPlugin * vsp)
{
  GDK_THREADS_ENTER();
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  int *value;

  g_debug ("%s\n", __PRETTY_FUNCTION__);
  advanced_backlight_plugin_update_icons (vsp);

  value = g_new (int, 1);
  *value = priv->brightness_level;
  g_debug ("priv->brightness_level=%d\n", priv->brightness_level);
  g_async_queue_push (priv->brightness_queue, value);
  GDK_THREADS_LEAVE();
  return FALSE;
}

static void
brightness_level_changing (GtkWidget * widget, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);
  priv->brightness_level = (gint) hildon_volumebar_get_level (HILDON_VOLUMEBAR (priv->brightness_slider));

  g_string_printf (priv->brightness_level_string, "%3d", priv->brightness_level);
  gtk_label_set_text (GTK_LABEL (priv->brightness_level_label), (priv->brightness_level_string)->str);

  if (priv->timer_id > 0)
  {
    g_source_remove(priv->timer_id);
  }
  priv->timer_id = g_timeout_add (BRIGHTNESS_CHANGE_DELAY, (GSourceFunc) brightness_level_changed, vsp);
}

static gboolean
volume_beep (GtkWidget * w, GdkEventButton * e, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  g_debug ("%s\n", __PRETTY_FUNCTION__);
  if (priv->volume_beep)
    hildon_play_system_sound (DEFAULT_BEEP);
  //priv->timer_id = 0;
  return FALSE;
}
static void
advanced_backlight_plugin_update_volume_value (AdvancedBacklightPlugin * vsp, gint volume)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  g_debug ("%s\n", __PRETTY_FUNCTION__);
  gconf_client_set_int (priv->gconf_client, GCONF_MASTER_VOLUME, volume, NULL);
  g_debug ("%s ->\n", __PRETTY_FUNCTION__);
}

/*
static gpointer
advanced_backlight_plugin_update_volume_thread (gpointer d)
{
  AdvancedBacklightPlugin *vsp = (AdvancedBacklightPlugin *) d;
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint i = 0, j, value, prev_value = -101;
  gpointer gip;

  g_async_queue_ref (priv->volume_queue);
  while ((gip = g_async_queue_pop (priv->volume_queue)) && !priv->all_done)
    {
      g_debug ("%s [woke up]\n", __PRETTY_FUNCTION__);
      value = *(int *) gip;
      g_free (gip);

      j = 0;
      g_async_queue_lock (priv->volume_queue);
      while ((gip = g_async_queue_try_pop_unlocked (priv->volume_queue)) != NULL)
        {
          value = *(int *) gip;
          g_free (gip);
          j++;
        }
      g_async_queue_unlock (priv->volume_queue);
      if (0)
        g_debug ("after queue pop j: %d %d\n", j, i);
      if (prev_value != value)
        {
          g_debug ("%s setting volume\n", __PRETTY_FUNCTION__);
          advanced_backlight_plugin_update_volume_value (vsp, value);
          prev_value = value;
        }
    }
  return g_thread_self ();
}
*/
/*
 Update the system-volume according to the volumebar level
 */
static void
advanced_backlight_plugin_update_volume (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint volume;
  //int *value;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  priv->mute = hildon_volumebar_get_mute (HILDON_VOLUMEBAR (priv->volume_slider));
  volume = (gint) hildon_volumebar_get_level (HILDON_VOLUMEBAR (priv->volume_slider));
  if (priv->mute)
    volume *= -1;
  g_string_printf (priv->volume_level_string, "%4d", volume);
  gtk_label_set_text (GTK_LABEL (priv->volume_level_label), (priv->volume_level_string)->str);
  priv->volume_level = volume;
  advanced_backlight_plugin_update_icons (vsp);

  //value = g_new (int, 1);
  //*value = volume;
  //g_async_queue_push (priv->volume_queue, value);
  advanced_backlight_plugin_update_volume_value (vsp, volume);
}

static void
volume_level_changed (GtkWidget * widget, AdvancedBacklightPlugin * vsp)
{
  //AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  advanced_backlight_plugin_update_volume (vsp);
  g_debug ("%s\n", __PRETTY_FUNCTION__);
  /* We only want to beep on changes with the slider */
  /*
     if (priv->timer_id > 0)
     {
     g_source_remove (priv->timer_id);
     }
     priv->timer_id = g_timeout_add (VOLUME_BEEP_DELAY, (GSourceFunc) volume_beep, vsp);
   */
}

static void
volume_mute_toggled (GtkWidget * widget, AdvancedBacklightPlugin * vsp)
{
  advanced_backlight_plugin_update_volume (vsp);
}

static void
get_volume_scale (GtkWidget * widget, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  if (GTK_IS_SCALE (widget))
    {
      priv->volume_scale = widget;
    }
}

static void
get_brightness_scale (GtkWidget * widget, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  if (GTK_IS_SCALE (widget))
    {
      priv->brightness_scale = widget;
    }
}

static void
advanced_backlight_plugin_update_volume_fields (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint volume = gconf_client_get_int (priv->gconf_client, GCONF_MASTER_VOLUME, NULL);
  priv->mute = (volume < 0 || (volume == 0 && priv->mute));
  priv->volume_level = (priv->mute ? -1 : 1) * volume;
}

static void
gconf_notify_volume (GConfClient * gclient, guint notify_id, GConfEntry * entry, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);
  advanced_backlight_plugin_update_volume_fields (vsp);
  g_object_freeze_notify (G_OBJECT (priv->volume_slider));
  hildon_volumebar_set_mute (HILDON_VOLUMEBAR (priv->volume_slider), priv->mute);
  hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->volume_slider), priv->volume_level);
  g_object_thaw_notify (G_OBJECT (priv->volume_slider));
  advanced_backlight_plugin_update_icons (vsp);
}


/*
 Create the menu widget
 */
static void
advanced_backlight_plugin_create_menu (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  GtkWidget *hbox, *vbox, *label, *hbox2, *hbox3;
  PangoFontDescription *small_pfd;
  gint i;
  GError *err;


  g_debug ("%s\n", __PRETTY_FUNCTION__);

  /* Create menu window and set its style */
  priv->menu = gtk_window_new (GTK_WINDOW_POPUP);
  gtk_window_set_type_hint (GTK_WINDOW (priv->menu), GDK_WINDOW_TYPE_HINT_MENU);
  gtk_window_set_is_temporary (GTK_WINDOW (priv->menu), TRUE);
  gtk_widget_set_name (priv->menu, "hildon-status-bar-popup");
  gtk_window_set_resizable (GTK_WINDOW (priv->menu), FALSE);
  gtk_window_set_decorated (GTK_WINDOW (priv->menu), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (priv->menu), HILDON_STATUS_BAR_BORDER_WIDTH);

  g_signal_connect (G_OBJECT (priv->menu), "key-press-event", G_CALLBACK (key_pressed), vsp);
  g_signal_connect (G_OBJECT (priv->menu), "delete_event", G_CALLBACK (delete_event), vsp);

  g_signal_connect (G_OBJECT (priv->menu), "map-event", G_CALLBACK (map_event), vsp);
  g_signal_connect (G_OBJECT (priv->menu), "button-release-event", G_CALLBACK (window_event), vsp);

  gconf_client_add_dir (priv->gconf_client, GCONF_OSSO_SOUND_DIR, GCONF_CLIENT_PRELOAD_ONELEVEL, NULL);
  priv->notify_id = gconf_client_notify_add (priv->gconf_client, GCONF_MASTER_VOLUME,
                                             (GConfClientNotifyFunc) (gconf_notify_volume), vsp, NULL, &err);
  if (!priv->notify_id)
    {
      g_critical ("Unable to add notification: %s\n", err->message);
      g_error_free (err);
    }

  /* Create slider for brightness and volume */
  priv->brightness_slider = hildon_hvolumebar_new ();
  gtk_container_forall (GTK_CONTAINER (priv->brightness_slider), (GtkCallback) get_brightness_scale, vsp);
  gtk_range_set_increments (GTK_RANGE (priv->brightness_scale), 1, 1);
  gtk_range_set_range (GTK_RANGE (priv->brightness_scale), (priv->allow_zero && !priv->is_770) ? 0 : 1, priv->max_brightness);
  //gtk_range_set_update_policy (GTK_RANGE (priv->brightness_scale), GTK_UPDATE_DELAYED);
  hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->brightness_slider), priv->brightness_level);
  g_object_set (G_OBJECT (priv->brightness_slider), "has_mute", FALSE, NULL);
  g_signal_connect (G_OBJECT (priv->brightness_slider), "level-changed", G_CALLBACK (brightness_level_changing), vsp);
  //gtk_widget_add_events(GTK_WIDGET(priv->brightness_scale), GDK_BUTTON_RELEASE_MASK);
  //g_signal_connect (G_OBJECT (priv->brightness_scale), "button-release-event", G_CALLBACK (brightness_level_changed), vsp);

  priv->volume_slider = hildon_hvolumebar_new ();
  gtk_container_forall (GTK_CONTAINER (priv->volume_slider), (GtkCallback) get_volume_scale, vsp);
  gtk_range_set_increments (GTK_RANGE (priv->volume_scale), 1, 1);
  hildon_volumebar_set_mute (HILDON_VOLUMEBAR (priv->volume_slider), priv->mute);
  hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->volume_slider), priv->volume_level);
  g_signal_connect (G_OBJECT (priv->volume_slider), "level-changed", G_CALLBACK (volume_level_changed), vsp);
  g_signal_connect (G_OBJECT (priv->volume_slider), "mute-toggled", G_CALLBACK (volume_mute_toggled), vsp);
  gtk_widget_add_events (GTK_WIDGET (priv->volume_scale), GDK_BUTTON_RELEASE_MASK);
  g_signal_connect (G_OBJECT (priv->volume_scale), "button-release-event", G_CALLBACK (volume_beep), vsp);


  /* Create rotation buttons with rotated labels */
  if (priv->has_rotation)
    {
      for (i = 0; i < 4; i++)
        {
          priv->rotation_button[i] = gtk_button_new_with_label ("A");
          gtk_button_set_relief (GTK_BUTTON (priv->rotation_button[i]), GTK_RELIEF_HALF);
          g_signal_connect (G_OBJECT (priv->rotation_button[i]), "clicked", G_CALLBACK (rotation_button_clicked), vsp);
        }
    }

  /* Create the menu item in an eventbox */
  priv->eventbox = gtk_event_box_new ();
  priv->menu_settings = gtk_menu_item_new_with_label (_("Settings..."));
  gtk_widget_add_events (priv->eventbox, GDK_BUTTON_PRESS_MASK | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK);
  gtk_container_add (GTK_CONTAINER (priv->eventbox), priv->menu_settings);

  g_signal_connect (G_OBJECT (priv->eventbox), "enter-notify-event", G_CALLBACK (enter_notify_event), vsp);
  g_signal_connect (G_OBJECT (priv->eventbox), "leave-notify-event", G_CALLBACK (leave_notify_event), vsp);

  g_signal_connect (G_OBJECT (priv->menu_settings), "activate", G_CALLBACK (activate_settings_item), vsp);
  g_signal_connect (G_OBJECT (priv->menu_settings), "button-release-event", G_CALLBACK (button_released_on_settings), vsp);


  /* Align everything in an vbox */
  vbox = gtk_vbox_new (FALSE, 10);

  small_pfd = pango_font_description_copy (priv->menu->style->font_desc);
  pango_font_description_set_size (small_pfd, (pango_font_description_get_size (small_pfd) * 6) / 7);

  hbox2 = gtk_hbox_new (TRUE, 0);

  label = gtk_label_new (_("Brightness"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_modify_font (GTK_WIDGET (label), small_pfd);
  gtk_box_pack_start (GTK_BOX (hbox2), label, TRUE, TRUE, 0);

  priv->brightness_level_string = g_string_new ("   ");
  g_string_printf (priv->brightness_level_string, "%3d", priv->brightness_level);
  priv->brightness_level_label = gtk_label_new ((priv->brightness_level_string)->str);
  gtk_misc_set_alignment (GTK_MISC (priv->brightness_level_label), 1.0, 0.5);
  gtk_widget_modify_font (GTK_WIDGET (priv->brightness_level_label), small_pfd);
  gtk_box_pack_start (GTK_BOX (hbox2), priv->brightness_level_label, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), priv->brightness_slider, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), gtk_separator_menu_item_new (), FALSE, FALSE, 0);

  hbox3 = gtk_hbox_new (TRUE, 0);

  label = gtk_label_new (_("Volume"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_widget_modify_font (GTK_WIDGET (label), small_pfd);
  gtk_box_pack_start (GTK_BOX (hbox3), label, TRUE, TRUE, 0);

  priv->volume_level_string = g_string_new ("   ");
  g_string_printf (priv->volume_level_string, "%4d", 50);
  priv->volume_level_label = gtk_label_new ((priv->volume_level_string)->str);
  gtk_misc_set_alignment (GTK_MISC (priv->volume_level_label), 1.0, 0.5);
  gtk_widget_modify_font (GTK_WIDGET (priv->volume_level_label), small_pfd);
  gtk_box_pack_start (GTK_BOX (hbox3), priv->volume_level_label, TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX (vbox), hbox3, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), priv->volume_slider, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), gtk_separator_menu_item_new (), FALSE, FALSE, 0);

  if (priv->has_rotation)
    {
      label = gtk_label_new (_("Rotation"));
      gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
      gtk_widget_modify_font (GTK_WIDGET (label), small_pfd);
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);

      hbox = gtk_hbox_new (TRUE, 5);
      for (i = 0; i < 4; i++)
        {
          gtk_box_pack_start (GTK_BOX (hbox), priv->rotation_button[i], TRUE, TRUE, 0);
        }
      gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

      gtk_box_pack_start (GTK_BOX (vbox), gtk_separator_menu_item_new (), FALSE, FALSE, 0);
    }
  pango_font_description_free (small_pfd);

  gtk_box_pack_start (GTK_BOX (vbox), priv->eventbox, FALSE, FALSE, 0);

  /* ...and add it to the main menu container */
  gtk_container_add (GTK_CONTAINER (priv->menu), GTK_WIDGET (vbox));

  advanced_backlight_plugin_update_icons (vsp);
}


/*
 When we press the status bar icon, the menu will be shown.
 */
static void
show_menu (GtkButton * widget, AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  gboolean activate_button;
  gint x, y;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  if (!GTK_IS_WINDOW (priv->menu))
    {
      advanced_backlight_plugin_create_menu (vsp);
      activate_button = TRUE;
    }
  else                          /* We have already created the menu */
    activate_button = !gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->button));

  /* Position the menu and show it */
  if (activate_button)
    {
      /* Update rotation icons */
      advanced_backlight_plugin_update_rotate_button_orientation (vsp);

      /* Position and size */
      gdk_window_get_origin (priv->button->window, &x, &y);
      x += priv->button->allocation.x + priv->button->allocation.width - MENU_WINDOW_WIDTH;
      y += priv->button->allocation.y + priv->button->allocation.height + 10;
      if (x < 0)
        x = 0;
      gtk_window_move (GTK_WINDOW (priv->menu), x, y);
      gtk_widget_set_size_request (GTK_WIDGET (priv->menu), MENU_WINDOW_WIDTH, -1);

      /* Show */
      gtk_widget_realize (GTK_WIDGET (priv->menu));
      gtk_window_set_keep_above (GTK_WINDOW (priv->menu), TRUE);
      gtk_widget_show_all (priv->menu);

      /* Set focus */
      gtk_grab_add (priv->menu);
      gtk_widget_grab_focus (priv->menu);

      /* Activate button */
      priv->button_released = FALSE;
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->button), TRUE);
    }
}


#ifdef XRANDR
static void
advanced_backlight_plugin_check_init_rotation (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  Display *dpy;
  int xrr_event_base, xrr_error_base;
  int xrr_major, xrr_minor;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  dpy = XOpenDisplay (NULL);
  if (dpy == NULL)
    {
      fprintf (stderr, "Couldn't open display\n");
      priv->has_rotation = FALSE;
      return;
    }

  if (XRRQueryExtension (dpy, &xrr_event_base, &xrr_error_base) == False
      || XRRQueryVersion (dpy, &xrr_major, &xrr_minor) == 0 || xrr_major != 1 || xrr_minor < 1)
    priv->has_rotation = FALSE;
  else
    priv->has_rotation = TRUE;

  XCloseDisplay (dpy);
}


static gint
advanced_backlight_plugin_get_rotation (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint rotation;
  XRRScreenConfiguration *rr_screen;
  Rotation current_rotation;
  Display *dpy;
  gint screen;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  if (!priv->has_rotation)
    return 0;

  dpy = XOpenDisplay (NULL);
  screen = DefaultScreen (dpy);

  rr_screen = XRRGetScreenInfo (dpy, RootWindow (dpy, screen));
  XRRRotations (dpy, screen, &current_rotation);

  XRRFreeScreenConfigInfo (rr_screen);

  switch (current_rotation)
    {
    case RR_Rotate_270:
      rotation = 3;
      break;
    case RR_Rotate_180:
      rotation = 2;
      break;
    case RR_Rotate_90:
      rotation = 1;
      break;
    case RR_Rotate_0:
      rotation = 0;
      break;
    default:
      fprintf (stderr, "Unknown RANDR rotation: %d\n", current_rotation);
      rotation = 0;
      break;
    }

  XCloseDisplay (dpy);

  return rotation;
}


static void
advanced_backlight_plugin_set_rotation (AdvancedBacklightPlugin * vsp, gint rotation)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  Rotation sc_rotation;
  XRRScreenConfiguration *scr_config;
  Rotation current_rotation;
  gint size;
  int i;
  Display *dpy;
  gint screen;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  if (!priv->has_rotation)
    return;

  dpy = XOpenDisplay (NULL);
  screen = DefaultScreen (dpy);

  scr_config = XRRGetScreenInfo (dpy, RootWindow (dpy, screen));
  size = XRRConfigCurrentConfiguration (scr_config, &current_rotation);

  switch (rotation)
    {
    case 3:
      sc_rotation = RR_Rotate_270;
      break;
    case 2:
      sc_rotation = RR_Rotate_180;
      break;
    case 1:
      sc_rotation = RR_Rotate_90;
      break;
    case 0:
      sc_rotation = RR_Rotate_0;
      break;
    default:
      fprintf (stderr, "Unknown RANDR rotation: %d\n", rotation);
      sc_rotation = RR_Rotate_0;
      break;
    }

  /* Rotate d-pad */
  for (i = 0; i < 4; i++)
    {
      KeySym keysym = XStringToKeysym (keyname[(rotation + i) % 4]);
      XChangeKeyboardMapping (dpy, keycode[i], 1, &keysym, 1);
    }

  XRRSetScreenConfig (dpy, scr_config, RootWindow (dpy, screen), size, sc_rotation, CurrentTime);
  XRRFreeScreenConfigInfo (scr_config);
  XCloseDisplay (dpy);

}


#else /* XRANDR */

static void
advanced_backlight_plugin_check_init_rotation (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  priv->has_rotation = FALSE;
}

static gint
advanced_backlight_plugin_get_rotation (AdvancedBacklightPlugin * vsp)
{
  return 0;
}

static void
advanced_backlight_plugin_set_rotation (AdvancedBacklightPlugin * vsp, gint rotation)
{
}
#endif /* XRANDR */

static void
advanced_backlight_plugin_dsme_update_brightness_value (AdvancedBacklightPluginPrivate * priv, gint value)
{
  struct dsme_message message;
  gint conn;

  g_debug ("%s\n", __PRETTY_FUNCTION__);
  g_debug ("Setting brightness to %d\n", value);

  if (!priv->has_dsme)
    return;

  value = (priv->brightness_level * 2) + 1;

  conn = dsmesock_connect ();
  message.len = 0x0c;
  message.request = 0x289;
  message.value = (void *) value;
  dsmesock_send (conn, &message);
  dsmesock_close (conn);
}

/*
static void
advanced_backlight_plugin_dsme_update_brightness (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  struct dsme_message message;
  gint conn, value;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  if (!priv->has_dsme)
    return;

  value = (priv->brightness_level * 2);
  if (value <= 2 && !priv->allow_zero)
    value = 3;
  if (value > 254)
    value = 254;

  conn = dsmesock_connect ();
  message.type = 0x0c;
  message.request = 0x289;
  message.value = (void *) value;
  dsmesock_send (conn, &message);
  dsmesock_close (conn);
}
*/
static gpointer
advanced_backlight_plugin_dsme_update_brightness_thread (gpointer d)
{
  AdvancedBacklightPlugin *vsp = (AdvancedBacklightPlugin *) d;
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint i = 0, j, value, prev_value = -1;
  gpointer gip;

  g_async_queue_ref (priv->brightness_queue);
  while ((gip = g_async_queue_pop (priv->brightness_queue)) && !priv->all_done)
    {
      value = *(int *) gip;
      g_free (gip);
      j = 0;
      g_async_queue_lock (priv->brightness_queue);
      while ((gip = g_async_queue_try_pop_unlocked (priv->brightness_queue)) != NULL)
        {
          value = *(int *) gip;
          g_free (gip);
          j++;
        }
      g_async_queue_unlock (priv->brightness_queue);
      if (0)
        g_debug ("after queue pop j: %d %d\n", j, i);
      if (prev_value != value)
        {
          g_debug ("priv->brightness_level=%d\n", value);

          if (priv->is_770)
            advanced_backlight_plugin_770_update_brightness_value (priv, value);
          else
            advanced_backlight_plugin_dsme_update_brightness_value (priv, value);
          prev_value = value;
        }
    }
  return g_thread_self ();
}

static void
advanced_backlight_plugin_770_update_brightness_value (AdvancedBacklightPluginPrivate * priv, gint value)
{
  FILE *myfile;
  value = value * 256 + 128;    /* Math stolen directly from Fanoush's blset script. was 256 + 128 */

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  myfile = fopen (SYSFS_BRIGHTNESS, "w");

  if (myfile != NULL)
  {
    g_debug ("Setting 770 brightness to %d\n", value);
    fprintf (myfile, "%d\n", value);
    fclose (myfile);
  }
}

/*
 TODO: Find a way to fetch the brightness...
static int
advanced_backlight_plugin_dsme_get_brightness (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  if (!priv->has_dsme)
    return 0;


  return 0;

}
*/

static gboolean
is_n810 ()
{
  char version;

  FILE *fp = fopen ("/proc/component_version", "r");
  if (!fp)
    return FALSE;
  fseek (fp, sizeof (char) * 15, 0);    /* The important character is the 16th in the file. */
  fread (&version, sizeof (char), 1, fp);
  fclose (fp);

  return (version == '4');      /* 4 means n810, 3 means n800. */
}

static gboolean
is_770 ()
{
  char version;

  FILE *fp = fopen ("/proc/component_version", "r");
  if (!fp)
    return FALSE;
  fseek (fp, sizeof (char) * 15, 0);    /* The important character is the 16th in the file. */
  fread (&version, sizeof (char), 1, fp);
  fclose (fp);

  return (version == '1');      /* 4 means n810, 3 means n800. */
}

static void
advanced_backlight_plugin_load_settings (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  GConfValue *conf_value;
  int *value;

  /* Brightness level */
  conf_value = gconf_client_get (priv->gconf_client, GCONF_KEY_BRIGHTNESS, NULL);
  if (conf_value)
    {
      priv->brightness_level = gconf_value_get_int (conf_value);
      /* advanced_backlight_plugin_dsme_update_brightness (vsp); */
      value = g_new (int, 1);
      *value = priv->brightness_level;
      g_async_queue_push (priv->brightness_queue, value);
      gconf_value_free (conf_value);
    }
  else
    priv->brightness_level = 100;

  /* Brightness allow zero */
  conf_value = gconf_client_get (priv->gconf_client, GCONF_KEY_ALLOW_ZERO, NULL);
  if (conf_value)
    {
      priv->allow_zero = gconf_value_get_bool (conf_value);
      gconf_value_free (conf_value);
    }
  else
    priv->allow_zero = 0;

  /* Selected theme */
  priv->current_theme = gconf_client_get_string (priv->gconf_client, GCONF_KEY_THEME, NULL);
  if (!priv->current_theme)
    priv->current_theme = "Default";

  /* Brightness jump */
  conf_value = gconf_client_get (priv->gconf_client, GCONF_KEY_BRIGHTNESS_JUMP, NULL);
  if (conf_value)
    {
      priv->brightness_jump = gconf_value_get_int (conf_value);
      gconf_value_free (conf_value);
    }
  else
    priv->brightness_jump = 10;

  /* Volume jump */
  conf_value = gconf_client_get (priv->gconf_client, GCONF_KEY_VOLUME_JUMP, NULL);
  if (conf_value)
    {
      priv->volume_jump = gconf_value_get_int (conf_value);
      gconf_value_free (conf_value);
    }
  else
    priv->volume_jump = 20;

  /* Volume beep */
  conf_value = gconf_client_get (priv->gconf_client, GCONF_KEY_VOLUME_BEEP, NULL);
  if (conf_value)
    {
      priv->volume_beep = gconf_value_get_bool (conf_value);
      gconf_value_free (conf_value);
    }
  else
    priv->volume_beep = TRUE;
}

static void
advanced_backlight_plugin_save_settings (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);

  /* Brightness level */
  gconf_client_set_int (priv->gconf_client, GCONF_KEY_BRIGHTNESS, priv->brightness_level, NULL);

  /* Brightness allow zero */
  gconf_client_set_bool (priv->gconf_client, GCONF_KEY_ALLOW_ZERO, priv->allow_zero, NULL);

  /* Selected theme */
  gconf_client_set_string (priv->gconf_client, GCONF_KEY_THEME, priv->current_theme, NULL);

  /* Brightness jump */
  gconf_client_set_int (priv->gconf_client, GCONF_KEY_BRIGHTNESS_JUMP, priv->brightness_jump, NULL);

  /* Volume jump */
  gconf_client_set_int (priv->gconf_client, GCONF_KEY_VOLUME_JUMP, priv->volume_jump, NULL);

  /* Volume beep */
  gconf_client_set_bool (priv->gconf_client, GCONF_KEY_VOLUME_BEEP, priv->volume_beep, NULL);
}

static void
advanced_backlight_plugin_configure_applet (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  GtkWidget *dialog;
  GtkWidget *confirmation_dialog;
  gint response;
  gint response_allow_zero;
  GtkWidget *vbox;
  GtkWidget *vbox2;
  GtkWidget *allow_zero_checkbox;
  GtkWidget *allow_zero_caption;
  GtkWidget *theme_caption;
  GtkWidget *brightness_spinner;
  GtkWidget *volume_spinner;
  GtkWidget *brightness_caption;
  GtkWidget *volume_caption;
  GtkWidget *choose_theme;
  GtkWidget *verifytext;
  GtkWidget *volume_beep_checkbox;
  GtkWidget *volume_beep_caption;

  GtkSizeGroup *size_group;

  size_group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  dialog =
    gtk_dialog_new_with_buttons ("Advanced Backlight", NULL, GTK_DIALOG_MODAL,
                                 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
  vbox = gtk_vbox_new (FALSE, 0);

  allow_zero_checkbox = gtk_check_button_new ();
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (allow_zero_checkbox), priv->allow_zero);
  allow_zero_caption =
    hildon_caption_new (NULL, _("Allow zero backlight\n(N800 users beware!)"), allow_zero_checkbox, NULL,
                        HILDON_CAPTION_MANDATORY);
  hildon_caption_set_size_group (HILDON_CAPTION (allow_zero_caption), GTK_SIZE_GROUP (size_group));

  volume_beep_checkbox = gtk_check_button_new ();
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (volume_beep_checkbox), priv->volume_beep);
  volume_beep_caption =
    hildon_caption_new (NULL, _("Beep on volume change"), volume_beep_checkbox, NULL, HILDON_CAPTION_MANDATORY);
  hildon_caption_set_size_group (HILDON_CAPTION (volume_beep_caption), GTK_SIZE_GROUP (size_group));

  choose_theme = gtk_combo_box_new_text ();
  advanced_backlight_plugin_fill_theme_list (vsp, choose_theme);

  theme_caption = hildon_caption_new (NULL, _("Theme"), choose_theme, NULL, HILDON_CAPTION_MANDATORY);
  hildon_caption_set_size_group (HILDON_CAPTION (theme_caption), GTK_SIZE_GROUP (size_group));

  brightness_spinner = hildon_number_editor_new (1, 255);
  hildon_number_editor_set_value (HILDON_NUMBER_EDITOR (brightness_spinner), priv->brightness_jump);
  brightness_caption = hildon_caption_new (NULL, _("Brightness step size"), brightness_spinner, NULL, HILDON_CAPTION_MANDATORY);
  hildon_caption_set_size_group (HILDON_CAPTION (brightness_caption), GTK_SIZE_GROUP (size_group));

  volume_spinner = hildon_number_editor_new (1, 100);
  hildon_number_editor_set_value (HILDON_NUMBER_EDITOR (volume_spinner), priv->volume_jump);
  volume_caption = hildon_caption_new (NULL, _("Volume step size"), volume_spinner, NULL, HILDON_CAPTION_MANDATORY);
  hildon_caption_set_size_group (HILDON_CAPTION (volume_caption), GTK_SIZE_GROUP (size_group));

  gtk_box_pack_start (GTK_BOX (vbox), theme_caption, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), brightness_caption, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), volume_caption, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), volume_beep_caption, FALSE, FALSE, 0);
  if (!priv->is_770)
    gtk_box_pack_start (GTK_BOX (vbox), allow_zero_caption, FALSE, FALSE, 0);

  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);

  gtk_widget_show_all (dialog);
  response = gtk_dialog_run (GTK_DIALOG (dialog));

  if (response == GTK_RESPONSE_ACCEPT)
    {
      if (!priv->is_770)
        priv->allow_zero = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (allow_zero_checkbox));
      else
        priv->allow_zero = 0;
      if (!priv->is_n810 && priv->allow_zero)
        {
          confirmation_dialog =
            gtk_dialog_new_with_buttons (_("Warning"), NULL, GTK_DIALOG_MODAL, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
          verifytext =
            gtk_label_new (_
                           ("Are you sure you want to allow the backlight to turn off?\nIf the backlight turns off, it may be very difficult to turn it\nback on, as the screen will no longer be visible."));

          vbox2 = gtk_vbox_new (FALSE, 0);
          gtk_box_pack_start (GTK_BOX (vbox2), verifytext, FALSE, FALSE, 0);
          gtk_container_add (GTK_CONTAINER (GTK_DIALOG (confirmation_dialog)->vbox), vbox2);
          gtk_widget_show_all (confirmation_dialog);
          response_allow_zero = gtk_dialog_run (GTK_DIALOG (confirmation_dialog));

          if (response_allow_zero != GTK_RESPONSE_ACCEPT)
            priv->allow_zero = FALSE;

          gtk_widget_destroy (confirmation_dialog);
        }

      gtk_range_set_range (GTK_RANGE (priv->brightness_scale), priv->allow_zero ? 0 : 1, priv->max_brightness);
      if (!priv->allow_zero && priv->brightness_level == 0)
        {
          /* If allow zero was disabled, ensure that the actual value is greater than zero */
          priv->brightness_level = 1;
          hildon_volumebar_set_level (HILDON_VOLUMEBAR (priv->brightness_slider), priv->brightness_level);
        }

      priv->brightness_jump = hildon_number_editor_get_value (HILDON_NUMBER_EDITOR (brightness_spinner));
      priv->volume_jump = hildon_number_editor_get_value (HILDON_NUMBER_EDITOR (volume_spinner));
      priv->volume_beep = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (volume_beep_checkbox));

      if (priv->current_theme &&
          g_strcasecmp (priv->current_theme, gtk_combo_box_get_active_text (GTK_COMBO_BOX (choose_theme))) != 0)
        {
          priv->current_theme = gtk_combo_box_get_active_text (GTK_COMBO_BOX (choose_theme));
          advanced_backlight_plugin_load_icons (vsp);
        }

      advanced_backlight_plugin_update_icons (vsp);
    }

  gtk_widget_destroy (dialog);
  advanced_backlight_plugin_save_settings (vsp);
}

static void
advanced_backlight_plugin_fill_theme_list (AdvancedBacklightPlugin * vsp, GtkWidget * combobox)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  DIR *d;
  struct dirent *de;
  int count = 0;

  d = opendir (ICON_DIR);

  if (d)
    {
      while ((de = readdir (d)) != NULL)
        {
          if (g_strcasecmp (de->d_name, ".") != 0 && g_strcasecmp (de->d_name, "..") != 0)
            {
              gtk_combo_box_append_text (GTK_COMBO_BOX (combobox), de->d_name);
              if (priv->current_theme && g_strcasecmp (priv->current_theme, de->d_name) == 0)
                gtk_combo_box_set_active (GTK_COMBO_BOX (combobox), count);
              count++;
            }
        }
    }
  closedir (d);
}


static void
advanced_backlight_plugin_load_icons (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint i, j;
  gchar icon_name[30];
  gchar *file_name;
  GError *err;

  for (i = 0; i < NUM_BRIGHTNESS_ICONS; i++)
    {
      for (j = 0; j < NUM_VOLUME_ICONS; j++)
        {
          if (j == NUM_VOLUME_ICONS - 1)
            snprintf (icon_name, 30, "adv-backlight-icon-%d.m.png", i + 1);
          else
            snprintf (icon_name, 30, "adv-backlight-icon-%d.%d.png", i + 1, j);

          file_name = g_build_filename (ICON_DIR, priv->current_theme, icon_name, NULL);
          if (priv->pixbuf[i][j] != NULL)
            {
              gdk_pixbuf_unref (priv->pixbuf[i][j]);
            }
          err = NULL;
          if (!(priv->pixbuf[i][j] = gdk_pixbuf_new_from_file (file_name, &err)))
            {
              g_critical ("Unable to read file: %s\n", err->message);
              g_error_free (err);
            }

          g_free (file_name);
        }
    }
}

static void
advanced_backlight_plugin_update_icons (AdvancedBacklightPlugin * vsp)
{
  AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp);
  gint volume, volume_icon, brightness_icon;

  g_debug ("%s\n", __PRETTY_FUNCTION__);

  volume = priv->volume_level;
  if (priv->mute)
    volume_icon = NUM_VOLUME_ICONS - 1;
  else
    {
      volume_icon = volume / (100 / (NUM_VOLUME_ICONS - 1));
      if (volume_icon > NUM_VOLUME_ICONS - 2)
        volume_icon = NUM_VOLUME_ICONS - 2;
    }

  brightness_icon = ((NUM_BRIGHTNESS_ICONS - 1) * priv->brightness_level) / priv->max_brightness;
  if (brightness_icon > NUM_BRIGHTNESS_ICONS - 1)
    brightness_icon = NUM_BRIGHTNESS_ICONS - 1;

  if (!priv->icon)
    priv->icon = gtk_image_new_from_pixbuf (priv->pixbuf[brightness_icon][volume_icon]);
  else
    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->icon), priv->pixbuf[brightness_icon][volume_icon]);

}

static int
get_light_sensor (AdvancedBacklightPlugin * vsp)
{
  /*AdvancedBacklightPluginPrivate *priv = ADVANCED_BACKLIGHT_PLUGIN_GET_PRIVATE (vsp); */
  int value = -1;
  FILE *fp;

  fp = fopen ("/sys/devices/platform/i2c_omap.2/i2c-0/0-0029/lux", "r");

  if (fp)
    {
      fscanf (fp, "%d", &value);
      fclose (fp);
    }

  return (value);
}

#ifdef TEST_PLUGIN
/* used for testing the plugin and debugging */
int
main (int argc, char *argv[])
{

  /* Create needed variables */
  HildonProgram *program;
  HildonWindow *window;
  GtkWidget *main_layout;
  GtkWidget *vsp;

  /* Initialize TLS */
#ifdef ENABLE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (GETTEXT_PACKAGE, LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif

  gtk_init (&argc, &argv);
  program = HILDON_PROGRAM (hildon_program_get_instance ());
  g_set_application_name ("Advanced-Backlight Testbed");
  window = HILDON_WINDOW (hildon_window_new ());
  g_signal_connect (G_OBJECT (window), "delete_event", G_CALLBACK (gtk_main_quit), NULL);
  hildon_program_add_window (program, window);
  main_layout = gtk_fixed_new ();
  gtk_container_add (GTK_CONTAINER (window), main_layout);
  vsp = advanced_backlight_plugin_new ();
  gtk_fixed_put (GTK_FIXED (main_layout), vsp, 300, 0);
  gtk_widget_set_size_request (GTK_WIDGET (vsp), 45, 45);
  gtk_widget_show_all (GTK_WIDGET (window));
  gtk_main ();

  /* Exit */
  return 0;
}
#endif /* TEST_PLUGIN */
