/*
 * haze-plugin: An account plugin for telepathy-haze
 * Copyright (C) 2007 Collabora Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include <config.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <dbus/dbus.h>
#include <libaccounts/account-error.h>

#include "haze-plugin.h"
#include "libosso-rtcom-accounts/rtcom-accounts.h"


enum
{
    PROP_0,
};

ACCOUNT_DEFINE_PLUGIN (HazePlugin, haze_plugin, RTCOM_TYPE_ACCOUNT_PLUGIN);

#define HAZE_PLUGIN_PRIV(ap) (G_TYPE_INSTANCE_GET_PRIVATE ((ap), \
                              HAZE_TYPE_PLUGIN, \
                              HazePluginPrivate))

typedef struct {
} HazePluginPrivate;

static void
haze_plugin_init (HazePlugin *self)
{
    RtcomAccountPlugin *rtcom_plugin = RTCOM_ACCOUNT_PLUGIN (self);
    GList *profiles, *list;

    rtcom_plugin->name = "haze";
    profiles = mc_profiles_list ();
    for (list = profiles; list; list = list->next)
    {
        McProfile *profile = MC_PROFILE (list->data);
        const gchar *name, *cfg_ui;

        cfg_ui = mc_profile_get_configuration_ui (profile);
        if (cfg_ui != NULL && strcmp (cfg_ui, "haze-plugin") == 0)
        {
            name = mc_profile_get_unique_name (profile);
            rtcom_account_plugin_add_service (rtcom_plugin, name);
        }
    }
    mc_profiles_free_list (profiles);
    glade_init ();
}

static void
haze_plugin_dispose (GObject *object)
{
    G_OBJECT_CLASS (haze_plugin_parent_class)->dispose (object);
}

static void
haze_plugin_finalize (GObject *object)
{
    G_OBJECT_CLASS (haze_plugin_parent_class)->finalize (object);
}

static void
on_password_can_next (RtcomWidget *password, GParamSpec *pspec)
{
    gboolean can_next;
    g_object_get (password, "can-next", &can_next, NULL);
    if (!can_next)
        rtcom_widget_set_msg_next (password, _("accountwizard_ib_enter_password_first"));
}

static RtcomPage *
username_page (RtcomContext *context)
{
    GladeXML *xml;
    GtkWidget *page, *label;
    const gchar *format, *profile_name;
    gchar text[200];
    AccountItem *account;
    AccountService *service;

    xml = glade_xml_new (PLUGIN_XML_DIR "/haze-existing-account.glade", "page", GETTEXT_PACKAGE);
    page = glade_xml_get_widget(xml, "page");

    glade_xml_signal_connect (xml, "on_password_can_next",
                              G_CALLBACK (on_password_can_next));

    account = account_edit_context_get_account (ACCOUNT_EDIT_CONTEXT (context));
    service = account_item_get_service (account);
    profile_name = account_service_get_display_name (service);
    label = glade_xml_get_widget(xml, "label1");
    format = _("accountwizard_ia_usename_enter_username_and_password");
    g_snprintf (text, sizeof(text), format, profile_name);
    gtk_label_set_label (GTK_LABEL (label), text);

    return RTCOM_PAGE (page);
}

static RtcomPage *
details_page (RtcomContext *context)
{
    GladeXML *xml;
    GtkWidget *page, *alias;
    GtkLabel *label;
    McProfile *profile;
    McProfileCapabilityFlags caps;
    AccountItem *account;

    xml = glade_xml_new (PLUGIN_XML_DIR "/haze-details.glade", "page", GETTEXT_PACKAGE);
    page = glade_xml_get_widget(xml, "page");

    account = account_edit_context_get_account (ACCOUNT_EDIT_CONTEXT (context));
    profile = rtcom_account_item_get_mc_profile (RTCOM_ACCOUNT_ITEM (account));
    label = GTK_LABEL (glade_xml_get_widget (xml, "label1"));
    alias = glade_xml_get_widget (xml, "capt_alias");
    caps = mc_profile_get_capabilities (profile);
    if (caps & MC_PROFILE_CAPABILITY_SUPPORTS_ALIAS)
    {
        gtk_label_set_label (label, _("accountwizard_ia_details_enter_account_name"));
        gtk_widget_show (alias);
    }
    else
    {
        gtk_label_set_label (label, _("accountwizard_ia_details_enter_account_name_no_nick"));
        gtk_widget_hide (alias);
    }
    g_object_unref (profile);

    return RTCOM_PAGE (page);
}

static void
create_editing_pages (RtcomContext *context, gboolean existing)
{
    gboolean has_roster;
    RtcomPage *next_page;

    next_page = username_page (context);
    rtcom_context_append_page (context, next_page);

    next_page = details_page (context);
    rtcom_context_append_page (context, next_page);

    has_roster = !account_edit_context_get_editing
        (ACCOUNT_EDIT_CONTEXT (context));
    next_page = RTCOM_PAGE (rtcom_lastpage_new (has_roster));
    rtcom_context_append_page (context, next_page);
}

static void
context_dim_finish (AccountWizardContext *context)
{
    account_wizard_context_set_can_finish (context, FALSE);
}

static void
context_init (RtcomAccountPlugin *plugin, RtcomContext *context)
{
    RtcomPage *page;
    gboolean editing;

    editing = account_edit_context_get_editing (ACCOUNT_EDIT_CONTEXT (context));
    create_editing_pages (context, editing);
    if (editing)
    {
        account_wizard_context_set_can_finish (ACCOUNT_WIZARD_CONTEXT (context),
                                               TRUE);
    }
}

static void
haze_plugin_class_init (HazePluginClass *klass)
{
    GObjectClass* object_class = G_OBJECT_CLASS (klass);
    RtcomAccountPluginClass *rtcom_class = RTCOM_ACCOUNT_PLUGIN_CLASS (klass);

    object_class->dispose = haze_plugin_dispose;
    object_class->finalize = haze_plugin_finalize;

    rtcom_class->context_init = context_init;

    g_type_class_add_private (object_class, sizeof (HazePluginPrivate));
}


