import gtk
import pango
import hildon

from youamp.ui.browser import Browser

class SearchView(gtk.VBox):
    def __init__(self, config, player, library):
        gtk.VBox.__init__(self)
        self.set_spacing(5)
        
        self._config = config
        self._library = library
        self._player = player
        self._is_browser = self._config["is-browser"]
        
        # search entry
        navi = gtk.HBox()
        navi.set_spacing(5)

        # Navi Toggle Button
        navi_button = gtk.Button()
        navi_button.connect("clicked", self._on_navi_toggle)
        navi_button.unset_flags(gtk.CAN_FOCUS)
        navi_button.set_size_request(100, -1)
        navi.pack_start(navi_button, expand=False)
        
        navi.pack_start(gtk.VSeparator(), expand=False)
        
        # Navi Artist Button
        self._artist = gtk.Button()
        self._artist.connect("clicked", self._reset_artist)
        self._artist.set_size_request(200, -1)
        label = gtk.Label()
        label.set_ellipsize(pango.ELLIPSIZE_END)
        self._artist.add(label)
        navi.pack_start(self._artist, expand=False)
        
        # Navi Album Button
        self._album = gtk.Button()
        self._album.set_size_request(200, -1)
        label = gtk.Label()
        label.set_ellipsize(pango.ELLIPSIZE_END)
        self._album.add(label)
        self._album.connect("clicked", self._reset_album)
        navi.pack_start(self._album, expand=False)
        
        # Navi Search Entry
        self._search_entry = gtk.Entry()
        self._search_entry.connect("activate", self._search)
        self._search_entry.set_text(self._config["search-str"])
        navi.pack_start(self._search_entry, expand=True)
        
        # Shuffle Button
        self.shuffle = gtk.ToggleButton(_("shuffle"))
        self.shuffle.unset_flags(gtk.CAN_FOCUS)
        self.shuffle.set_active(self._config["shuffle"])
        self.shuffle.connect("toggled", self._on_shuffle_toggled)
        
        navi.pack_end(self.shuffle, expand=False)
        self.pack_start(navi, expand=False)
        
        # Scrolled View
        self._scroll = gtk.ScrolledWindow()
        self._scroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        self._scroll.set_shadow_type(gtk.SHADOW_IN)
        hildon.hildon_helper_set_thumb_scrollbar(self._scroll, True)
        
        self._playlist = PlaylistView(player.playlist, self._config)
        self._playlist.connect("row-activated", self._on_row_activated)
        
        self._browser = Browser(self._config, self._library)
        
        self._scroll.add(self._playlist)

        self.pack_start(self._scroll)

        self.show_all()
 
        # Restore View
        if self._config["is-browser"]:
            navi_button.set_label(_("Search"))
            self._artist.get_child().set_text(self._config["search-artist"])
            self._album.get_child().set_text(self._config["search-album"])
            self._browser._has_artist = True
            self._search_entry.hide()
        else:
            navi_button.set_label(_("Browse"))
            self._artist.hide()
            self._album.hide()
        
        self._restore_playlist()
            
        self._player.connect("song-changed", self._on_song_changed)
        
        # browser callbacks
        self._config.notify_add("search-artist", self._on_search_artist)
        self._config.notify_add("search-album", self._on_search_album)

    def _on_shuffle_toggled(self, caller):
        self._config["shuffle"] = not self._config["shuffle"]
        self._player.playlist.shuffle(self._config["shuffle"])
    
    def _on_navi_toggle(self, button):
        if self._is_browser:
            button.set_label(_("Browse"))
            self._artist.hide()
            self._album.hide()
            self._search_entry.show()
            
            if self._browser.get_parent() is not None:
                self._show_playlist()
        else:
            button.set_label(_("Search"))
            self._search_entry.hide()
            self._show_browser()
            self._reset_artist()

        self._is_browser = not self._is_browser
    
    def _restore_playlist(self):
        if self._config["is-browser"]:
            playlist = self._library.get_tracks(self._config["search-artist"], self._config["search-album"])
        else:
            playlist = self._library.search(self._config["search-str"])

        self._player.playlist.update(playlist)
        
        if self._config["shuffle"]:
            self._player.playlist.shuffle(True)

    def _browse_complete(self):
        playlist = self._library.get_tracks(self._config["search-artist"], self._config["search-album"])
        self._set_playlist(playlist)
        
        self._search_entry.set_text("")
        self._config["is-browser"] = True
        
        self._show_playlist()
 
    def _search(self, caller):
        self._config["search-str"] = caller.get_text()
        self._config["shuffle"] = self.shuffle.get_active()
                
        playlist = self._library.search(self._config["search-str"])
        
        self._set_playlist(playlist)
        
        self._config.notify("search-str")
        
        self._config["is-browser"] = False
    
    def _set_playlist(self, data):
        self._config["pos"] = 0
        self._playlist.set_cursor(0)

        self._player.playlist.update(data)
        
        if self._config["shuffle"]:
            self._player.playlist.shuffle(True)

    def _show_playlist(self):
        if self._playlist.get_parent() is None:
            self._scroll.remove(self._browser)
            self._scroll.add(self._playlist)

    def _show_browser(self):
        if self._browser.get_parent() is None:
            self._scroll.remove(self._playlist)
            self._scroll.add(self._browser)

    def _on_search_artist(self, client, cxn_id, entry, data):
        try:
            val = entry.get_value().get_string()
        except AttributeError:
            return
        
        val = val if val != "" else _("All Artists")
        
        self._artist.get_child().set_text(val)
        self._artist.show()

    def _on_search_album(self, client, cxn_id, entry, data):
        try:
            val = entry.get_value().get_string()
        except AttributeError:
            return
        
        val = val if val != "" else _("All Albums")
        
        self._album.get_child().set_text(val)
        self._album.show()
        
        self._browse_complete()
    
    def _reset_album(self, caller):
        self._album.hide()
        
        self._show_browser()
        self._browser.show_albums()
    
    def _reset_artist(self, caller=None):
        self._show_browser()
        self._browser.show_artists()
        
        self._artist.hide()
        self._album.hide()

    def _on_song_changed(self, caller, newsong):
        self._playlist.set_cursor(self._config["pos"])

    def _on_row_activated(self, caller, path, column):
        self._player.goto_pos(path[0])

class PlaylistView(gtk.TreeView):
    def __init__(self, model, config):        
        gtk.TreeView.__init__(self, model)
    
        self.set_rules_hint(True)
        self.set_headers_visible(True)
        self.set_fixed_height_mode(True)
        keys = ("title", "artist", "album")
        cols = (_("Title"), _("Artist"), _("Album"))
        
        cell = gtk.CellRendererText()
        cell.set_property("ellipsize", pango.ELLIPSIZE_END)
            
        for i in xrange(len(keys)):
            col = gtk.TreeViewColumn(cols[i], cell)
            col.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
            col.set_fixed_width(200)
            col.set_cell_data_func(cell, self.data_func, keys[i])
            self.append_column(col)

        self.set_cursor(config["pos"])
    
    def data_func(self, col, cell, model, itr, key):
        cell.set_property("text", model[itr][key])

