import gtk
import hildon
import pango
import gobject

from youamp import Playlist, IS_MAEMO

from youamp.ui.controls import Controls
from youamp.ui.playlistview import SearchView
from youamp.ui.about import AboutDialog
from youamp.ui.preferences import Preferences

from youamp.library import Library
from youamp.player import Player
from youamp.config import Config

class Window(hildon.Window):
    def __init__(self):
        hildon.Window.__init__(self)

        self.set_default_size(800, 480)
        self.set_title(_("No Track"))
        
        if IS_MAEMO:
            gtk.set_application_name("YouAmp")
        
        # config
        self._is_fullscreen = False
        self._is_browser = False
        self._config = Config()
        self._player = Player(self._config)
        self._player.playlist = Playlist()

        # programm
        program = hildon.Program()
        program.add_window(self)
        program.set_property("can-hibernate", True)
        
        # dialogs
        self._about_dialog = None
        self._preferences = None
        
        # menu
        menu = gtk.Menu()
        
        prefs = gtk.ImageMenuItem(gtk.STOCK_PREFERENCES)
        prefs.connect("activate", self._show_preferences)
        menu.append(prefs)
        
        about = gtk.ImageMenuItem(gtk.STOCK_ABOUT)
        about.connect("activate", self._show_about_dialog)
        menu.append(about)
        menu.show_all()
        self.set_menu(menu)
        
        # Content Box
        vbox = gtk.VBox()
        vbox.set_spacing(5)
        
        # Label
        self._label = gtk.Label()
        self._label.set_alignment(0, 0)
        self._label.set_justify(gtk.JUSTIFY_LEFT)
        self._label.set_ellipsize(pango.ELLIPSIZE_END)
        self._label.set_selectable(False)
        self._label.set_markup("<b><big>%s</big></b>" % _("No Track"))
        self._label.show_all()
        vbox.pack_start(self._label, expand=False)

        # Controls
        self._controls = Controls(self._player, self._config)
        self._controls.show_all()
        vbox.pack_start(self._controls, expand=False)

        # set up library
        self._library = Library()
        self._library.connect("update-complete", self._destroy_banner)

        vbox.pack_start(SearchView(self._config, self._player, self._library))
        vbox.show()
        self.add(vbox)

        # signals
        self.connect("destroy", self.quit)
        self.connect("key-press-event", self._on_key_press)
        
        self._player.connect("tags-updated", self._on_tags_updated)
        
        self.show()

        # restore player
        if self._player.tracks() > 0:
            self._player.load_track()
   
        self._banner = hildon.hildon_banner_show_animation(self, "qgn_indi_pball_a", "Updating database")

        # start updating thread
        self._library.update()

        
    def _show_about_dialog(self, caller):
        if self._about_dialog is None:
            self._about_dialog = AboutDialog()
        
        self._about_dialog.show()
    
    def _show_preferences(self, caller):
        if self._preferences is None:
            self._preferences = Preferences(self, self._config)
            self._preferences.show_all()
    
        self._preferences.show()

    def _destroy_banner(self, caller):
        gobject.idle_add(self._banner.destroy)
    
    def _on_key_press(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "Right":
            self._controls.on_next()
        elif key == "Left":
            self._controls.on_prev()
        if key == "F7":
            self._on_vol_up()
        elif key == "F8":
            self._on_vol_down()
        elif key == "F6":
            self._on_toggle_size()
    
    def _on_vol_down(self):
        vol = self._config["volume"]
        self._config["volume"] = max(0.0, vol - 0.2) # cut to range
    
    def _on_vol_up(self):
        vol = self._config["volume"]
        self._config["volume"] = min(1.0, vol + 0.2) # cut to range
    
    def _on_toggle_size(self):
        if self._is_fullscreen:
            self.unfullscreen()
            self._label.set_padding(0, 0)
        else:
            self.fullscreen()
            self._label.set_padding(10, 10)
        
        self._is_fullscreen = not self._is_fullscreen
    
    def _on_tags_updated(self, caller, newsong):
        self.set_title("%s - %s" % (newsong["title"], newsong["artist"]))
                
        label_txt = "<b><big>%s</big></b> %s <i>%s</i> %s <i>%s</i>" % (newsong["title"], _("by"), newsong["artist"], _("from"), newsong["album"])
        label_txt = label_txt.replace("&", "&amp;") # escape the xml escape char
        
        if not self._player.has_rg_info():
            label_txt = """<span foreground="red">%s</span>""" % label_txt
        
        self._label.set_markup(label_txt)
       
    def quit(self, widget=None):
        gtk.main_quit()
    
    def main(self):
        gtk.main() 
