/*
  WifiInfo - Show current Wifi sttaus
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/OS/Probe.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Boolean.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/AppInfo.h>
#include <Lum/Boolean.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/FuelGauge.h>
#include <Lum/Label.h>
#include <Lum/LED.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Slider.h>
#include <Lum/Tab.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>

#include "config.h"

#if defined(CAN_USE_STATUS_HILDON)
  #include "StatusHildon.h"
#elif defined(CAN_USE_STATUS_LINUX)
  #include "StatusLinux.h"
#else
  #include "StatusNone.h"
#endif

#define NETWORK_REFRESH    10
#define POLL_ERROR_TIMEOUT 60

#include <iostream>
static Lum::AppInfo info;

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::StringRef    type;
  Lum::Model::StringRef    essid;
  Lum::Model::StringRef    accesspoint;
  Lum::Model::StringRef    client;
  Lum::Model::DoubleRef    quality;
  Lum::Model::StringRef    signalNoise;
  Lum::Model::StringRef    channel;
  Lum::Model::BooleanRef   powerSaving;

  Lum::Model::IntRef       refreshTime;

  Lum::Model::ListTableRef tableModel;

  Lum::Model::BooleanRef   scanForNetworks;
  Lum::Model::ActionRef    powerSavingOnAction;
  Lum::Model::ActionRef    powerSavingOffAction;

  Lum::Model::ActionRef    statusTimer;
  Lum::Model::ActionRef    statusErrorTimer;
  Lum::Model::ActionRef    networksTimer;
  Lum::Model::ActionRef    networksErrorTimer;

  Lum::Model::ActionRef    aboutAction;

#if defined(CAN_USE_STATUS_HILDON)
    StatusHildon           status;
#elif defined(CAN_USE_STATUS_LINUX)
    StatusLinux            status;
#else
    StatusNone             status;
#endif

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textView;

    textView=Lum::TextValue::Create(model,Lum::TextValue::left,true,false);
    textView->SetWidth(Lum::Base::Size::stdCharWidth,size);

    return textView;
  }

  void ParseArguments()
  {
    size_t i=0;

    while (i<Lum::OS::display->GetArgCount()) {
      if (Lum::OS::display->GetArg(i)==L"-d" &&
          i+1<Lum::OS::display->GetArgCount()) {
        i++;
        status.SetDefaultInterface(Lum::Base::WStringToString(Lum::OS::display->GetArg(i)));
      }

      i++;
    }
  }

  void RequestStatus()
  {
    Lum::OS::display->RemoveTimer(statusTimer);
    Lum::OS::display->RemoveTimer(statusErrorTimer);

    if (status.UpdateStatus()) {
      Lum::OS::display->AddTimer(refreshTime->Get()+POLL_ERROR_TIMEOUT,0,statusErrorTimer);
    }
    else {
      status.type=Status::typeNone;
      UpdateStatus();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,statusTimer);
    }
  }

  void CancelNetworks()
  {
    Lum::OS::display->RemoveTimer(networksTimer);
    Lum::OS::display->RemoveTimer(networksErrorTimer);
  }

  void RequestNetworks()
  {
    CancelNetworks();

    if (status.SupportsNetworkRetrieval()) {
      if (status.UpdateNetworks()) {
        Lum::OS::display->AddTimer(NETWORK_REFRESH+POLL_ERROR_TIMEOUT,0,networksErrorTimer);
      }
      else {
        tableModel->Clear();
        Lum::OS::display->AddTimer(NETWORK_REFRESH,0,networksTimer);
      }
    }
  }

  void UpdateStatus()
  {
    if (status.type==Status::typeNone || status.bitrate==0) {
      type->Set(L"-");
      essid->Set(L"-");
      channel->Set(L"-");
      accesspoint->Set(L"-");
      client->Set(L"-");
      quality->SetNull();
      signalNoise->Set(L"-");
      powerSaving->SetNull();
      return;
    }

    std::wstring tmr;

    essid->Set(status.essid);

    tmr=status.typeName;

    tmr.append(L" ");

    switch (status.type) {
    case Status::typeAuto:
      tmr.append(L"Auto");
      break;
    case Status::typeAdHoc:
      tmr.append(L"Ad-Hoc");
      break;
    case Status::typeInfrastructure:
      tmr.append(L"Infrastructure");
      break;
    case Status::typeMaster:
      tmr.append(L"Master");
      break;
    case Status::typeRepeater:
    tmr.append(L"Repeater");
      break;
    case Status::typeSecond:
      tmr.append(L"Secondary master");
      break;
    case Status::typeMonitor:
      tmr.append(L"Passive monitor");
      break;
    case Status::typeMesh:
      tmr.append(L"Mesh");
      break;
    default:
      tmr.append(L"-");
    }

    tmr.append(L" ");

    if (status.bitrate>1000000 && status.bitrate==(status.bitrate/1000000)*1000000) {
      tmr.append(Lum::Base::NumberToWString(status.bitrate/1000000));
      tmr.append(L"Mb/s");
    }
    else if (status.bitrate>1000 && status.bitrate==(status.bitrate/1000)*1000) {
      tmr.append(Lum::Base::NumberToWString(status.bitrate/1000));
      tmr.append(L"Kb/s");
    }
    else {
    tmr.append(Lum::Base::NumberToWString(status.bitrate));
      tmr.append(L"b/s");
    }

    type->Set(tmr);

    if (status.channel>=0) {
      channel->Set(Lum::Base::NumberToWString(status.channel));
    }
    else {
      channel->Set(L"-");
    }

    accesspoint->Set(status.accesspoint);

    client->Set(std::wstring(status.ip)+L" ("+status.mac+L")");

    if (status.quality>=0 && status.quality<=100) {
      quality->Set(status.quality);
    }
    else {
      quality->SetNull();
    }

    signalNoise->Set(Lum::Base::NumberToWString(status.signal-0x100)+L" dBm / "+Lum::Base::NumberToWString(status.noise-0x100)+L" dBm");

    switch (status.powerSaving) {
    case Status::powerSavingUnknown:
      powerSaving->SetNull();
      break;
    case Status::powerSavingOn:
      powerSaving->Set(true);
      break;
    case Status::powerSavingOff:
      powerSaving->Set(false);
      break;
    }
  }

  void UpdateNetworks()
  {
    std::cout << "Updating networks..." << std::endl;
    std::list<Status::Network>::const_iterator iter;

    // First try to update existing entries in the current table
    iter=status.networks.begin();
    while (iter!=status.networks.end()) {
      size_t i;

      // Find the right entry
      for (i=1; i<=tableModel->GetRows(); i++) {
        if (iter->essid==tableModel->GetEntry(i)->GetString(1)) {
          break;
        }
      }

      if (i<=tableModel->GetRows()) { // Found an entry => Update
        Lum::Model::ListTable::Entry    *entry=tableModel->GetEntry(i);
        Lum::FuelGauge                  *gauge;
        Lum::Model::DoubleRef           data;

        gauge=dynamic_cast<Lum::FuelGauge*>(entry->GetObject(3));
        data=dynamic_cast<Lum::Model::Double*>(gauge->GetModel());

        data->Set(iter->quality);
      }
      else { // No entry => add
        Lum::FuelGauge                  *gauge;
        Lum::Model::DoubleRef           data;
        Lum::Model::ListTable::StdEntry *entry;

        gauge=new Lum::FuelGauge();
        gauge->SetFlex(true,false);

        data=new Lum::Model::Double();
        data->SetRange(0,100);
        data->Set(iter->quality);
        gauge->SetModel(data);

        entry=new Lum::Model::ListTable::StdEntry(tableModel);
        entry->SetString(1,iter->essid);
        entry->SetString(2,Lum::Base::NumberToWString(iter->channel));
        entry->SetObject(3,gauge);

        tableModel->Append(entry);
      }

      ++iter;
    }

    // Now see, if entries in the table have been removed and must
    // be deleted
    size_t i=1;

    while(i<=tableModel->GetRows()) {
      std::wstring                               essid=tableModel->GetEntry(i)->GetString(1);
      std::list<Status::Network>::const_iterator iter;

      // Find essid from table entry in network list
      iter=status.networks.begin();
      while (iter!=status.networks.end()) {
        if (iter->essid==essid) {
          break;
        }

        ++iter;
      }

      if (iter==status.networks.end()) { // table entry is not in list
        tableModel->Delete(i);
      }
      else {
        i++;
      }
    }
    std::cout << "Updating networks done." << std::endl;
  }

public:
  MyWindow()
  : type(new Lum::Model::String()),
    essid(new Lum::Model::String()),
    accesspoint(new Lum::Model::String()),
    client(new Lum::Model::String()),
    quality(new Lum::Model::Double()),
    signalNoise(new Lum::Model::String()),
    channel(new Lum::Model::String()),
    powerSaving(new Lum::Model::Boolean()),
    refreshTime(new Lum::Model::Int(5)),
    tableModel(new Lum::Model::ListTable()),
    scanForNetworks(new Lum::Model::Boolean(false)),
    powerSavingOnAction(new Lum::Model::Action()),
    powerSavingOffAction(new Lum::Model::Action()),
    statusTimer(new Lum::Model::Action()),
    statusErrorTimer(new Lum::Model::Action()),
    networksTimer(new Lum::Model::Action()),
    networksErrorTimer(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action())
  {
    quality->SetRange(0,100);
    refreshTime->SetRange(0,10);

    powerSaving->Disable();

    Observe(scanForNetworks);
    Observe(powerSavingOnAction);
    Observe(powerSavingOffAction);

    Observe(refreshTime);
    Observe(statusTimer);
    Observe(statusErrorTimer);
    Observe(networksTimer);
    Observe(networksErrorTimer);
    Observe(GetOpenedAction());
    Observe(aboutAction);

    Observe(status.statusChangedAction);
    Observe(status.networksChangedAction);
  }

  virtual void PreInit()
  {
    Lum::Label       *label;
    Lum::Panel       *hPanel,*hPanel2;
    Lum::Slider      *slider;
    Lum::Tab         *tab;

    std::wstring     tmp;

    tab=Lum::Tab::Create(true,true);

    hPanel=Lum::HPanel::Create(true,true);

    label=Lum::Label::Create(true,true);
    label->AddLabel(L"ESSID:",CreateTextValue(essid,32));
    label->AddLabel(L"Channel:",CreateTextValue(channel,2));
    label->AddLabel(L"Type:",CreateTextValue(type,32));
    label->AddLabel(L"AP:",CreateTextValue(accesspoint,17));
    label->AddLabel(L"Client:",CreateTextValue(client,28));
    label->AddLabel(L"Quality:",Lum::FuelGauge::Create(quality,true,false));
    label->AddLabel(L"SNR:",CreateTextValue(signalNoise,18));

    hPanel2=Lum::HPanel::Create();
    hPanel2->Add(new Lum::LED(powerSaving));

    if (status.CanChangePowerSaving()) {
      hPanel2->AddSpace();
      hPanel2->Add(Lum::Button::Create(L"_On",powerSavingOnAction));
      hPanel2->AddSpace();
      hPanel2->Add(Lum::Button::Create(L"O_ff",powerSavingOffAction));
    }

    label->AddLabel(L"Power saving:",hPanel2);

    slider=new Lum::HSlider();
    slider->SetFlex(true,false);
    slider->SetScale(true);
    slider->SetModel(refreshTime);
    label->AddLabel(L"Poll (s):",slider);

    hPanel->Add(label);

    tab->Add(L"Current",hPanel);

    if (status.SupportsNetworkRetrieval()) {
      Lum::Panel                         *panel;
      Lum::Table                         *table;
      Lum::Model::HeaderRef              headerModel;
      Lum::Model::SingleLineSelectionRef selection;

      panel=Lum::VPanel::Create(true,true);
      panel->Add(Lum::Label::Create(true,false)
                 ->AddLabel(L"Scan for networks:",Lum::Boolean::Create(scanForNetworks)));
      panel->AddSpace();

      selection=new Lum::Model::SingleLineSelection;

      headerModel=new Lum::Model::HeaderImpl();
      headerModel->AddColumn(L"ESSID",Lum::Base::Size::stdCharWidth,15,true);
      headerModel->AddColumn(L"C",Lum::Base::Size::stdCharWidth,2);
      headerModel->AddColumn(L"Quality",Lum::Base::Size::stdCharWidth,10);

      table=new Lum::Table();
      table->SetFlex(true,true);
      table->SetHeaderModel(headerModel);
      table->SetModel(tableModel);
      //table->SetSelection(selection);
      //table->SetSelectionAction(showDetail);
      //table->SetDoubleClickAction(showDetail);
      table->SetShowHeader(true);
      table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,4);
      table->GetTableView()->SetAutoFitColumns(true);
      table->GetTableView()->SetAutoHSize(true);

      panel->Add(table);

      tab->Add(L"All",panel);
    }

    SetMain(tab);

    Lum::MenuDesc *menu=new Lum::MenuDesc();

    menu
      ->AddMenuItemSub(_ld(menuProject))
        ->AddMenuItemAction(_ld(menuProjectQuit),Lum::OS::qualifierControl,L"q",GetClosedAction());
    menu
      ->AddMenuItemSub(_ld(menuHelp))
        ->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->AddMenuItemAction(_ld(menuHelpAbout),aboutAction);

    SetMenu(menu);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      ParseArguments();
      UpdateStatus();
      RequestStatus();
    }
    else if (model==scanForNetworks) {
      if (*scanForNetworks) {
        RequestNetworks();
      }
      else {
        CancelNetworks();
        UpdateNetworks();
      }
    }
    else if (model==powerSavingOnAction && powerSavingOnAction->IsFinished()) {
      status.SetPowerSaving(true);
    }
    else if (model==powerSavingOffAction && powerSavingOffAction->IsFinished()) {
      status.SetPowerSaving(false);
    }
    else if ((model==statusTimer && statusTimer->IsFinished()) ||
             (model==statusErrorTimer && statusErrorTimer->IsFinished())) {
      RequestStatus();
    }
    else if ((model==networksTimer && networksTimer->IsFinished()) ||
             (model==networksErrorTimer && networksErrorTimer->IsFinished())) {
      RequestNetworks();
    }
    else if (model==refreshTime && IsOpen()) {
      RequestStatus();
    }
    else if (model==status.statusChangedAction && status.statusChangedAction->IsFinished()) {
      Lum::OS::display->RemoveTimer(statusErrorTimer);
      UpdateStatus();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,statusTimer);
    }
    else if (model==status.networksChangedAction && status.networksChangedAction->IsFinished()) {
      Lum::OS::display->RemoveTimer(networksErrorTimer);
      UpdateNetworks();
      Lum::OS::display->AddTimer(NETWORK_REFRESH,0,networksTimer);
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MyWindow>
{
public:
  Main()
  {
    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"How is your Wifi?");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2004, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

