/*
  WifiInfo - Show current Wifi sttaus
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "StatusHildon.h"

#include <Lum/Base/String.h>

#include <Lum/OS/X11/Display.h>

#include <iostream>

DBusHandlerResult DBusMsgHandler(DBusConnection *connection, DBusMessage *msg, void *data)
{
  StatusHildon *statusHildon=static_cast<StatusHildon*>(data);

  statusHildon->HandleMessage(connection,msg);

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

void callback(DBusPendingCall* call, void* /*data*/)
{
  DBusMessage *reply;

  reply=dbus_pending_call_steal_reply(call);

  std::cout << "Call finished with reply of type " << dbus_message_type_to_string(dbus_message_get_type(reply)) << std::endl;

  if (dbus_message_get_type(reply)==DBUS_MESSAGE_TYPE_ERROR) {
    std::cout << "Error: " << dbus_message_get_error_name(reply) << std::endl;
  }

  dbus_message_unref(reply);
}

StatusHildon::StatusHildon()
{
  dbus_connection_add_filter(GetDBusConnection(),DBusMsgHandler,this,NULL);
}

std::string StatusHildon::GetDefaultInterface() const
{
  if (StatusLinux::GetDefaultInterface().empty()) {
    return "wlan0";
  }
  else {
    return StatusLinux::GetDefaultInterface();
  }
}

DBusConnection* StatusHildon::GetDBusConnection()
{
  return dynamic_cast<Lum::OS::X11::Display*>(Lum::OS::display)->dbusSystem;
}

bool StatusHildon::HandleMessage(DBusConnection *connection, DBusMessage *msg)
{
  if (dbus_message_get_type(msg)==DBUS_MESSAGE_TYPE_METHOD_CALL) {
    DBusMessage *response;
    std::string appName;
    std::string error;

    response=dbus_message_new_error(msg,"Message not implemented","Message not implemented");
    dbus_connection_send(connection,response,NULL);
    dbus_message_unref(response);
    return true;
  }
  else if (dbus_message_get_type(msg) == DBUS_MESSAGE_TYPE_SIGNAL) {
    if (strcmp(dbus_message_get_interface(msg),"com.nokia.wlancond.signal")!=0 ||
        strcmp(dbus_message_get_path(msg),"/com/nokia/wlancond/signal")!=0 ||
        strcmp(dbus_message_get_member(msg),"scan_results")!=0) {
      return false;
    }
  }
  else {
    return false;
  }

  std::cout << "Handling network scan result..." << std::endl;

  Lum::OS::Guard<Lum::OS::Mutex> guard(mutex);

  networks.clear();

  DBusMessageIter iter;
  int             type;
  dbus_int32_t    entries;
  dbus_int32_t    intValue;
  dbus_uint32_t   uintValue;
  std::string     strValue;

  dbus_message_iter_init(msg,&iter);

  type=dbus_message_iter_get_arg_type(&iter);
  if (type!=DBUS_TYPE_INT32) {
    std::cerr << "Expected number of entries" << std::endl;
    return true;
  }

  dbus_message_iter_get_basic(&iter,&entries);
  dbus_message_iter_next(&iter);

  for (int i=0; i<entries; i++) {
    Network         network;
    DBusMessageIter iter2;

    //
    // ESSID
    //

    type=dbus_message_iter_get_arg_type(&iter);
    if (type!=DBUS_TYPE_ARRAY) {
      std::cerr << "Expected ESSID" << std::endl;
      return true;
    }

    dbus_message_iter_recurse(&iter,&iter2);
    strValue.clear();

    while ((type=dbus_message_iter_get_arg_type(&iter2))!=DBUS_TYPE_INVALID) {
      std::string tmp;

      if (type!=DBUS_TYPE_BYTE) {
        std::cerr << "Wrong type for ESSID" << std::endl;
        return true;
      }

      char value;
      dbus_message_iter_get_basic(&iter2,&value);

      strValue.append(1,value);

      dbus_message_iter_next(&iter2);
    }

    if (strValue.length()>0 && strValue[strValue.length()-1]=='\0') {
      strValue.erase(strValue.length()-1);
    }

    network.essid=Lum::Base::StringToWString(strValue);

    dbus_message_iter_next(&iter);


    type=dbus_message_iter_get_arg_type(&iter);
    if (type!=DBUS_TYPE_ARRAY) {
      std::cerr << "Expected BSSID" << std::endl;
      return true;
    }
    dbus_message_iter_next(&iter);

    type=dbus_message_iter_get_arg_type(&iter);
    if (type!=DBUS_TYPE_INT32) {
      std::cerr << "Expected RSSI" << std::endl;
      return true;
    }
    dbus_message_iter_get_basic(&iter,&intValue);

    if (noise<0) {
      std::cerr << "Do not have global noise value, quitting" << std::endl;
      return true;
    }

    if (noise==0)  {
      network.quality=GetQualityFromSignalNoiseDbm(intValue,-95);
    }
    else {
      network.quality=GetQualityFromSignalNoiseDbm(intValue,noise-0x100);
    }

    dbus_message_iter_next(&iter);

    type=dbus_message_iter_get_arg_type(&iter);
    if (type!=DBUS_TYPE_UINT32) {
      std::cerr << "Expected CHANNEL" << std::endl;
      return true;
    }
    dbus_message_iter_get_basic(&iter,&uintValue);

    network.channel=uintValue;

    dbus_message_iter_next(&iter);

    type=dbus_message_iter_get_arg_type(&iter);
    if (type!=DBUS_TYPE_UINT32) {
      std::cerr << "Expected CAPABILITIES" << std::endl;
      return true;
    }
    dbus_message_iter_get_basic(&iter,&uintValue);
    if (uintValue & (1 << 0)) {
      network.type=typeInfrastructure;
    }
    else if (uintValue & (1 << 1)) {
      network.type=typeAdHoc;
    }
    else if (uintValue & (1 << 2)) {
      network.type=typeAuto;
    }

    /*
    if (uintValue & (1 << 16)) {
      std::cout << "Bitrate: 10" << std::endl;
    }
    else if (uintValue & (1 << 17)) {
      std::cout << "Bitrate: 20" << std::endl;
    }
    else if (uintValue & (1 << 18)) {
      std::cout << "Bitrate: 55" << std::endl;
    }
    else if (uintValue & (1 << 19)) {
      std::cout << "Bitrate: 60" << std::endl;
    }
    else if (uintValue & (1 << 20)) {
      std::cout << "Bitrate: 90" << std::endl;
    }
    else if (uintValue & (1 << 21)) {
      std::cout << "Bitrate: 110" << std::endl;
    }
    else if (uintValue & (1 << 22)) {
      std::cout << "Bitrate: 120" << std::endl;
    }
    else if (uintValue & (1 << 23)) {
      std::cout << "Bitrate: 180" << std::endl;
    }
    else if (uintValue & (1 << 24)) {
      std::cout << "Bitrate: 260" << std::endl;
    }
    else if (uintValue & (1 << 25)) {
      std::cout << "Bitrate: 540" << std::endl;
    }*/

    dbus_message_iter_next(&iter);

    if (!network.essid.empty()) {
      networks.push_back(network);
    }
  }

  Lum::OS::display->QueueActionForEventLoop(networksChangedAction);

  return true;
}

bool StatusHildon::SupportsNetworkRetrieval() const
{
  return true;
}

bool StatusHildon::UpdateNetworks()
{
  DBusMessage     *msg;
  DBusPendingCall *call;
  dbus_int32_t    power=4;
  dbus_int32_t    flags=2;
  unsigned char   a[] = { 0 };
  unsigned char*  ap=a;

  std::cout << "Requesting network scan..." << std::endl;

  msg=dbus_message_new_method_call("com.nokia.wlancond",
                                   "/com/nokia/wlancond/request",
                                   "com.nokia.wlancond.request",
                                   "scan");

  dbus_message_append_args(msg,
                           DBUS_TYPE_INT32,&power,
                           DBUS_TYPE_ARRAY,DBUS_TYPE_BYTE,&ap,0,
                           DBUS_TYPE_UINT32,&flags,
                           DBUS_TYPE_INVALID);

  if (!dbus_connection_send_with_reply(GetDBusConnection(),msg,&call,-1)) {
    std::cout << "Cannot send with reply" << std::endl;
    return false;
  }

  dbus_pending_call_set_notify(call,callback,NULL,NULL);

  std::cout << "Requesting network scan done." << std::endl;

  dbus_message_unref(msg);

  return true;
}

bool StatusHildon::CanChangePowerSaving() const
{
  return true;
}

bool StatusHildon::SetPowerSaving(bool savePower)
{
  std::cout << "Requesting power saving switch... " << savePower << std::endl;

  DBusMessage     *msg;
  DBusPendingCall *call;
  bool            result;

  msg=dbus_message_new_method_call("com.nokia.wlancond",
                                   "/com/nokia/wlancond/request",
                                   "com.nokia.wlancond.request",
                                   "set_powersave");

  dbus_message_append_args(msg,
                           DBUS_TYPE_BOOLEAN,&savePower,
                           DBUS_TYPE_INVALID);

  result=dbus_connection_send_with_reply(GetDBusConnection(),msg,&call,-1);

  dbus_pending_call_set_notify(call,callback,NULL,NULL);

  dbus_message_unref(msg);

  if (result) {
    UpdateStatus();
  }
  else {
    std::cout << "Cannot send" << std::endl;
  }

  return result;
}

