/*
  DisplayInfo - a demo program for the Illimination library
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#if defined(linux)
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <arpa/inet.h>
#include <fcntl.h>

extern "C" {
  #include <linux/if.h>
  #include <linux/wireless.h>
}

#include <fstream>

#endif

#include <cstdlib>
#include <cstring>
#include <iostream>

#include <Lum/OS/Probe.h>

#include <Lum/Base/String.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/Dialog.h>
#include <Lum/FuelGauge.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Slider.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/WindowGroup.h>

class Status
{
public:
  class Network
  {
  public:
    std::wstring essid;
    int          mode;
    unsigned     bitrate;
    int          quality;

    Network()
    {
      Clear();
    }

    void Clear()
    {
      mode=-1;
      bitrate=0;
      quality=-1;
    }
  };

public:
  std::wstring       type;
  int                mode;
  std::wstring       essid;
  unsigned           bitrate;
  int                quality;
  std::wstring       accesspoint;
  std::wstring       ip;
  std::wstring       mac;
  std::list<Network> networks;
};

bool GetStatus(Status &status)
{
  status.type.clear();
  status.mode=-1;
  status.essid.clear();
  status.bitrate=0;
  status.quality=-1;
  status.accesspoint.clear();
  status.mac.clear();
  status.ip.clear();
  status.networks.clear();

#if defined(linux)
         std::ifstream file;
         std::string   line,interface;
         int           handle;
  struct ifreq         ifr;
  struct iwreq         data;
  struct iw_statistics stats;
         char          buffer[1024];

  file.open("/proc/net/wireless",std::ios::in);

  if (!file) {
    return false;
  }

  if (std::getline(file,line) && std::getline(file,line) && std::getline(file,line)) {
    std::string::size_type start,end;

    start=line.find_first_not_of(" ");
    end=line.find(":");

    if (start!=std::string::npos && end!=std::string::npos) {
      interface=line.substr(start,end-start);
    }
  }

  file.close();

  if (interface.empty()) {
    return false;
  }

  handle=socket(AF_INET,SOCK_DGRAM,0);

  if (handle<0) {
    return false;
  }

  strcpy(ifr.ifr_name,interface.c_str());

  if (ioctl(handle,SIOCGIFADDR,&ifr)>=0) {

    status.ip=Lum::Base::StringToWString(inet_ntoa(((struct sockaddr_in*)&ifr.ifr_addr)->sin_addr));
  }

  if (ioctl(handle,SIOCGIFHWADDR,&ifr)>=0) {

    status.mac=Lum::Base::StringToWString(ether_ntoa((ether_addr*)ifr.ifr_hwaddr.sa_data));
  }

  // Type

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWNAME,&data)>=0) {
    status.type=Lum::Base::StringToWString(data.u.name);
  }

  // Mode

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWMODE,&data)>=0) {
    status.mode=data.u.mode;
  }

  // ESSID

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  memset(buffer,0,1024);
  data.u.essid.pointer=buffer,
  data.u.essid.length=1024;

  if (ioctl(handle,SIOCGIWESSID,&data)>=0) {
    status.essid=Lum::Base::StringToWString(buffer);
  }

  // Bitrate

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWRATE,&data)>=0) {
    status.bitrate=data.u.bitrate.value;
  }

  // Quality

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  data.u.data.pointer=(caddr_t)&stats;
  data.u.data.length=sizeof(iw_statistics);
  data.u.data.flags=1;

  if (ioctl(handle,SIOCGIWSTATS,&data)>=0) {
    if (!(IW_QUAL_QUAL_INVALID & data.u.data.flags)) {
      status.quality=stats.qual.qual;
    }
  }

  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());

  if (ioctl(handle,SIOCGIWAP,&data)>=0) {
    ether_addr *in=(ether_addr*)&data.u.ap_addr.sa_data;

    status.accesspoint=Lum::Base::StringToWString(ether_ntoa(in));
  }

  // Scan
#if 0
  memset(&data,0,sizeof(struct iwreq));
  sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
  data.u.essid.pointer=(caddr_t)NULL;
  data.u.essid.flags=IW_SCAN_THIS_ESSID;
  data.u.essid.length=0;

  if (ioctl(handle,SIOCSIWSCAN,&data)>=0) {
    int    counter=3;
    size_t buflen=IW_SCAN_MAX_DATA;
    char   *buffer=(char*)malloc(IW_SCAN_MAX_DATA);
    bool   success=false;

    memset(buffer,0,IW_SCAN_MAX_DATA);

    while (counter>0) {
      std::cerr << "Try reading scan results..." << counter << std::endl;
      memset(&data,0,sizeof(struct iwreq));
      sprintf(data.ifr_ifrn.ifrn_name,interface.c_str());
      data.u.data.pointer=buffer;
      data.u.data.flags=0;
      data.u.data.length=IW_SCAN_MAX_DATA;

      if (ioctl(handle,SIOCGIWSCAN,&data)<0) {
        if (errno==E2BIG) {
          std::cerr << "To much data!" << std::endl;
          if (data.u.data.length>buflen) {
            buflen=data.u.data.length;
          }
          else {
            buflen*=2;
          }

          char* newBuffer=(char*)realloc(buffer,buflen);
          if (newBuffer==NULL) {
            std::cerr << "Cannot allocate buffer of size " << buflen << " for scan result!" << std::endl;
            free(buffer);
            buffer=NULL;
            break;
          }
          else {
            buffer=newBuffer;
          }
        }
        else if (errno==EAGAIN) {
          std::cout << "We have to wait for the result..." << std::endl;
          counter--;
          sleep(1);
        }
        else {
          std::cout << "Error: " << errno << " text: " << strerror(errno) << std::endl;
          break;
        }
      }
      else {
        std::cerr << "Sucessfully scanned..." << std::endl;
        success=true;
        break;
      }
    }

    if (success) {
      std::cout << "successfully read data block of size " << data.u.data.length << "..." << std::endl;
      if (data.u.data.length>0) {
        size_t          readLength=0;
        Status::Network network;
        bool            filled=false;

        while (readLength+IW_EV_LCP_LEN<data.u.data.length) {
          struct iw_event iwe;

          memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,IW_EV_LCP_LEN);

          //std::cout << "command: " << iwe.cmd << " data length: " << iwe.len << std::endl;

          if (iwe.len<=IW_EV_LCP_LEN) {
            std::cerr << " Invalid event length, quitting..." << std::endl;
            break;
          }

          if (iwe.cmd==SIOCGIWAP) {
            std::cout << "------------" << std::endl;
            if (filled) {
              if (!network.essid.empty()) {
                status.networks.push_back(network);
              }
              network.essid.clear();
              filled=false;
            }
          }
          else if (iwe.cmd==SIOCGIWESSID) {
            memcpy((char*)&iwe+IW_EV_LCP_LEN+IW_EV_POINT_OFF,
                   ((char*)data.u.data.pointer)+readLength+IW_EV_LCP_LEN,IW_EV_POINT_LEN);

            iwe.u.essid.pointer=(char*)data.u.data.pointer+readLength+IW_EV_LCP_LEN+IW_EV_POINT_LEN-IW_EV_POINT_OFF;

            if (iwe.u.essid.pointer!=NULL && iwe.u.essid.length>0) {
              std::string tmp;

              if (((const char*)iwe.u.essid.pointer)[iwe.u.essid.length-1]=='\0') {
                tmp.assign((const char*)iwe.u.essid.pointer,iwe.u.essid.length-1);
              }
              else {
                tmp.assign((const char*)iwe.u.essid.pointer,iwe.u.essid.length);
              }

              network.essid=Lum::Base::StringToWString(tmp);
              std::cout << "ESSID: '" << tmp << "'" << std::endl;
            }
            else {
              std::cerr << "No essid!" << std::endl;
            }
            filled=true;
          }
          else if (iwe.cmd==SIOCGIWMODE) {
            memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,iwe.len);

            network.mode=iwe.u.mode;
            //std::cout << "Mode: " << network.mode << std::endl;
          }
          else if (iwe.cmd==SIOCGIWRATE) {
            memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,IW_EV_LCP_LEN+IW_EV_PARAM_LEN);

            network.bitrate=iwe.u.bitrate.value;
            std::cout << "Bitrate: " << network.bitrate << std::endl;
          }
          else if (iwe.cmd==SIOCGIWFREQ) {
            //std::cout << "Frequence: " << std::endl;
          }
          else if (iwe.cmd==SIOCGIWENCODE) {
            //std::cout << "Encoding: " << std::endl;
          }
          else if (iwe.cmd==IWEVQUAL) {
            memcpy((char*)&iwe,((char*)data.u.data.pointer)+readLength,IW_EV_LCP_LEN+IW_EV_QUAL_LEN);

            network.quality=iwe.u.qual.qual;
            std::cout << "Quality: " << network.quality << std::endl;
          }
          else if (iwe.cmd==IWEVCUSTOM) {
            std::cout << "Custom: " << std::endl;
          }
          else if (iwe.cmd==IWEVGENIE) {
            std::cout << "Genie: " << std::endl;
          }
          else {
            std::cout << "Read command " << iwe.cmd << " length " << iwe.len << std::endl;
          }

          readLength+=iwe.len;
        }

        std::cout << "Scanning done!" << std::endl;

        if (filled) {
          if (!network.essid.empty()) {
            status.networks.push_back(network);
          }
          network.essid.clear();
          filled=false;
        }
      }
    }
    else {
      std::cout << "Timeout!" << std::endl;
    }

    free(buffer);
  }
  else if (errno!=EPERM) {
    std::cerr << "Cannot initiate scan: " << strerror(errno) << std::endl;
  }
  else {
    std::cerr << "Scanning network not allowed!" << std::endl;
  }
#endif
  close(handle);

  return true;
#else
  return false;
#endif
}

class MyWindow : public Lum::Dialog
{
private:
  Lum::Model::StringRef    type;
  Lum::Model::StringRef    essid;
  Lum::Model::StringRef    accesspoint;
  Lum::Model::StringRef    client;
  Lum::Model::DoubleRef    quality;
  Lum::Model::IntRef       refreshTime;
  Lum::Model::ListTableRef tableModel;

  Lum::Model::ActionRef    timer;

private:
  Lum::TextValue* CreateTextValue(Lum::Model::StringRef& model, size_t size)
  {
    Lum::TextValue *textView;

    textView=new Lum::TextValue();
    textView->SetFlex(true,false);
    textView->SetWidth(Lum::Base::Size::stdCharWidth,size);
    textView->SetModel(model);

    return textView;
  }

  void UpdateDisplay()
  {
    Status       status;
    std::wstring tmp;

    if (GetStatus(status) && status.bitrate>0 && !status.essid.empty()) {
      essid->Set(status.essid);

      if (status.type.empty()) {
        type->Set(L"");
      }
      else {
        std::wstring tmr;

        tmr=status.type;

        tmr.append(L" ");

        switch (status.mode) {
#if defined(linux)
        case IW_MODE_AUTO:
          tmr.append(L"Auto");
          break;
        case IW_MODE_ADHOC:
          tmr.append(L"Ad-Hoc");
          break;
        case IW_MODE_INFRA:
          tmr.append(L"Infrastructure");
          break;
        case IW_MODE_MASTER:
          tmr.append(L"Master");
          break;
        case IW_MODE_REPEAT:
        tmr.append(L"Repeater");
          break;
        case IW_MODE_SECOND:
          tmr.append(L"Secondary master");
          break;
        case IW_MODE_MONITOR:
          tmr.append(L"Passive monitor");
          break;
#endif
        default:
          tmr.append(L"-");
        }

        tmr.append(L" ");

        if (status.bitrate==0) {
          tmr.append(L"-");
        }
        else if (status.bitrate>1000000 && status.bitrate==(status.bitrate/1000000)*1000000) {
          tmr.append(Lum::Base::NumberToWString(status.bitrate/1000000));
          tmr.append(L"Mb/s");
        }
        else if (status.bitrate>1000 && status.bitrate==(status.bitrate/1000)*1000) {
          tmr.append(Lum::Base::NumberToWString(status.bitrate/1000));
          tmr.append(L"Kb/s");
        }
        else {
        tmr.append(Lum::Base::NumberToWString(status.bitrate));
          tmr.append(L"b/s");
        }

        type->Set(tmr);
      }

      accesspoint->Set(status.accesspoint);

      client->Set(std::wstring(status.ip)+L" ("+status.mac+L")");

      if (status.quality>=0 && status.quality<=100) {
        quality->Set(status.quality);
      }
      else {
        quality->SetNull();
      }
    }
    else {
      type->Set(L"-");
      essid->Set(L"-");
      accesspoint->Set(L"-");
      client->Set(L"-");
      quality->SetNull();
    }

    std::list<Status::Network>::const_iterator iter;

    // First try to update existing entries in the current table
    iter=status.networks.begin();
    while (iter!=status.networks.end()) {
      size_t i;

      // Find the right entry
      for (i=1; i<=tableModel->GetRows(); i++) {
        if (iter->essid==tableModel->GetEntry(i)->GetString(1)) {
          break;
        }
      }

      if (i<=tableModel->GetRows()) { // Found an entry => Update
        Lum::Model::ListTable::Entry    *entry=tableModel->GetEntry(i);
        Lum::FuelGauge                  *gauge;
        Lum::Model::DoubleRef           data;

        gauge=dynamic_cast<Lum::FuelGauge*>(entry->GetObject(2));
        data=dynamic_cast<Lum::Model::Double*>(gauge->GetModel());

        data->Set(iter->quality);
      }
      else { // No entry => add
        Lum::FuelGauge                  *gauge;
        Lum::Model::DoubleRef           data;
        Lum::Model::ListTable::StdEntry *entry;

        gauge=new Lum::FuelGauge();
        gauge->SetFlex(true,true);
        //gauge->SetMinWidth(Lum::Base::Size::stdCharWidth,30);

        data=new Lum::Model::Double();
        data->SetRange(0,100);
        data->Set(iter->quality);
        gauge->SetModel(data);

        entry=new Lum::Model::ListTable::StdEntry(tableModel);
        entry->SetString(1,iter->essid);
        entry->SetObject(2,gauge);

        tableModel->Append(entry);
      }

      ++iter;
    }

    // Now see, if entries in the table have been removed and must
    // be deleted
    size_t i=1;

    while(i<=tableModel->GetRows()) {
      std::wstring                               essid=tableModel->GetEntry(i)->GetString(1);
      std::list<Status::Network>::const_iterator iter;

      // Find essid from table entry in network list
      iter=status.networks.begin();
      while (iter!=status.networks.end()) {
        if (iter->essid==essid) {
          break;
        }

        ++iter;
      }

      if (iter==status.networks.end()) { // table entry is not in list
        tableModel->Delete(i);
      }
      else {
        i++;
      }
    }
  }

public:
  MyWindow()
  : type(new Lum::Model::String()),
    essid(new Lum::Model::String()),
    accesspoint(new Lum::Model::String()),
    client(new Lum::Model::String()),
    quality(new Lum::Model::Double()),
    refreshTime(new Lum::Model::Int(5)),
    tableModel(new Lum::Model::ListTable(2)),
    timer(new Lum::Model::Action())
  {
    quality->SetRange(0,100);
    refreshTime->SetRange(0,10);

    AttachModel(refreshTime);
    AttachModel(timer);
    AttachModel(GetOpenedAction());
  }

  ~MyWindow()
  {
    UnattachModel(GetOpenedAction());
    UnattachModel(timer);
    UnattachModel(refreshTime);
  }

  virtual void PreInit()
  {
    Lum::FuelGauge   *fuelgauge;
    Lum::Label       *label;
    Lum::Panel       *hPanel,*vPanel;
    Lum::Slider      *slider;
    //Lum::Table                         *table;
    //Lum::Model::SingleLineSelectionRef selection;
    Lum::Text        *text;
    Lum::WindowGroup *wGroup;
    std::wstring     tmp;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    hPanel=new Lum::HPanel();
    hPanel->SetFlex(true,true);

    label=new Lum::Label();
    label->SetFlex(true,true);

    label->AddLabel(L"ESSID:",CreateTextValue(essid,IW_ESSID_MAX_SIZE));
    label->AddLabel(L"Type:",CreateTextValue(type,42));
    label->AddLabel(L"AP:",CreateTextValue(accesspoint,17));
    label->AddLabel(L"Client:",CreateTextValue(client,28));

    fuelgauge=new Lum::FuelGauge();
    fuelgauge->SetFlex(true,false);
    fuelgauge->SetModel(quality);
    label->AddLabel(L"Quality:",fuelgauge);

    /*
    selection=new Lum::Model::SingleLineSelection;

    tableModel->SetLabel(1,L"ESSID");
    tableModel->SetColumnWidth(1,Lum::Base::Size::stdCharWidth,15,true);
    tableModel->SetLabel(2,L"Size");
    tableModel->SetColumnWidth(2,Lum::Base::Size::stdCharWidth,15);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetModel(tableModel);
    //table->SetSelection(selection);
    //table->SetSelectionAction(showDetail);
    //table->SetDoubleClickAction(showDetail);
    table->SetShowHeader(true);
    table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,4);
    table->GetTableView()->SetAutoHFit(true);
    label->AddLabel(L"Networks:",table);*/

    hPanel->Add(label);

    hPanel->Add(new Lum::HSpace());

    vPanel=new Lum::VPanel();
    vPanel->SetFlex(false,true);

    text=new Lum::Text();
    text->SetFlex(true,false);
    text->SetAlignment(Lum::Text::centered);
    text->SetText(L"Poll (s)");
    vPanel->Add(text);

    vPanel->Add(new Lum::VSpace());

    slider=new Lum::VSlider();
    slider->SetFlex(false,true);
    slider->SetScale(true);
    slider->SetModel(refreshTime);

    vPanel->Add(slider);

    hPanel->Add(vPanel);

    wGroup->SetMain(hPanel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      UpdateDisplay();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==timer && timer->IsFinished()) {
      UpdateDisplay();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==refreshTime && IsOpen()) {
      Lum::OS::display->RemoveTimer(timer);
      if (refreshTime->Get()>0) {
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }
    }

    Dialog::Resync(model,msg);
  }
};

int main(int argc, char* argv[])
{
  if (Lum::OS::prober->Open(L"WifiInfo",argc,argv)) {

    Lum::Dialog *window;

    window=new MyWindow;

    if (window->Open()) {
      window->SetExitAction(window->GetClosedAction());
      window->EventLoop();
      window->Close();
    }

    delete window;

    Lum::OS::display->Close();
  }
}
