/*
  This file is part of "WhoAmI" - A "Guess whats on the photo" riddle game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Control.h>

#include <Lum/OS/Driver.h>

#define PLATE_ROWS    5
#define PLATE_COLUMNS 5

Control::Control()
 : showAll(false)
{
  // no code
}

Control::~Control()
{
  // delete bitmap;
}

void Control::SetImage(Lum::Images::Image *image)
{
  this->image=image;

  plateState.resize(PLATE_ROWS*PLATE_COLUMNS);
  for (size_t i=0; i<PLATE_ROWS*PLATE_COLUMNS; i++) {
    plateState[i]=false;
  }

  Redraw();
}

void Control::ToggleShowAll()
{
  showAll=!showAll;

  Redraw();
}

void Control::CalcSize()
{
  width=250;
  height=200;

  minWidth=width;
  minHeight=height;

  Object::CalcSize();
}

void Control::Draw(int x, int y, size_t w, size_t h)
{
  Object::Draw(x,y,w,h);

  if (!Intersect(x,y,w,h)) {
    return;
  }

  /* --- */

  if (!image.Valid()) {
    DrawBackground(x,y,w,h);
    return;
  }

  Lum::OS::DrawInfo *draw;
  Lum::OS::Bitmap   *bitmap=Lum::OS::driver->CreateBitmap(width,height);
  size_t nw,nh;

  if (bitmap==NULL) {
    DrawBackground(x,y,w,h);
    return;
  }

  draw=bitmap->GetDrawInfo();

  draw->PushForeground(Lum::OS::display->GetColor(Lum::OS::Display::blackColor));
  draw->FillRectangle(0,0,width,height);
  draw->PopForeground();

  nw=image->GetWidth();
  nh=image->GetHeight();

  if (nw<width) {
    nh=(nh*width)/nw;
    nw=width;
  }

  if (nh<height) {
    nw=(nw*height)/nh;
    nh=height;
  }

  if (nw>width) {
    nh=(nh*width)/nw;
    nw=width;
  }

  if (nh>height) {
    nw=(nw*height)/nh;
    nh=height;
  }

  image->Resize(nw,nh);
  image->Draw(draw,(width-image->GetWidth())/2,(height-image->GetHeight())/2);

  if (!showAll) {
    for (size_t i=0; i<PLATE_ROWS*PLATE_COLUMNS; i++) {
      if (!plateState[i]) {
        draw->PushForeground(Lum::OS::display->GetColor(Lum::OS::Display::blackColor));
        draw->FillRectangle((i%PLATE_COLUMNS)*(width/PLATE_COLUMNS),
                            (i/PLATE_COLUMNS)*(height/PLATE_ROWS),
                            width/PLATE_COLUMNS,height/PLATE_ROWS);
        draw->PopForeground();
      }
    }
  }

  GetDrawInfo()->CopyFromBitmap(bitmap,x-this->x,y-this->y,w,h,x,y);

  delete bitmap;
}

bool Control::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (showAll || !image.Valid()) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down &&
      PointIsIn(event) &&
      event.button==Lum::OS::MouseEvent::button1) {

    for (size_t i=0; i<PLATE_ROWS*PLATE_COLUMNS; i++) {
      if(event.x>=this->x+(int)((i%PLATE_COLUMNS)*width/PLATE_COLUMNS) &&
         event.y>=this->y+(int)((i/PLATE_COLUMNS)*height/PLATE_ROWS) &&
         event.x<this->x+(int)((i%PLATE_COLUMNS)*width/PLATE_COLUMNS+width/PLATE_COLUMNS) &&
         event.y<this->y+(int)((i/PLATE_COLUMNS)*height/PLATE_ROWS+height/PLATE_ROWS)) {
        plateState[i]=!plateState[i];
        Redraw(this->x+(i%PLATE_COLUMNS)*width/PLATE_COLUMNS,
               this->y+(i/PLATE_COLUMNS)*height/PLATE_ROWS,
               width/PLATE_COLUMNS,height/PLATE_ROWS);
        return true;
      }
    }
  }

  return false;
 }

