/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Statistics.h"

#include <Lum/Base/L10N.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>

#include "Configuration.h"
#include "Util.h"

Statistics::Statistics()
{
  SetExitAction(GetClosedAction());
}

Statistics::~Statistics()
{
  // no code
}

void Statistics::PreInit()
{
  Lum::Label       *label;
  Lum::Panel       *panel;

  size_t           minWeight=0,maxWeight=0;
  Value            minValue=0, maxValue=0;
  Value            avgValue=0;
  size_t           avgValueDays=0;

  for (std::map<Lum::Base::Calendar,Day*>::const_iterator iter=data.data.begin();
       iter!=data.data.end();
       ++iter) {
    Value value=0;

    for (std::list<Entry>::const_iterator entry=iter->second->entries.begin();
         entry!=iter->second->entries.end();
         ++entry) {
      if (entry->item.empty()) {
        if (minWeight==0) {
          minWeight=entry->amount;
          maxWeight=entry->amount;
        }
        else {
          minWeight=std::min(minWeight,entry->amount);
          maxWeight=std::max(maxWeight,entry->amount);
        }
      }
      else {
        value+=entry->amount*entry->value;
      }
    }

    if (minValue==0 && maxValue==0) {
      minValue=value;
      maxValue=value;
    }
    else {
      minValue=std::min(minValue,value);
      maxValue=std::max(maxValue,value);
    }

    avgValue+=value;
    avgValueDays++;
  }

  minValue/=10;
  maxValue/=10;

  panel=new Lum::VPanel();

  label=new Lum::Label();
  label->SetFlex(true,true);

  std::wstring tmp;

  label->AddLabel(_(L"STAT_LABEL_WEIGHT_MIN",L"Weight minimum:"),AmountToString(minWeight)+L" "+weightUnit->Get());
  label->AddLabel(_(L"STAT_LABEL_WEIGHT_MAX",L"Weight maximum:"),AmountToString(maxWeight)+L" "+weightUnit->Get());
  label->AddLabel(_(L"STAT_LABEL_WEIGHT_DIFF",L"Weight difference:"),AmountToString(maxWeight-minWeight)+L" "+weightUnit->Get());
  label->AddLabel(_(L"STAT_LABEL_VALUE_MIN",L"Minimum value:"),ValueToString(minValue)+L" "+valueUnit->Get());
  if (avgValueDays>0) {
    label->AddLabel(_(L"STAT_LABEL_VALUE_AVG",L"Average value:"),ValueToString(avgValue/avgValueDays)+L" "+valueUnit->Get());
  }
  label->AddLabel(_(L"STAT_LABEL_VALUE_MAX",L"Maximum value:"),ValueToString(maxValue)+L" "+valueUnit->Get());

  panel->Add(label);
  panel->AddSpace();
  panel->Add(Lum::ButtonRow::CreateClose(GetClosedAction(),true,false));

  SetMain(panel);

  Dialog::PreInit();
}

void Statistics::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  Dialog::Resync(model,msg);
}
