/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DataInput.h"

#include <Lum/Base/L10N.h>

#include <Lum/Array.h>
#include <Lum/Button.h>
#include <Lum/DateSelect.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/TextValue.h>
#include <Lum/TimeSelect.h>

#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Values.h>

#include <Lum/Model/Header.h>
#include <Lum/Model/String.h>

#include "Configuration.h"
#include "Trend.h"
#include "Util.h"

#include <iostream>
static DataInput::Prefs* prefs=new DataInput::Prefs();

class WeightEdit : public Lum::Dlg::ValuesInput
{
public:
  WeightEdit(Lum::Model::Time* time,
             Lum::Model::String* weight)
  {
    Lum::String     *string;
    Lum::TimeSelect *timeSelect;
    Value           v;
    std::wstring    tmp;

    timeSelect=new Lum::TimeSelect();
    timeSelect->SetFlex(false,false);

    v.label=_(L"WEIGHT_EDIT_LABEL_TIME",L"Time:");
    v.model=time;
    v.control=timeSelect;
    v.validator=NULL;
    AddValue(v);

    string=new Lum::String();
    string->SetFlex(false,false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,6);

    tmp=_(L"WEIGHT_EDIT_LABEL_WEIGHT",L"Weight ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",weightUnit->Get());
    v.label=tmp;
    v.model=weight;
    v.control=string;
    v.validator=new AmountValidator();
    AddValue(v);
  }

  static bool EditWeight(Lum::OS::Window* parent,
                         const std::wstring& caption,
                         const std::wstring& text,
                         Lum::Model::Time* time,
                         Lum::Model::String* weight)
  {
    assert(time!=NULL && weight!=NULL);

    WeightEdit *dlg;
    bool       res;

    dlg=new WeightEdit(time,weight);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodEdit : public Lum::Dlg::ValuesInput
{
public:
  FoodEdit(Lum::Model::String* name,
           Lum::Model::String* unit,
           Lum::Model::String* value)
  {
    Lum::String  *string;
    Value        v;
    std::wstring tmp;

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,20);

    v.label=_(L"FOOD_EDIT_LABEL_NAME",L"Name:");
    v.model=name;
    v.control=string;
    v.validator=new Lum::Dlg::InputValidatorStringNotEmpty();
    AddValue(v);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,6);

    v.label=_(L"FOOD_EDIT_LABEL_UNIT",L"Unit:");
    v.model=unit;
    v.control=string;
    v.validator=new Lum::Dlg::InputValidatorStringNotEmpty();
    AddValue(v);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);

    tmp=_(L"FOOD_EDIT_LABEL_VALUE",L"Value ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",valueUnit->Get());
    v.label=tmp;
    v.model=value;
    v.control=string;
    v.validator=new ValueValidator();
    AddValue(v);
  }

  static bool EditFood(Lum::OS::Window* parent,
                       const std::wstring& caption,
                       const std::wstring& text,
                       Lum::Model::String* name,
                       Lum::Model::String* unit,
                       Lum::Model::String* value)
  {
    assert(name!=NULL && unit!=NULL && value!=NULL);

    FoodEdit *dlg;
    bool     res;

    dlg=new FoodEdit(name,unit,value);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class FoodRegister : public Lum::Dlg::ValuesInput
{
public:
  FoodRegister(Lum::Model::Time* time,
          Lum::Model::String* amount,
          const std::wstring& unit)
  {
    Lum::String     *string;
    Lum::TimeSelect *timeSelect;
    Value           v;
    std::wstring    tmp;

    timeSelect=new Lum::TimeSelect();
    timeSelect->SetFlex(false,false);

    v.label=_(L"FOOD_REGISTER_LABEL_TIME",L"Time:");
    v.model=time;
    v.control=timeSelect;
    v.validator=NULL;
    AddValue(v);

    string=new Lum::String();
    string->SetFlex(false,false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,4);

    tmp=_(L"FOOD_REGISTER_LABEL_AMOUNT",L"Amount ($unit):");
    Lum::Base::L10N::Substitute(tmp,L"$unit",unit);
    v.label=tmp;
    v.model=amount;
    v.control=string;
    v.validator=new AmountValidator();
    AddValue(v);
  }

  static bool RegisterFood(Lum::OS::Window* parent,
                           const std::wstring& caption,
                           const std::wstring& text,
                           Lum::Model::Time* time,
                           Lum::Model::String* amount,
                           const std::wstring& unit)
  {
    assert(time!=NULL && amount!=NULL);

    FoodRegister *dlg;
    bool    res;

    dlg=new FoodRegister(time,amount,unit);
    dlg->SetParent(parent);
    dlg->SetTitle(L""/*caption*/);
    dlg->SetCaption(caption);
    dlg->SetText(text);

    if (dlg->Open()) {
      dlg->EventLoop();
      dlg->Close();
    }

    res=dlg->HasResult();

    delete dlg;

    return res;
  }
};

class EntriesEntry : public Lum::Model::ListTable::Entry
{
public:
  ::Entry *entry;

public:
  EntriesEntry(Lum::Model::ListTable *table, ::Entry* entry)
  : Lum::Model::ListTable::Entry(table),
    entry(entry)
  {
    // no code
  }

  std::wstring GetString(size_t column) const
  {
    if (column==1) {
      std::wstring res;

      res=Lum::Base::NumberToWString(entry->time.GetHour());
      res.append(L":");
      if (entry->time.GetMinute()<10) {
        res.append(L"0");
      }
      else {
        res.append(Lum::Base::NumberToWString(entry->time.GetMinute()/10));
      }
      res.append(Lum::Base::NumberToWString(entry->time.GetMinute()%10));

      return res;
    }
    else if (column==2) {
      if (entry->item.empty()) {
        return _(L"LABEL_WEIGHT",L"Weight");
      }
      else {
        return entry->item+L" ("+AmountToString(entry->amount)+L" "+entry->unit+L")";
      }
    }
    else if (column==3) {
      if (entry->item.empty()) {
        return AmountToString(entry->amount)+L" "+weightUnit->Get();
      }
      else {
        return ValueToString((((Value)entry->amount)*entry->value)/10)+L" "+valueUnit->Get();
      }
    }

    assert(false);
  }

  bool IsGreater(const Entry* other, size_t column)
  {
    return entry->time>dynamic_cast<const EntriesEntry*>(other)->entry->time;
  }
};

class FoodsEntry : public Lum::Model::ListTable::Entry
{
public:
  const ::Food *entry;

public:
  FoodsEntry(Lum::Model::ListTable *table,
             const ::Food* entry)
  : Lum::Model::ListTable::Entry(table),
    entry(entry)
  {
    // no code
  }

  std::wstring GetString(size_t column) const
  {
    if (column==1) {
      return entry->name+L" ("+entry->unit+L")";
    }
    else if (column==2) {
      return ValueToString(entry->value)+L" "+valueUnit->Get();
    }

    assert(false);
  }

  bool IsGreater(const Entry* other, size_t column)
  {
    return entry->name>dynamic_cast<const FoodsEntry*>(other)->entry->name;
  }
};

class FoodFormatProvider : public Lum::TableView::FormatProvider
{
private:
  Lum::Model::ListTable *table;
  Lum::OS::ColorRef     goodFoodColor;
  Lum::OS::ColorRef     badFoodColor;
  Lum::OS::FillRef      goodFoodBackground;
  Lum::OS::FillRef      badFoodBackground;

public:
  FoodFormatProvider(Lum::Model::ListTable* table)
   : table(table),
     goodFoodColor(0.85,0.95,0.85,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     badFoodColor(0.95,0.85,0.85,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
     goodFoodBackground(new Lum::OS::PlainFill(goodFoodColor)),
     badFoodBackground(new Lum::OS::PlainFill(badFoodColor))
  {
    // no code
  }

  Lum::OS::Fill* GetBackground(size_t column, size_t row) const
  {
    FoodsEntry* entry=(FoodsEntry*)table->GetEntry(row);

    if (entry->entry->value<=(Value)goodValueLimit->Get()) {
      return goodFoodBackground.Get();
    }
    else if (entry->entry->value>=(Value)badValueLimit->Get()) {
      return badFoodBackground.Get();
    }
    else {
      return Lum::TableView::FormatProvider::GetBackground(column,row);
    }
  }
};

DataInput::Prefs::Prefs()
 : Lum::Component::Prefs(L"DataInput")
{
  // no code
}

void DataInput::Prefs::Initialize()
{
  Lum::Component::Prefs::Initialize();

  frame=new Lum::OS::EmptyFrame();
}

DataInput::DataInput()
 : prevDayAction(new Lum::Model::Action()),
   todayAction(new Lum::Model::Action()),
   nextDayAction(new Lum::Model::Action()),

   addFoodAction(new Lum::Model::Action()),
   removeFoodAction(new Lum::Model::Action()),
   editFoodAction(new Lum::Model::Action()),
   registerFoodAction(new Lum::Model::Action()),

   addWeightAction(new Lum::Model::Action()),
   removeItemAction(new Lum::Model::Action()),
   editItemAction(new Lum::Model::Action()),

   foodSearch(new Lum::Model::String(L"")),
   averageDiff(new Lum::Model::String(L"")),
   summary(new Lum::Model::String(L"")),
   time(new Lum::Model::Time()),
   foods(new Lum::Model::ListTable(3)),
   entries(new Lum::Model::ListTable(3)),
   foodSelection(new Lum::Model::SingleLineSelection()),
   entrySelection(new Lum::Model::SingleLineSelection()),

   searchTimerAction(new Lum::Model::Action()),
   trend(new Lum::Model::Int(Trend::none))
{
  SetPrefs(::prefs);

  date->SetToCurrent();
  time->SetToNow();

  AttachModel(foodSearch);
  AttachModel(date);

  AttachModel(prevDayAction);
  AttachModel(todayAction);
  AttachModel(nextDayAction);

  AttachModel(addFoodAction);
  AttachModel(removeFoodAction);
  AttachModel(editFoodAction);
  AttachModel(registerFoodAction);

  AttachModel(addWeightAction);
  AttachModel(removeItemAction);
  AttachModel(editItemAction);

  AttachModel(searchTimerAction);

  AttachModel(goodValueLimit);
  AttachModel(badValueLimit);

  AttachModel(trendType);
  AttachModel(trendDirection);
  AttachModel(refValue);
}

DataInput::~DataInput()
{
  UnattachModel(refValue);
  UnattachModel(trendDirection);
  UnattachModel(trendType);

  UnattachModel(goodValueLimit);
  UnattachModel(badValueLimit);

  UnattachModel(editItemAction);
  UnattachModel(removeItemAction);
  UnattachModel(addWeightAction);

  UnattachModel(registerFoodAction);
  UnattachModel(editFoodAction);
  UnattachModel(removeFoodAction);
  UnattachModel(addFoodAction);

  UnattachModel(nextDayAction);
  UnattachModel(todayAction);
  UnattachModel(prevDayAction);

  UnattachModel(date);
  UnattachModel(foodSearch);

  UnattachModel(searchTimerAction);
}

void DataInput::CalcSize()
{
  Lum::Button           *button;
  Lum::DateSelect       *dateSelect;
  Lum::Menu             *menu;
  Lum::Panel            *ft;
  Lum::Panel            *et;
  Lum::Panel            *ds;
  Lum::Panel            *sb;
  Lum::Panel            *horiz;
  Lum::Panel            *vert;
  Lum::String           *string;
  Lum::Table            *table;
  Lum::TextValue        *text;
  Lum::Model::HeaderRef headerModel;
  Trend                 *trend;

  // Search box

  sb=new Lum::VPanel();
  sb->SetFlex(true,false);

  string=new Lum::String();
  string->SetFlex(true,false);
  string->SetModel(foodSearch);
  sb->Add(string);

  // Food table

  ft=new Lum::HPanel();
  ft->SetFlex(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"FOOD_TABLE_HEADER_FOOD",L"Food"),Lum::Base::Size::stdCharWidth,15,true);
  headerModel->AddColumn(_(L"FOOD_TABLE_HEADER_VALUE",L"Value"),Lum::Base::Size::stdCharWidth,8);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  table->SetHeaderModel(headerModel);
  table->SetModel(foods);
  table->SetSelection(foodSelection);
  table->SetDoubleClickAction(registerFoodAction);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoFitColumns(true);
  //table->GetTableView()->SetAutoHSize(true);
  table->GetTableView()->SetFormatProvider(new FoodFormatProvider(foods));
  ft->Add(table);

  menu=new Lum::Menu();
  menu->SetParent(GetWindow());
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_ADD",L"_Add entry..."),addFoodAction);
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_EDIT",L"_Edit entry..."),editFoodAction);
  menu->AddActionItem(_(L"FOOD_TABLE_MENU_REMOVE",L"_Remove entry"),removeFoodAction);
  table->SetMenu(menu->GetWindow());

  // Date select

  ds=new Lum::HPanel();
  ds->SetFlex(true,false);

  button=new Lum::Button(L"_<");
  button->SetModel(prevDayAction);
  ds->Add(button);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));

  dateSelect=new Lum::DateSelect();
  dateSelect->SetModel(date);
  ds->Add(dateSelect);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));

  button=new Lum::Button(L"_>");
  button->SetModel(nextDayAction);
  ds->Add(button);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));

  trend=new Trend();
  trend->SetModel(this->trend);
  ds->Add(trend);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));

  text=new Lum::TextValue();
  text->SetFlex(true,false);
  text->SetAlignment(Lum::TextValue::right);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,6);
  text->SetModel(averageDiff);
  ds->Add(text);

  ds->Add(new Lum::HSpace(Lum::Space::sizeSmall));

  text=new Lum::TextValue();
  text->SetFlex(true,false);
  text->SetAlignment(Lum::TextValue::right);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,9);
  text->SetModel(summary);
  ds->Add(text);

  // Entries table

  et=new Lum::HPanel();
  et->SetFlex(true,true);

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(_(L"ENTRIES_TABLE_HEADER_TIME",L"Time"),Lum::Base::Size::stdCharWidth,5);
  headerModel->AddColumn(_(L"ENTRIES_TABLE_HEADER_ITEM",L"Item"),Lum::Base::Size::stdCharWidth,15,true);
  headerModel->AddColumn(_(L"ENTRIES_TABLE_HEADER_VALUE",L"Value"),Lum::Base::Size::stdCharWidth,8);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,30);
  table->SetHeaderModel(headerModel);
  table->SetModel(entries);
  table->SetSelection(entrySelection);
  table->SetDoubleClickAction(editItemAction);
  table->SetShowHeader(true);
  table->GetTableView()->SetAutoFitColumns(true);
  //table->GetTableView()->SetAutoHSize(true);
  et->Add(table);

  menu=new Lum::Menu();
  menu->SetParent(GetWindow());
  menu->AddActionItem(_(L"ENTRIES_TABLE_MENU_ADD",L"_Add weight..."),addWeightAction);
  menu->AddActionItem(_(L"ENTRIES_TABLE_MENU_EDIT",L"_Edit entry..."),editItemAction);
  menu->AddActionItem(_(L"ENTRIES_TABLE_MENU_REMOVE",L"_Remove entry"),removeItemAction);
  table->SetMenu(menu->GetWindow());

  // Layout comonents

  horiz=new Lum::HPanel();
  horiz->SetFlex(true,true);

  vert=new Lum::VPanel();
  vert->SetFlex(true,true);

  vert->Add(sb);
  vert->Add(new Lum::VSpace());
  vert->Add(ft);


  horiz->Add(vert);
  horiz->Add(new Lum::HSpace());

  vert=new Lum::VPanel();
  vert->SetFlex(true,true);

  vert->Add(ds);
  vert->Add(new Lum::VSpace());
  vert->Add(et);

  horiz->Add(vert);

  container=horiz;

  container->SetParent(this);
  container->CalcSize();

  minWidth=container->GetOMinWidth();
  minHeight=container->GetOMinHeight();
  width=container->GetOWidth();
  height=container->GetOHeight();

  Component::CalcSize();
}

void DataInput::UpdateEntries()
{
  if (!entries.Valid() || !date.Valid()) {
    return;
  }

  entries->Clear();

  // Entries

  Day *day=data.GetDay(date->Get());


  if (day==NULL) {
    return;
  }

  entries->Off();

  for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
    entries->Append(new EntriesEntry(entries,&*iter));
  }

  entries->Sort(1);

  entries->On();
}

void DataInput::UpdateFoods()
{
  std::wstring filter=foodSearch->Get();
  // Food
  foods->Off();

  foods->Clear();
  for (std::set<Food>::iterator iter=data.foods.begin(); iter!=data.foods.end(); ++iter) {
    if (filter.empty() || iter->name.find(filter)!=std::wstring::npos) {
      foods->Append(new FoodsEntry(foods,&*iter));
    }
  }

  foods->Sort(1);

  foods->On();
}

void DataInput::UpdateSummary()
{
  Value current=0;

  Day *day=data.GetDay(date->Get());

  if (day!=NULL) {
    for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
      if (!iter->item.empty()) {
        current+=((Value)iter->amount)*iter->value;
      }
    }
  }

  if (current>0) {
    current=(current+5)/100; // Round
  }
  else {
    current=(current-5)/100; // Round
  }

  summary->Set(Lum::Base::NumberToWString(current)+
               L"/"+
               Lum::Base::NumberToWString(((int)maxValue->Get())/10-current));

  if (trendType->Get()==trendRefRel) {
    if (day!=NULL) {
      for (std::list<Entry>::reverse_iterator iter=day->entries.rbegin();
           iter!=day->entries.rend();
           ++iter) {
        if (iter->item.empty()) {
          if (iter->amount>=refValue->Get()) {
            averageDiff->Set(AmountToString(iter->amount-refValue->Get())+weightUnit->Get());
            if (trendDirection->Get()!=1) {
              trend->Set(Trend::up);
            }
            else {
              trend->Set(Trend::down);
            }
          }
          else {
            averageDiff->Set(std::wstring(L"-")+AmountToString(refValue->Get()-iter->amount)+weightUnit->Get());
            if (trendDirection->Get()!=1) {
              trend->Set(Trend::down);
            }
            else {
              trend->Set(Trend::up);
            }
          }
          return;
        }
      }
    }
    trend->Set(Trend::none);
    averageDiff->Set(L"?");
  }
  else if (trendType->Get()==trendTwoWeeks) {
    Lum::Base::Calendar d=date->Get();
    size_t weightLastWeek=0;
    size_t weightLastWeekCount=0;
    size_t weightPriorWeek=0;
    size_t weightPriorWeekCount=0;

    for (size_t i=1; i<=7; i++) {
      Day *day=data.GetDay(d);

      if (day!=NULL) {
        for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
          if (iter->item.empty()) {
            weightLastWeek+=iter->amount;
            weightLastWeekCount++;
          }
        }
      }

      d.AddDays(-1);
    }

    for (size_t i=1; i<=7; i++) {
      Day *day=data.GetDay(d);

      if (day!=NULL) {
        for (std::list<Entry>::iterator iter=day->entries.begin(); iter!=day->entries.end(); ++iter) {
          if (iter->item.empty()) {
            weightPriorWeek+=iter->amount;
            weightPriorWeekCount++;
          }
        }
      }

      d.AddDays(-1);
    }

    if (weightLastWeekCount==0 || weightPriorWeekCount==0) {
      averageDiff->Set(L"?");
      trend->Set(Trend::none);
    }
    else {
      weightLastWeek=weightLastWeek/weightLastWeekCount;
      weightPriorWeek=weightPriorWeek/weightPriorWeekCount;

      if (weightPriorWeek<=weightLastWeek) {
        averageDiff->Set(AmountToString(weightLastWeek-weightPriorWeek)+weightUnit->Get());
        if (trendDirection->Get()!=1) {
          trend->Set(Trend::up);
        }
        else {
          trend->Set(Trend::down);
        }
      }
      else {
        averageDiff->Set(std::wstring(L"-")+AmountToString(weightPriorWeek-weightLastWeek)+weightUnit->Get());
        if (trendDirection->Get()!=1) {
          trend->Set(Trend::down);
        }
        else {
          trend->Set(Trend::up);
        }
      }
    }
  }
  else {
    averageDiff->Set(L"???");
    trend->Set(Trend::none);
  }
}

void DataInput::AddFood()
{
  Lum::Model::StringRef name;
  Lum::Model::StringRef unit;
  Lum::Model::StringRef value;
  Value                 v;
  std::wstring          tmp;

  name=new Lum::Model::String(L"");
  unit=new Lum::Model::String(L"");
  value=new Lum::Model::String(L"");

  tmp=_(L"FOOD_ADD_TEXT",L"Please enter the name of the new food component together\n"
        L"with its standard portion unit and its value in $unit!");
  Lum::Base::L10N::Substitute(tmp,L"$unit",valueUnit->Get());
  if (!FoodEdit::EditFood(this->GetWindow(),
                          _(L"FOOD_ADD_TITLE",L"Please enter a new food component..."),
                          tmp,
                          name,unit,value)) {
    return;
  }

  if (name->IsNull() || unit->IsNull() || value->IsNull() ||
      name->Empty() || unit->Empty() || value->Empty()) {
    return;
  }

  if (!StringToValue(value->Get(),v)) {
    return;
  }

  Food       food;
  const Food *result;

  food.name=name->Get();
  food.unit=unit->Get();
  food.value=v;

  result=data.AddFood(food);

  if (result==NULL) {
    Lum::Dlg::Msg::ShowOk(this->GetWindow(),
                          _(L"DUPLICATE_TITLE",L"Duplicate!"),
                          _(L"DUPLICATE_TEXT",
                            L"You tried to insert a piece of food while there is already\n"
                            L"an entry with the identical name in the database.\n"
                            L"\n"
                            L"Please check and and give unique names to the existing\n"
                            L"and/or the new entry if you like to keep them both!"));
    return;
  }

  foods->Append(new FoodsEntry(foods,result));

  foods->Sort(1);
}

void DataInput::EditFood()
{
  Food       oldFood;
  FoodsEntry *entry=(FoodsEntry*)foods->GetEntry(foodSelection->GetLine());

  Lum::Model::StringRef name;
  Lum::Model::StringRef unit;
  Lum::Model::StringRef value;
  Value                 v;

  oldFood=*entry->entry;

  name=new Lum::Model::String(entry->entry->name);
  unit=new Lum::Model::String(entry->entry->unit);
  value=new Lum::Model::String(ValueToString(entry->entry->value));

  if (!FoodEdit::EditFood(this->GetWindow(),
                              L"Adding a new food component to the list...",
                              L"Please enter the name of the new food component together\n"
                              L"with its standard portion unit and its value!",
                              name,unit,value)) {
    return;
  }

  if (name->IsNull() || unit->IsNull() || value->IsNull() ||
      name->Empty() || unit->Empty() || value->Empty()) {
    return;
  }

  if (!StringToValue(value->Get(),v)) {
    return;
  }

  foods->Off();

  data.DeleteFood(*dynamic_cast<FoodsEntry*>(foods->GetEntry(foodSelection->GetLine()))->entry);
  foods->Delete(foodSelection->GetLine());

  Food       food;
  const Food *result;

  food.name=name->Get();
  food.unit=unit->Get();
  food.value=v;

  result=data.AddFood(food);

  if (result==NULL) {
    Lum::Dlg::Msg::ShowOk(this->GetWindow(),
                          _(L"DUPLICATE_TITLE",L"Duplicate!"),
                          _(L"DUPLICATE_TEXT",
                            L"You tried to insert a piece of food while there is already\n"
                            L"an entry with the identical name in the database.\n"
                            L"\n"
                            L"Please check and and give unique names to the existing\n"
                            L"and/or the new entry if you like to keep them both!"));

    result=data.AddFood(oldFood);
  }

  foods->Append(new FoodsEntry(foods,result));
  foods->Sort(1);

  foods->On();
}

void DataInput::RemoveFood()
{
  data.DeleteFood(*dynamic_cast<FoodsEntry*>(foods->GetEntry(foodSelection->GetLine()))->entry);
  foods->Delete(foodSelection->GetLine());
}

void DataInput::RegisterFood()
{
  FoodsEntry *entry=(FoodsEntry*)foods->GetEntry(foodSelection->GetLine());

  Lum::Model::StringRef value;
  size_t                amount;
  std::wstring          t1,t2;

  value=new Lum::Model::String(L"1");

  t1=_(L"FOOD_REGISTER_TITLE",L"I have consumed $food...");
  Lum::Base::L10N::Substitute(t1,L"$food",entry->entry->name);
  t2=_(L"FOOD_REGISTER_TEXT",L"Please enter the amount of $food\nyou have consumed!");
  Lum::Base::L10N::Substitute(t2,L"$food",entry->entry->name);
  if (!FoodRegister::RegisterFood(this->GetWindow(),
                                  t1,t2,time,value,entry->entry->unit)) {
    return;
  }

  if (value->IsNull() || value->Empty()) {
    return;
  }

  if (!StringToAmount(value->Get(),amount)) {
    return;
  }

  Entry e;

  e.time=time->Get();
  e.item=entry->entry->name;
  e.unit=entry->entry->unit;
  e.value=entry->entry->value;
  e.amount=amount;

  entries->Append(new EntriesEntry(entries,data.AddEntry(date->Get(),e)));

  entries->Sort(1);

  UpdateSummary();
}

void DataInput::AddWeight()
{
  Lum::Model::StringRef value;
  size_t                amount;

  value=new Lum::Model::String(L"");

  if (!WeightEdit::EditWeight(this->GetWindow(),
                              _(L"WEIGHT_EDIT_TITLE",L"My weight is..."),
                              _(L"WEIGHT_EDIT_TEXT",
                                L"Please enter your weight at the given time!\n"
                                L"\n"
                                L"(The weight should be entered as a decimal number\n"
                                L"with a maximum of one digit after the colon.)"),
                              time,
                              value)) {
    return;
  }

  if (value->IsNull()) {
    return;
  }

  if (!StringToAmount(value->Get(),amount)) {
    return;
  }

  Entry e;

  e.time=time->Get();
  e.item=L"";
  e.unit=L"";
  e.value=0;
  e.amount=amount;

  entries->Append(new EntriesEntry(entries,data.AddEntry(date->Get(),e)));

  entries->Sort(1);

  UpdateSummary();
}

void DataInput::RemoveItem()
{
  data.DeleteEntry(date->Get(),
                   dynamic_cast<EntriesEntry*>(entries->GetEntry(entrySelection->GetLine()))->entry);
  entries->Delete(entrySelection->GetLine());

  UpdateSummary();
}

void DataInput::EditItem()
{
  EntriesEntry *entry=(EntriesEntry*)entries->GetEntry(entrySelection->GetLine());

  if (entry->entry->item.empty()) {
    Lum::Model::StringRef value;
    size_t                amount;

    time->Push();
    time->Set(entry->entry->time);
    value=new Lum::Model::String(AmountToString(entry->entry->amount));

    if (!WeightEdit::EditWeight(this->GetWindow(),
                                _(L"WEIGHT_EDIT_TITLE",L"My weight is..."),
                                _(L"WEIGHT_EDIT_TEXT",
                                  L"Please enter your weight at the given time!\n"
                                  L"\n"
                                  L"(The weight should be entered as a decimal number\n"
                                  L"with a maximum of one digit after the colon.)"),
                                time,
                                value)) {
      time->Pop();
      return;
    }

    if (value->IsNull()) {
      time->Pop();
      return;
    }

    if (!StringToAmount(value->Get(),amount)) {
      time->Pop();
      return;
    }

    entry->entry->amount=amount;
    entry->entry->time=time->Get();
    entries->RedrawRow(entrySelection->GetLine());

    entries->Sort(1);

    time->Pop();
  }
  else {
    Lum::Model::StringRef value;
    size_t                amount;
    std::wstring          t1,t2;

    time->Push();
    time->Set(entry->entry->time);
    value=new Lum::Model::String(AmountToString(entry->entry->amount));

    t1=_(L"FOOD_REGISTER_TITLE",L"I have consumed $food...");
    Lum::Base::L10N::Substitute(t1,L"$food",entry->entry->item);
    t2=_(L"FOOD_REGISTER_TEXT",L"Please enter the amount of $food\nyou have consumed!");
    Lum::Base::L10N::Substitute(t2,L"$food",entry->entry->item);
    if (!FoodRegister::RegisterFood(this->GetWindow(),
                                    t1,t2,time,value,entry->entry->unit)) {

      time->Pop();
      return;
    }

    if (value->IsNull() || value->Empty()) {
      time->Pop();
      return;
    }

    if (!StringToAmount(value->Get(),amount)) {
      time->Pop();
      return;
    }

    entry->entry->time=time->Get();
    entry->entry->amount=amount;
    entries->RedrawRow(entrySelection->GetLine());

    entries->Sort(1);

    time->Pop();
  }

  UpdateSummary();
}

void DataInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==date) {
    UpdateEntries();
    UpdateSummary();
  }
  if (model==prevDayAction && prevDayAction->IsFinished()) {
    Lum::Base::Calendar d=date->Get();

    d.AddDays(-1);
    date->Set(d);
  }
  else if (model==todayAction && todayAction->IsFinished()) {
    date->SetToCurrent();
  }
  else if (model==nextDayAction && nextDayAction->IsFinished()) {
    Lum::Base::Calendar d=date->Get();

    d.AddDays(1);
    date->Set(d);
  }

  else if (model==addFoodAction && addFoodAction->IsFinished()) {
    AddFood();
  }
  else if (model==removeFoodAction && removeFoodAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      RemoveFood();
    }
  }
  else if (model==editFoodAction && editFoodAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      EditFood();
    }
  }
  else if (model==registerFoodAction && registerFoodAction->IsFinished()) {
    if (foodSelection->HasSelection()) {
      RegisterFood();
    }
  }

  else if (model==addWeightAction && addWeightAction->IsFinished()) {
    AddWeight();
  }
  else if (model==removeItemAction && removeItemAction->IsFinished()) {
    if (entrySelection->HasSelection()) {
      RemoveItem();
    }
  }
  else if (model==editItemAction && editItemAction->IsFinished()) {
    if (entrySelection->HasSelection()) {
      EditItem();
    }
  }

  else if (model==foodSearch) {
    Lum::OS::display->RemoveTimer(searchTimerAction);
    Lum::OS::display->AddTimer(1,500000,searchTimerAction);
  }
  else if (model==searchTimerAction && searchTimerAction->IsFinished()) {

    if (foodSearch->Get()!=lastFoodSearchFilter) {
      UpdateFoods();
      lastFoodSearchFilter=foodSearch->Get();
    }
  }

  else if (model==goodValueLimit || model==badValueLimit) {
    UpdateFoods();
  }

  else if (model==trendType || model==trendDirection || model==refValue) {
    UpdateSummary();
  }

  Lum::Component::Resync(model,msg);
}

void DataInput::InitializeGUI()
{
  UpdateEntries();
  UpdateFoods();
  UpdateSummary();
}
