/*
  This file is part of "WeightJinni" - A program to control your weight.
  Copyright (C) 2008  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dialog.h>
#include <Lum/Menu.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Tab.h>
#include <Lum/View.h>
#include <Lum/WindowGroup.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/File.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/Main.h>

#include "config.h"

#include "Configuration.h"
#include "Diagram.h"
#include "DataInput.h"
#include "ImportExport.h"
#include "Preferences.h"
#include <iostream>
static Lum::AppInfo info;

class MainDialog : public Lum::Dialog
{
private:
  Lum::Model::ActionRef importAction;
  Lum::Model::ActionRef exportAction;
  Lum::Model::ActionRef backupAction;
  Lum::Model::ActionRef restoreAction;
  Lum::Model::ActionRef configAction;
  Lum::Model::ActionRef aboutAction;

  DataInput             *dataInput;
  Diagram               *diagram;

public:
  MainDialog()
  : importAction(new Lum::Model::Action()),
    exportAction(new Lum::Model::Action()),
    backupAction(new Lum::Model::Action()),
    restoreAction(new Lum::Model::Action()),
    configAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action())
  {
    AttachModel(GetOpenedAction());
    AttachModel(GetClosedAction());

    AttachModel(importAction);
    AttachModel(exportAction);
    AttachModel(backupAction);
    AttachModel(restoreAction);
    AttachModel(configAction);
    AttachModel(aboutAction);
  }

  ~MainDialog()
  {
    UnattachModel(aboutAction);
    UnattachModel(configAction);
    UnattachModel(restoreAction);
    UnattachModel(backupAction);
    UnattachModel(exportAction);
    UnattachModel(importAction);

    UnattachModel(GetClosedAction());
    UnattachModel(GetOpenedAction());
  }

  void PreInit()
  {
    Lum::Menu        *menu;
    Lum::MenuStrip   *menuStrip;
    Lum::Panel       *panel;
    Lum::Tab         *tab;
    Lum::View        *view;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    panel=new Lum::VPanel();
    panel->SetFlex(true,true);

    tab=new Lum::Tab();
    tab->SetFlex(true,true);
    tab->SetModel(currentTab);

    dataInput=new DataInput();
    dataInput->SetFlex(true,true);

    tab->Add(_(L"MAIN_TAB_INPUT",L"Input"),dataInput);

    view=new Lum::View();
    view->SetFlex(true,true);

    diagram=new Diagram();
    diagram->SetFlex(true,true);
    view->SetObject(diagram);

    tab->Add(_(L"MAIN_TAB_DIAGRAM",L"Diagram"),view);

    panel->Add(tab);

    wGroup->SetMain(panel);

    menuStrip=new Lum::MenuStrip();

    menu=new Lum::Menu();
    menu->AddActionItem(_(L"MENU_IMPORT_FOOD_COMPONENTS",L"_Import food components..."),importAction,Lum::OS::qualifierControl,L"i");
    menu->AddActionItem(_(L"MENU_EXPORT_FOOD_COMPONENTS",L"_Export food components..."),exportAction,Lum::OS::qualifierControl,L"e");
    menu->AddSeparator();
    menu->AddActionItem(_(L"MENU_BACKUP",L"_Backup..."),backupAction);
    menu->AddActionItem(_(L"MENU_RESTORE",L"_Restore..."),restoreAction);
    menu->AddSeparator();
    menu->AddActionItem(_ld(menuProjectQuit),GetClosedAction(),Lum::OS::qualifierControl,L"q");
    menuStrip->AddPullDownMenu(_ld(menuProject),menu);

    menu=new Lum::Menu();
    menu->AddActionItem(_ld(menuEditSettings),configAction,Lum::OS::qualifierControl,L"p");
    menuStrip->AddPullDownMenu(_ld(menuEdit),menu);

    menu=new Lum::Menu();
    menu->AddActionItem(_ld(menuHelpAbout),aboutAction);
    menuStrip->AddPullDownMenu(_ld(menuHelp),menu);

    wGroup->SetMenuStrip(menuStrip);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      if (!LoadConfig(Lum::Base::Path::GetApplicationConfigPath())) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Cannot load configuration!",
                              L"Cannot load configuration!");
      }

      dataInput->InitializeGUI();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      if (!SaveConfig(Lum::Base::Path::GetApplicationConfigPath())) {
        Lum::Dlg::Msg::ShowOk(this,
                              L"Cannot save configuration!",
                              L"Cannot save configuration!");
      }
    }
    else if (model==importAction && importAction->IsFinished()) {
      Import(this);
      dataInput->UpdateFoods();
    }
    else if (model==exportAction && exportAction->IsFinished()) {
      Export(this);
    }
    else if (model==backupAction && backupAction->IsFinished()) {
      Lum::Base::Path path;
      std::wstring    file;

      path.SetNativeDir(Lum::Base::Path::GetHome());
      path.SetBaseName(L"WeightJinni.bak");

      file=path.GetPath();

      if (Lum::Dlg::File::SaveFile(GetWindow(),
                                   L"Store backup to...",
                                   file)) {
        if (!SaveConfig(file)) {
          Lum::Dlg::Msg::ShowOk(this,
                                L"Cannot save configuration!",
                                L"Cannot save configuration!");
        }
      }
    }
    else if (model==restoreAction && restoreAction->IsFinished()) {
      Lum::Base::Path path;
      std::wstring    file;

      path.SetNativeDir(Lum::Base::Path::GetHome());
      path.SetBaseName(L"WeightJinni.bak");

      file=path.GetPath();

      if (Lum::Dlg::File::GetFile(GetWindow(),
                                   L"Restore from...",
                                   file)) {
        if (!LoadConfig(file)) {
          Lum::Dlg::Msg::ShowOk(this,
                                L"Cannot load configuration!",
                                L"Cannot load configuration!");
        }
        else {
          dataInput->InitializeGUI();
          diagram->Redraw();
        }
      }
    }
    else if (model==configAction && configAction->IsFinished()) {
      Preferences *dialog;

      dialog=new Preferences();
      dialog->SetParent(this);

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Control your weight..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License (GPL)");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }

  void Cleanup()
  {
    Lum::OS::MainDialog<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))

