/* maemo vpnc-gui
 * 
 * Copyright (c) 2007 Michael "ScriptKiller" Arndt
 * http://scriptkiller.de/
 * <scriptkiller@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 */

/*
 * Changelog: See debian/changelog
 *
 */

/* Includes */
#include "hildon.h"

#include <gtk/gtkmain.h>
#include <gtk/gtkradiobutton.h>

#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkradiomenuitem.h>
#include <gtk/gtktext.h>


#include <libosso.h>


#include <signal.h>
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>

#include "configuration.h"
#include "save_buffer.h"
#include "gpl.h"

#define VPNC_WRAPPER_CONNECT    "/usr/bin/vpnc-gui-connect"
#define VPNC_WRAPPER_DISCONNECT "/usr/bin/vpnc-gui-disconnect"

#define VERSION "0.3.4"

/* widget for displaying infos */
GtkWidget *text=NULL;
gint input_tag=-1;

/* program and main window */
HildonProgram *program;
HildonWindow *window;




void show_about_cb(GtkBin *action, gpointer data) {

  GtkWindow *win=(GtkWindow *)data;

  const gchar *authors[3] = { "Michael Arndt <scriptkiller@gmx.de", 0 };

  //const gchar *documenters[] = { 0 };
  gtk_show_about_dialog(win,
			"name", "vpnc GUI",
			"version", VERSION,
			"copyright", "Copyright(C) 2007 Michael Arndt",
			//"comments", "A frontend for vpnc",
			"authors", authors,
			//"documenters", documenters,
			//"translator-credits", 0,
			"website", "http://scriptkiller.de/",
			"website-label", "Home page",
			"license", GPL,
			//"logo-icon-name", "blubb",
			0);
}


void child_input_cb(gpointer data, int fd, GdkInputCondition cond) {

  ssize_t bytes;
  char buf[1024];
  int errnosave;
  
  while( (bytes=read(fd, buf, sizeof(buf)-1)) > 0) {
    /* save errno from read */
    errnosave=errno;
    buf[bytes]='\0';
    printf("Read: %s\n", buf);
    gtk_text_insert(text,
		    NULL,
		    NULL,
		    NULL,
		    buf,
		    bytes);
  }

#ifdef DEBUG
  perror("read");
#endif

  if(errnosave != EAGAIN && errnosave != 0) {
    /* we probably hit EOF */
#ifdef DEBUG
    puts("removing io");
#endif
    gdk_input_remove(input_tag);
    input_tag=-1;
  }

}


/* kill zombies */
void reap_children(int sig) {

  pid_t pid;
  int status;

  while((pid = waitpid(-1, &status, WNOHANG)) > 0) {

    if(WIFEXITED(status) != 0) {
#ifdef DEBUG
      printf("SIGCHLD: Child with PID %u terminated with status %d\n", pid, WEXITSTATUS(status));
#endif
    }
    else {
#ifdef DEBUG
      printf("SIGCHLD: Child with PID %u terminated mysteriously\n", pid);
#endif
    }

  }

  signal(SIGCHLD, reap_children);

}

/* callback for connect button */
void button_connect_cb() {
  
#ifdef DEBUG
  printf("Starting vpnc (conf=%s)...\n", config_path);
#endif


  int stdout_pipe[2];

  if(pipe(stdout_pipe)==-1) {
    perror("pipe");
    return;
  }


  pid_t pid=fork();
  if(pid==-1) {
    perror("fork");
    return;
  }

  if(pid==0) {
    

    /* replace stdout with out side of pipe */
    dup2(stdout_pipe[1], 1);

    /* close in side of pipe */
    close(stdout_pipe[0]);
    
    
    //setvbuf(stdout,(char*)NULL,_IONBF,0);	/* Set non-buffered output on stdout */

    /* replace stderr with out side of pipe */
    dup2(stdout_pipe[1], 2);

    

    if(execl(VPNC_WRAPPER_CONNECT,
	     "vpnc", config_path /* configuration.c */, NULL)==-1)
      perror("execl");
    exit(1);
  }
  else {



    /* Close the write end of stdout */
    close(stdout_pipe[1]);


    /* make read end non blocking */
    if(fcntl(stdout_pipe[0], F_SETFL, O_NONBLOCK) == -1) {
      perror("fcntl failed");
      return;
    }


    if(input_tag > -1)
      gtk_input_remove(input_tag);


    /* use gdk to monitor read end of stdout */
    input_tag=
      gdk_input_add(stdout_pipe[0],
		    GDK_INPUT_READ,
		    child_input_cb,
		    NULL);

  }

}

/* callback for disconnect button */
void button_disconnect_cb() {

#ifdef DEBUG
  printf("Stopping vpnc ...\n");
#endif

  int stdout_pipe[2];

  if(pipe(stdout_pipe)==-1) {
    perror("pipe");
    return;
  }


  pid_t pid=fork();
  if(pid==-1) {
    perror("fork");
    return;
  }

  if(pid==0) {

    /* replace stdout with out side of pipe */
    dup2(stdout_pipe[1], 1);
    /* close in side of pipe */
    close(stdout_pipe[0]);
    /* replace stderr with out side of pipe */
    dup2(stdout_pipe[1], 2);


    if(execl(VPNC_WRAPPER_DISCONNECT, "vpnc-disconnect", NULL)==-1)
      perror("execl");
    exit(1);
  }
  else {

    /* Close the write end of stdout */
    close(stdout_pipe[1]);

    /* make read end non blocking */
    if(fcntl(stdout_pipe[0], F_SETFL, O_NONBLOCK) == -1) {
      perror("fcntl failed");
      return;
    }

    if(input_tag > -1)
      gtk_input_remove(input_tag);


    /* use gdk to monitor read end of stdout */
    input_tag=
      gdk_input_add(stdout_pipe[0],
		    GDK_INPUT_READ,
		    child_input_cb,
		    NULL);
  }

}


/* Callback for "Close" menu entry */
void item_close_cb()
{
#ifdef DEBUG
    g_print("Closing application ...\n");
#endif

    gtk_main_quit();
}

/* Callback for "Configure" menu entry */
void item_configure_cb()
{
#ifdef DEBUG
    g_print("configuring application ...\n");
#endif
    configuration_dialog_show();

}

/* Callback for "Save-Buffer" menu entry */
void item_save_buffer_cb()
{

#ifdef DEBUG
    g_print("saving buffer ...\n");
#endif

    save_buffer_show_dialog(window, text);
}

/* Create the menu items needed for the main view */
static void create_menu(HildonWindow * main_window) 
{
    /* Create needed variables */
    GtkWidget *main_menu;
    GtkWidget *item_configure;
    GtkWidget *item_save_buffer;
    GtkWidget *item_close;
    GtkWidget *item_about;
    GtkWidget *item_separator;

    /* Create new main menu */
    main_menu = gtk_menu_new();

    /* Create menu items */
    item_configure = gtk_menu_item_new_with_label("Configure");
    item_save_buffer = gtk_menu_item_new_with_label("Save buffer");
    item_close = gtk_menu_item_new_with_label("Close");
    item_about= gtk_menu_item_new_with_label("About");
    item_separator = gtk_separator_menu_item_new();


    /* Add menu items to right menus */
    gtk_menu_append(main_menu, item_configure);
    gtk_menu_append(main_menu, item_save_buffer);
    gtk_menu_append(main_menu, item_about);
    gtk_menu_append(main_menu, item_separator);
    gtk_menu_append(main_menu, item_close);

    hildon_window_set_menu(HILDON_WINDOW(main_window), GTK_MENU(main_menu));


    /* Attach the callback functions to the activate signal */
    g_signal_connect(G_OBJECT(item_close), "activate",
                     GTK_SIGNAL_FUNC(item_close_cb), NULL);

    g_signal_connect(G_OBJECT(item_configure), "activate",
                     GTK_SIGNAL_FUNC(item_configure_cb), NULL);

    g_signal_connect(G_OBJECT(item_save_buffer), "activate",
                     GTK_SIGNAL_FUNC(item_save_buffer_cb), NULL);

    g_signal_connect(G_OBJECT(item_about), "activate",
                     GTK_SIGNAL_FUNC(show_about_cb), main_window);

    /* Make all menu widgets visible */
    gtk_widget_show_all(GTK_WIDGET(main_menu));
}


int main(int argc, char *argv[]) {

  /* Add signal handlers */
  signal(SIGCHLD, reap_children);

  /* init config */
  configuration_init();


  //  g_set_application_name (_("vpnc GUI"));


  /* Initialize the GTK. */
  gtk_init(&argc, &argv);


  /*
  osso_context_t  *osso_context = osso_initialize("vpnc-gui", "0.0", TRUE, NULL);
  if (osso_context == NULL) {
      g_printerr("osso_initialize() failed!\n");
      exit(1);
  }
  */



  /* Create the hildon program and setup the title */
  program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("vpnc GUI");

  /* Create HildonWindow and set it to HildonProgram */
  window = HILDON_WINDOW(hildon_window_new());
  hildon_program_add_window(program, window);


  GtkWidget *vbox=gtk_vbox_new(FALSE, 10);

  text=gtk_text_new(NULL, NULL);
  gtk_text_set_editable(text, FALSE);

  gtk_box_pack_start(vbox, text, TRUE, TRUE, 0);

  //
  GtkWidget *hbox=gtk_hbox_new(FALSE, 10);

  GtkWidget *connect=gtk_button_new_with_label("Connect");
  GtkWidget *disconnect=gtk_button_new_with_label("Disconnect");

  gtk_box_pack_start(hbox, connect,
		     TRUE, TRUE, 0);
  gtk_box_pack_end(hbox, disconnect,
		   TRUE, TRUE, 0);

  g_signal_connect(G_OBJECT(connect), "clicked",
		   GTK_SIGNAL_FUNC(button_connect_cb), NULL);

  g_signal_connect(G_OBJECT(disconnect), "clicked",
		   GTK_SIGNAL_FUNC(button_disconnect_cb), NULL);

  //

  gtk_box_pack_end(vbox, hbox, TRUE, TRUE, 0);



  /* Add box to window */
  gtk_container_add(GTK_CONTAINER(window),
		    vbox);

  /* add menu */
  create_menu(window);

  /* Begin the main application */
  gtk_widget_show_all(GTK_WIDGET(window));

  /* Connect signal to X in the upper corner */
  g_signal_connect(G_OBJECT(window), "delete_event",
		   G_CALLBACK(gtk_main_quit), NULL);

  gtk_main();

  /* Exit */
  return 0;
}
