#!/usr/bin/python
# -*- coding: utf-8 -*-
#
#
#   Copyright (c) 2007 Pasi Keärnen <pasi.k.keranen@gmail.com>
#
#   This program is free software; you can redistribute it and/or
#   modify it under the terms of the GNU General Public License as
#   published by the Free Software Foundation; either version 2 of the
#   License, or (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
#   General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
#   02111-1307, USA.

# Based on CookBook recipe from Christos Georgiou
# See http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/496984
import struct
from PIL import Image
import StringIO

FLAGS = CONTAINER, SKIPPED, TAGITEM, IGNORED, DOUBLEINT, DATAITEM = [2**_ for _ in xrange(6)]

# CONTAINER: contains other atoms
# SKIPPED:   ignore first 4 bytes of data
# TAGITEM:   tag item
# DOUBLEINT: data is 2 4-byte big-endian integers
# DATAITEM:  data is a triplet of "size" (4-bytes), "name" (4-bytes), "data" (rest of the bytes)
CALLBACK= TAGITEM | DATAITEM
FLAGS.append(CALLBACK)

ARTWORK_TYPE_NONE = None
ARTWORK_TYPE_JPG  = "jpg"
ARTWORK_TYPE_PNG  = "png"

# Dictionary definitions in the vein of ID3 to make this class interoperable
ATTRIBUTE_KEY_TRACKNUM    = "TRACKNUMBER"
ATTRIBUTE_KEY_ARTIST   = "ARTIST"
ATTRIBUTE_KEY_TITLE    = "TITLE"
ATTRIBUTE_KEY_ALBUM    = "ALBUM"
ATTRIBUTE_KEY_YEAR     = "YEAR"
ATTRIBUTE_KEY_GENRE    = "GENRE"
ATTRIBUTE_KEY_COMMENT  = "COMMENT"
ATTRIBUTE_KEY_WRITER   = "WRITER"
ATTRIBUTE_KEY_TOOL     = "TOOL"
ATTRIBUTE_KEY_COVERART_TYPE = "ART_TYPE"
ATTRIBUTE_KEY_FILENAME        = "FILE_NAME"
ATTRIBUTE_KEY_COVERART_OFFSET = "ART_OFFSET"
ATTRIBUTE_KEY_COVERART_LENGTH = "ART_LENGTH"

# Definition of known tagtypes (there are more, but at the moment we don't care)
TAGTYPES= (
        ('ftyp', 0),
        ('moov', CONTAINER),
        ('mdat', 0),
        ('udta', CONTAINER),
        ('meta', CONTAINER|SKIPPED),
        ('ilst', CONTAINER),
        ('\xa9ART', TAGITEM),
        ('\xa9nam', TAGITEM),
        ('\xa9too', TAGITEM),
        ('\xa9alb', TAGITEM),
        ('\xa9day', TAGITEM),
        ('\xa9gen', TAGITEM),
        ('\xa9wrt', TAGITEM),
        ('covr', DATAITEM),
        ('trkn', TAGITEM|DOUBLEINT),
        ('\xa9cmt', TAGITEM),
        ('trak', CONTAINER),
        ('----', DATAITEM),
        ('mdia', CONTAINER),
        ('minf', CONTAINER),
)

# Used when printing log output from parsing
inset = 0
# Whether or not log output is printed from parsing
VERBOSE = False

flagged= {}
for flag in FLAGS:
    flagged[flag]= frozenset(_[0] for _ in TAGTYPES if _[1] & flag)

def _dataitem(strData, globalOffset):
    """Convert '----' and 'covr' atom data into dictionaries"""
    offset= 0
    result= {}
    # TODO: Don't read the data, just return a tuple containing the offset and length to it -> whoever needs it can read it later using those!
    while offset < len(strData):
        atomsize= struct.unpack("!i", strData[offset:offset+4])[0]
        atomtype= strData[offset+4:offset+8]
        if atomtype == "data":
            if VERBOSE: print "Reading "+str(atomsize)+" bytes of data"
            result[atomtype] = ( globalOffset+offset+16, globalOffset+offset+atomsize-16, strData[offset+16:offset+atomsize] )
            #result[atomtype]= strData[offset+16:offset+atomsize]
        else:
            #result[atomtype]= strData[offset+12:offset+atomsize]
            result[atomtype] = ( globalOffset+offset+12, globalOffset+offset+atomsize-12, strData[offset+12:offset+atomsize] )
        offset+= atomsize
    return result

def _analyse(fp, startOffset, endOffset):
    """Walk the atom tree in the MP4 file"""
    global inset
    offset= startOffset
    while offset < endOffset:
        fp.seek(offset)
        atomsize= struct.unpack("!i", fp.read(4))[0]
        atomtype= fp.read(4)
        if inset > 0:
            for i in range(inset):
                if VERBOSE: print "+",
        if VERBOSE: print "Atomtype:\""+atomtype+"\""
        if atomtype in flagged[CONTAINER]:
            data= ''
            inset += 1
            for reply in _analyse( fp, offset+( atomtype in flagged[SKIPPED] and 12 or 8 ), offset+atomsize ):
                yield reply
            inset -= 1
        else:
            fp.seek(offset+8)
            if atomtype in flagged[TAGITEM]:
                data=fp.read(atomsize-8)[16:]
                if atomtype in flagged[DOUBLEINT]:
                    data= struct.unpack("!ii", data)
            elif atomtype in flagged[DATAITEM]:
                data= _dataitem(fp.read(atomsize-8),offset+8)
            else:
                data= fp.read(min(atomsize-8, 32))
        if not atomtype in flagged[IGNORED]: yield atomtype, atomsize, data
        offset+= atomsize

def mp4_atoms(pathname):
    """Opens the given file and starts parsing"""
    fp= open(pathname, "rb")
    fp.seek(0,2)
    size=fp.tell()
    for atom in _analyse(fp, 0, size):
        yield atom
    fp.close()


def PyAtom_getAlbumArt( filename, offset = 0, length = 0 ):
    """Returns any embedded artworkfile as PIL.Image"""
    if length > 0:
        # We know the location of the data, so just grab it.
        fp= open(filename, "rb")
        fp.seek(offset)
        artdata = fp.read(length)
        file = StringIO.StringIO( artdata )
        image = Image.open( file )
        return image

    # Search for the data
    for atomtype, atomsize, atomdata in mp4_atoms(filename):
        if atomtype == "covr":
            # Recognize the artwork type and if known, return the image
            artdata = atomdata['data'][2]
            if artdata.startswith( "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A" ) or artdata.startswith( "\xFF\xD8\xFF\xE0" ):
                try:
                    #_saveFile( artworkfile, value, ARTWORK_TYPE_PNG )
                    file = StringIO.StringIO( artdata )
                    image = Image.open( file )
                    return image
                except IOError:
                    print "Image load failed on IOError"
                    pass
            else:
                # Unknown image type
                if VERBOSE: print "Unknown image type, can't handle"
    # Failed to load any of the embedded images
    return None

class PyAtom(dict):
    """Reads M4A files tags and stores them in a dictionary. Only the type info of the coverart is stored at this stage."""
    cvt= {
            'trkn'   : ATTRIBUTE_KEY_TRACKNUM,
            '\xa9ART': ATTRIBUTE_KEY_ARTIST,
            '\xa9nam': ATTRIBUTE_KEY_TITLE,
            '\xa9alb': ATTRIBUTE_KEY_ALBUM,
            '\xa9day': ATTRIBUTE_KEY_YEAR,
            '\xa9gen': ATTRIBUTE_KEY_GENRE,
            '\xa9cmt': ATTRIBUTE_KEY_COMMENT,
            '\xa9wrt': ATTRIBUTE_KEY_WRITER,
            '\xa9too': ATTRIBUTE_KEY_TOOL,
            'covr'   : ATTRIBUTE_KEY_COVERART_TYPE
    }

    def __init__(self, mp4filename=None):
        super(dict, self).__init__()
        self.filename = mp4filename
        if mp4filename is None:
            return
        self[ATTRIBUTE_KEY_FILENAME] = mp4filename
        for atomtype, atomsize, atomdata in mp4_atoms(self.filename):
            self.atom2tag(atomtype, atomsize, atomdata)

    def atom2tag(self, atomtype, atomsize, atomdata):
        """Insert items using descriptive key instead of atomtype"""
        if atomtype == "----":
            # Ignore
            #key  = atomdata['name'].title()
            #value= atomdata['data'].decode("utf-8")
            return
        try:
            key= self.cvt[atomtype]
        except KeyError:
            return

        if atomtype == "covr":
            # Recognize the artwork type
            artdata = atomdata['data'][2]
            if artdata.startswith( "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A" ):
                value = ARTWORK_TYPE_PNG
                self[ATTRIBUTE_KEY_COVERART_OFFSET]= atomdata['data'][0]
                self[ATTRIBUTE_KEY_COVERART_LENGTH]= atomdata['data'][1]
            elif artdata.startswith( "\xFF\xD8\xFF\xE0" ):
                value = ARTWORK_TYPE_JPG
                self[ATTRIBUTE_KEY_COVERART_OFFSET]= atomdata['data'][0]
                self[ATTRIBUTE_KEY_COVERART_LENGTH]= atomdata['data'][1]
            else:
                # Unknown image type
                return
        elif atomtype == "trkn":
            # Track number must be handled differently
            value= atomdata[0]
        else:
            # Other types can be handled with the generic code
            try:
                value= atomdata.decode("utf-8")
            except AttributeError:
                if VERBOSE: print `atomtype`, `atomdata`
                #raise

        # Store the value under the key
        self[key]= value

    def getArtworkImage( self ):
        """Returns any embedded artworkfile as PIL.Image"""
        if not self.has_key( ATTRIBUTE_KEY_COVERART_TYPE ):
            return None
        if self[ATTRIBUTE_KEY_COVERART_TYPE] is None:
            return None
        if not self.has_key(ATTRIBUTE_KEY_FILENAME):
            return None

        offset = 0
        if self.has_key(ATTRIBUTE_KEY_COVERART_OFFSET): offset = self[ATTRIBUTE_KEY_COVERART_OFFSET]
        length = 0
        if self.has_key(ATTRIBUTE_KEY_COVERART_LENGTH): length = self[ATTRIBUTE_KEY_COVERART_LENGTH]
        return PyAtom_getAlbumArt( self[ATTRIBUTE_KEY_FILENAME], offset, length )

if __name__=="__main__":
    import sys, pprint
    info= PyAtom(sys.argv[1]) # pathname of an .mp4/.m4a file as first argument
    print info
    if info.has_key( ATTRIBUTE_KEY_ALBUM ): print "\""+str(info[ATTRIBUTE_KEY_ALBUM])+"\""
    if info.has_key( ATTRIBUTE_KEY_COVERART_TYPE ): print "\""+str(info[ATTRIBUTE_KEY_COVERART_TYPE])+"\""
    if info.has_key( ATTRIBUTE_KEY_ARTIST ): print "\""+str(info[ATTRIBUTE_KEY_ARTIST])+"\""
    if info.has_key( ATTRIBUTE_KEY_TRACKNUM ): print "\""+str(info[ATTRIBUTE_KEY_TRACKNUM])+"\""
    if info.has_key( ATTRIBUTE_KEY_TITLE ): print "\""+str(info[ATTRIBUTE_KEY_TITLE])+"\""
    if info.has_key( ATTRIBUTE_KEY_YEAR ): print "\""+str(info[ATTRIBUTE_KEY_YEAR])+"\""
    if info.has_key( ATTRIBUTE_KEY_GENRE ): print "\""+str(info[ATTRIBUTE_KEY_GENRE])+"\""
    if info.has_key( ATTRIBUTE_KEY_COMMENT ): print "\""+str(info[ATTRIBUTE_KEY_COMMENT])+"\""
    if info.has_key( ATTRIBUTE_KEY_WRITER ): print "\""+str(info[ATTRIBUTE_KEY_WRITER])+"\""
    if info.has_key( ATTRIBUTE_KEY_TOOL ): print "\""+str(info[ATTRIBUTE_KEY_TOOL])+"\""
    song = info[ATTRIBUTE_KEY_TITLE]
    if isinstance(song, unicode): print "Song is unicode"
    song="0"+str(info[ATTRIBUTE_KEY_TRACKNUM])+" "+song
    if isinstance(song, unicode): print "Song is still unicode"
    print "\""+song+"\""
