#include "Preferences.hpp"
#include "FSFile.hpp"

#include <pugixml/pugixml.hpp>

//static
std::string Preferences::TimbreTypeToName(Preferences::E_Timbre timbre){
	switch(timbre){
		default:
		case SINE_PLUS_ADSR:
			return "SineWave";
			break;
		case PLUCKED_STRING:
			return "PluckedString";
			break;
	}
}



//static
Preferences::E_Timbre Preferences::TimbreNameToType(std::string name){
	for(ting::uint t = 0; t < ting::uint(NUM_TIMBRES); ++t){
		if(Preferences::TimbreTypeToName(E_Timbre(t)).compare(name) == 0)
			return E_Timbre(t);
	}
	return SINE_PLUS_ADSR;
}



void Preferences::ClampBasePitch(){
	if(this->basePitch + 12 * this->numOctaves > (DHighestOctave() + 1) * 12){
		this->basePitch = (DHighestOctave() - (this->numOctaves)) * 12;
	}
}



void Preferences::LoadFromFile(){
//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): enter" << std::endl)

	file::FSFile f(file::File::GetHomeDir() + ".theremin/" + this->prefsFileName);

//	TRACE(<< "MainWindow::LoadPreferencesFromFile(): reading file" << std::endl)

	pugi::xml_document doc;
	{
		ting::Array<ting::byte> rawFile;
		try{
			rawFile = f.LoadWholeFileIntoMemory();
		}catch(file::File::Exc &e){
			LOG(
					<< "MainWindow::LoadPreferencesFromFile(): could not load preferences file." << std::endl
					<< "\t" << " e = " << e.What() << std::endl
				)
			return;
		}

		doc.parse(reinterpret_cast<char*>(rawFile.Buf()));
	}

	{
		pugi::xml_node opts = doc.child("Options");
		if(!opts.empty()){
			pugi::xml_attribute numOctavesAttr = opts.attribute("numOctaves");
			if(!numOctavesAttr.empty()){
				ting::uint num = numOctavesAttr.as_uint();
				if(num != 0 && num <= 3)
					this->SetNumOctaves(num);
			}

			pugi::xml_attribute timbreAttr = opts.attribute("timbre");
			if(!timbreAttr.empty()){
				this->SetTimbre(Preferences::TimbreNameToType(timbreAttr.value()));
			}

			pugi::xml_attribute lockCurOctaveAttr = opts.attribute("lockCurrentOctave");
			if(!timbreAttr.empty()){
				this->SetLockCurrentOctave(lockCurOctaveAttr.as_bool());
			}
		}
	}

	{
		pugi::xml_node tun = doc.child("Tuning");
		if(!tun.empty()){
			pugi::xml_attribute basePitchAttr = tun.attribute("basePitch");
			if(!basePitchAttr.empty()){
				this->SetBasePitch(basePitchAttr.as_uint());
			}
		}
		TRACE(<< "Load prefs, new basePitch = " << this->GetBasePitch() << std::endl)
	}
}



void Preferences::SaveToFile(){
	pugi::xml_document doc;

	{
		pugi::xml_node n = doc.append_child(pugi::node_declaration);
		n.set_name("xml");
		n.append_attribute("version").set_value("1.0");
	}

	//TODO: move tag names to separate file
	{
		pugi::xml_node opts = doc.append_child();
		opts.set_name("Options");
		opts.append_attribute("numOctaves") = this->GetNumOctaves();
		opts.append_attribute("timbre").set_value(
				Preferences::TimbreTypeToName(this->GetTimbre()).c_str()
			);
		opts.append_attribute("lockCurrentOctave") = this->GetLockCurrentOctave();
	}

	{
		pugi::xml_node tun = doc.append_child();
		tun.set_name("Tuning");
		tun.append_attribute("basePitch") = this->GetBasePitch();
	}

	class PrefsWriter : public pugi::xml_writer{
		file::FSFile f;
	public:
		PrefsWriter(const std::string& fileName) :
				f(fileName)
		{
			f.Open(file::File::CREATE);
		}

		//override
		void write(const void *data, size_t size){
			ting::Array<ting::byte> buf(size);
			memcpy(buf.Buf(), data, size);
			f.Write(buf);
		}
	} writer(file::File::GetHomeDir() + ".theremin/" + this->prefsFileName);
	
    doc.print(writer);
}
