#! /usr/bin/env python2.5
# -*- coding: UTF8 -*-
################################################################################
#  ShareOnOvi - A Python module for interacting with Nokia's "Share on Ovi".   #
#  Version 1.0, September 18, 2008                                             #
#                                                                              #
# Copyright (C) 2008 by Daniel Martin Yerga                                    #
# <dyerga@gmail.com>                                                           #
#                                                                              #
# This program is free software; you can redistribute it and/or modify         #
# it under the terms of the GNU General Public License as published by         #
# the Free Software Foundation; either version 2 of the License, or            #
# (at your option) any later version.                                          #
#                                                                              #
# This program is distributed in the hope that it will be useful               #
# but WITHOUT ANY WARRANTY; without even the implied warranty of               #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the                 #
# GNU General Public License for more details.                                 #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with this program; if not, write to the Free Software                  #
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.   #
#                                                                              #
# This python library is based in the PHP class written by Peter Rukavina:     #
# http://ruk.ca/w/index.php/Share_on_Ovi_API                                   #
#                                                                              #
#                                                                              #
# * Usage example:                                                             #
# import share_on_ovi                                                          #
# shareonovi = share_on_ovi.ShareOnOvi(username, password)                     #
# shareonovi.upload(filename, title, summary, channel, tags, filetype)         #
#                                                                              #
# Where:                                                                       #
# filename: the complete path to the file to upload. '/home/user/photo.jpg'    #
# title: title for the file uploaded, works like in Share on Ovi service.      #
# summary: summary for the file uploaded, works like in Share on Ovi service.  #
# channel: 'mymedia' for the private channel, 'public' for the public one.     #
# tags: tag list for the file uploaded. ['tag1', 'tag2', 'tag3']               #
# filetype: file type for the file uploaded. 'image/jpeg'                      #
#                                                                              #
# channel and filetype are obligatory parameters. Others are optional.         #
#                                                                              #
# * Example to get the channel list:                                           #
# shareonovi.getChannels()                                                     #
#                                                                              #
# it return a list similar to -> ['public', 'mymedia', 'channel1']             #
################################################################################

import pycurl
import urllib
from xml.dom import minidom
import sys
import StringIO
import os

#This is a list of the actually supported formats on Share On Ovi
supported_formats = {
'3g2': 'video/3gpp2',
'3gp': 'video/3gp',
'3gp2': 'video/3gpp2',
'3gpp': 'video/3gp',
'3gpp2': 'video/3gpp2',
'aac': 'audio/x-aac',
'aif': 'audio/x-aiff',
'aifc': 'audio/x-aiff',
'aiff': 'audio/x-aiff',
'amr': 'audio/amr',
'asf': 'video/x-ms-asf',
'asr': 'video/x-ms-asf',
'asx': 'video/x-ms-asf',
'au': 'audio/basic',
'avi': 'video/x-msvideo',
'bmp': 'image/x-ms-bmp',
'cmx': 'image/x-cmx',
'cod': 'application/vnd.rim.cod',
'css': 'text/css',
'dib': 'image/bmp',
'doc': 'application/msword',
'dot': 'application/msword',
'dtd': 'application/xml',
'dwf': 'drawing/x-dwf',
'fla': 'application/octet-stream',
'flv': 'video/x-flv',
'gif': 'image/gif',
'ico': 'image/x-icon',
'ief': 'image/ief',
'ifo': 'application/octet-stream',
'ipx': 'application/x-ipix',
'jfif': 'image/jpeg',
'jpe': 'image/jpeg',
'jpeg': 'image/jpeg',
'jpg': 'image/jpeg',
'kml': 'application/vnd.google-earth.kml+xml',
'kmz': 'application/vnd.google-earth.kmz',
'lsf': 'video/x-la-asf',
'm1a': 'audio/mpeg',
'm1v': 'video/mpeg',
'm3u': 'audio/x-mpegurl',
'm4a': 'audio/mpeg',
'm4p': 'application/octet-stream',
'm4v': 'video/mp4',
'mdi': 'image/vnd.ms-modi',
'mid': 'audio/midi',
'midi': 'audio/midi',
'mov': 'video/quicktime',
'movie': 'video/x-sgi-movie',
'mp2': 'audio/mpeg',
'mp3': 'audio/mpeg',
'mp4': 'video/mp4',
'mpa': 'video/mpeg',
'mpe': 'video/mpeg',
'mpeg': 'video/mpeg',
'mpg': 'video/mpeg',
'mpp': 'application/vnd.ms-project',
'mpv2': 'video/mpeg',
'mqv': 'video/quicktime',
'nsc': 'application/x-conference',
'odf': 'application/vnd.oasis.opendocument.formula',
'odg': 'application/vnd.oasis.opendocument.graphics',
'odp': 'application/vnd.oasis.opendocument.presentation',
'ods': 'application/vnd.oasis.opendocument.spreadsheet',
'odt': 'application/vnd.oasis.opendocument.text',
'ogg': 'audio/ogg',
'pbm': 'image/x-portable-bitmap',
'pdf': 'application/pdf',
'pgm': 'image/x-portable-graymap',
'png': 'image/png',
'pot': 'text/plain',
'ppm': 'image/x-portable-pixmap',
'pps': 'application/vnd.ms-powerpoint',
'ppt': 'application/vnd.ms-powerpoint',
'psd': 'image/x-photoshop',
'pub': 'application/x-pub',
'qt': 'video/quicktime',
'ra': 'audio/x-realaudio',
'ram': 'audio/x-pn-realaudio',
'ras': 'image/x-cmu-raster',
'rgb': 'image/x-rgb',
'rm': 'audio/x-pn-realaudio',
'rmi': 'audio/midi',
'rtf': 'application/rtf',
'rtx': 'text/richtext',
'scd': 'application/vnd.ms-schedule',
'snd': 'audio/basic',
'swf': 'application/x-shockwave-flash',
'tif': 'image/tiff',
'tiff': 'image/tiff',
'tsv': 'text/tab-separated-values',
'txt': 'text/plain',
'vsd': 'application/vnd.visio',
'vss': 'application/vnd.visio',
'vst': 'application/vnd.visio',
'vsx': 'application/vnd.visio',
'vtx': 'application/vnd.visio',
'wav': 'audio/x-wav',
'wax': 'audio/x-ms-wax',
'wbmp': 'image/vnd.wap.wbmp',
'wm': 'video/x-ms-wm',
'wma': 'audio/x-ms-wma',
'wml': 'text/vnd.wap.wml',
'wmp': 'image/x-wmp',
'wmv': 'video/x-ms-wmv',
'wmx': 'video/x-ms-wmx',
'wri': 'application/x-wri',
'xbm': 'image/x-xbitmap',
'xla': 'application/vnd.ms-excel',
'xlc': 'application/vnd.ms-excel',
'xlm': 'application/vnd.ms-excel',
'xls': 'application/vnd.ms-excel',
'xlt': 'application/vnd.ms-excel',
'xlw': 'application/vnd.ms-excel',
'xml': 'application/xml',
'xsd': 'application/xml',
'xsl': 'application/xml',
'xslt': 'application/xml',
}

class ShareOnOvi:

    def __init__(self, username, password):
    
        self.username = username
        self.password = password
        
        #Not very important, is our application
        self.useragent = "SharePy 0.1"
        
        #Atom API Url from Share on Ovi
        self.endpoint = "http://share.ovi.com/api/atom/1.0/"
    
    def upload(self, filename, title, summary, channel, tags, filetype):
    
        mycurl = pycurl.Curl() 
        
        #Set Atom URL of the channel to upload.
        mycurl.setopt(pycurl.URL, self.endpoint + self.username + '.' + channel)        
        
        headers = ["Content-Type: %s" % filetype, "User-Agent: %s" % self.useragent]

        #Get the binary data from the file to upload
        a = open(filename, 'r')
        upload = a.read()
        a.close()

        mycurl.setopt(pycurl.USERPWD, self.username+':'+self.password) 

        # Add the headers to the cURL object.
        mycurl.setopt(pycurl.HTTPHEADER, headers)

        # Don't return the headers as part of the response.
        mycurl.setopt(pycurl.HEADER, 0)
        
        # This is just a straight binary POST.
        mycurl.setopt(pycurl.POST, 1)
        
        # Set the binary data as the POST payload.
        mycurl.setopt(pycurl.POSTFIELDS, upload)        

        # Do the actual cURL magic.
        # StringIO for get the output in a variable
        b = StringIO.StringIO()
        mycurl.setopt(pycurl.WRITEFUNCTION, b.write)
        mycurl.perform()
        output = b.getvalue()
        
        print output
        
        # Close the cURL object.  
        mycurl.close()
        
        # If we got back a simple 'Authentication failed.' response
        if (output == "Authentication failed."):
            print "ERROR: Share on Ovi authentication failed.\n"
            return None

        elif (output.find('unavailable') > 0):
            print "ERROR: Share on Ovi is unavailable.\n"
            return None            
            
            
        # Take the response we got from cURL as XML      
        xml_document = minidom.parseString(output)
        linklist = xml_document.getElementsByTagName('link')
        
        # Look for the links with attributes of 'edit' (for PUTing metadata) 
        # and 'alternate' (the URL for the new object on OVI)
        editURL = ''
        objectURL = ''
        for i in linklist:
            rel = i.attributes['rel']
            href = i.attributes['href']
            if rel.value == 'edit':
                editURL = href.value
            if rel.value == 'alternate':
                objectURL = href.value

        #print editURL, objectURL

        # If we didn't get a link in the XML returned to PUT the metadata.
        if editURL == '':
            print "ERROR: No endpoint returned from Share on Ovi to PUT image metadata.\n"
            return None
            
        # Contruct a new XML object, using the 'metadata-skeleton' as a template.
        template = self.makeMetadataSkeleton()
        xml = minidom.parseString(str(template))       

        # Add image title element, with attribute type=text.
        title_el = xml.createElement('title')
        title_el.appendChild(xml.createTextNode(title))
        xml.documentElement.appendChild(title_el)

        # Add image summary element.
        summary_el = xml.createElement('summary')
        summary_el.appendChild(xml.createTextNode(summary))
        xml.documentElement.appendChild(summary_el) 

        # Add a category element, setting privacy to public.
        xml_term = xml.createElement('category')        
        xml_term.setAttribute("term", channel)
        xml_term.setAttribute("scheme","http://schemas.nokia.com/onlinesharing/2006/category#privacylevels")
        xml.documentElement.appendChild(xml_term) 

        # Add category elements for each tag.
        for tag in tags:
            tag_el = xml.createElement('category')
            tag_el.setAttribute("term", tag)
            tag_el.setAttribute("scheme","http://schemas.nokia.com/onlinesharing/2006/category#tags")
            xml.documentElement.appendChild(tag_el)            

        
        # Convert the XML object into XML text.
        xml_request = xml.toxml()
        
        #print '\nxml_request: ', xml_request
        content_length = len(xml_request)
        
        # Create a new cURL object.
        mycurl = pycurl.Curl()
        
        mycurl.setopt(pycurl.URL, str(editURL))

        # The headers for the HTTP POST.
        headers = ["Content-Type: application/atom+xml; charset=utf-8",
                         "Content-Length: %s" % content_length,
                         "User-Agent: %s" % self.endpoint]

        mycurl.setopt(pycurl.USERPWD, self.username+':'+self.password) 

        # Add the headers to the cURL object.
        mycurl.setopt(mycurl.HTTPHEADER, headers)

        # Don't return the headers as part of the response.
        mycurl.setopt(mycurl.HEADER, 0)

        # This is going to be an HTTP PUT
        mycurl.setopt(mycurl.CUSTOMREQUEST, "PUT")

        # Set the binary data as the PUT payload.
        mycurl.setopt(mycurl.POSTFIELDS, str(xml_request))
        
        # Do the actual cURL magic.        
        b = StringIO.StringIO()
        mycurl.setopt(pycurl.WRITEFUNCTION, b.write)
        mycurl.perform()
        output = b.getvalue()
        
        # Close the cURL object.  
        mycurl.close()
        
        # Return the Share on Ovi URL of the object we created.
        return objectURL
    
    def getChannels(self):
        mycurl = pycurl.Curl()
        mycurl.setopt(pycurl.URL, 'http://share.ovi.com/api/atom/1.0')

        mycurl.setopt(pycurl.USERPWD, self.username+':'+self.password)
        mycurl.setopt(pycurl.HEADER, 0)
        mycurl.setopt(pycurl.HTTPGET, 1)
        
        b = StringIO.StringIO()
        mycurl.setopt(pycurl.WRITEFUNCTION, b.write)
        mycurl.perform()
        output = b.getvalue() 

        channels = []

        doc = minidom.parseString(output)
        wkslist = doc.getElementsByTagName('workspace')
        chlist = wkslist[0].getElementsByTagName('collection')
        for channelxml in chlist:
            url = channelxml.attributes['href']
            channels.append(os.path.splitext(url.value)[1][1:])

        return channels
        
    def makeMetadataSkeleton(self):
        xml = """<?xml version="1.0" encoding="utf-8" ?>
<entry xmlns="http://www.w3.org/2005/Atom"
        xmlns:atom03="http://purl.org/atom/ns#"
        xmlns:thr="http://purl.org/syndication/thread/1.0"
        xmlns:georss="http://www.georss.org/georss"
        xmlns:nokia="http://schemas.nokia.com/onlinesharing/2006">
<atom03:generator>ShareOnOvi Python/1.0</atom03:generator>
</entry>"""
        return xml
