/*
  This file is part of "Scopa" - An italian card game.
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "GameStartDialog.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/ButtonRow.h>
#include <Lum/Combo.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Space.h>

GameStartDialog::GameStartDialog(size_t points, size_t player)
 : commitAction(new Lum::Model::Action()),
   points(new Lum::Model::String(Lum::Base::NumberToWString(points))),
   player(new Lum::Model::SizeT(player)),
   success(false)
{
  commitAction->Disable();

  Observe(GetClosedAction());

  Observe(commitAction);
  Observe(this->points);
}

void GameStartDialog::PreInit()
{
  Lum::ButtonRow           *row;
  Lum::TextEditCombo       *pointsCombo;
  Lum::IndexCombo          *playerCombo;
  Lum::Label               *label;
  Lum::Panel               *vert;
  Lum::Model::ListTableRef tableModel;

  vert=new Lum::VPanel();

  label=new Lum::Label();
  label->SetFlex(true,false);

  tableModel=new Lum::Model::ListTable(1);
  for (size_t i=11; i<=21; i++) {
    tableModel->AppendString(Lum::Base::NumberToWString(i));
  }

  pointsCombo=new Lum::TextEditCombo();
  pointsCombo->SetTableModel(tableModel);
  pointsCombo->SetModel(points);
  pointsCombo->GetString()->SetAlignment(Lum::String::right);
  label->AddLabel(_(L"GAME_START_LABEL_POINTS",L"Required points to win:"),pointsCombo);

  tableModel=new Lum::Model::ListTable(1);
  tableModel->AppendString(_(L"GAME_START_JUDGE",L"Mrs. Judge"));
  tableModel->AppendString(_(L"GAME_START_GOAL",L"Mr. Goal"));
  tableModel->AppendString(_(L"GAME_START_GREEDY",L"Mr. Mc Greedy"));
  tableModel->AppendString(_(L"GAME_START_RANDOM",L"Mr. Random"));

  playerCombo=new Lum::IndexCombo();
  playerCombo->SetTableModel(tableModel);
  playerCombo->SetModel(player);
  label->AddLabel(_(L"GAME_START_LABEL_PLAYER",L"Computer opponent:"),playerCombo);

  vert->Add(label);

  vert->Add(new Lum::VSpace());

  row=Lum::ButtonRow::CreateOkCancel(commitAction,GetClosedAction());
  row->SetFlex(true,false);
  vert->Add(row);

  SetMain(vert);

  Dialog::PreInit();
}

void GameStartDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==commitAction && commitAction->IsFinished()) {
    success=true;
    Exit();
  }
  else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    Exit();
  }
  else if (model==points) {
    size_t tmp;

    if (Lum::Base::WStringToNumber(points->Get(),tmp) && tmp>0) {
      commitAction->Enable();
    }
    else {
      commitAction->Disable();
    }
  }

  Dialog::Resync(model,msg);
}

bool GameStartDialog::GetSuccess() const
{
  return success;
}

size_t GameStartDialog::GetPoints() const
{
  size_t tmp=0;

  if ((!Lum::Base::WStringToNumber(points->Get(),tmp) && tmp>0)) {
    assert(false);
  }

  return tmp;
}
size_t GameStartDialog::GetPlayer() const
{
  return player->Get();
}

bool ShowGameStartDialog(Lum::Dialog* parent, size_t& points, size_t& player)
{
  GameStartDialog *dialog;
  bool            success;

  dialog=new GameStartDialog(points,player);
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  success=dialog->GetSuccess();
  if (success) {
    points=dialog->GetPoints();
    player=dialog->GetPlayer();
  }

  delete dialog;

  return success;
}

