#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import evas
import logging
import math

__all__ = ("Box", "HBox", "VBox", "Grid")

log = logging.getLogger("terra.ui.layout")

class Box(evas.ClippedSmartObject):
    """Base class for layouts.

    Box will clip its contents to its size and can also modulate color
    of inner objects to its own color/clip.

    Subclasses should implement L{_setup_gui()} and L{required_size_get()}
    in order to work. The former will be used to actually move the children,
    while the later is used to return how much space it needs to be fully
    displayed.
    """
    def __init__(self, canvas, children=None, halign=0.5, valign=0.5):
        if self.__class__ is Box:
            raise TypeError("should not instantiate 'Box' directly")
        self._children = []
        self._frozen = 0
        self._dirty = True
        self.halign = float(halign)
        self.valign = float(valign)
        evas.ClippedSmartObject.__init__(self, canvas)
        if children:
            self.extend(children)

    def _setup_gui(self, x, y, w, h):
        """Virtual implementation to layout children in the given geometry."""
        raise NotImplementedError("%s._setup_gui() not implemented" %
                                  self.__class__.__name__)

    def freeze(self):
        self._frozen += 1

    def thaw(self):
        if self._frozen > 1:
            self._frozen -= 1
        elif self._frozen == 1:
            self._frozen = 0
            if self._dirty:
                self._setup_gui(*self.geometry_get())
        else:
            log.warning("thaw more than freeze!")

    def request_reconfigure(self, x, y, w, h):
        """Request object to reconfigure its children to fit given geometry."""
        self._dirty = True
        if self._frozen == 0:
            self._setup_gui(x, y, w, h)

    def required_size_get(self):
        """Virtual implementation to return the required size to fit.

        @return: C{(width, height)} required to fully display this box.
        @rtype: tuple
        """
        raise NotImplementedError("required_size_get() not implemented")

    def resize(self, w, h):
        if self.size == (w, h):
            return
        x, y = self.pos
        self.request_reconfigure(x, y, w, h)

    def append(self, obj):
        """Add new child to this box.

        If object is already member of some other object, it will be removed
        from it's parent and then added as member of this box.

        @warning: it will trigger object reconfiguration, use L{extend()} to
                  insert multiple.
        @see: L{extend()}
        """
        self.member_add(obj)
        self._children.append(obj)
        self.request_reconfigure(*self.geometry_get())

    def extend(self, children):
        """Add a bunch of children to this box.

        This behaves like L{append()} but is meant for multiple objects.
        @see: L{append()}
        """
        self.freeze()
        for c in children:
            self.append(c)
        self.thaw()

    def remove(self, obj):
        """Remove children from this box.

        @warning: it will trigger object reconfiguration, use L{clear()} to
                  remove all children.

        @return: C{True} is the object was found and removed or C{False}
                 if it was not found.
        @rtype: bool
        """
        try:
            self._children.remove(obj)
        except ValueError, e:
            return False

        self.member_del(obj)
        self.request_reconfigure(*self.geometry_get())
        return True

    def clear(self):
        """Remove all children from this box."""
        if not self._children:
            return

        for c in self._children:
            self.member_del(c)
        del self._children[:]
        self.request_reconfigure(*self.geometry_get())

    def children_get(self):
        """Returns an interator over box children.

        @warning: you should not add or remove any object while iterating.

        @return: iterator over children.
        @rtype: iter
        """
        return self._children.__iter__()


class HBox(Box):
    """Horizontal layout of items."""
    def __init__(self, canvas, children=None, hpadding=1, halign=0.5,
                 valign=0.5):
        self.hpadding = int(hpadding)
        Box.__init__(self, canvas, children, halign, valign)

    def required_size_get(self):
        if not self._children:
            return (0, 0)
        total_pad = self.hpadding * (len(self._children) - 1)
        total_width = sum(c.size[0] for c in self._children)
        max_height = max(c.size[1] for c in self._children)
        return (total_width + total_pad, max_height)

    def _setup_gui(self, x, y, w, h):
        if not self._dirty:
            return

        if not self._children:
            self.clipper.hide()
            self._dirty = False
            return

        self.clipper.visible = self.visible

        req_w, req_h = self.required_size_get()
        if req_w < w:
            x += int((w - req_w) * self.halign)

        for c in self._children:
            cy = int((h - c.size[1]) * self.valign)
            c.move(x, y + cy)
            x += c.size[0] + self.hpadding

        self._dirty = False


class VBox(Box):
    """Vertical layout of items."""
    def __init__(self, canvas, children=None, vpadding=1, halign=0.5,
                 valign=0.5):
        self.vpadding = int(vpadding)
        Box.__init__(self, canvas, children, halign, valign)

    def required_size_get(self):
        if not self._children:
            return (0, 0)
        total_pad = self.vpadding * (len(self._children) - 1)
        max_width = max(c.size[0] for c in self._children)
        total_height = sum(c.size[1] for c in self._children)
        return (max_width, total_height + total_pad)

    def _setup_gui(self, x, y, w, h):
        if not self._dirty:
            return

        if not self._children:
            self.clipper.hide()
            self._dirty = False
            return

        self.clipper.visible = self.visible

        req_w, req_h = self.required_size_get()
        if req_h < h:
            y += int((h - req_h) * self.valign)

        for c in self._children:
            cx = int((w - c.size[0]) * self.halign)
            c.move(x + cx, y)
            y += c.size[1] + self.vpadding

        self._dirty = False


class Grid(Box):
    """Grid layout of items.

    This component uses VBox with inner HBox, taking care of size
    to flow items properly.

    @warning: current implementation is not good for huge amount of items!
              It clears all rows and then layout all items again.
    """
    def __init__(self, canvas, children=None, hpadding=1, vpadding=1,
                 halign=0.5, valign=0.5):
        self.hpadding = int(hpadding)
        self.vpadding = int(vpadding)
        self._dirty_children = True
        self._last_grid_size = (0, 0)
        self._vbox = VBox(canvas, None, vpadding, halign, valign)
        Box.__init__(self, canvas, children, halign, valign)
        self.member_add(self._vbox)
        self._vbox.show()

    def freeze(self):
        self._vbox.freeze()
        Box.freeze(self)

    def thaw(self):
        if self._dirty_children:
            self._dirty = True
        Box.thaw(self)
        self._vbox.thaw()

    def required_size_get(self):
        return self._vbox.required_size_get()

    def request_reconfigure_contents(self):
        self._dirty_children = True
        if self._frozen == 0:
            self._setup_gui(*self.geometry_get())

    def resize(self, w, h):
        if self.size == (w, h):
            return
        self._vbox.resize(w, h)
        Box.resize(self, w, h)

    def append(self, obj):
        self._children.append(obj)
        self.request_reconfigure_contents()

    def remove(self, obj):
        try:
            self._children.remove(obj)
        except ValueError, e:
            return False

        self.request_reconfigure_contents()
        return True

    def clear(self):
        del self._children[:]
        self.request_reconfigure_contents()

    def get_grid_size(self, w, h):
        """Get how many (rows, columns) will fit in given area.

        @return: C{(rows, columns)} fits in the given area.
        @rtype: tuple
        """
        if not self._children:
            return (0, 0)

        if w <= 0 or h <= 0:
            return (0, 0)

        n_items = len(self._children)
        max_w = 0
        for c in self._children:
            cw, ch = c.size_get()
            max_w = max(max_w, cw)

        if max_w == 0:
            log.warning("could not find max size for children.")
            return (0, 0)

        max_w += self.hpadding

        n_horiz = w / max_w
        if n_horiz == 0:
            log.debug("no items fit on one line, forcing one")
            return (1, n_items)
        elif n_horiz > n_items:
            return (n_items, 1)
        else:
            n_vert = int(math.ceil(float(n_items) / float(n_horiz)))
            if n_vert > 1:
                last_line = n_items - ((n_vert - 1) * n_horiz)
                if 0 < last_line < n_horiz:
                    slots = n_horiz - last_line
                    reducing = slots / n_vert
                    n_horiz -= reducing

            return (n_horiz, n_vert)

    def _setup_gui(self, x, y, w, h):
        if not self._dirty and not self._dirty_children:
            return

        grid_size = self.get_grid_size(w, h)
        if not self._dirty_children and self._last_grid_size == grid_size:
            self._dirty = False
            return

        d_vert = grid_size[1] - self._last_grid_size[1]
        self._last_grid_size = grid_size
        n_horiz, n_vert = grid_size

        self._vbox.freeze()
        self._vbox.resize(w, h)

        for hbox in self._vbox.children_get():
            hbox.freeze()
            hbox.clear()

        if d_vert < 0:
            hboxes = tuple(self._vbox.children_get())
            for hbox in hboxes[:-d_vert]:
                self._vbox.remove(hbox)
                hbox.delete()
        elif d_vert > 0:
            for i in xrange(d_vert):
                hbox = HBox(self.evas, hpadding=self.hpadding, valign=0.0)
                hbox.freeze()
                self._vbox.append(hbox)
                hbox.show()

        offset = 0
        for hbox in self._vbox.children_get():
            row_children = self._children[offset : offset + n_horiz]
            offset += n_horiz

            if row_children:
                hbox.extend(row_children)
                hbox.size = hbox.required_size_get()

            hbox.thaw()

        self._vbox.request_reconfigure(x, y, w, h)
        self._vbox.thaw()
        self._dirty = False
        self._dirty_children = False
