#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import edje
import os.path

__all__ = ("Widget", "EdjeWidget", "EdjeSettingsWidget")

class Widget(object):
    """Base to be used as Terra widget.

    This class provides theme support, with dynamic change by calling
    L{theme_changed()}.

    This class is not an Evas Object, so you can use it as a mix-in
    (multiple inheritance) with any Evas Object you want.
    """
    def __init__(self, parent=None, theme=None):
        self._parent_widget = parent
        self._theme = theme

    def theme_get(self):
        """Get the theme.

        If object has no explicit theme set, then it will be acquired
        from parent.

        @return: filename.
        @rtype: str
        """
        if self._theme is not None:
            return self._theme
        elif self._parent_widget is not None:
            return self._parent_widget.theme

    def theme_set(self, theme, end_callback=None):
        """Set the theme and calls end_callback when it's done.

        @param theme: file name/path to desired theme.
        @type theme: str or None
        @param end_callback: function to callback when theme change is finished
               (it can be an animation).
        @type end_callback: function(widget)

        @warning: if theme is equal to current, then it will just call
                  L{end_callback(self)}.
        """
        if self._theme != theme:
            self._theme = theme
            self.theme_changed(end_callback)
        elif end_callback is not None:
            end_callback(self)

    theme = property(theme_get, theme_set)

    def theme_changed(self, end_callback=None):
        """Virtual method used to implement theme change.

        It will be called by L{theme_set()} or by parent to request widget
        to reload it's internal objects using the new theme.
        """
        raise NotImplementedError("%s.theme_changed() not implemented." %
                                  self.__class__.__name__)


DEFAULT_THEME_NAME = "default.edj"
def file_set_fallback(obj, theme, group):
    try:
        edje.Edje.file_set(obj, theme, group)
    except edje.EdjeLoadError, e:
        dir, name = os.path.split(theme)
        if name == DEFAULT_THEME_NAME:
            raise
        else:
            fallback = os.path.join(dir, DEFAULT_THEME_NAME)
            edje.Edje.file_set(obj, fallback, group)


class EdjeWidget(Widget, edje.Edje):
    "Edje object implementing Widget interface."
    def __init__(self, canvas, group, parent=None, theme=None):
        Widget.__init__(self, parent=parent, theme=theme)
        edje.Edje.__init__(self, canvas, file=self.theme, group=group)
        self.size_set(*self.size_min_get())

    def theme_changed(self, end_callback=None):
        "Reset Edje file with new theme."
        f, group = self.file_get()
        self.file_set(self.theme, group)
        if end_callback is not None:
            end_callback(self)

    def file_set(self, file, group):
        file_set_fallback(self, file, group)

    def Edje(self, group, **params):
        """Edje factory using the same theme as this object.

        @return: L{edje.Edje} instance using the same theme file as this widget.
        @rtype: L{edje.Edje}
        """
        e = edje.Edje(self.evas, **params)
        file_set_fallback(e, self.theme, group)
        e.show()
        return e


def extra_file_set_fallback(obj, theme, group, plugin):
    fset = edje.Edje.file_set
    plugin_theme = "%s-%s.edj" % (theme[:-4], plugin)

    try:
        fset(obj, plugin_theme, group)
        return
    except edje.EdjeLoadError:
        pass # ok, see in the main theme file

    try:
        fset(obj, theme, group)
        return
    except edje.EdjeLoadError:
        dir, name = os.path.split(theme)
        if name == DEFAULT_THEME_NAME:
            raise
        else:
            default_theme = os.path.join(dir, DEFAULT_THEME_NAME)

    plugin_default = "%s-%s.edj" % (default_theme[:-4], plugin)
    try:
        fset(obj, plugin_default, group)
        return
    except edje.EdjeLoadError:
        pass

    # last try, default theme
    fset(obj, default_theme, group)


class PluginThemeMixin(object):
    "Mixin for EdjeWidgets that look for plugin-specific themes."
    plugin = None

    def file_set(self, file, group):
        assert self.plugin is not None
        extra_file_set_fallback(self, file, group, self.plugin)

    def Edje(self, group, **params):
        """Edje factory using the same theme as this object.

        @return: L{edje.Edje} instance using the same theme file as this widget.
        @rtype: L{edje.Edje}
        """
        e = edje.Edje(self.evas, **params)
        extra_file_set_fallback(e, self.theme, group, self.plugin)
        e.show()
        return e

    def PluginEdjeWidget(self, group, parent=None):
        """PluginEdjeWidget factory using the same theme as this object.

        @return: L{PluginEdjeWidget} instance using the same theme file as this widget.
        @rtype: L{PluginEdjeWidget}
        """
        if parent is None:
            parent = self
        e = PluginEdjeWidget(self.evas, group, parent, plugin=self.plugin)
        e.show()
        return e


class PluginEdjeWidget(PluginThemeMixin, EdjeWidget):
    "EdjeWidget that look for plugin-specific themes."

    def __init__(self, canvas, group, parent=None, theme=None, plugin=None):
        if plugin is not None:
            self.plugin = plugin
        EdjeWidget.__init__(self, canvas, group, parent, theme)

    def PluginEdjeWidget(self, group, parent=None):
        """PluginEdjeWidget factory using the same theme as this object.

        @return: L{PluginEdjeWidget} instance using the same theme file as this widget.
        @rtype: L{PluginEdjeWidget}
        """
        if parent is None:
            parent = self
        e = PluginEdjeWidget(self.evas, group, parent, plugin=self.plugin)
        e.show()
        return e


class EdjeSettingsWidget(EdjeWidget):
    """
    Base class used by settings widgets, such as Panel and Modal.
    """
    def __init__(self, main_window, title, group, theme=None):
        EdjeWidget.__init__(self, main_window.evas, group,
                            main_window, theme)
        self.title_set(title)

    def title_get(self):
        return self._title

    def title_set(self, title):
        self._title = title
        self.part_text_set("title", title)

    title = property(title_get, title_set)

    def contents_get(self):
        return self._contents

    def contents_set(self, contents):
        self._contents = contents
        self.part_swallow("contents", contents)

    contents = property(contents_get, contents_set)

