#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
import re
import logging

from terra_object import TerraObject, TERRA_TYPE_ATTR_NAME
from module_loader import ModuleLoader


__all__ = ("PluginManager",)


log = logging.getLogger("terra.core.plugin_manager")


class Plugin(object):
    def __init__(self, basedir, conf):
        self._conf = conf
        self._ldr = ModuleLoader(os.path.join(basedir, conf.moddir),
                                 conf.modname)
        self.classes = {}               # maps filters to classes
        self.loaded = False

    def _get_name(self):
        return self._conf.name

    name = property(_get_name)

    def _is_enabled(self):
        return self._conf.enabled

    enabled = property(_is_enabled)

    def _register_filters(self, mod):
        for check_type, cls_name in self._conf.filter_map:
            try:
                cls = getattr(mod, cls_name)
            except AttributeError:
                log.debug("plugin %r doesn't have class %r",
                          self.name, cls_name)
                raise

            if not isinstance(cls, type) or \
               not issubclass(cls, TerraObject) or \
               cls is TerraObject:
                msg = "plugin %r: class %r is not a Terra object" % \
                      (self.name, cls_name)
                log.debug(msg)
                raise ValueError(msg)

            try:
                terra_type = getattr(cls, TERRA_TYPE_ATTR_NAME)
            except AttributeError:
                log.debug("plugin %r: class %r doesn't have %r attribute",
                          self.name, cls_name, TERRA_TYPE_ATTR_NAME)
                raise

            if terra_type != check_type:
                msg = "plugin %r: class %r type %r differs from config " \
                      "file (%r)" % (self.name, cls_name, terra_type, check_type)
                log.debug(msg)
                raise ValueError(msg)

            if terra_type in self.classes:
                msg = "plugin %r tried to register another class for " \
                      "filter %r" % (self.name, terra_type)
                log.debug(msg)
                raise ValueError(msg)

            self.classes[terra_type] = cls

    def load(self):
        if self.loaded:
            log.warning("plugin %r already loaded", self.name)
            return self.classes

        self.loaded = True              # don't load me again, never!

        mod = self._ldr.load()
        self._register_filters(mod)

        return self.classes


class FilterEntry(object):
    def __init__(self):
        self.classes = []
        self.complete = False


class PluginManager(object):
    def __init__(self, basedir, conf_mger):
        self._basedir = basedir
        self._conf_mger = conf_mger
        self._plugins = {}               # maps filters to Plugin
        self._filters = {}               # maps filters to FilterEntry

        self._load_config()

    def _load_config(self):
        try:
            confs = self._conf_mger.load()
        except Exception, e:
            log.error("couldn't load plugins config file: %s", e)
            raise

        for c in confs:
            plg = Plugin(self._basedir, c)
            for f, cls_name in c.filter_map:
                self._plugins.setdefault(f, []).append(plg)

        def sort_by_rank(plg1, plg2):
            r1 = plg1._conf.rank
            r2 = plg2._conf.rank

            if r1 > r2:
                return -1
            if r1 < r2:
                return 1
            return 0

        for plgs_list in self._plugins.itervalues():
            plgs_list.sort(cmp=sort_by_rank)

    def _load_plugins(self, filter):
        for plg in self._plugins.get(filter, []):
            if plg.loaded:
                continue

            if not plg.enabled:
                log.debug_warning("plugin %r disabled, skipping it ...",
                                  plg.name)
                continue

            try:
                classes = plg.load()
            except Exception, e:
                log.debug_error("[%s] while loading plugin %r, skipping it ...",
                                e, plg.name, exc_info=True)
                continue

            log.info("loaded plugin %r with filters %r",
                     plg.name, classes.keys())

            for f, cls in classes.iteritems():
                fmap = self._filters.setdefault(f, FilterEntry())
                fmap.classes.append(cls)

    def load(self, filter):
        if not isinstance(filter, basestring):
            raise TypeError("filter should be a string")

        if filter not in self._plugins:
            log.debug_warning("no plugin implements filter %r", filter)
            return None

        fmap = self._filters.setdefault(filter, FilterEntry())
        if fmap.complete:
            return fmap.classes

        fmap.complete = True            # don't load this filter ever again!

        self._load_plugins(filter)

        return fmap.classes

    def load_by_regexp(self, regexp, unique=False):
        if not isinstance(regexp, basestring):
            raise TypeError("regexp should be a string")

        re_comp = re.compile(regexp)

        classes = []
        for f in self._plugins:
            if not re_comp.match(f):
                continue
            ncls = self.load(f)
            if not unique:
                classes.extend(ncls)
            elif ncls:
                classes.append(ncls[0])

        return classes

    def load_all(self):
        classes = []
        for f in self._plugins:
            classes.extend(self.load(f))

        return classes

    def get(self, filter):
        if filter not in self._plugins:
            raise ValueError("no plugin implements filter %r" % filter)

        fmap = self._filters.get(filter, None)
        if fmap is not None and fmap.complete:
            return fmap.classes

        raise RuntimeError("plugins for filter %r not loaded yet" % filter)
