#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging

from manager import Manager
from terra_object import TerraObject


log = logging.getLogger("terra.core.controller")


class Controller(TerraObject):
    """Base class for all Terra controllers."""
    terra_type = "Controller"

    def __init__(self, model, canvas, parent):
        TerraObject.__init__(self)
        self.model = model
        self.evas = canvas
        self.parent = parent
        self._refcount = 0
        self.is_suspended = False

    def back(self):
        raise NotImplementedError("must be implemented by subclasses")

    def go_home(self):
        raise NotImplementedError("must be implemented by subclasses")

    def do_suspend(self):
        pass

    def suspend(self):
        if self.is_suspended:
            return

        self.is_suspended = True
        self.do_suspend()

    def do_resume(self):
        pass

    def resume(self):
        if not self.is_suspended:
            return

        self.is_suspended = False
        self.do_resume()

    def delete(self):
        raise NotImplementedError("must be implemented by subclasses")

    def ref(self):
        assert self._refcount >= 0
        self._refcount += 1
        self.model.hold()

    def unref(self):
        assert self._refcount > 0
        self._refcount -= 1
        self.model.release()
        if self._refcount == 0:
            self.delete()

    def __str__(self):
        return "%s(model=%r, canvas=%s, parent=%s)" % \
               (self.__class__.__name__, self.model, bool(self.evas),
                bool(self.parent))

    __repr__ = __str__


def _controller_type_from_string(model_type):
    model_name = model_type
    if model_type.startswith("Model/"):
        model_name = model_type[6:]

    return "Controller/%s" % model_name


def _get_parent_type(type_name):
    try:
        type_name = type_name[:type_name.rindex("/")]
    except ValueError:
        return None

    return type_name


def get_controller_class_for_model(model):
    return get_controller_class_for_model_type(model.terra_type)


def get_controller_class_for_model_type(type):
    mger = Manager()
    type_name = _controller_type_from_string(type)

    while type_name:
        log.debug("looking for controller class with filter %r ...", type_name)

        try:
            cls = mger.get_class(type_name)
            log.debug("found controller class with filter %r!", type_name)
            return cls
        except ValueError:
            pass

        type_name = _get_parent_type(type_name)

    return None


def get_controller_for_model(model, canvas, parent):
    """Create a new controller based on a model type.
    This works using model's terra_type string and does inheritance based on it.

    Example: if you request Model/Folder/Audio but nothing matches exactly, it
    will try Model/Folder, then Model.
    """
    log.debug("looking for controller for model with filter %r ...",
              model.terra_type)

    cls = get_controller_class_for_model(model)
    if cls is not None:
        return cls(model, canvas, parent)

    return None
