/*
 * This file is part of pwsafe
 *
 * Copyright (C) 2005 HolisTech Limited.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
 
#include <src/crypto.h>
#include <src/sha1.h>
#include <src/BlowFish.h>
#include <src/interface.h>

#include <src/preferences.h>

#include <gtk/gtk.h>
/* Gnome VFS for file i/o */
#include <libgnomevfs/gnome-vfs.h>
#include <string.h>


/* private */
char readbytes(MainView* mainview, GnomeVFSHandle *handle, unsigned char *ptr, unsigned int bytes);
char *readcbc(MainView* mainview, GnomeVFSHandle *handle, BLOWFISH_CTX *blowCTX, guint32 iv[2], int *bytes, unsigned char *rec_type);
void GenRandhash(const gchar *pass, const unsigned char* randstuff, unsigned char* result_hash);
char testpassword(MainView* mainview, GnomeVFSHandle *handle);
gint rec_comparer(const rec *a, const rec *b);
void writepassword(MainView* mainview, GnomeVFSHandle *handle);
void writecbc(MainView* mainview, GnomeVFSHandle *handle, BLOWFISH_CTX *blowCTX, gint32 iv[2], const gchar *data, int length, const unsigned char rec_type, const gboolean writeNull);
void new_uuid(rec *entry);
gboolean pwsafe_file_open_error(MainView *mainview, GnomeVFSResult vfs_result);

inline void trashMemory(char *start, unsigned int length) {
/* The original password safe overwrites 90 times... I really don't understand why.
Memory isn't magnetic, and it would be far far easier to log keypresses to get the password than 
grab some kind of echo (?) from memory. A 90x loop seems like a huge waste of processor for 
anything battery powered. Is there something I don't know about?*/
	memset(start, 0, length);
}


char readbytes(MainView* mainview, GnomeVFSHandle *handle, unsigned char *ptr, unsigned int bytes) {
/* reads the specified number of byted into ptr, returns true for ok, on fail flags error to user and returns false */
	GnomeVFSFileSize in_bytes;
	gnome_vfs_read(handle, ptr, bytes, &in_bytes);
	if (in_bytes!=bytes) {
		interface_error(PWSAFE_ERROR_OPEN_FAILED, mainview);
		gnome_vfs_close(handle);
		return FALSE;
	}
	return TRUE;
}

void close_file(MainView *mainview) {
	if (!mainview) return;
	mainview->pass=NULL;		/*  make sure file can't be overwritten V1.2 change */
	mainview->file_edited=FALSE;
	mainview->file_name=NULL;
	GList *onrec=NULL;
	empty_iconView(mainview);
	if (recs!=NULL) {
		unshow_rec(mainview);  /* close any open rec window */
		onrec=g_list_first(recs);
		while (onrec!=NULL) {
			delete_rec(onrec->data);
			onrec=g_list_first(recs); /* first ratehr than next, as we just deleted the first... */
		}
		g_list_free (recs); /* should be empty, but just incase */
		recs=NULL;
	}
	gtk_clipboard_set_text(mainview->clipboard, "", -1);
	mainview->preferences=NULL;
	trashFreeMemory(mainview->pass, -1);
	interface_show_state(mainview);
}

void delete_rec(rec *arec) {
	if (arec!=NULL) {
		recs = g_list_remove(recs, arec);
		trashFreeMemory(arec->uuid, 16);
		trashFreeMemory(arec->group, -1);
		trashFreeMemory(arec->title, -1);
		trashFreeMemory(arec->user, -1);
		trashFreeMemory(arec->notes, -1);
		trashFreeMemory(arec->password, -1);
		trashFreeMemory((char *)arec, sizeof(arec));
	}
}



void trashFreeMemory(char *start, int length) {
/* is length<0 take strlen(start) */
	if (start!=NULL) {
		if (length<0) length=strlen(start);
		trashMemory(start, length);
		g_free(start);
	}
}

char *readcbc(MainView* mainview, GnomeVFSHandle *handle, BLOWFISH_CTX *blowCTX, guint32 iv[2], int *bytes, unsigned char *rec_type) {
/* allocates and fills buffer with next data block. returns pointer to data read or NULL on error */
/* caller must always g_free(data) even when 0 bytes returned */
/* warning... sloppy coding below... wots with all that casting */
	guint32 len[2];
	guint32 newiv[2];
	guint32 x, x1, BlockLength;
	guint32 *buffer=NULL;
	/* read and decrypt the file */
	if (!readbytes(mainview, handle, (unsigned char *)len, 8)) return NULL;
	/* I think the compiler will do better with 2x32bit arithmentic rather than 8 byte memcpys as in original */
	/* is register juggling faster than small memory copies on gcc-arm? */
	newiv[0]=len[0];
	newiv[1]=len[1];
	Blowfish_Decrypt(blowCTX, &len[0], &len[1]);
	len[0]^=iv[0];	/* length of block */
	len[1]^=iv[1];	/* type of block */
	iv[0]=newiv[0];
	iv[1]=newiv[1];
	
	if (len[0]>0x80000000) return NULL;  /* length is signed - no negatives allowed */
	/* if (len[0]>0x1400000) return; but on maemo we won't even try a block bigger than 20MB */
	
	(*bytes)=len[0];
	(*rec_type)=len[1] & 0xff;	/* type is first byte only... other 3 are unused (random). */

	BlockLength = ((len[0]+7)/8)*8;
	if (BlockLength==0) BlockLength=8;	/* yep - read a block even if length is 0 */
	buffer = (guint32 *) g_malloc0(BlockLength+1);
	if (!readbytes(mainview, handle, (unsigned char *)buffer, BlockLength)) return NULL;
	BlockLength/=4;
	for (x=0; x<BlockLength; x+=2) {
		x1=x+1;
		newiv[0]=buffer[x];
		newiv[1]=buffer[x1];
		Blowfish_Decrypt(blowCTX, &buffer[x], &buffer[x1]);
		buffer[x]^=iv[0];
		buffer[x1]^=iv[1];
		iv[0]=newiv[0];
		iv[1]=newiv[1];
	}
/*	{
		char txt[300];
		sprintf(txt, "t: %ld len: %ld %08lX %08lX %08lX %08lX", len[1], len[0], buffer[0], buffer[1], buffer[2], buffer[3]);
		msgbox (mainview, txt);
	}
*/
	/* blank the unneeded stuff... we could leak newiv... but that doesn't look too dangerous too me. */
	return (char *)buffer;
}

void GenRandhash(const gchar *pass, const unsigned char* randstuff, unsigned char* result_hash) {
/* returns a password safe hash of the password (combined with the randstuff) in result_hash */
/* all a bit obscure really... security through...? */
	guint32 tempbuf[3];  /* 12 bytes */
	unsigned char tempSalt[20];
	int x;
	SHA1_CTX keyHash;
	BLOWFISH_CTX blowCTX;
	/*     tempSalt <- H(a_randstuff + a_passkey)    */
	SHA1Init(&keyHash);
	SHA1Update(&keyHash, randstuff, 10);
	SHA1Update(&keyHash, (unsigned char *) pass, strlen((char *)pass));
	SHA1Final(tempSalt, &keyHash);
	/*     tempbuf <- a_randstuff encrypted 1000 times using tempSalt as key?   */
	Blowfish_Init(&blowCTX, tempSalt, sizeof(tempSalt));
	memcpy((char*)tempbuf, (char*)randstuff, 10);
	for (x=0; x<1000; x++) Blowfish_Encrypt(&blowCTX, &tempbuf[0], &tempbuf[1]);
	/*      hmm - seems we're not done with this context we throw the tempbuf into the hasher, and extract a_randhash */
	SHA1Update(&keyHash, (unsigned char*)tempbuf, 10);
	SHA1Final(result_hash, &keyHash);
	/* security tidy */
	trashMemory((char*) &keyHash, sizeof(keyHash));
	trashMemory((char*) &blowCTX, sizeof(blowCTX));
	trashMemory((char*) &tempSalt, sizeof(tempSalt));
	trashMemory((char*) &tempbuf, sizeof(tempbuf));
}

gint rec_comparer(const rec *a, const rec *b) {
	gboolean agroup, bgroup;
	agroup=(a->group) && (a->group[0]!='\0');
	bgroup=(b->group) && (b->group[0]!='\0');
	if (agroup && bgroup) {
		int result;
		result=strcmp(a->group, b->group);
		if (result!=0) return result;
		return strcmp(emptynull(a->title), emptynull(b->title));
	}
	if (!(agroup || bgroup)) strcmp(emptynull(a->title), emptynull(b->title));
	if (agroup) return -1;
	return 1;
}

void resort_rec(rec *entry) {
/* resort entry into list */
	recs=g_list_remove(recs, entry);
	recs=g_list_insert_sorted(recs, entry, (GCompareFunc) rec_comparer);
}

void new_uuid(rec *entry) {
	int i;
	guint32 uuid[4];
	for (i=0; i<4; i++) {
		uuid[i]=(guint32) g_random_int();
	}
	entry->uuid=(char *)g_malloc(16);
	memcpy(entry->uuid, uuid, 16);
}

rec * add_new_record(gchar *title, gchar *group) {
	rec *entry;
	entry=g_malloc0(sizeof(rec));
	
	entry->title=g_strdup(title);
	entry->group=g_strdup(group);
	new_uuid(entry);
	entry->user=entry->notes=entry->password=NULL;
	recs=g_list_insert_sorted(recs, entry, (GCompareFunc) rec_comparer); /*  add record to list */
	return entry;
}

char testpassword(MainView* mainview, GnomeVFSHandle *handle) {
/* expects handle to be a ready opened file (at the start) - reads out the crypted password and compares it with pass */
	unsigned char rnd[10];
	unsigned char rndhash[20];
	unsigned char rndhash2[20];
	if (!readbytes(mainview, handle, rnd, 8)) return FALSE;
	rnd[8]=rnd[9]='\0'; 
	if (!readbytes(mainview, handle, rndhash, sizeof(rndhash))) return FALSE;
	/* test password */
	GenRandhash(mainview->pass, rnd, rndhash2);
	if (memcmp(rndhash, rndhash2, sizeof(rndhash))!=0) {
		interface_error(PWSAFE_ERROR_BADPASS, mainview);
		gnome_vfs_close(handle);
		return FALSE;
	}
	return TRUE;
}

void writepassword(MainView* mainview, GnomeVFSHandle *handle) {
	gint32 rnd[3]; /* first 10 bytes used... */
	unsigned char rndhash[20];
	GnomeVFSFileSize bWritten;
	rnd[0]=g_random_int();
	rnd[1]=g_random_int();
	rnd[2]=0;
	gnome_vfs_write(handle, rnd, 8, &bWritten); /* write 8 bytes of random number */
	GenRandhash(mainview->pass, (unsigned char *) rnd, rndhash);  /* generate password hash */
	gnome_vfs_write(handle, rndhash, sizeof(rndhash), &bWritten);
}

void write_pwsafe_file(MainView* main) {
/* TODO: fix tiny memory leak on file write error */
	gint32 salt[5];
	gint32 iv[2];
	gchar passkey[20];
	SHA1_CTX sha1;
	BLOWFISH_CTX blowCTX;
	GList *onrec=NULL;
	GnomeVFSHandle *handle = NULL;
	int i;
	GnomeVFSFileSize bWritten;
	GnomeVFSResult vfs_result;
	
	if (main->pass==NULL) return; /* no password, no save */
	/* prompt if there's no file name */
	if (main->file_name==NULL) {
		main->file_name=interface_file_chooser(main, GTK_FILE_CHOOSER_ACTION_SAVE);
	}
	main->file_edited=FALSE;
	/* init vfs just incase... */
	gnome_vfs_init();
	/* create file to write */
	gchar *file_name_new = g_strconcat(main->file_name, ".new", NULL);
	vfs_result = gnome_vfs_create(&handle, file_name_new, GNOME_VFS_OPEN_WRITE, FALSE, GNOME_VFS_PERM_USER_ALL);
	if (vfs_result!=GNOME_VFS_OK) {
		interface_error(PWSAFE_ERROR_SAVE_FAILED, main);
		main->file_edited=TRUE;
		return;
	}
	/* writepassword... */
	writepassword(main, handle);
	/* write salt and iv */
	iv[0]=g_random_int();
	iv[1]=g_random_int();
	for (i=0; i<5; i++) {
		salt[i]=g_random_int();
	}
	gnome_vfs_write(handle, salt, sizeof(salt), &bWritten);
	gnome_vfs_write(handle, iv, sizeof(iv), &bWritten);

	/* init the blowfish */
	SHA1Init(&sha1);
	SHA1Update(&sha1, (unsigned char *) main->pass, strlen(main->pass));
	SHA1Update(&sha1, (unsigned char *) salt, 20);
	SHA1Final((unsigned char *) passkey, &sha1);
	Blowfish_Init(&blowCTX, (unsigned char *) passkey, sizeof(passkey));
	trashMemory((char *) passkey, sizeof(passkey));
	trashMemory((char *) &sha1, sizeof(sha1));
	
	/* write header */
	writecbc(main, handle, &blowCTX, iv, " !!!Version 2 File Format!!! Please upgrade to PasswordSafe 2.0 or later", 72, NAME, TRUE);
	writecbc(main, handle, &blowCTX, iv, "2.0", 3, PASSWORD, TRUE);
	writecbc(main, handle, &blowCTX, iv, main->preferences, -1, NOTES, TRUE);

	onrec=g_list_first(recs);
	while (onrec!=NULL) {
		rec* thisrec=onrec->data;
		if (thisrec!=NULL) {
			/* write record... original pwsafe seems to write nulls rather than omit unneeded fields, so we will too */
			writecbc(main, handle, &blowCTX, iv, thisrec->uuid, 16, UUID, TRUE);
			writecbc(main, handle, &blowCTX, iv, thisrec->group, -1, GROUP, TRUE);
			writecbc(main, handle, &blowCTX, iv, thisrec->title, -1, TITLE, TRUE);
			writecbc(main, handle, &blowCTX, iv, thisrec->user, -1, USER, TRUE);
			writecbc(main, handle, &blowCTX, iv, thisrec->notes, -1, NOTES, TRUE);
			writecbc(main, handle, &blowCTX, iv, thisrec->password, -1, PASSWORD, TRUE);
			writecbc(main, handle, &blowCTX, iv, "", 0, END, TRUE);
		}
		onrec=g_list_next(onrec);
	}
	/* close file */
	if (GNOME_VFS_OK==gnome_vfs_close(handle)) {	/* Only replace original file if it closed OK */
		gnome_vfs_move(file_name_new, main->file_name, TRUE);
	} else {
		interface_error(PWSAFE_ERROR_SAVE_FAILED, main);
	}
	g_free(file_name_new);
	trashMemory((char *) &blowCTX, sizeof(blowCTX));
}

void vfs_write_error_test(MainView *mainview, GnomeVFSHandle *handle, GnomeVFSResult result) {
	if (result!=GNOME_VFS_OK) {
		interface_error(PWSAFE_ERROR_SAVE_FAILED, mainview);
		mainview->file_edited=TRUE;
		gnome_vfs_close(handle);		/* nasty way of making sure the error is noticed before replacing the original file */
	}
}

void writecbc(MainView* mainview, GnomeVFSHandle *handle, BLOWFISH_CTX *blowCTX, gint32 iv[2], 
			const gchar *data, int length, const unsigned char rec_type, const gboolean writeNull) {
/* if length<0 then get the string length... */
	int blockLength;
	int x;
	guint32 len[2];
	GnomeVFSFileSize bWritten;
	
	if ((!writeNull) && ((data==NULL) || (data[0]=='\0'))) return;
	if (length<0) length= (data==NULL) ? 0 : strlen(data);
	blockLength = ((length+7)/8)*8;
	if (blockLength==0) blockLength = 8; /* write a block for 0 bytes... */

	/* First encrypt and write the length of the buffer */
	/* Fill unused bytes of type with random data */
	len[0]=length;
	len[1]=(g_random_int() & 0xffffff00) + rec_type;
	len[0]^=iv[0];
	len[1]^=iv[1];
	Blowfish_Encrypt(blowCTX, &len[0], &len[1]);
	iv[0]=len[0];
	iv[1]=len[1];
	vfs_write_error_test(mainview, handle, gnome_vfs_write(handle, len, 8, &bWritten));

	/* Now, encrypt and write the buffer */
	for (x=0; x<blockLength; x+=8) {
		/* check for last uneven block */
		if ((length==0) || ((length%8 != 0) && (length-x<8))) {
			len[0]=0;
			len[1]=0;
			if ((length % 8)!=0) memcpy(len, data+x, length % 8);
		} else { /* not uneven, just copy 8 bytes to encrypt */
			memcpy(len, data+x, 8);
		}
		len[0]^=iv[0];
		len[1]^=iv[1];
		Blowfish_Encrypt(blowCTX, &len[0], &len[1]);
		iv[0]=len[0];
		iv[1]=len[1];
		vfs_write_error_test(mainview, handle, gnome_vfs_write(handle, len, 8, &bWritten));
	}
	/* nothing worth scrubbing here - we encrypted it all, and will reuse the blowfish context */
}

gboolean pwsafe_file_open_error(MainView *mainview, GnomeVFSResult vfs_result) {
	if (vfs_result!=GNOME_VFS_OK) {
		gchar *txt;
		txt=g_strjoin(NULL, "Failed to Open '", mainview->file_name, "' (", gnome_vfs_result_to_string(vfs_result), ")", NULL);
		msgbox(mainview, txt);
		g_free(txt);
		mainview->file_name=NULL;
		mainview->pass=NULL;
		return TRUE;
	}
	return FALSE;
}

void read_pwsafe_file(MainView* mainview) {
	/* get password */
	gchar *file_utf8;
	g_assert(mainview!=NULL);
	if (!mainview->file_name) return;
	file_utf8=g_filename_to_utf8(mainview->file_name, -1, NULL, NULL, NULL);
	mainview->pass=get_password(mainview, file_utf8);
	g_free(file_utf8);

	if (mainview->pass) {
		GnomeVFSResult vfs_result;
		GnomeVFSHandle *handle = NULL;
		GnomeVFSFileInfo finfo;
		char *buffer=NULL;
		int buffer_len;
		unsigned char salt[20];
		guint32 iv[2];
		unsigned char rec_type;
		char *filetitle=NULL;
		char *fileversion=NULL;
		gboolean uuiderr=FALSE;
		SHA1_CTX keyHash;
		BLOWFISH_CTX blowCTX;
		rec *entry;
		GnomeVFSFileSize filepos;

		/* init vfs just incase... */
		gnome_vfs_init();
		/* get file info (test's file exists, and gets length into finfo) */
		vfs_result = gnome_vfs_get_file_info(mainview->file_name, &finfo, GNOME_VFS_FILE_INFO_FOLLOW_LINKS);
		if (pwsafe_file_open_error(mainview, vfs_result)) return;
		/* open file to read */
		vfs_result = gnome_vfs_open(&handle, mainview->file_name, GNOME_VFS_OPEN_READ);
		if (pwsafe_file_open_error(mainview, vfs_result)) return;
		if (!testpassword(mainview, handle)) return;
		/* read salt and iv */
		if (!readbytes(mainview, handle, salt, sizeof(salt))) return;
		if (!readbytes(mainview, handle, (unsigned char *) iv, sizeof(iv))) return;
		/* generate decryption key */
		SHA1Init(&keyHash);
		SHA1Update(&keyHash, (unsigned char *) mainview->pass, strlen(mainview->pass));
		SHA1Update(&keyHash, salt, sizeof(salt));
		SHA1Final(salt, &keyHash);
		/* initialize blowfish with the key */
		Blowfish_Init(&blowCTX, salt, sizeof(salt));
		
		buffer=readcbc(mainview, handle, &blowCTX, iv, &buffer_len, &rec_type);
		if (buffer==NULL) return;
		filetitle=buffer;
		buffer=readcbc(mainview, handle, &blowCTX, iv, &buffer_len, &rec_type);
		if (buffer==NULL) return;
		fileversion=buffer;
		if (strcmp("2.0", fileversion)!=0) msgbox(mainview, "Unexpected file version (not 2.0)");
		buffer=readcbc(mainview, handle, &blowCTX, iv, &buffer_len, &rec_type);
		mainview->preferences=buffer;
		/* create an entry record, with everything pointing to an empty string */
		entry=g_malloc(sizeof(rec));
		entry->uuid=entry->group=entry->title=entry->user=entry->notes=entry->password=NULL;
		recs=NULL;
		/* read and decrypt the blocks. store the settings for each one, and write them to the tree on each END */
		buffer=readcbc(mainview, handle, &blowCTX, iv, &buffer_len, &rec_type);
		do {
			switch (rec_type) {
				case UUID:
					if (buffer_len!=16) {
						if (!uuiderr) msgbox(mainview, "file error - UUID not 16 bytes. Recreating");
						uuiderr=TRUE;
						g_free(buffer);
						new_uuid(entry);
					} else {
						entry->uuid=buffer;
					}
					break;
				case GROUP:
					entry->group=buffer;
					break;
				case TITLE:
					entry->title=buffer;
					break;
				case USER:
					entry->user=buffer;
					break;
				case NOTES:
					entry->notes=buffer;
					break;
				case PASSWORD:
					entry->password=buffer;
					break;
				case END:
					g_free(buffer);
					recs=g_list_insert_sorted(recs, entry, (GCompareFunc) rec_comparer);
					entry=g_malloc(sizeof(rec));
					entry->uuid=entry->group=entry->title=entry->user=entry->notes=entry->password=NULL;
					break;
				default: /* unknown block... just delete it and move on. */
					if (buffer_len>0) g_free(buffer);
			}
			gnome_vfs_tell(handle, &filepos);
			if (finfo.size<=filepos) break;
			buffer=readcbc(mainview, handle, &blowCTX, iv, &buffer_len, &rec_type);
		} while (buffer!=NULL);
		gnome_vfs_close(handle);
		g_free(filetitle);
		g_free(fileversion);
		mainview->last_file=mainview->file_name;
		conf_save_string(CONF_LASTFILE, mainview->last_file);
		populate_iconView(mainview, "");
	}
}
