/*
 * This file is part of pwsafe
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2005 HolisTech Limited.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
                     
#include <src/callbacks.h>
#include <src/interface.h>
#include <src/crypto.h>
#include <gtk/gtk.h>
#include <libintl.h>
/* strlen needed from string.h */
#include <string.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if HILDON == 1
#include <hildon/hildon.h>
#else
#include <hildon-widgets/hildon-app.h>
#endif
#include <gdk/gdkkeysyms.h>

/* Privates: */
void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview);
void maybe_save_file(MainView *mainview);
char *get_confirmed_pass(MainView *mainview);

/* activate grid */
void callback_activate_iconView(GtkIconView *iconview, GtkTreePath *path, MainView *mainview) {
	GtkTreeIter iter;
	rec *onrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	if ( !gtk_tree_model_get_iter(GTK_TREE_MODEL(mainview->store), &iter, path) ) {
		return;
	}
	gtk_tree_model_get(GTK_TREE_MODEL(mainview->store), &iter,
	    DATA_COLUMN, &onrec,
	    -1);
	/*
	onrec=g_object_get_data (G_OBJECT(&iter), "r");
	*/
	if (onrec==NULL) {
		populate_iconView(mainview, ""); 
	} else {
		if ((onrec->group==NULL) || (strcmp(showing_group, onrec->group)==0)) {
			show_rec(mainview, onrec);
		} else {
			populate_iconView(mainview, onrec->group); 
		}
	}
}

gboolean callback_key_press(GtkWidget* widget, GdkEventKey* event,  MainView *mainview) {
	switch (event->keyval) {
		case GDK_F6: /* fullscreen */
			interface_fullscreen(mainview);
			return TRUE;
		case GDK_F7: /* button + */
			return TRUE;
		case GDK_F8: /* button - */
			return TRUE;
	}
	return FALSE;
}

void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview) {
/* scrub and delete oldstr, malloc and duplicate newstr, return pointer and tag file as changed... */
/* but only if the strings are different (and be careful of null old string pointers) */
	if ((*oldstr)==NULL) {
		if (newstr[0]!='\0') {
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	} else {
		if (strcmp((*oldstr),newstr)!=0) { /* check they differ before copying stuff around */
			trashMemory((*oldstr), strlen((*oldstr)));
			g_free((*oldstr));
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	}
}

/* record detail events */
void callback_rec_response(GtkDialog *dialog, gint arg1, MainView *mainview) {
	RecWindow *recwindow = NULL;
	rec *rec=NULL;
	GtkTextBuffer *buffer=NULL;
	GtkTextIter start;
	GtkTextIter end;

	g_assert(mainview != NULL && mainview->data != NULL );
	
	recwindow = mainview->rec_win;
	rec=recwindow->rec;
	switch (arg1) {
		case 1: /* save any changes */
			safeReplaceString(&rec->title, gtk_entry_get_text(GTK_ENTRY(recwindow->title)), mainview);
			safeReplaceString(&rec->group, gtk_entry_get_text(GTK_ENTRY(recwindow->group)), mainview);
			safeReplaceString(&rec->user, gtk_entry_get_text(GTK_ENTRY(recwindow->username)), mainview);
			safeReplaceString(&rec->password, gtk_entry_get_text(GTK_ENTRY(recwindow->password)), mainview);
			buffer=gtk_text_view_get_buffer (GTK_TEXT_VIEW(recwindow->notes));
			if (gtk_text_buffer_get_modified(buffer)) { /* update notes if buffer has changed */
				gtk_text_buffer_get_start_iter (buffer, &start);
				gtk_text_buffer_get_end_iter (buffer, &end);
				safeReplaceString(&rec->notes, gtk_text_buffer_get_text (buffer, &start, &end, FALSE), mainview);
			}
			resort_rec(rec);
			unshow_rec(mainview);  /* close rec window */
			populate_iconView(mainview, showing_group); /* redraw grid */
			break;
		case 2:
			gtk_clipboard_set_text(mainview->clipboard, rec->user, -1);
			break;
		case 3:
			gtk_clipboard_set_text(mainview->clipboard, rec->password, -1);
			break;
		case 4:
			if ((recwindow->editable) && (CONFRESP_YES==interface_yes_no_note(mainview, "Confirm Delete?"))) {
				unshow_rec(mainview);
				delete_rec(rec);
				populate_iconView(mainview, showing_group); /* redraw grid */
			}
			break;
		case 5:
			interface_set_rec_editable(recwindow, TRUE);
			break;
	}
}

/* Callback for hardware D-BUS events */
void hw_event_handler(osso_hw_state_t *state, MainView *mainview) {
 	g_assert(mainview != NULL && mainview->data != NULL );

	if ((state->shutdown_ind)	/* system shutdown */
			|| (state->system_inactivity_ind)) {	/* system sleeping */
		close_file(mainview);
	}
	if (state->memory_low_ind) {	/* memory low... */
		unshow_rec(mainview);	/* close the rec window... changes not saved... */
	}
	if (state->save_unsaved_data_ind && mainview->file_edited) {	/* save unsaved data... */
		write_pwsafe_file(mainview);
/*		if (CONFRESP_YES==interface_save_changes_note(mainview)) {
		} */
	} 
}

/* application forced exit callback  */
//void exit_event_handler (gboolean die_now, MainView *mainview) {
gboolean exit_event_handler( GtkWidget *widget, GdkEvent  *event, MainView *mainview) {
	if (mainview && mainview->data) close_file(mainview);
	gtk_main_quit();
	return FALSE;
}


void maybe_save_file(MainView *mainview) {
/* save changes if file changed */
	if (mainview->pass && mainview->file_edited) {
		if (CONFRESP_YES==interface_yes_no_note(mainview, "Save Changes?")) {
			if (!mainview->file_name) {
				mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
			}
			write_pwsafe_file( mainview );
		}
	}
}

/* add entry */
void callback_add_entry ( GtkAction * action, MainView *mainview ) {
	rec* newrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	newrec=add_new_record("new", showing_group);
	populate_iconView(mainview, showing_group);
	show_rec(mainview, newrec);
}

/* change password */
void callback_change_password(GtkAction * action, MainView *mainview) {
	char *oldpass=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	oldpass=get_password(mainview, "Old Password");
	if (oldpass && mainview->pass && (strcmp(oldpass, mainview->pass)==0)) {
		char *newpass=get_confirmed_pass(mainview);
		if (newpass) mainview->pass=newpass;
	}
}

char *get_confirmed_pass(MainView *mainview) {
	char *result=NULL;
	char *confirm=NULL;
	result=get_password(mainview, "New Password");
	if (result==NULL) return NULL;
	confirm=get_password(mainview, "Confirm Password");
	if (confirm==NULL) {
		trashFreeMemory(result, -1);
		return NULL;
	}
	if (strcmp(result, confirm)==0) {
		trashFreeMemory(confirm, -1);
		return result;
	}
	trashFreeMemory(confirm, -1);
	trashFreeMemory(result, -1);
	msgbox(mainview, "Password Mismatch");
	return NULL;
}

/* new */
void callback_file_new(GtkAction * action, MainView *mainview) {
	char *newpass;
	g_assert(mainview != NULL && mainview->data != NULL );
	maybe_save_file(mainview);
	close_file(mainview);
	newpass=get_confirmed_pass(mainview);
	if (newpass) {
		mainview->pass=newpass;
		showing_group="";
		add_new_record("new", showing_group);
		populate_iconView(mainview, "");
		maybe_save_file(mainview);
	}
}

/* open */
void callback_file_open(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	maybe_save_file(mainview);
	mainview->file_name = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_OPEN );
	read_pwsafe_file(mainview);
}

void callback_file_open_last ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	maybe_save_file(mainview);
	mainview->file_name = mainview->last_file;
	read_pwsafe_file(mainview);
}

/* close file */
void callback_file_close ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
	maybe_save_file(mainview);
	close_file(mainview);
}

/* save */
void callback_file_save(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	write_pwsafe_file(mainview);
}


/* save as... */
void callback_file_saveas(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	mainview->file_name=NULL;
	write_pwsafe_file(mainview);
}


/* fullscreen */
void callback_fullscreen( GtkAction * action, gpointer data ) {
	interface_fullscreen((MainView *) data);
}

/* icon size */
void callback_large_icons( GtkAction * action, MainView *mainview ) {
	interface_set_icon_size(mainview, "largeicons");
}

/* icon size */
void callback_small_icons( GtkAction * action, MainView *mainview ) {
	interface_set_icon_size(mainview, "smallicons");
}
