/*
    Copyright 2008, Eion Robb <eionrobb@gmail.com>
    
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*

	Donate money to me at http://tinyurl.com/6crkya

*/


#define PURPLE_PLUGIN

#include <glib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <glib/gi18n.h>
#include <sys/types.h>

#ifndef G_GNUC_NULL_TERMINATED
#  if __GNUC__ >= 4
#    define G_GNUC_NULL_TERMINATED __attribute__((__sentinel__))
#  else
#    define G_GNUC_NULL_TERMINATED
#  endif /* __GNUC__ >= 4 */
#endif /* G_GNUC_NULL_TERMINATED */

#include <proxy.h>
#include <sslconn.h>
#include <prpl.h>
#include <debug.h>
#include <connection.h>
#include <request.h>
#include <dnsquery.h>
#include <accountopt.h>

#ifdef _WIN32
#	include <win32dep.h>
#else
#	include <arpa/inet.h>
#	include <sys/socket.h>
#	include <netinet/in.h>
#endif

#define LAST_MESSAGE_MAX 10

typedef struct _FacebookAccount {
	PurpleAccount *account;
	PurpleConnection *gc;
	gchar *login_challenge;
	GHashTable *cookie_table;
	gchar *post_form_id;
	gint32 uid;
	guint buddy_list_timeout;
	guint friend_request_timeout;
	gchar *channel_number;
	guint message_fetch_sequence;
	gint64 last_messages[LAST_MESSAGE_MAX];
	guint16 next_message_pointer;
	GSList *auth_buddies;
	GHashTable *hostname_ip_cache;
	PurpleConnectionState state;
	guint notifications_timeout;
	time_t last_messages_download_time;
	guint new_messages_check_timeout;
	gchar *last_status_message;
} FacebookAccount;

typedef void (*FacebookProxyCallbackFunc)(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data);

typedef struct _FacebookProxyData {
	FacebookAccount *fba;
	char *data;
	FacebookProxyCallbackFunc callback;
	gpointer user_data;
	GString *response;
	PurpleProxyConnectData *purple_proxy_connect_data;
	guint input_timeout;
	gboolean connection_keepalive;
} FacebookProxyData;

typedef struct _FacebookBuddy {
	FacebookAccount *fba;
	PurpleBuddy *buddy;
	gint32 uid;
	gchar *name;
	gchar *status;
	gchar *status_rel_time;
	gchar *thumb_url;
} FacebookBuddy;

typedef struct _FacebookOutgoingMessage {
	FacebookAccount *fba;
	gchar *who;
	time_t time;
	gchar *message;
	gint msg_id;
	guint retry_count;
} FacebookOutgoingMessage;

const char *facebookim_list_icon(PurpleAccount *account, PurpleBuddy *buddy);
void facebookim_post_or_get_readdata_cb(gpointer data, gint source, PurpleInputCondition cond);
void facebookim_post_or_get_connect_cb(gpointer data, gint source, const gchar *error_message);
void buddy_icon_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer buddy_data);
void set_buddies_offline(PurpleBuddy *buddy, GSList *online_buddies_list);
void got_buddy_list_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata);
void got_new_messages(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata);
void got_form_id_page(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata);
gboolean facebookim_get_buddy_list(PurpleAccount *account);
gboolean facebookim_get_new_messages(FacebookAccount *fba);
void facebookim_fetch_login_cb(FacebookAccount *fba, gchar *url_text, gsize data_len, gpointer userdata);
void facebookim_login(PurpleAccount *acct);
void facebookim_close(PurpleConnection *gc);
void facebookim_send_im_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data);
int facebookim_send_im(PurpleConnection *gc, const gchar *who, const gchar *message, PurpleMessageFlags flags);
GList *facebookim_statuses(PurpleAccount *acct);
gchar *facebookim_status_text(PurpleBuddy *buddy);
void facebookim_tooltip_text(PurpleBuddy *buddy, PurpleNotifyUserInfo *userinfo, gboolean full);
gboolean plugin_load(PurplePlugin *plugin);
gboolean plugin_unload(PurplePlugin *plugin);
void facebookim_keepalive(PurpleConnection *gc);
void facebookim_buddy_free(PurpleBuddy *buddy);
void facebookim_keepalive_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data);
unsigned int facebookim_send_typing(PurpleConnection *gc, const gchar *name, PurpleTypingState state);
gboolean facebookim_check_friend_requests(FacebookAccount *fba);
void facebookim_check_friend_request_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data);
void facebookim_auth_reject(gpointer data);
void facebookim_auth_accept(gpointer data);
GList *facebookim_actions(PurplePlugin *plugin, gpointer context);
void facebook_update_cookies(FacebookAccount *fba, const gchar *headers);
gchar *facebook_cookies_to_string(FacebookAccount *fba);
void facebookim_get_info(PurpleConnection *gc, const gchar *uid);
void facebookim_get_info_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data);
void facebookim_set_status_p(PurpleAccount *account, PurpleStatus *status);
gboolean facebookim_get_notifications_feed(FacebookAccount *fba);
gboolean facebookim_send_im_fom(FacebookOutgoingMessage *msg);
void facebookim_get_post_form_id(FacebookAccount *fba);
GHashTable *facebookim_get_account_text_table(PurpleAccount *account);

/******************************************************************************/


const char *
facebookim_list_icon(PurpleAccount *account, PurpleBuddy *buddy)
{
	return "facebook";
}

gchar *
facebook_convert_unicode(const gchar *input)
{
	// \u00e9t\u00e9 should be été
	
	gunichar unicode_char;
	gchar unicode_char_str[6];
	gint unicode_char_len;
	gchar *next_pos;
	gchar *input_string;
	gchar *output_string;
	
	if (input == NULL)
		return NULL;
	
	next_pos = input_string = g_strdup(input);
	
	//purple_debug_info("facebook", "unicode convert: in: %s\n", input);
	while((next_pos = strstr(next_pos, "\\u")))
	{
		//grab the unicode
		sscanf(next_pos, "\\u%4x", &unicode_char);
		//turn it to a char*
		unicode_char_len = g_unichar_to_utf8(unicode_char, unicode_char_str);
		//shove it back into the string
		g_memmove(next_pos, unicode_char_str, unicode_char_len);
		//move all the data after the \u0000 along
		g_stpcpy(next_pos+unicode_char_len, next_pos+6);
	}
	//purple_debug_info("facebook", "unicode convert: out: %s\n", input);
	output_string = g_strcompress(input_string);
	g_free(input_string);
	return output_string;
}

/* Like purple_strdup_withhtml, but escapes htmlentities too */
gchar *
facebook_strdup_withhtml(const gchar *src)
{
	gulong destsize, i, j;
	gchar *dest;

	g_return_val_if_fail(src != NULL, NULL);

	/* New length is (length of src) + (number of \n's * 3) + (number of &'s * 5) + (number of <'s * 4) + (number of >'s *4) + (number of "'s * 6) - (number of \r's) + 1 */
	destsize = 1;
	for (i = 0; src[i] != '\0'; i++)
	{
		if (src[i] == '\n' || src[i] == '<' || src[i] == '>')
			destsize += 4;
		else if (src[i] == '&')
			destsize += 5;
		else if (src[i] == '"')
			destsize += 6;
		else if (src[i] != '\r')
			destsize++;
	}

	dest = g_malloc(destsize);

	/* Copy stuff, ignoring \r's, because they are dumb */
	for (i = 0, j = 0; src[i] != '\0'; i++) {
		if (src[i] == '\n') {
			strcpy(&dest[j], "<BR>");
			j += 4;
		} else if (src[i] == '<') {
			strcpy(&dest[j], "&lt;");
			j += 4;
		} else if (src[i] == '>') {
			strcpy(&dest[j], "&gt;");
			j += 4;
		} else if (src[i] == '&') {
			strcpy(&dest[j], "&amp;");
			j += 5;
		} else if (src[i] == '"') {
			strcpy(&dest[j], "&quot;");
			j += 6;
		} else if (src[i] != '\r')
			dest[j++] = src[i];
	}

	dest[destsize-1] = '\0';

	return dest;
}

struct _PurpleProxyConnectData{void *handle;};

void
facebookim_post_or_get_readdata_cb(gpointer data, gint source, PurpleInputCondition cond)
{
	FacebookProxyData *proxy_data = data;
	PurpleConnection *gc = NULL;
	struct _PurpleProxyConnectData *tempdata;
	gchar response[3096];
	gchar *temp;
	int len;
	
	len = read(source, response, sizeof(response)-1);
	if (len < 0)
	{
		//There's an error of some kind
		if (errno != EAGAIN && errno != EWOULDBLOCK)
		{
			close(source);
			purple_input_remove(proxy_data->input_timeout);
			if (proxy_data->response)
				g_string_free(proxy_data->response, TRUE);
			g_free(proxy_data);
		}
		return;
	}

	if (proxy_data && proxy_data->purple_proxy_connect_data)
	{
		tempdata = proxy_data->purple_proxy_connect_data;
		gc = tempdata->handle;
	}

	if (len != 0)
	{
		if (proxy_data->response == NULL)
		{
			proxy_data->response = g_string_new_len(response, len);
		} else
		{
			proxy_data->response = g_string_append_len(proxy_data->response, response, len);
		}
	}
	else
	//if (len == 0)
	{
		if (errno)
		{
			purple_debug_info("facebook", "Errno: %d\n", errno);
		}
		if (!proxy_data->connection_keepalive || errno)
		{
			//close the connection if we're not using it any more
			proxy_data->connection_keepalive = FALSE;
			close(source);
			purple_input_remove(proxy_data->input_timeout);
		}
		if (proxy_data->response != NULL)
		{
			len = proxy_data->response->len;
			temp = g_strstr_len(proxy_data->response->str, len, "\r\n\r\n");
			if (temp == NULL)
			{
				temp = g_strndup(proxy_data->response->str, len);
			} else {
				temp += 4;
				len -= temp - proxy_data->response->str;
				temp = g_memdup(temp, len+1);
				temp[len] = '\0';
				proxy_data->response->str[proxy_data->response->len - len] = '\0';
				purple_debug_info("facebook", "response headers %s\n", proxy_data->response->str);
				//if (gc && PURPLE_CONNECTION_IS_CONNECTED(gc))
					facebook_update_cookies(proxy_data->fba, proxy_data->response->str);
				if (strstr(proxy_data->response->str, "Location: http://www.new.facebook.com/"))
				{
					//this person is using the new layout and has their host set to 'old' facebook
					if (proxy_data->fba && proxy_data->fba->account)
						purple_account_set_string(proxy_data->fba->account, "host", "www.new.facebook.com");
				}
			}
			g_string_free(proxy_data->response, TRUE);
		} else {
			temp = g_strdup("");
			len = 0;
		}
		if (proxy_data->callback != NULL && len)
			proxy_data->callback(proxy_data->fba, temp, len, proxy_data->user_data);
		proxy_data->response = NULL;
		if (!proxy_data->connection_keepalive)
			g_free(proxy_data);
		g_free(temp);
#ifdef _WIN32
		//put a pause in here to fix sometimes-high cpu usage
		Sleep(1);
#endif
	}
}

void
facebookim_post_or_get_connect_cb(gpointer data, gint source, const gchar *error_message)
{
	FacebookProxyData *proxy_data = data;
	int len;
	
	if (error_message)
	{
		purple_debug_error("facebook", "post_or_get_connect_cb %s\n", error_message);
		if (proxy_data->data)
			g_free(proxy_data->data);
		close(source);
		purple_input_remove(proxy_data->input_timeout);
		g_free(proxy_data);
		return;
	} else {
		purple_debug_info("facebook", "post_or_get_connect_cb\n");
	}
	len = write(source, proxy_data->data, strlen(proxy_data->data));
	g_free(proxy_data->data);
	proxy_data->data = NULL;

	proxy_data->input_timeout = purple_input_add(source, PURPLE_INPUT_READ, facebookim_post_or_get_readdata_cb, data);
}

void
facebook_host_lookup_cb(GSList *hosts, gpointer data, const char *error_message)
{
	GSList *host_lookup_list = data;
	struct sockaddr_in *sin;
	gchar *hostname;
	gchar *ip_address;
	FacebookAccount *fba = host_lookup_list->data;
	
	if (error_message && strlen(error_message))
	{
		purple_debug_info("facebook", "host lookup error: %s\n", error_message);
		return;
	}
	
	hosts = g_slist_remove(hosts, hosts->data);
	sin = hosts->data;
	hosts = g_slist_remove(hosts, hosts->data);
	
	host_lookup_list = g_slist_remove(host_lookup_list, fba);
	hostname = host_lookup_list->data;
	host_lookup_list = g_slist_remove(host_lookup_list, hostname);
	
	ip_address = g_strdup(inet_ntoa(sin->sin_addr));
	
	if (fba && fba->account && !fba->account->disconnecting)
	{
		purple_debug_info("facebook", "Host %s has IP %s\n", hostname, ip_address);
		
		if (fba->hostname_ip_cache)
			g_hash_table_insert(fba->hostname_ip_cache, hostname, ip_address);
	} else {
		g_free(hostname);
		g_free(ip_address);
	}
}

static void
facebookim_post(FacebookAccount *fba, const gchar *host, const gchar *url, gchar *postdata, FacebookProxyCallbackFunc callback_func, gpointer user_data, gboolean keepalive)
{
	gchar *httpdata;
	gchar *cookies;
	FacebookProxyData *data;
	gchar *host_ip;
	GSList *host_lookup_list = NULL;
	gchar *real_url;
	gboolean is_proxy = FALSE;
	
	if (host == NULL)
	{
		if (!fba || !fba->account)
			host = "www.facebook.com";
		else
			host = purple_account_get_string(fba->account, "host", "www.facebook.com");
	}

	if (fba && fba->account && fba->account->proxy_info && 
		(fba->account->proxy_info->type == PURPLE_PROXY_HTTP ||
		(fba->account->proxy_info->type == PURPLE_PROXY_USE_GLOBAL &&
			purple_global_proxy_get_info() &&
			purple_global_proxy_get_info()->type == PURPLE_PROXY_HTTP)))
	{
		real_url = g_strdup_printf("http://%s%s", host, url);
		is_proxy = TRUE;
	} else {
		real_url = g_strdup(url);
	}
	
	cookies = facebook_cookies_to_string(fba);
	httpdata = g_strdup_printf("POST %s HTTP/1.0\r\n"
							"Host: %s\r\n"
							"Connection: %s\r\n"
							"Content-Type: application/x-www-form-urlencoded\r\n"
							"User-Agent: Opera/9.50 (Windows NT 5.1; U; en-GB)\r\n"
							"Content-length: %d\r\n"
							"Accept: */*\r\n"
							"Cookie: isfbe=false;%s\r\n"
							"\r\n%s",
							real_url, host, (keepalive?"Keep-Alive":"close"), ((int)strlen(postdata)), cookies, postdata);
	g_free(cookies);
	g_free(real_url);
	
	data = g_new(FacebookProxyData, 1);
	data->data = httpdata;
	data->fba = fba;
	data->callback = callback_func;
	data->user_data = user_data;
	data->response = NULL;
	data->connection_keepalive = keepalive;

	purple_debug_info("facebookim", "%s\n", httpdata);

	if (!is_proxy)
	{
		//only cache dns for non-proxy connections, since dns should be handled by proxy
		if ((host_ip = g_hash_table_lookup(fba->hostname_ip_cache, host)) != NULL)
		{
			host = host_ip;
		} else if (fba->account && !fba->account->disconnecting) {
			host_lookup_list = g_slist_prepend(host_lookup_list, g_strdup(host));
			host_lookup_list = g_slist_prepend(host_lookup_list, fba);
			purple_dnsquery_a(host, 80, facebook_host_lookup_cb, host_lookup_list);
		}
	}
	
	data->purple_proxy_connect_data = purple_proxy_connect(fba->gc, fba->account, host, 80, facebookim_post_or_get_connect_cb, data);
}

static void
facebookim_get(FacebookAccount *fba, const gchar *host, const gchar *url, FacebookProxyCallbackFunc callback_func, gpointer user_data, gboolean keepalive)
{
	gchar *httpdata;
	FacebookProxyData *data;
	gchar *cookies;
	gchar *host_ip;
	GSList *host_lookup_list = NULL;
	gchar *real_url;
	gboolean is_proxy = FALSE;
	
	if (host == NULL)
	{
		if (!fba || !fba->account)
			host = "www.facebook.com";
		else
			host = purple_account_get_string(fba->account, "host", "www.facebook.com");
	}
	
	if (fba && fba->account && fba->account->proxy_info && 
		(fba->account->proxy_info->type == PURPLE_PROXY_HTTP ||
		(fba->account->proxy_info->type == PURPLE_PROXY_USE_GLOBAL &&
			purple_global_proxy_get_info() &&
			purple_global_proxy_get_info()->type == PURPLE_PROXY_HTTP)))
	{
		real_url = g_strdup_printf("http://%s%s", host, url);
		is_proxy = TRUE;
	} else {
		real_url = g_strdup(url);
	}
	
	cookies = facebook_cookies_to_string(fba);
	httpdata = g_strdup_printf("GET %s HTTP/1.0\r\n"
							"Host: %s\r\n"
							"Connection: %s\r\n"
							"User-Agent: Opera/9.50 (Windows NT 5.1; U; en-GB)\r\n"
							"Accept: */*\r\n"
							"Cookie: isfbe=false;%s\r\n"
							"\r\n",
							real_url, host, (keepalive?"Keep-Alive":"close"), cookies);
	
	g_free(cookies);
	g_free(real_url);
	
	data = g_new(FacebookProxyData, 1);
	data->data = httpdata;
	data->fba = fba;
	data->callback = callback_func;
	data->user_data = user_data;
	data->response = NULL;
	data->connection_keepalive = keepalive;

	purple_debug_info("facebookim", "%s\n", httpdata);

	if (!is_proxy)
	{
		//only cache dns for non-proxy connections, since dns should be handled by proxy
		if ((host_ip = g_hash_table_lookup(fba->hostname_ip_cache, host)) != NULL)
		{
			host = host_ip;
		} else if (fba->account && !fba->account->disconnecting) {
			host_lookup_list = g_slist_prepend(host_lookup_list, g_strdup(host));
			host_lookup_list = g_slist_prepend(host_lookup_list, fba);
			purple_dnsquery_a(host, 80, facebook_host_lookup_cb, host_lookup_list);
		}
	}
	
	data->purple_proxy_connect_data = purple_proxy_connect(fba->gc, fba->account, host, 80, facebookim_post_or_get_connect_cb, data);
}

void
buddy_icon_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer buddy_pointer)
{
	PurpleBuddy *buddy = buddy_pointer;
	gpointer buddy_icon_data;
	
	purple_debug_info("facebook", "buddy icon for buddy %s %" G_GSIZE_FORMAT "\n", buddy->name, data_len);
	//purple_debug_info("facebook", "%s\n", data);

	buddy_icon_data = g_memdup(data, data_len);

	purple_buddy_icons_set_for_user(fba->account, buddy->name, buddy_icon_data, data_len, NULL);
}

void
set_buddies_offline(PurpleBuddy *buddy, GSList *online_buddies_list)
{
	if (g_slist_find(online_buddies_list, buddy) == NULL && PURPLE_BUDDY_IS_ONLINE(buddy))
	{
		purple_prpl_got_user_status(buddy->account, buddy->name, purple_primitive_get_id_from_type(PURPLE_STATUS_OFFLINE), NULL);
	}
}

void
got_buddy_list_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata)
{
	GSList *buddies_list;
	GSList *online_buddies_list = NULL;
	PurpleBuddy *buddy;
	FacebookBuddy *fbuddy;
	gchar *uid;
	gchar *name;
	gchar *status_text;
	gchar *status_time_text;
	gchar *buddy_icon_url;
	guint32 error_number;
	
	gchar *search_start;
	gchar *search_temp;
	gchar *temp;
	gchar *largest_buddy_search_point = NULL;

	PurpleGroup *facebook_group = NULL;

	purple_debug_info("facebook", "buddy list %s\n", data);

	if (fba == NULL)
		return;
		
	//Check if the facebook group already exists (fixes #13)
	facebook_group = purple_find_group("Facebook");

	//if logged out, this comes up
	// for (;;);{"error":1357001,"errorSummary":"Not Logged In","errorDescription":"You must be logged in to do that.","payload":null,"bootload":[{"name":"js\/common.js.pkg.php","type":"js","src":"http:\/\/static.ak.fbcdn.net\/rsrc.php\/pkg\/59\/98561\/js\/common.js.pkg.php"}]}
	temp = g_strstr_len(data, data_len, "\"error\":");
	if (temp != NULL)
	{
		temp += 9;
		temp = g_strndup(temp, strchr(temp, ',')-temp);
		error_number = atoi(temp);
		g_free(temp);
		if (error_number)
		{
			//error :(
			temp = g_strstr_len(data, data_len, "\"errorDescription\":");
			temp += 20;
			temp = g_strndup(temp, strchr(temp, '"')-temp);
			purple_connection_error(fba->gc, temp);
			g_free(temp);
			return;
		}
	}

	//look for "userInfos":{ ... },
	search_start = strstr(data, "\"userInfos\":{");
	if (search_start == NULL)
		return;
	search_start += 13;

	while(*search_start != '}' && (search_start - data < data_len))
	{
		temp = strchr(search_start, ':');
		uid = g_strndup(search_start+1, temp-search_start-2);
		purple_debug_info("facebook", "uid: %s\n", uid);

		search_start += strlen(uid) + 2;

		search_temp = strstr(search_start, "\"name\":") + 8;
		if (search_temp > largest_buddy_search_point)
			largest_buddy_search_point = search_temp;
		search_temp = g_strndup(search_temp, strchr(search_temp, '"')-search_temp);
		name = facebook_convert_unicode(search_temp);
		g_free(search_temp);
		purple_debug_info("facebookim", "name: %s\n", name);
		
		//try updating the alias, just in case it was removed locally
		serv_got_alias(fba->gc, uid, name);
		
		//look for "uid":{"i":_____}
		temp = g_strdup_printf("\"%s\":{\"i\":", uid);
		search_temp = g_strstr_len(data, data_len, temp);
		if (search_temp != NULL)
		{
			search_temp += strlen(temp);	
			if (search_temp > largest_buddy_search_point)
				largest_buddy_search_point = search_temp;
			search_temp = g_strndup(search_temp, strchr(search_temp, '}')-search_temp);
			purple_debug_info("facebook", "buddy idle: %s\n", search_temp);
			buddy = purple_find_buddy(fba->account, uid);
			if (g_str_equal(search_temp, "true"))
			{
				purple_prpl_got_user_idle(fba->account, uid, TRUE, -1);
				if (buddy && !purple_presence_is_idle(purple_buddy_get_presence(buddy)))
					purple_prpl_got_user_status(fba->account, uid, purple_primitive_get_id_from_type(PURPLE_STATUS_AVAILABLE), NULL);
			} else {
				purple_prpl_got_user_idle(fba->account, uid, FALSE, 0);
				if (buddy && purple_presence_is_idle(purple_buddy_get_presence(buddy)))
					purple_prpl_got_user_status(fba->account, uid, purple_primitive_get_id_from_type(PURPLE_STATUS_AVAILABLE), NULL);
			}
			g_free(search_temp);
		}
		g_free(temp);

		//Set the buddy status text and time
		search_temp = strstr(search_start, "\"status\":") + 10;
		if ( *( search_temp - 1 ) == '"' )
		{
			if (search_temp > largest_buddy_search_point)
				largest_buddy_search_point = strstr(search_temp, ",\"statusTime");
			search_temp = g_strndup(search_temp, strstr(search_temp, ",\"statusTime")-1-search_temp);
			status_text = facebook_convert_unicode(search_temp);
			g_free(search_temp);
		} else {
			status_text = NULL;
		}
		
		//is this us?
		if (atoi(uid) == fba->uid)
		{
			//has the account alias been set?
			if (!purple_account_get_alias(fba->account))
				purple_account_set_alias(fba->account, name);
			
			//set our last known status so that we don't re-set it
			if (status_text && !fba->last_status_message)
				fba->last_status_message = g_strdup(status_text);
			
			//check that we don't want to show ourselves
			if (purple_account_get_bool(fba->account, "facebook_hide_self", TRUE))
			{
				g_free(status_text);
				g_free(name);
				g_free(uid);
			
				//Move pointer to the end of the buddy entry
				search_start = strchr(largest_buddy_search_point, '}') + 1;
				while (*search_start == ',' && (search_start - data < data_len))
					search_start++;
				//go on to the next buddy
				continue;
			}
		}
		
		//Is this a new buddy?
		if (!(buddy = purple_find_buddy(fba->account, uid)))
		{
			buddy = purple_buddy_new(fba->account, uid, name);
			if (facebook_group == NULL)
			{
				facebook_group = purple_group_new("Facebook");
				purple_blist_add_group(facebook_group, NULL);
			}
			purple_blist_add_buddy(buddy, NULL, facebook_group, NULL);
		}
		
		//Set the FacebookBuddy structure
		if (buddy->proto_data == NULL)
		{
			fbuddy = g_new0(FacebookBuddy, 1);
			fbuddy->buddy = buddy;
			fbuddy->fba = fba;
			fbuddy->uid = atoi(uid);
			fbuddy->name = g_strdup(name);
			buddy->proto_data = fbuddy;
		} else {
			fbuddy = buddy->proto_data;
		}
		
		g_free(uid);
		g_free(name);
		
		if (status_text != NULL)
		{
			temp = facebook_strdup_withhtml(status_text);
			g_free(status_text);
			status_text = temp;
			purple_debug_info("facebook", "status: %s\n", status_text);
		
			search_temp = strstr(search_start, "\"statusTimeRel\":") + 17;
			if (search_temp > largest_buddy_search_point)
				largest_buddy_search_point = strchr(search_temp, '"');
			search_temp = g_strndup(search_temp, strchr(search_temp, '"')-search_temp);
			status_time_text = facebook_convert_unicode(search_temp);
			g_free(search_temp);
			
			if (g_str_equal(status_time_text, "ull,"))
			{
				g_free(status_time_text);
				status_time_text = NULL;
			}
			if (fbuddy->status_rel_time != NULL)
				g_free(fbuddy->status_rel_time);
			if (status_time_text != NULL)
			{
				fbuddy->status_rel_time = facebook_strdup_withhtml(status_time_text);
				g_free(status_time_text);
				purple_debug_info("facebook", "status time: %s\n", fbuddy->status_rel_time);
			} else {
				fbuddy->status_rel_time = NULL;
			}
			
			//if the buddy status has changed, update the contact list
			if (fbuddy->status == NULL || !g_str_equal(fbuddy->status, status_text))
			{
				temp = fbuddy->status;
				fbuddy->status = status_text;
				if (temp != NULL)
					g_free(temp);
				purple_prpl_got_user_status(fba->account, buddy->name, purple_primitive_get_id_from_type(PURPLE_STATUS_AVAILABLE), NULL);
			} else {
				g_free(status_text);
			}
		} else {
			if (fbuddy->status != NULL)
			{
				g_free(fbuddy->status);
				fbuddy->status = NULL;
				//update the status in the contact list
				purple_prpl_got_user_status(fba->account, buddy->name, purple_primitive_get_id_from_type(PURPLE_STATUS_AVAILABLE), NULL);
			}
		}
		
		//Set the buddy icon (if it hasn't changed)
		search_temp = strstr(search_start, "\"thumbSrc\":") + 12;
		if (search_temp > largest_buddy_search_point)
			largest_buddy_search_point = search_temp;
		buddy_icon_url = g_strndup(search_temp, strchr(search_temp, '"')-search_temp);
		if (fbuddy->thumb_url == NULL || !g_str_equal(fbuddy->thumb_url, buddy_icon_url))
		{
			if (fbuddy->thumb_url != NULL)
				g_free(fbuddy->thumb_url);
			fbuddy->thumb_url = g_strdup(buddy_icon_url);
			temp = g_strcompress(buddy_icon_url);
			
			//small icon at  http://profile.ak.facebook.com/profile6/1845/74/q800753867_2878.jpg
			//bigger icon at http://profile.ak.facebook.com/profile6/1845/74/n800753867_2878.jpg
			search_temp = strstr(temp, "/q");
			if (search_temp) *(search_temp+1)='n';
			purple_debug_info("facebook", "buddy_icon_url: %s\n", temp);
			
			facebookim_get(fba, "profile.ak.facebook.com", temp+strlen("http://profile.ak.facebook.com"), buddy_icon_cb, buddy, FALSE);
			g_free(temp);
		}
		g_free(buddy_icon_url);

		//Add buddy to the list of online buddies
		online_buddies_list = g_slist_append(online_buddies_list, buddy);
		
		//Update the display of the buddy in the buddy list and make the user online
		if (!PURPLE_BUDDY_IS_ONLINE(buddy))
			purple_prpl_got_user_status(fba->account, buddy->name, purple_primitive_get_id_from_type(PURPLE_STATUS_AVAILABLE), NULL);


		//Move pointer after any user configurable data
		search_start = search_temp;
		//Move pointer to the end of the buddy entry
		search_start = strchr(largest_buddy_search_point, '}') + 1;
		while (*search_start == ',' && (search_start - data < data_len))
			search_start++;
	}
	
	buddies_list = purple_find_buddies(fba->account, NULL);
	if (buddies_list != NULL)
	{
		g_slist_foreach(buddies_list, (GFunc)set_buddies_offline, online_buddies_list); 
		g_slist_free(buddies_list);
	}
	g_slist_free(online_buddies_list);
}

gboolean
facebookim_get_buddy_list(PurpleAccount *account)
{
	FacebookAccount *fba;
	gchar *postdata;
	
	if (account == NULL)
		return FALSE;
	
	fba = account->gc->proto_data;
	
	postdata = g_strdup_printf("user=%d&popped_out=false&force_render=true&buddy_list=1", fba->uid);
	facebookim_post(fba, NULL, "/ajax/presence/update.php", postdata, got_buddy_list_cb, NULL, FALSE);
	g_free(postdata);
	
	return TRUE;
}

void
got_new_messages(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata)
{
	gchar *message_text;
	gchar *message_time;
	gchar *from;
	gchar *to;
	gchar *temp;
	gchar *type;
	gchar *start;
	gchar *end;
	gint64 msgID;
	int i;
	PurpleConnection *gc = userdata;

	purple_debug_info("facebook", "got new messages: %s\n", data);
	
	
	//purple_debug_info("facebook", "fba: %d\n", fba);
	//purple_debug_info("facebook", "account: %d\n", fba->account);
	if (!PURPLE_CONNECTION_IS_CONNECTED(gc))
		return;
	
	fba->last_messages_download_time = time(NULL);
	
	//for (;;);{"t":"msg","c":"p_800753867","ms":[{"type":"msg","msg":{"text":"yes","time":1211176515861,"clientTime":1211176514750,"msgID":"367146364"},"from":596176850,"to":800753867,"from_name":"Jeremy Lawson","to_name":"Eion Robb","from_first_name":"Jeremy","to_first_name":"Eion"}]}
	//for (;;);{"t":"refresh"}
	//for (;;);{"t":"msg","c":"p_800753867","ms":[{"type":"msg","msg":{"text":"porn head                                ","time":1211177326689,"clientTime":1211177325,"msgID":"-1992480367"},"from":800753867,"to":596176850,"from_name":"Eion Robb","to_name":"Jeremy Lawson","from_first_name":"Eion","to_first_name":"Jeremy"}]}
//for (;;);{"t":"msg","c":"p_800753867","ms":[{"type":"typ","st":1,"from":596176850,"to":800753867},{"type":"msg","msg":{"text":"nubile!","time":1211177334019,"clientTime":1211177326690,"msgID":"696260545"},"from":596176850,"to":800753867,"from_name":"Jeremy Lawson","to_name":"Eion Robb","from_first_name":"Jeremy","to_first_name":"Eion"},{"type":"msg","msg":{"text":"test2","time":1211177336688,"clientTime":1211177326691,"msgID":"1527815367"},"from":596176850,"to":800753867,"from_name":"Jeremy Lawson","to_name":"Eion Robb","from_first_name":"Jeremy","to_first_name":"Eion"},{"type":"msg","msg":{"text":"ahhhhhhh!","time":1211177344361,"clientTime":1211177326692,"msgID":"4028916254"},"from":596176850,"to":800753867,"from_name":"Jeremy Lawson","to_name":"Eion Robb","from_first_name":"Jeremy","to_first_name":"Eion"}]}
	//for (;;);{"t":"msg","c":"p_800753867","ms":[{"type":"msg","msg":{"text":"2","time":1211178167261,"clientTime":1211178164656,"msgID":"3382240259"},"from":596176850,"to":800753867,"from_name":"Jeremy Lawson","to_name":"Eion Robb","from_first_name":"Jeremy","to_first_name":"Eion"}]}
	//for (;;);{"t":"refresh", "seq":1}	
	
	//look for the start of the JSON, and ignore any proxy headers
	data = g_strstr_len(data, data_len, "for (;;);");
	if (!data)
	{
		//something weird happened and we didn't get back real information
		return;
	}
	
	//refresh means that the session or post_form_id is invalid
	if (g_str_equal(data, "for (;;);{\"t\":\"refresh\"}"))
	{
		facebookim_get_post_form_id(fba);
		facebookim_get_new_messages(fba);
		return;
	}
	
	//continue means that the server wants us to remake the connection
	if (g_str_equal(data, "for (;;);{\"t\":\"continue\"}"))
	{
		facebookim_get_new_messages(fba);
		return;
	}
	
	temp = strstr(data, "\"seq\":");
	if (temp != NULL)
	{
		temp = temp + 6;
		temp = g_strndup(temp, strchr(temp, '}')-temp);
		purple_debug_info("facebook", "new seq number: %s\n", temp);
		fba->message_fetch_sequence = atoi(temp);
		g_free(temp);
	} else {
		fba->message_fetch_sequence++;
	}
	
	if (strncmp(data, "for (;;);{\"t\":\"msg\"", 19) == 0)
	{
		start = g_strstr_len(data, data_len, "\"ms\":[");
		if (!start)
			return;
		start += 6;
		while (*start != ']')
		{
			temp = strstr(start, "\"type\":\"");
			if (temp)
			{
				temp += 8;
				type = g_strndup(temp, strchr(temp, '"')-temp);
				purple_debug_info("facebook","type: %s\n", type);
			} else {
				type = g_strdup("unknown");
			}
			
			temp = strstr(start, "\"from\":");
			if (temp)
			{
				temp += 7;
				from = g_strndup(temp, strchr(temp, ',')-temp);
				if (from[0] == '"')
					snprintf(from, strlen(from), "%d", atoi(from+1));
				purple_debug_info("facebook","from: %s\n", from);
			} else {
				from = NULL;
			}
			temp = strstr(start, "\"to\":");
			if (temp)
			{
				temp += 5;
				end = strchr(temp, ',');
				if (end == NULL || strchr(temp, '}') < end)
					end = strchr(temp, '}');
				to = g_strndup(temp, end-temp);
				if (to[0] == '"')
					snprintf(to, strlen(to), "%d", atoi(to+1));
				purple_debug_info("facebook","to: %s\n", to);
			} else {
				to = NULL;
			}
			
			if (from && to && g_str_equal(type, "msg"))
			{
				//IM message
				if (fba->uid != atoi(from) || fba->uid == atoi(to))
				{
					temp = strstr(start, "\"msgID\":");
					temp += 9;
					temp = g_strndup(temp, strchr(temp, '"')-temp);
					msgID = atoll(temp);
					purple_debug_info("facebook", "message id: %s %" G_GINT64_FORMAT " %lld\n", temp, msgID, atoll(temp));
					g_free(temp);
					//loop through all the previous messages that we have stored
					//to make sure that we havn't already received this message
					for(i = 0; i < LAST_MESSAGE_MAX; i++)
					{
						purple_debug_info("facebook", "last_messages[%d] = %" G_GINT64_FORMAT "\n", i, fba->last_messages[i]);
						if (fba->last_messages[i] == msgID)
							break;
					}
					purple_debug_info("facebook", "i: %d\n", i);
					if (i == LAST_MESSAGE_MAX)
					{
						//if we're here, it must be a new message
						fba->last_messages[fba->next_message_pointer++] = msgID;
						if (fba->next_message_pointer >= LAST_MESSAGE_MAX)
							fba->next_message_pointer = 0;
						
						temp = strstr(start, "\"text\":\"");
						temp += 8;
						temp = g_strndup(temp, strstr(temp, "\",\"time\":")-temp);
						message_text = facebook_convert_unicode(temp);
						g_free(temp);
						temp = facebook_strdup_withhtml(message_text);
						g_free(message_text);
						message_text = temp;
						
						purple_debug_info("facebook", "text: %s\n", message_text);
						temp = strstr(start, "\"time\":");
						temp += 7;
						message_time = g_strndup(temp, strchr(temp, ',')-temp-3);
						purple_debug_info("facebook", "time: %s\n", message_time);
						
						serv_got_im(gc, from, message_text, PURPLE_MESSAGE_RECV, atoi(message_time));
						
						g_free(message_text);
						g_free(message_time);
					}
				}
				start = strchr(start, '}')+1;
			} else if (from && g_str_equal(type, "typ"))
			{
				//typing notification
				temp = strstr(start, "\"st\":");
				if (temp != NULL)
				{
					temp += 5;
					if (*temp == '0')
						serv_got_typing(gc, from, 10, PURPLE_TYPED);
					else
						serv_got_typing(gc, from, 10, PURPLE_TYPING);
				}	
			}
			
			//we've received something from a buddy, assume they're online
			//only if it's not from ourselves
			if (from && fba->uid != atoi(from))
				purple_prpl_got_user_status(fba->account, from, purple_primitive_get_id_from_type(PURPLE_STATUS_AVAILABLE), NULL);
			
			if (from != NULL)
				g_free(from);
			if (to != NULL)
				g_free(to);
			g_free(type);
			
			start = strchr(start, '}')+1;
			while (*start == ',')
				start++;
		}
	}
	
	facebookim_get_new_messages(fba);
}

gboolean
facebookim_get_new_messages(FacebookAccount *fba)
{
	gchar *fetch_url;
	gchar *fetch_server;
	
	if (fba == NULL)
		return FALSE;
	purple_debug_info("facebook", "getting new messages\n");
	
	//use the current time in the url to get past any transparent proxy caches
	fetch_url = g_strdup_printf("/x/%d/false/p_%d=%d", ((int)time(NULL)), fba->uid, fba->message_fetch_sequence);
	fetch_server = g_strdup_printf("%d.channel%s.facebook.com", 0, fba->channel_number);
	
	facebookim_get(fba, fetch_server, fetch_url, got_new_messages, fba->gc, TRUE);
	
	g_free(fetch_url);
	g_free(fetch_server);
	
	return TRUE;
}

void
got_form_id_page(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata)
{
	gchar *start_text = "id=\"post_form_id\" name=\"post_form_id\" value=\"";
	gchar *post_form_id;
	gchar *channel_number;
	gchar *temp;
	gpointer freepoint;
	
	temp = g_strstr_len(data, data_len, start_text);
	if (temp == NULL)
	{
		purple_debug_info("facebook", "couldn't find post_form_id\n");
		//must be a bad password/username
		fba->gc->wants_to_die = TRUE;
		purple_connection_error(fba->gc, _("Invalid username or password"));
		return;
	}
	temp += strlen(start_text);
	post_form_id = g_strndup(temp, strchr(temp, '"')-temp);

	if (fba->post_form_id)
	{
		freepoint = fba->post_form_id;
		fba->post_form_id = NULL;
		g_free(freepoint);
	}
	fba->post_form_id = post_form_id;
	
	//dodgy as search for channel server number
	start_text = "\", \"channel";
	temp = g_strstr_len(data, data_len, start_text);
	if (temp == NULL)
	{
		purple_debug_info("facebook", "couldn't find channel\n");
		fba->gc->wants_to_die = TRUE;
		purple_connection_error(fba->gc, _("Invalid username or password"));
		return;
	}
	temp += strlen(start_text);
	channel_number = g_strndup(temp, strchr(temp, '"')-temp);
	
	if (fba->channel_number)
	{
		freepoint = fba->channel_number;
		fba->channel_number = NULL;
		g_free(freepoint);
	}
	fba->channel_number = channel_number;
	
	temp = g_strdup_printf("visibility=true&post_form_id=%s", post_form_id);
	facebookim_post(fba, "apps.facebook.com", "/ajax/chat/settings.php", temp, NULL, NULL, FALSE);
	g_free(temp);
	
	fba->state = PURPLE_CONNECTED;
}

void
facebookim_get_post_form_id(FacebookAccount *fba)
{
	facebookim_get(fba, NULL, "/home.php", got_form_id_page, NULL, FALSE);
}

void
find_feed_url_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer userdata)
{
	const gchar *search_string = "<link rel=\"alternate\" type=\"application/rss+xml\" title=\"Your &quot;Facebook Notifications Feed\" href=\"";
	gchar *feed_url;
	gchar *stripped;
	
	feed_url = g_strstr_len(data, data_len, search_string);
	if (!feed_url)
		return;
	feed_url += strlen(search_string);
	
	feed_url = g_strndup(feed_url, strchr(feed_url, '"')-feed_url);
	//convert &amp; to &
	stripped = purple_unescape_html(feed_url);
	g_free(feed_url);
	//strip the host and protocol off url
	feed_url = g_strdup(strstr(stripped, "/feeds"));
	g_free(stripped);

	if (feed_url && strlen(feed_url) && fba->account)
	{
		purple_account_set_string(fba->account, "facebook_notifications_feed_url", feed_url);
		facebookim_get_notifications_feed(fba);
	}
}

void
facebookim_find_feed_url(FacebookAccount *fba)
{
	facebookim_get(fba, NULL, "/notifications.php", find_feed_url_cb, NULL, FALSE);
}

void
facebookim_got_notifications_cb(FacebookAccount *fba, gchar *url_text, gsize len, gpointer userdata)
{
	time_t last_fetch_time;
	time_t time_of_message;
	time_t newest_message = 0;
	gchar *temp;
	gchar month_string[4], weekday[4];
	guint year, month, day, hour, minute, second;
	long timezone;
	gchar *subject, *url;
	gchar *search_start = (gchar *)url_text;

	month_string[3] = weekday[3] = '\0';
	year = month = day = hour = minute = second = 0;
	
	if (!url_text || !len)
		return;
	
	last_fetch_time = purple_account_get_int(fba->account, "facebook_notifications_last_fetch", 0);
	purple_debug_info("facebook", "last fetch time: %ld\n", (long)last_fetch_time);
	
	while (search_start && (search_start = strstr(search_start, "<item>")))
	{
		search_start = search_start + 6;
		
		temp = strstr(search_start, "<pubDate>");
		if (!temp)
		{
			purple_debug_error("facebook","couldn't find date in rss feed\n");
			return;
		}
		temp += 9;
		temp = g_strndup(temp, strchr(temp, '<')-temp);
		
		//rss times are in Thu, 19 Jun 2008 15:51:25 -1100 format
		//purple_debug_info("facebook", "pre time: %s\n", temp);
		sscanf(temp, "%3s, %2u %3s %4u %2u:%2u:%2u %5ld", (char*)&weekday, &day, (char*)&month_string, &year, &hour, &minute, &second, &timezone);
		if (g_str_equal(month_string, "Jan")) month = 0;
		else if (g_str_equal(month_string, "Feb")) month = 1;
		else if (g_str_equal(month_string, "Mar")) month = 2;
		else if (g_str_equal(month_string, "Apr")) month = 3;
		else if (g_str_equal(month_string, "May")) month = 4;
		else if (g_str_equal(month_string, "Jun")) month = 5;
		else if (g_str_equal(month_string, "Jul")) month = 6;
		else if (g_str_equal(month_string, "Aug")) month = 7;
		else if (g_str_equal(month_string, "Sep")) month = 8;
		else if (g_str_equal(month_string, "Oct")) month = 9;
		else if (g_str_equal(month_string, "Nov")) month = 10;
		else if (g_str_equal(month_string, "Dec")) month = 11;
		g_free(temp);
		
		//try using pidgin's functions
		temp = g_strdup_printf("%04u%02u%02uT%02u%02u%02u%05ld", year, month, day, hour, minute, second, timezone);
		time_of_message = purple_str_to_time(temp, FALSE, NULL, NULL, NULL);
		g_free(temp);
		
		if (time_of_message <= 0)
		{
			//there's no cross-platform, portable way of converting string to time
			//that doesn't need a new version of glib, so just cheat
			time_of_message = second + 60*minute + 3600*hour + 86400*day + 2592000*month + 31536000*(year-1970);
		}
		
		purple_debug_info("facebook", "time of message: %ld\n", (long)time_of_message);
		//purple_debug_info("facebook", "time of message: %s\n", ctime(&time_of_message));
		
		if (time_of_message > newest_message)
		{
			//we'll keep the newest message to save
			newest_message = time_of_message;
		}
		
		if (time_of_message <= last_fetch_time)
		{
			//fortunatly, rss messages are ordered from newest to oldest
			//so if this message is older than the last one, ignore rest
			break;
		}
		
		temp = strstr(search_start, "<link>");
		if (!temp)
		{
			url = g_strdup("");
		} else {
			temp += 6;
			url = g_strndup(temp, strchr(temp, '<')-temp);
			//convert &amp; to &
			temp = purple_unescape_html(url);
			g_free(url);
			url = temp;
		}
		
		temp = strstr(search_start, "<title>");
		if (!temp)
		{
			subject = g_strdup("");
		} else {
			temp += 7;
			subject = g_strndup(temp, strchr(temp, '<')-temp);
		}
		
		purple_debug_info("facebook", "subject: %s\n", subject);
		
		purple_notify_email(fba->gc, subject, NULL, fba->account->username, url, NULL, NULL);
		g_free(subject);
		g_free(url);
		
		search_start = strstr(search_start, "</item>");
	}
	
	if (newest_message > last_fetch_time)
	{
		//update the last fetched time if we had newer messages
		purple_account_set_int(fba->account, "facebook_notifications_last_fetch", newest_message);
	}
}

gboolean
facebookim_get_notifications_feed(FacebookAccount *fba)
{
	const gchar *feed_url;
	
	feed_url = purple_account_get_string(fba->account, "facebook_notifications_feed_url", NULL);
	if (!feed_url)
	{
		facebookim_find_feed_url(fba);
		return TRUE;
	}
	
	if (purple_account_get_bool(fba->account, "facebook_get_notifications", TRUE))
		facebookim_get(fba, NULL, feed_url, facebookim_got_notifications_cb, NULL, FALSE);
	
	return TRUE;
}

gboolean
facebookim_new_messages_check_timeout(PurpleConnection *gc)
{
	FacebookAccount *fba = gc->proto_data;
	
	if (!PURPLE_CONNECTION_IS_CONNECTED(gc))
		return FALSE;
	if (!fba)
		return FALSE;
	if (fba->state == PURPLE_DISCONNECTED)
		return FALSE;
	
	if (fba->last_messages_download_time < (time(NULL) - 3*60))
	{
		//messages havn't been downloaded for at least 3 minutes... something's gone wrong
		facebookim_get_new_messages(fba);
	}
	
	return TRUE;
}

static void
facebookim_login_response_cb(gpointer data, PurpleSslConnection *ssl, PurpleInputCondition cond)
{
	FacebookAccount *fba = data;
	PurpleConnection *gc = fba->gc;
	gchar response[2048];
	int len;
	gchar *temp;
	gchar *header_end;
	gchar *headers;
	
	len = purple_ssl_read(ssl, response, sizeof(response)-1);

	if (len < 0 && errno == EAGAIN) {
		/* Try again later */
		return;
	} else if (len < 0) {
		purple_ssl_close(ssl);
		if (gc)
			purple_connection_error(gc, _("Read error"));
		return;
	} else if (len == 0) {
		purple_ssl_close(ssl);
		if (gc)
			purple_connection_error(gc, _("Server has disconnected"));
		return;
	}
	
	purple_ssl_close(ssl);
	
	if (fba->account && fba->account->disconnecting)
	{
		return;
	}
	
	//we only want the headers
	header_end = g_strstr_len(response, len, "\r\n\r\n");
	headers = g_strndup(response, header_end-response);
	purple_debug_info("facebook", "Headers: %s\n", headers);
	facebook_update_cookies(fba, headers);
	g_free(headers);

	//look for our uid
	temp = g_hash_table_lookup(fba->cookie_table, "c_user");
	if (temp == NULL) {
		//wasn't given a c_user, so must be a bad password/username
		gc->wants_to_die = TRUE;
		purple_connection_error(gc, _("Invalid username or password"));
		return;
	}
	fba->uid = atoi(temp);
	purple_debug_info("facebook", "uid %d\n", fba->uid);

	//ok, we're logged in now!
	purple_connection_set_state(gc, PURPLE_CONNECTED);

	facebookim_get_post_form_id(fba);
	facebookim_get_buddy_list(fba->account);
	facebookim_get_new_messages(fba);
	facebookim_check_friend_requests(fba);
	facebookim_get_notifications_feed(fba);
	
	//periodically check for people adding you to their facebook friend list
	fba->friend_request_timeout = purple_timeout_add_seconds(60*5, (GSourceFunc)facebookim_check_friend_requests, fba);
	//periodically check for updates to your buddy list
	fba->buddy_list_timeout = purple_timeout_add_seconds(60, (GSourceFunc)facebookim_get_buddy_list, fba->account);
	//periodically check for new notifications
	fba->notifications_timeout = purple_timeout_add_seconds(60, (GSourceFunc)facebookim_get_notifications_feed, fba);
	//periodically make sure we're downloading new messages
	fba->new_messages_check_timeout = purple_timeout_add_seconds(60*4, (GSourceFunc)facebookim_new_messages_check_timeout, gc);
}

void
facebook_cookie_foreach_cb(gchar *cookie_name, gchar *cookie_value, GString *return_string)
{
	g_string_append_printf(return_string, "%s=%s;", cookie_name, cookie_value);
}

gchar *
facebook_cookies_to_string(FacebookAccount *fba)
{
	//convert fba->cookie_table to a string
	GString *return_string;
	
	return_string = g_string_new(NULL);
	
	g_hash_table_foreach(fba->cookie_table, (GHFunc)facebook_cookie_foreach_cb, return_string);

	return g_string_free(return_string, FALSE);
}

void
facebook_update_cookies(FacebookAccount *fba, const gchar *headers)
{
	gchar *cookie_start;
	gchar *cookie_end;
	gchar *cookie_name;
	gchar *cookie_value;
	int header_len;
	
	g_return_if_fail(headers != NULL);
	
	header_len = strlen(headers);
	
	//Check that we're not logged out or in the process of logging out
	if (!fba || fba->state == PURPLE_DISCONNECTED)
		return;

	//look for the next "Set-Cookie: "
	//grab the data up until ';'
	cookie_start = (gchar *)headers;
	while((cookie_start = strstr(cookie_start, "Set-Cookie: ")) && (headers-cookie_start) < header_len)
	{
		cookie_start += 12;
		cookie_end = strchr(cookie_start, '=');
		cookie_name = g_strndup(cookie_start, cookie_end-cookie_start);
		cookie_start = cookie_end + 1;
		cookie_end = strchr(cookie_start, ';');
		cookie_value= g_strndup(cookie_start, cookie_end-cookie_start);
		cookie_start = cookie_end;

		purple_debug_info("facebook", "got cookie %s=%s;\n", cookie_name, cookie_value);
		if (fba && fba->account && !fba->account->disconnecting && fba->cookie_table)
			g_hash_table_replace(fba->cookie_table, cookie_name, cookie_value);
	}
}

static void
facebookim_login_cb(gpointer data, PurpleSslConnection *ssl, PurpleInputCondition cond)
{
	FacebookAccount *fba = data;
	gchar *postdata;
	gchar *httpdata;
	gchar *username_temp, *password_temp, *challenge_temp;
	
	purple_debug_info("facebook", "login_cb\n");
	
	if (!fba || fba->state == PURPLE_DISCONNECTED || !fba->account || fba->account->disconnecting)
	{
		purple_ssl_close(ssl);
		return;
	}
	
	//purple_url_encode can't be used more than once on the same line
	username_temp = g_strdup(purple_url_encode(purple_account_get_username(fba->account)));
	password_temp = g_strdup(purple_url_encode(purple_account_get_password(fba->account)));
	
	if (fba->login_challenge)
	{
		challenge_temp = g_strdup(purple_url_encode(fba->login_challenge));
		postdata = g_strdup_printf("challenge=%s&md5pass=1&noerror=1&email=%s&pass=%s&charset_test=%%E2%%AC%%C2%%B4%%E2%%82%%AC%%C2%%B4%%E6%%B0%%B4%%D0%%94%%D0%%84", challenge_temp, username_temp, password_temp);
		g_free(challenge_temp);
	} else {
		postdata = g_strdup_printf("email=%s&pass=%s&persistent=1&login=Login&charset_test=%%E2%%AC%%C2%%B4%%E2%%82%%AC%%C2%%B4%%E6%%B0%%B4%%D0%%94%%D0%%84", username_temp, password_temp);
	}
	g_free(username_temp);
	g_free(password_temp);
	
	httpdata = g_strdup_printf("POST /login.php HTTP/1.0\r\n"
								"Host: login.facebook.com\r\n"
								"Connection: close\r\n"
								"Content-Type: application/x-www-form-urlencoded\r\n"
								"User-Agent: Opera/9.50 (Windows NT 5.1; U; en-GB)\r\n"
								"Content-length: %d\r\n"
								"Cookie: test_cookie=1;\r\n"
								"Accept: */*\r\n"
								"\r\n%s",
								((int)strlen(postdata)), postdata);
	g_free(postdata);
	
	purple_ssl_write(ssl, httpdata, strlen(httpdata));
	purple_ssl_input_add(ssl, facebookim_login_response_cb, fba);
	
	g_free(httpdata);
}

static void
facebookim_login_error(PurpleSslConnection *ssl, PurpleSslErrorType errortype, gpointer data)
{
	FacebookAccount *fba = data;

	//ssl error is after 2.3.0
	//purple_connection_ssl_error(fba->gc, errortype);
	purple_connection_error(fba->gc, _("SSL Error"));
}

void
facebookim_fetch_login_cb(FacebookAccount *fba, gchar *url_text, gsize data_len, gpointer userdata)
{
	//FacebookAccount *fba = user_data;

	gchar *start_text = "id=\"challenge\" name=\"challenge\" value=\"";
	gchar *challenge = NULL;
	gchar *start;
	gchar *end;
	
	//look for the challenge in url_text
	//start at <input type="hidden" id="challenge" name="challenge" value="
	//finish at "
	start = strstr(url_text, start_text);
	if (start != NULL)
	{
		start += strlen(start_text);
		end   = strchr(start, '"');
		challenge = g_strndup(start, end-start);
		//printf("challenge: %s\n", challenge);
	}
	fba->login_challenge = challenge;
	purple_debug_info("facebook", "challenge %s\n", challenge);
	
	purple_ssl_connect(fba->account, "login.facebook.com", 443, facebookim_login_cb, facebookim_login_error, fba);
}

void 
facebookim_login(PurpleAccount *acct)
{
	FacebookAccount *fba;
	guint16 i;
	
	fba = g_new(FacebookAccount, 1);
	fba->account = acct;
	fba->gc = acct->gc;
	fba->login_challenge = NULL;
	fba->post_form_id = NULL;
	fba->uid = -1;
	fba->channel_number = g_strdup("01");
	fba->message_fetch_sequence = 0;
	fba->next_message_pointer = 0;
	fba->cookie_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
	fba->hostname_ip_cache = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
	fba->auth_buddies = NULL;
	fba->last_status_message = NULL;
	fba->state = PURPLE_CONNECTING;
	
	for (i = 0; i < LAST_MESSAGE_MAX; i++)
	{
		fba->last_messages[i] = 0l;
	}
	
	acct->gc->proto_data = fba;
	
	//first line grabs the challenge, second line tries without challenge
	//facebookim_get(fba, NULL, "/index.php", facebookim_fetch_login_cb, NULL, FALSE);
	purple_ssl_connect(fba->account, "login.facebook.com", 443, facebookim_login_cb, facebookim_login_error, fba);
}

void
facebookim_close(PurpleConnection *gc)
{
	FacebookAccount *fba = gc->proto_data;
	gc->proto_data = NULL;
	
	fba->state = PURPLE_DISCONNECTED;
	
	fba->account = NULL;
	fba->gc = NULL;
	
	purple_timeout_remove(fba->buddy_list_timeout);
	purple_timeout_remove(fba->friend_request_timeout);
	purple_timeout_remove(fba->notifications_timeout);
	
	//not sure which one of these lines is the right way to logout
	//facebookim_post(fba, "apps.facebook.com", "/ajax/chat/settings.php", "visibility=false", NULL, NULL, FALSE);
	//facebookim_post(fba, NULL, "/logout.php", "confirm=1", NULL, NULL, FALSE);
	
	g_hash_table_destroy(fba->cookie_table);
	fba->cookie_table = NULL;
	g_hash_table_destroy(fba->hostname_ip_cache);
	fba->hostname_ip_cache = NULL;
	if (fba->login_challenge)
	{
		g_free(fba->login_challenge);
		fba->login_challenge = NULL;
	}
	if (fba->post_form_id)
	{
		g_free(fba->post_form_id);
		fba->post_form_id = NULL;
	}
	if (fba->channel_number)
	{
		g_free(fba->channel_number);
		fba->channel_number = NULL;
	}
	g_slist_free(fba->auth_buddies);
	if (fba->last_status_message)
	{
		g_free(fba->last_status_message);
		fba->last_status_message = NULL;
	}
	
	g_free(fba);
}

void
facebookim_send_im_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data)
{
	FacebookOutgoingMessage *msg = user_data;
	gchar *error_summary = NULL;
	gchar *temp;
	
	//printf("send im response: %s\n", data);
	//purple_debug_info("facebook", "sent im response: %s\n", data);
	//for (;;);{"error":1356003,"errorSummary":"Send destination not online","errorDescription":"This person is no longer online.","payload":null,"bootload":[{"name":"js\/common.js.pkg.php","type":"js","src":"http:\/\/static.ak.fbcdn.net\/rsrc.php\/pkg\/59\/98936\/js\/common.js.pkg.php"}]}
	//for (;;);{"error":0,"errorSummary":"","errorDescription":"No error.","payload":[],"bootload":[{"name":"js\/common.js.pkg.php","type":"js","src":"http:\/\/static.ak.fbcdn.net\/rsrc.php\/pkg\/59\/98936\/js\/common.js.pkg.php"}]}
	
	temp = g_strstr_len(data, data_len, "\"errorSummary\":\"");
	if (temp != NULL)
	{
		temp += 16;
		error_summary = g_strndup(temp, strchr(temp, '"')-temp);
		purple_debug_info("facebook", "sent im error: %s\n", error_summary);
		if (strlen(error_summary))
		{
			//there was an error, either report it or retry
			if (msg->retry_count++ < purple_account_get_int(msg->fba->account, "facebook_max_msg_retry", 2))
			{
				purple_timeout_add_seconds(1, (GSourceFunc)facebookim_send_im_fom, msg);
				g_free(error_summary);
				return;
			}
			else
			{
				serv_got_im(msg->fba->gc, msg->who, error_summary, PURPLE_MESSAGE_ERROR, msg->time);
			}
		}
	}
	
	if (error_summary)
		g_free(error_summary);
	g_free(msg->who);
	g_free(msg->message);
	g_free(msg);
}

gboolean
facebookim_send_im_fom(FacebookOutgoingMessage *msg)
{
	gchar *postdata;
	gchar *message_temp;

	message_temp = g_strdup(purple_url_encode(msg->message));
	postdata = g_strdup_printf("msg_text=%s&msg_id=%d&to=%s&client_time=%ld&post_form_id=%s", message_temp, msg->msg_id, msg->who, ((long)msg->time), (msg->fba->post_form_id?msg->fba->post_form_id:"0"));
	g_free(message_temp);
	
	facebookim_post(msg->fba, NULL, "/ajax/chat/send.php", postdata, facebookim_send_im_cb, (gpointer)msg, FALSE);
	g_free(postdata);
	
	return FALSE;
}

int 
facebookim_send_im(PurpleConnection *gc, const gchar *who, const gchar *message, PurpleMessageFlags flags)
{
	FacebookOutgoingMessage *msg;

	msg = g_new0(FacebookOutgoingMessage, 1);
	msg->fba = gc->proto_data;

	//convert html to plaintext, removing trailing spaces
	msg->message = g_strchomp(purple_markup_strip_html(message));
	if (strlen(msg->message) > 999)
	{
		g_free(msg->message);
		g_free(msg);
		return -E2BIG;
	}

	msg->msg_id = g_random_int();
	msg->who = g_strdup(who);
	msg->time = time(NULL);
	msg->retry_count = 0;
	
	facebookim_send_im_fom(msg);
	
	return strlen(message);
}

GList *
facebookim_statuses(PurpleAccount *acct)
{
	GList *types = NULL;
	PurpleStatusType *status;
	
	//Online people have a status message and also a date when it was set	
	status = purple_status_type_new_with_attrs(PURPLE_STATUS_AVAILABLE, NULL, _("Online"), FALSE, TRUE, FALSE, "message", _("Message"), purple_value_new(PURPLE_TYPE_STRING), "message_date", _("Message changed"), purple_value_new(PURPLE_TYPE_STRING), NULL);
	types = g_list_append(types, status);
	
	//Offline people dont have messages
	status = purple_status_type_new_full(PURPLE_STATUS_OFFLINE, NULL, _("Offline"), FALSE, TRUE, FALSE);
	types = g_list_append(types, status);
	
	return types;
}

gchar *
facebookim_status_text(PurpleBuddy *buddy)
{
	FacebookBuddy *fbuddy = buddy->proto_data;
	
	if (fbuddy && fbuddy->status && strlen(fbuddy->status))
		return g_strdup(fbuddy->status);
	
	return NULL;
}

void
facebookim_tooltip_text(PurpleBuddy *buddy, PurpleNotifyUserInfo *userinfo, gboolean full)
{
	FacebookBuddy *fbuddy = buddy->proto_data;
	
	g_return_if_fail(fbuddy);

	if (fbuddy->status && strlen(fbuddy->status))
	{
		purple_notify_user_info_add_pair(userinfo, _("Message"), fbuddy->status);
		if (fbuddy->status_rel_time && strlen(fbuddy->status_rel_time))
		{
			purple_notify_user_info_add_pair(userinfo, _("Message changed"), fbuddy->status_rel_time);
		}
	}	
}

void
facebookim_buddy_free(PurpleBuddy *buddy)
{
	FacebookBuddy *fbuddy = buddy->proto_data;
	if (fbuddy != NULL)
	{
		buddy->proto_data = NULL;
		
		if (fbuddy->name != NULL)
			g_free(fbuddy->name);
		if (fbuddy->status != NULL)
			g_free(fbuddy->status);
		if (fbuddy->status_rel_time != NULL)
			g_free(fbuddy->status_rel_time);
		if (fbuddy->thumb_url != NULL)
			g_free(fbuddy->thumb_url);
		
		g_free(fbuddy);
	}
}

unsigned int
facebookim_send_typing(PurpleConnection *gc, const gchar *name, PurpleTypingState state)
{
	int typing_state;
	gchar *postdata;
	FacebookAccount *fba = gc->proto_data;
	gchar *name_temp;
	
	g_return_val_if_fail(fba, 0);
	g_return_val_if_fail(fba->post_form_id, 0);

	if (state == PURPLE_TYPING)
	{
		typing_state = 1;
	} else {
		typing_state = 0;		
	}
	
	//Don't send typing notifications to self
	if (atoi(name) != fba->uid)
	{
		name_temp = g_strdup(purple_url_encode(name));
		postdata = g_strdup_printf("typ=%d&to=%s&post_form_id=%s", typing_state, name_temp, fba->post_form_id);
		g_free(name_temp);
		facebookim_post(fba, NULL, "/ajax/chat/typ.php", postdata, NULL, NULL, FALSE);
		g_free(postdata);
	} else {
		serv_got_typing(gc, name, 10, state);
	}
	
	return 7;
}

void
facebookim_auth_accept(gpointer data)
{
	FacebookBuddy *fbuddy = data;
	FacebookAccount *fba = fbuddy->fba;
	gchar *postdata;
		
	g_return_if_fail(fba);
	g_return_if_fail(fba->post_form_id);
	g_return_if_fail(fbuddy->uid);

	postdata = g_strdup_printf("type=friend_add&id=%d&action=accept&post_form_id=%s", fbuddy->uid, fba->post_form_id);
	facebookim_post(fba, NULL, "/ajax/reqs.php", postdata, NULL, NULL, FALSE);
	g_free(postdata);
	
	fba->auth_buddies = g_slist_remove(fba->auth_buddies, GINT_TO_POINTER(fbuddy->uid));
	
	g_free(fbuddy);
}

void
facebookim_auth_reject(gpointer data)
{
	FacebookBuddy *fbuddy = data;
	FacebookAccount *fba = fbuddy->fba;
	gchar *postdata;
			
	g_return_if_fail(fba);
	g_return_if_fail(fba->post_form_id);
	g_return_if_fail(fbuddy->uid);
	
	postdata = g_strdup_printf("type=friend_add&id=%d&action=reject&post_form_id=%s", fbuddy->uid, fba->post_form_id);
	facebookim_post(fba, NULL, "/ajax/reqs.php", postdata, NULL, NULL, FALSE);
	g_free(postdata);
	
	fba->auth_buddies = g_slist_remove(fba->auth_buddies, GINT_TO_POINTER(fbuddy->uid));
	
	g_free(fbuddy);
}

void
facebookim_check_friend_request_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data)
{
	const char *uid_pre_text = "class=\"confirm\" id=\"friend_add_";
	const char *name_pre_text = "<td class=\"info\"><a ";
	const char *msg_pre_text = "<div class=\"personal_msg\"><span>";
	gchar *uid;
	gint32 uid_int;
	gchar *name;
	gchar *msg;
	gchar *msg_plain;
	FacebookBuddy *buddy;
	gchar *search_start = data;	
	
	//loop through the page data and look for confirm_friend_add_([0-9]*)"
	while((search_start = strstr(search_start, uid_pre_text)))
	{
		search_start += strlen(uid_pre_text);
		uid = g_strndup(search_start, strchr(search_start, '"')-search_start);
		purple_debug_info("facebook", "uid: %s\n", uid);
		
		uid_int = atoi(uid);
		
		if (g_slist_find(fba->auth_buddies, GINT_TO_POINTER(uid_int)) != NULL)
		{
			//we've already notified the user of this friend request
			g_free(uid);
			continue;
		}
		
		name = strstr(search_start, name_pre_text);
		if (name != NULL)
		{
			name += strlen(name_pre_text);
			name = strchr(name, '>')+1;
			name = g_strndup(name, strchr(name, '<')-name);
			purple_debug_info("facebook", "name: %s\n", name);
		} else {
			name = NULL;
		}
		
		msg = strstr(search_start, msg_pre_text);
		if (msg != NULL)
		{
			msg += strlen(msg_pre_text);
			msg = g_strndup(msg, strstr(msg, "</span></div>")-msg);
			msg_plain = purple_markup_strip_html(msg);
			g_free(msg);
			purple_debug_info("facebook", "msg: %s\n", msg_plain);
		} else {
			msg_plain = NULL;
		}
		
		buddy = g_new0(FacebookBuddy, 1);
		buddy->fba = fba;
		buddy->uid = uid_int;
		purple_account_request_authorization(fba->account, uid, NULL, name, msg_plain, TRUE,
											facebookim_auth_accept, facebookim_auth_reject, buddy);
		//add buddy to list of already displayed, so as not to double up
		fba->auth_buddies = g_slist_prepend(fba->auth_buddies, GINT_TO_POINTER(uid_int));
	}
}

gboolean
facebookim_check_friend_requests(FacebookAccount *fba)
{	
	if (fba == NULL)
		return FALSE;
	
	facebookim_get(fba, NULL, "/reqs.php", facebookim_check_friend_request_cb, NULL, FALSE);

	return TRUE;
}

void 
facebookim_add_buddy(PurpleConnection *gc, PurpleBuddy *buddy, PurpleGroup *group)
{
	gchar *postdata;
	gchar *url;
	FacebookAccount *fba = gc->proto_data;
	gchar *buddy_temp;
	
	if (atoi(buddy->name) == fba->uid)
	{
		purple_account_set_bool(fba->account, "facebook_hide_self", FALSE);
		return;
	}
	
	buddy_temp = g_strdup(purple_url_encode(buddy->name));
	postdata = g_strdup_printf("confirmed=1&add=Add+Friend&action=follow_up&uid=%s&flids=&flid_name=&source=search&is_from_whitelist=0&message=&failed_captcha=0&post_form_id=%s", buddy_temp, fba->post_form_id);
	url = g_strdup_printf("/ajax/addfriend.php?id=%s", buddy_temp);
	g_free(buddy_temp);
	
	facebookim_post(fba, NULL, url, postdata, NULL, NULL, FALSE);
	
	g_free(postdata);
	g_free(url);
}

void 
facebookim_remove_buddy(PurpleConnection *gc, PurpleBuddy *buddy, PurpleGroup *group)
{
	gchar *postdata;
	FacebookAccount *fba = gc->proto_data;
	
	if (atoi(buddy->name) == fba->uid)
	{
		purple_account_set_bool(fba->account, "facebook_hide_self", TRUE);
		return;
	}	
	
	postdata = g_strdup_printf("uid=%s&post_form_id=%s", buddy->name, fba->post_form_id);
	
	facebookim_post(fba, NULL, "/ajax/removefriend.php", postdata, NULL, NULL, FALSE);
	
	g_free(postdata);
}

void
facebookim_set_status_ok_cb(gpointer connection, gchar *status_text)
{
	PurpleConnection *gc = connection;
	FacebookAccount *fba = gc->proto_data;
	gchar *postdata;
	gchar *status_temp;
	
	g_return_if_fail(fba != NULL);
	g_return_if_fail(fba->post_form_id != NULL);
	
	g_strstrip(status_text);
	
	//don't set the status if it's idential to what we've already set
	if (fba->last_status_message && g_str_equal(fba->last_status_message, status_text))
	{
		return;
	}
	if (fba->last_status_message)
	{
		status_temp = fba->last_status_message;
		fba->last_status_message = NULL;
		g_free(status_temp);
	}
	fba->last_status_message = g_strdup(status_text);
	
	if (strlen(status_text))
	{
		status_temp = g_strdup(purple_url_encode(status_text));
		postdata = g_strdup_printf("status=%s&post_form_id=%s", status_temp, fba->post_form_id);
		g_free(status_temp);
	}
	else
		postdata = g_strdup_printf("clear=1&post_form_id=%s", fba->post_form_id);
	
	facebookim_post(fba, NULL, "/updatestatus.php", postdata, NULL, NULL, FALSE);
	
	g_free(postdata);
}

void
facebookim_set_status_p(PurpleAccount *account, PurpleStatus *status)
{
	const gchar *message;
	gchar *stripped;
	
	//first check that we actually want to set this through Pidgin
	if (!purple_account_get_bool(account, "facebook_set_status_through_pidgin", FALSE))
	{
		return;
	}
	
	message = purple_status_get_attr_string(status, "message");
	if (message == NULL)
		message = "";
		
	stripped = g_strstrip(purple_markup_strip_html(message));
	facebookim_set_status_ok_cb(account->gc, stripped);
	g_free(stripped);
}

void
facebookim_set_status_cb(PurplePluginAction *action)
{
	PurpleConnection *gc = action->context;
	FacebookAccount *fba = gc->proto_data;
	gchar *uid_str;
	
	uid_str = g_strdup_printf("%d", fba->uid);
	
	purple_request_input(gc, NULL, _("Set your Facebook status"), purple_account_get_alias(gc->account), "is ", 
							FALSE, FALSE, NULL, _("OK"), 
							G_CALLBACK(facebookim_set_status_ok_cb), _("Cancel"), 
							NULL, gc->account, uid_str, NULL, gc);
	
	g_free(uid_str);
}

void
facebook_searchresults_add_buddy(PurpleConnection *gc, GList *row, void *user_data)
{
	PurpleAccount *acct = purple_connection_get_account(gc);
	
	if (!purple_find_buddy(acct, g_list_nth_data(row, 0)))
		purple_blist_request_add_buddy(acct, g_list_nth_data(row, 0), NULL, NULL);
}

void
facebook_searchresults_info_buddy(PurpleConnection *gc, GList *row, void *user_data)
{
	//PurpleAccount *acct = purple_connection_get_account(gc);
	
	//if (purple_find_buddy(acct, g_list_nth_data(row, 0)))
		facebookim_get_info(gc, g_list_nth_data(row, 0));
}

void
facebook_found_friends(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data)
{
	PurpleNotifySearchResults *results;
	PurpleNotifySearchColumn *column;
	gchar *id, *temp, *stripped, *last_id_pos = 0, *id_pos = data;
	gchar *search_term = user_data;
	const gchar *id_search_term = "facebook.com/inbox/?compose&amp;id=";  // "
	const gchar *name_search_term = "class=\"url fn\">"; // <
	const gchar *network_search_term = "class=\"result_network\">"; // <

	if (!g_strstr_len(data, data_len, id_search_term))
	{
		//there's no friends found
		//notify as such
		temp = g_strdup_printf(_("No results found for %s"), search_term);
		purple_notify_error(fba->gc, NULL, temp, NULL);
		g_free(temp);
		g_free(search_term);
		return;
	}
	
	results = purple_notify_searchresults_new();
	if (results == NULL)
	{
		g_free(search_term);
		return;
	}
	
	//columns: Facebook ID, Name, Network
	column = purple_notify_searchresults_column_new(_("ID"));
	purple_notify_searchresults_column_add(results, column);
	column = purple_notify_searchresults_column_new(_("Name"));
	purple_notify_searchresults_column_add(results, column);
	column = purple_notify_searchresults_column_new(_("Network"));
	purple_notify_searchresults_column_add(results, column);
	column = purple_notify_searchresults_column_new(_("In List?"));
	purple_notify_searchresults_column_add(results, column);
	
	purple_notify_searchresults_button_add(results, PURPLE_NOTIFY_BUTTON_ADD, 
										facebook_searchresults_add_buddy);
	purple_notify_searchresults_button_add(results, PURPLE_NOTIFY_BUTTON_INFO, 
										facebook_searchresults_info_buddy);
	
	purple_debug_info("facebook", "found_friends\n");
	while ((id_pos = strstr(id_pos, id_search_term)))
	{
		//the row in the search results table
		//prepend to it backwards then reverse to speed up adds
		GList *row = NULL;
		
		//grab id
		id_pos += strlen(id_search_term);
		if (strchr(id_pos, '&') < strchr(id_pos, '"'))
		{
			//new layout repeats the id
			continue;
		}
		id = g_strndup(id_pos, strchr(id_pos, '"')-id_pos);
		purple_debug_info("facebook", "Found user with id: %s\n", id);
		row = g_list_prepend(row, id);
		
		//look for name
		temp = g_strrstr_len(data, id_pos-data, name_search_term);
		if (temp && temp > last_id_pos)
		{
			temp += strlen(name_search_term);
			temp = g_strndup(temp, strchr(temp, '<')-temp);
			stripped = purple_unescape_html(temp);
			g_free(temp);
			purple_debug_info("facebook", "With name: %s\n", stripped);
			row = g_list_prepend(row, stripped);
		} else {
			row = g_list_prepend(row, NULL);
		}
		
		//look for network
		temp = g_strrstr_len(data, id_pos-data, network_search_term);
		if (temp && temp > last_id_pos)
		{
			temp += strlen(network_search_term);
			temp = g_strndup(temp, strchr(temp, '<')-temp);
			stripped = purple_unescape_html(temp);
			g_free(temp);
			purple_debug_info("facebook", "With network: %s\n", stripped);
			row = g_list_prepend(row, stripped);
		} else {
			row = g_list_prepend(row, NULL);
		}
		
		if (purple_find_buddy(fba->account, id))
			row = g_list_prepend(row, g_strdup(_("Yes")));
		else
			row = g_list_prepend(row, g_strdup(_("No")));
		
		row = g_list_reverse(row);
		purple_notify_searchresults_row_add(results, row);
		
		last_id_pos = id_pos;
	}
	purple_debug_info("facebook", "dumping search results\n");
	purple_notify_searchresults(fba->gc, NULL, search_term, NULL, results, NULL, NULL);
	
	g_free(search_term);
}

void
facebookim_search_users_search_cb(gpointer connection, const gchar *search_text)
{
	PurpleConnection *gc = connection;
	FacebookAccount *fba = gc->proto_data;
	gchar *search_url;
	gchar *search_temp;
	gchar *sid_cookie_value;
	
	if (!search_text || !strlen(search_text))
		return;
	
	search_temp = g_strdup(purple_url_encode(search_text));
	sid_cookie_value = g_hash_table_lookup(fba->cookie_table, "sid");
	if (sid_cookie_value == NULL)
		sid_cookie_value = "1";
	search_url = g_strdup_printf("/s.php?q=%s&init=q&sid=%s", search_temp, sid_cookie_value);
	g_free(search_temp);
	
	facebookim_get(fba, NULL, search_url, facebook_found_friends, g_strdup(search_text), FALSE);
	
	g_free(search_url);
}

void
facebookim_search_users(PurplePluginAction *action)
{
	PurpleConnection *gc = (PurpleConnection *) action->context;
	
	purple_request_input(gc, _("Search for Friends"),
					   _("Search for Facebook Friends"),
					   _("Type the full name or e-mail address of the friend you are searching for."),
					   NULL, FALSE, FALSE, NULL,
					   _("_Search"), G_CALLBACK(facebookim_search_users_search_cb),
					   _("_Cancel"), NULL,
					   purple_connection_get_account(gc), NULL, NULL,
					   gc);
}

GList *
facebookim_actions(PurplePlugin *plugin, gpointer context)
{
	GList *m = NULL;
	PurplePluginAction *act;

	act = purple_plugin_action_new(_("Set Facebook status..."), facebookim_set_status_cb);
	m = g_list_append(m, act);
	
	act = purple_plugin_action_new(_("Search for buddies..."), facebookim_search_users);
	m = g_list_append(m, act);
	
	return m;
}

gchar *
facebook_remove_useless_stripped_links(const gchar *input)
{
	//removes stripped links like "(/s.php? ... )" from user info
	//as an artefact of purple_markup_strip_html
	
	gchar *output = g_strdup(input);
	gchar *i = output;
	gchar *end;
	
	while ((i = strstr(i, " (/")))
	{
		end = strchr(i, ')');
		if (end)
		{
			end += 1;
			//overwrite everything after the brackets to before it
			g_stpcpy(i, end);
		}
	}
	
	return output;
}

void
facebookim_get_new_info_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data)
{
	PurpleNotifyUserInfo *user_info;
	PurpleBuddyIcon *buddy_icon;
	size_t image_size;
	gconstpointer image_pointer;
	int icon_id = -1;
	gchar *uid = user_data;
	gchar *label_temp;
	gchar *value_temp;
	gchar *value_temp2;
	gchar *search_start;
	gchar *search_end;
	
	purple_debug_info("facebook", "get_new_info_cb\n");
	
	//look from <div id="info_tab" class="info_tab">
	//until </div></div></div></div>
	search_start = g_strstr_len(data, data_len, "<div id=\"info_tab\" class=\"info_tab\">");
	if(search_start == NULL)
	{
		user_info = purple_notify_user_info_new();
		purple_notify_user_info_add_pair(user_info, NULL, "Could not find profile");
		purple_notify_userinfo(fba->gc, uid, user_info, NULL, NULL);
		purple_notify_user_info_destroy(user_info);
		g_free(uid);
		return;
	}
	search_end = strstr(search_start, "</div></div></div></div>");
		
	user_info = purple_notify_user_info_new();

	value_temp = g_strstr_len(data, data_len, "<title>Facebook | ");
	if (value_temp)
	{
		value_temp = strchr(value_temp, '|')+2;
		value_temp2 = g_strndup(value_temp, strstr(value_temp, "</title>")-value_temp);
		value_temp = g_strchomp(purple_markup_strip_html(value_temp2));
		purple_notify_user_info_add_pair(user_info, _("Name"), value_temp);
		g_free(value_temp);
		g_free(value_temp2);
	}

	value_temp = g_strstr_len(data, data_len, "<span id=\"profile_status\"");
	if (value_temp)
	{
		value_temp2 = strstr(value_temp, "</span>");
		if (value_temp2)
		{
			value_temp = strchr(value_temp, '>')+1;
			value_temp2 = g_strndup(value_temp, strchr(value_temp, '<')-value_temp);
			value_temp = g_strchomp(purple_markup_strip_html(value_temp2));
			purple_notify_user_info_add_pair(user_info, _("Status"), value_temp);
			g_free(value_temp);
			g_free(value_temp2);
		}
	}
	
	buddy_icon = purple_buddy_icons_find(fba->account, uid);
	if (buddy_icon)
	{
		image_pointer = purple_buddy_icon_get_data(buddy_icon, &image_size);
		icon_id = purple_imgstore_add_with_id(g_memdup(image_pointer, image_size), image_size, NULL);
		value_temp = g_strdup_printf("<img id='%d'>", icon_id);
		purple_debug_info("facebook", "user info pic: '%s'\n", value_temp);
		purple_notify_user_info_add_pair(user_info, NULL, value_temp);
		g_free(value_temp);
	}

	while((search_start = strstr(search_start, "<dt>")) && search_start < search_end)
	{
		search_start += 4;
		if (search_start[0] == '<' && search_start[1] == '/' && search_start[2] == 'd' && search_start[3] == 't')
		{
			//the tag closes as soon as it opens (bad xhtml)
			continue;
		}
		
		label_temp = g_strndup(search_start, strchr(search_start, ':')-search_start);
		if (!strlen(label_temp))
		{
			g_free(label_temp);
			continue;
		}
		
		search_start = strstr(search_start, "<dd>");
		if (!search_start)
		{
			g_free(label_temp);
			break;
		}
		
		search_start += 4;
		value_temp = g_strndup(search_start, strstr(search_start, "</dd>")-search_start);
		if (!strlen(value_temp))
		{
			g_free(label_temp);
			g_free(value_temp);
			continue;
		}
		
		//turn html to plaintext
		value_temp2 = g_strchomp(purple_markup_strip_html(value_temp));
		g_free(value_temp);
		
		//remove the silly links
		value_temp = facebook_remove_useless_stripped_links(value_temp2);
		g_free(value_temp2);
		
		purple_debug_info("facebook", "label: %s\n", label_temp);
		purple_debug_info("facebook", "value: %s\n", value_temp);
		purple_notify_user_info_add_pair(user_info, label_temp, value_temp);
		g_free(label_temp);
		g_free(value_temp);
	}
	
	purple_notify_user_info_add_section_break(user_info);
	value_temp = g_strdup_printf("http://www.new.facebook.com/profile.php?id=%s", uid);
	purple_notify_user_info_add_pair(user_info, NULL, value_temp);
	g_free(value_temp);

	purple_notify_userinfo(fba->gc, uid, user_info, NULL, NULL);
	purple_notify_user_info_destroy(user_info);

	if (icon_id >= 0)
		purple_imgstore_unref_by_id(icon_id);
	
	g_free(uid);
}

void
facebookim_get_info_cb(FacebookAccount *fba, gchar *data, gsize data_len, gpointer user_data)
{
	PurpleNotifyUserInfo *user_info;
	PurpleBuddyIcon *buddy_icon;
	size_t image_size;
	gconstpointer image_pointer;
	int icon_id = -1;
	gchar *uid = user_data;
	gchar *label_temp;
	gchar *value_temp;
	gchar *value_temp2;
	gchar *search_start;
	gchar *search_end;
	
	purple_debug_info("facebook", "get_info_cb\n");
	
	//look from <div class="basic_info"><table class="profileTable" cellpadding="0" cellspacing="0">
	//until </table>
	search_start = g_strstr_len(data, data_len, "<div class=\"basic_info\"><table class=\"profileTable\" cellpadding=\"0\" cellspacing=\"0\">");
	if(search_start == NULL)
	{
		//try getting the profile from the new server
		return facebookim_get_new_info_cb(fba, data, data_len, user_data);
	}
	search_end = strstr(search_start, "</table>");
		
	user_info = purple_notify_user_info_new();
	
	value_temp = g_strstr_len(data, data_len, "<div class=\"profile_name\"");
	if (value_temp)
	{
		value_temp = strchr(value_temp, '>')+1;
		value_temp2 = g_strndup(value_temp, strstr(value_temp, "</div>")-value_temp);
		value_temp = g_strchomp(purple_markup_strip_html(value_temp2));
		purple_notify_user_info_add_pair(user_info, _("Name"), value_temp);
		g_free(value_temp);
		g_free(value_temp2);
	}
	
	value_temp = g_strstr_len(data, data_len, "<div class=\"profile_status\"");
	if (value_temp)
	{
		value_temp = strchr(value_temp, '>')+1;
		value_temp2 = g_strndup(value_temp, strstr(value_temp, "</div>")-value_temp);
		value_temp = g_strchomp(purple_markup_strip_html(value_temp2));
		purple_notify_user_info_add_pair(user_info, _("Status"), value_temp);
		g_free(value_temp);
		g_free(value_temp2);
	}
	
	buddy_icon = purple_buddy_icons_find(fba->account, uid);
	if (buddy_icon)
	{
		image_pointer = purple_buddy_icon_get_data(buddy_icon, &image_size);
		icon_id = purple_imgstore_add_with_id(g_memdup(image_pointer, image_size), image_size, NULL);
		value_temp = g_strdup_printf("<img id='%d'>", icon_id);
		purple_debug_info("facebook", "user info pic: '%s'\n", value_temp);
		purple_notify_user_info_add_pair(user_info, NULL, value_temp);
		g_free(value_temp);
	}
	/*
	fbuddy = buddy->proto_data;
	
	buddy_icon_url = g_strcompress(fbuddy->thumb_url);
	//small icon at  http://profile.ak.facebook.com/profile6/1845/74/q800753867_2878.jpg
	//bigger icon at http://profile.ak.facebook.com/profile6/1845/74/n800753867_2878.jpg
	label_temp = strstr(buddy_icon_url, "/q");
	if (label_temp) *(label_temp+1)='n';
	value_temp = g_strdup_printf("<img src='%s' >", buddy_icon_url);
	purple_debug_info("facebook", "user info pic: '%s'\n", value_temp);
	purple_notify_user_info_add_pair(user_info, NULL, value_temp);
	g_free(value_temp);
	g_free(buddy_icon_url);*/
	
	//purple_notify_user_info_add_section_header(user_info, _("Contact Info"));
	
	while((search_start = strstr(search_start, "<td class=\"label\">")) && search_start < search_end)
	{
		search_start += 18;
		label_temp = g_strndup(search_start, strchr(search_start, ':')-search_start);
		
		search_start = strstr(search_start, "<td class=\"data\">");
		if (!search_start)
			break;
		search_start += 17;
		value_temp = g_strndup(search_start, strstr(search_start, "</td>")-search_start);
		
		//turn html to plaintext
		value_temp2 = g_strchomp(purple_markup_strip_html(value_temp));
		g_free(value_temp);
		
		//remove the silly links
		value_temp = facebook_remove_useless_stripped_links(value_temp2);
		g_free(value_temp2);
		
		purple_notify_user_info_add_pair(user_info, label_temp, value_temp);
		g_free(label_temp);
		g_free(value_temp);
	}
	
	purple_notify_user_info_add_section_break(user_info);
	value_temp = g_strdup_printf("http://www.facebook.com/profile.php?id=%s", uid);
	purple_notify_user_info_add_pair(user_info, NULL, value_temp);
	g_free(value_temp);

	purple_notify_userinfo(fba->gc, uid, user_info, NULL, NULL);
	purple_notify_user_info_destroy(user_info);

	if (icon_id >= 0)
		purple_imgstore_unref_by_id(icon_id);
	
	g_free(uid);
}

void 
facebookim_get_info(PurpleConnection *gc, const gchar *uid)
{
	gchar *profile_url;
	
	profile_url = g_strdup_printf("/profile.php?id=%s&v=info", uid);
	
	facebookim_get(gc->proto_data, NULL, profile_url, facebookim_get_info_cb, g_strdup(uid), FALSE);
	
	g_free(profile_url);
}

GHashTable *
facebookim_get_account_text_table(PurpleAccount *account)
{
	GHashTable *table;
	
	table = g_hash_table_new(g_str_hash, g_str_equal);
	
	g_hash_table_insert(table, "login_label", (gpointer)_("Email Address..."));
	
	return table;
}

static void
plugin_init(PurplePlugin *plugin)
{
	PurpleAccountOption *option;
	PurplePluginInfo *info = plugin->info;
	PurplePluginProtocolInfo *prpl_info = info->extra_info;
	
	option = purple_account_option_string_new(_("Server"), "host", "www.facebook.com");
	prpl_info->protocol_options = g_list_append(prpl_info->protocol_options, option);

	option = purple_account_option_bool_new(_("Hide myself in the Buddy List"), "facebook_hide_self", TRUE);
	prpl_info->protocol_options = g_list_append(prpl_info->protocol_options, option);
	
	option = purple_account_option_bool_new(_("Set Facebook status through Pidgin status"), "facebook_set_status_through_pidgin", FALSE);
	prpl_info->protocol_options = g_list_append(prpl_info->protocol_options, option);
	
	option = purple_account_option_bool_new(_("Show Facebook notifications as e-mails in Pidgin"), "facebook_get_notifications", TRUE);
	prpl_info->protocol_options = g_list_append(prpl_info->protocol_options, option);
	
	option = purple_account_option_int_new(_("Number of retries to send message"), "facebook_max_msg_retry", 2);
	prpl_info->protocol_options = g_list_append(prpl_info->protocol_options, option);
}

gboolean
plugin_load(PurplePlugin *plugin)
{
	return TRUE;
}

gboolean
plugin_unload(PurplePlugin *plugin)
{
	return TRUE;
}

static PurplePluginProtocolInfo prpl_info = {
	/* options */
	OPT_PROTO_UNIQUE_CHATNAME,

	NULL,                   /* user_splits */
	NULL,                   /* protocol_options */
	//NO_BUDDY_ICONS          /* icon_spec */
	{"jpg",0,0,50,50,-1,PURPLE_ICON_SCALE_SEND},/* icon_spec */
	facebookim_list_icon,   /* list_icon */
	NULL,                   /* list_emblems */
	facebookim_status_text, /* status_text */
	facebookim_tooltip_text,/* tooltip_text */
	facebookim_statuses,    /* status_types */
	NULL,                   /* blist_node_menu */
	NULL,                   /* chat_info */
	NULL,                   /* chat_info_defaults */
	facebookim_login,       /* login */
	facebookim_close,       /* close */
	facebookim_send_im,     /* send_im */
	NULL,                   /* set_info */
	facebookim_send_typing, /* send_typing */
	facebookim_get_info,    /* get_info */
	facebookim_set_status_p,/* set_status */
	NULL,                   /* set_idle */
	NULL,                   /* change_passwd */
	facebookim_add_buddy,   /* add_buddy */
	NULL,                   /* add_buddies */
	NULL,                   /* remove_buddy */
	NULL,                   /* remove_buddies */
	NULL,                   /* add_permit */
	NULL,                   /* add_deny */
	NULL,                   /* rem_permit */
	NULL,                   /* rem_deny */
	NULL,                   /* set_permit_deny */
	NULL,                   /* join_chat */
	NULL,                   /* reject chat invite */
	NULL,                   /* get_chat_name */
	NULL,                   /* chat_invite */
	NULL,                   /* chat_leave */
	NULL,                   /* chat_whisper */
	NULL,                   /* chat_send */
	NULL,                   /* keepalive */
	NULL,                   /* register_user */
	NULL,                   /* get_cb_info */
	NULL,                   /* get_cb_away */
	NULL,                   /* alias_buddy */
	NULL,                   /* group_buddy */
	NULL,                   /* rename_group */
	facebookim_buddy_free,  /* buddy_free */
	NULL,                   /* convo_closed */
	purple_normalize_nocase,/* normalize */
	NULL,                   /* set_buddy_icon */
	NULL,                   /* remove_group */
	NULL,                   /* get_cb_real_name */
	NULL,                   /* set_chat_topic */
	NULL,                   /* find_blist_chat */
	NULL,                   /* roomlist_get_list */
	NULL,                   /* roomlist_cancel */
	NULL,                   /* roomlist_expand_category */
	NULL,                   /* can_receive_file */
	NULL,                   /* send_file */
	NULL,                   /* new_xfer */
	NULL,                   /* offline_message */
	NULL,                   /* whiteboard_prpl_ops */
	NULL,                   /* send_raw */
	NULL,                   /* roomlist_room_serialize */
	NULL,                   /* unregister_user */
	NULL,                   /* send_attention */
	NULL,                   /* attention_types */
	(gpointer)sizeof(PurplePluginProtocolInfo), /* struct_size */
	facebookim_get_account_text_table, /* get_account_text_table */
};

static PurplePluginInfo info = {
	PURPLE_PLUGIN_MAGIC,
/*	PURPLE_MAJOR_VERSION,
	PURPLE_MINOR_VERSION,
*/
	2, 1,
	PURPLE_PLUGIN_PROTOCOL, /* type */
	NULL, /* ui_requirement */
	0, /* flags */
	NULL, /* dependencies */
	PURPLE_PRIORITY_DEFAULT, /* priority */
	"prpl-bigbrownchunx-facebookim", /* id */
	"Facebook Chat", /* name */
	"0.1", /* version */
	"Connects to the Facebook Chat protocol", /* summary */
	"Connects to the Facebook Chat protocol", /* description */
	"Eion Robb <eionrobb@gmail.com>", /* author */
	"http://pidgin-facebookchat.googlecode.com/", /* homepage */
	plugin_load, /* load */
	plugin_unload, /* unload */
	NULL, /* destroy */
	NULL, /* ui_info */
	&prpl_info, /* extra_info */
	NULL, /* prefs_info */
	facebookim_actions, /* actions */
	NULL, /* padding */
	NULL,
	NULL,
	NULL
};


PURPLE_INIT_PLUGIN(facebookim, plugin_init, info);


