/*
  Copyright (c) 2007-2008 Austin Che

  Command-line utilities for doing window related things
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#define NUM_ELEM(x) ((int)(sizeof(x) / sizeof(x[0])))

typedef int (*command_main) (int argc, char **argv);
struct command_map
{
    char *name;
    char *shortname;
    command_main main;
    int min_args;
};

static const char *program_name = "powerwindows"; // name of this program

static int usage()
{
    printf("Usage: %s <cmd> [arguments]\n", program_name);
    printf("\nValid commands:\n");

    printf("active\t\tPrints the X window id of the currently active window\n");
    printf("list\t\tPrints a list of top level windows in stacking order\n");

    return 1;
}

static int do_get_active_window(int argc, char *argv[])
{
    GdkScreen *screen = gdk_screen_get_default();
    GdkWindow *active = gdk_screen_get_active_window(screen);
    if (!active)
    {
        g_printerr("Could not get active window\n");
        return 1;
    }

    printf("0x%.8lx\n", GDK_WINDOW_XID(active));
    g_object_unref(active);
    return 0;
}

static int do_list_windows(int argc, char *argv[])
{
    GdkScreen *screen = gdk_screen_get_default();
    GList *windows = gdk_screen_get_window_stack(screen);
    GList *ptr = windows;
    while (ptr)
    {
        GdkWindow *win = ptr->data;
        printf("0x%.8lx\n", GDK_WINDOW_XID(win));
        g_object_unref(win);
        ptr = g_list_next(ptr);
    }
    g_list_free(windows);
    return 0;
}

int main(int argc, char *argv[])
{
    struct command_map command_table[] =
    {
        { "active", "-a", do_get_active_window, 0 },
        { "list", "-l", do_list_windows, 0 },
    };
    char *cmd;
    int i;

    gtk_init(&argc, &argv);

    program_name = argv[0];     // save program name for usage()
    if (argc == 1)
        return usage();
    else
    {
        cmd = argv[1];
        // remove the program and command name
        argc -= 2; 
        argv += 2;
    }

    // find what command was given and dispatch to that handler
    for (i = 0; i < NUM_ELEM(command_table); i++)
    {
        if ((strcmp(cmd, command_table[i].name) == 0) ||
            (command_table[i].shortname && (strcmp(cmd, command_table[i].shortname) == 0)))
        {
            if (argc < command_table[i].min_args)
                return usage();
            else
                return (command_table[i].main)(argc, argv);
            return 0;
        }
    }

    return usage();
}
