/*
  Copyright (c) 2007 Austin Che

  Command-line utilities for doing various things
*/
#define MAEMO_CHANGES

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <X11/extensions/XTest.h>

#ifdef HAVE_CONIC
#include <conic.h>
#endif

#define NUM_ELEM(x) ((int)(sizeof(x) / sizeof(x[0])))

typedef int (*command_main) (int argc, char **argv);
struct command_map
{
    char *name;
    char *shortname;
    command_main main;
    int min_args;
};

static const char *program_name = "powertool"; // name of this program

static int usage()
{
    printf("Usage: %s <cmd> [arguments]\n", program_name);
    printf("\nValid commands:\n");

    printf("keyboard [show|hide]\t\tShow or hide the keyboard\n");
#ifdef HAVE_CONIC
    printf("wlan [on|off]       \t\tConnect or disconnect from wlan\n");
#endif
    return 1;
}

static int do_send_key(int argc, char *argv[])
{
    if (argc == 0)
        return usage();

    int event_base, error_base, major_version, minor_version;
    if (! XTestQueryExtension(gdk_display, &event_base, &error_base, &major_version, &minor_version))
    {
        g_critical("XTest not supported");
        return 1;
    }

    char *text = argv[0];

    while (*text)
    {
        KeyCode code = XKeysymToKeycode(gdk_display, *text);
        g_debug("Sending fake key '%c' with code %d", *text, code);
        XTestFakeKeyEvent(gdk_display, code, True, CurrentTime); /* key press */
        XTestFakeKeyEvent(gdk_display, code, False, CurrentTime); /* key release */
        text++;
    }
    XFlush(gdk_display);

    return 0;
}

static int do_keyboard(int argc, char *argv[])
{
    int event_base, error_base, major_version, minor_version;
    if (! XTestQueryExtension(gdk_display, &event_base, &error_base, &major_version, &minor_version))
    {
        g_critical("XTest not supported");
        return 1;
    }

    GdkScreen *screen = gdk_screen_get_default();
    GdkWindow *active = gdk_screen_get_active_window(screen);
    if (!active)
    {
        g_critical("Could not get active window");
        return 1;
    }

    // this doesn't work properly, shows the keyboard but keys not sent properly

    GtkWindow *window = GTK_WINDOW(g_object_new(GTK_TYPE_WINDOW,
                                                "decorated", FALSE,
                                                "default-height", 0,
                                                "default-width", 0,
                                                "type-hint", GDK_WINDOW_TYPE_HINT_UTILITY,
                                                NULL));

    //g_signal_connect(G_OBJECT(window), "focus_out_event", G_CALLBACK(event_focus_change), NULL);

    gpointer widget_class = g_type_class_ref(GTK_TYPE_WIDGET); // have to first load this to avoid errors
	GtkIMContext *im_context = gtk_im_multicontext_new(); // the context for the keyboard input method
    gtk_widget_show_all(GTK_WIDGET(window));
    gtk_im_context_set_client_window(im_context, GTK_WIDGET(window)->window);

    if (strcmp(argv[0], "hide") == 0)
        hildon_gtk_im_context_hide(im_context);
    else
        hildon_gtk_im_context_show(im_context);

    //XTestFakeKeyEvent(gdk_display, keycode, is_press, CurrentTime);

    gtk_main();
    g_object_unref(active);
    g_object_unref(widget_class);
    //g_object_unref(im_context);
    return 0;
}

#ifdef HAVE_CONIC
static void network_connected(ConIcConnection *cnx, ConIcConnectionEvent *event, gpointer user_data)
{
    switch (con_ic_connection_event_get_status(event)) 
    {
        case CON_IC_STATUS_CONNECTED:
            break;
        case CON_IC_STATUS_DISCONNECTED:
        case CON_IC_STATUS_DISCONNECTING:
        default:
            break;
    }
}

static int do_wlan(int argc, char *argv[])
{
    g_debug("wlan");
    ConIcConnection *con = con_ic_connection_new();
    if (!con)
        g_critical("Could not create conic object");

    //g_signal_connect(ic, "connection-event", (GCallback)network_connected, NULL);

    GSList *list = con_ic_connection_get_all_iaps(con);
    GSList *ptr = list;
    while (ptr)
    {
        ConIcIap *iap = ptr->data;
        printf("%s\t%s\t%s\n", con_ic_iap_get_bearer_type(iap), con_ic_iap_get_id(iap), con_ic_iap_get_name(iap));
        ptr = ptr->next;
        g_object_unref(iap);
    }
    g_slist_free(list);

    if (strcmp(argv[0], "on") == 0)
        con_ic_connection_connect(con, CON_IC_CONNECT_FLAG_UNMANAGED);
    else
        con_ic_connection_disconnect(con); /* cannot disconnect if we didn't connect, and this crashes */
    g_object_unref(con);
    return 0;
}
#endif

int main(int argc, char *argv[])
{
    struct command_map command_table[] =
    {
        { "keyboard", "-k", do_keyboard, 1 },
        { "sendkey", "-s", do_send_key, 1 },
#ifdef HAVE_CONIC
        { "wlan", "-w", do_wlan, 1 },
#endif
    };
    char *cmd;
    int i;

    gtk_init(&argc, &argv);

    program_name = argv[0];     // save program name for usage()
    if (argc == 1)
        return usage();
    else
    {
        cmd = argv[1];
        // remove the program and command name
        argc -= 2; 
        argv += 2;
    }

    // find what command was given and dispatch to that handler
    for (i = 0; i < NUM_ELEM(command_table); i++)
    {
        if ((strcmp(cmd, command_table[i].name) == 0) ||
            (command_table[i].shortname && (strcmp(cmd, command_table[i].shortname) == 0)))
        {
            if (argc < command_table[i].min_args)
                return usage();
            else
                return (command_table[i].main)(argc, argv);
            return 0;
        }
    }

    return usage();
}
