/*
  Copyright (c) 2007-2008 Austin Che

  For communicating with powered and powerlaunch via dbus
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "power-common.h"

typedef struct 
{
    const gchar *search;
    gchar *current_interface;
    gchar *found_interface;
    gchar *found_method;
} IntrospectParseState;

static void print_dbus_message_args(DBusMessage *msg)
{
    DBusMessageIter iter;
    if (!dbus_message_iter_init(msg, &iter))
        return;

    g_print("Reply:\n"); 

    gchar *str;
    gboolean b;
    guint32 ui32;
    gint32 i32;
    double d;
    
    int type;
    int i = 0;
    while ((type = dbus_message_iter_get_arg_type(&iter)) != DBUS_TYPE_INVALID)
    {
        g_print("  %d: ", i);
        switch (type)
        {
            case DBUS_TYPE_STRING:
                dbus_message_iter_get_basic(&iter, &str);
                g_print("string - %s", str);
                break;

            case DBUS_TYPE_BOOLEAN:
                dbus_message_iter_get_basic(&iter, &b);
                g_print("boolean - %s", b ? "TRUE" : "FALSE");
                break;

            case DBUS_TYPE_UINT32:
                dbus_message_iter_get_basic(&iter, &ui32);
                g_print("uint32 - %u", ui32);
                break;

            case DBUS_TYPE_INT32:
                dbus_message_iter_get_basic(&iter, &i32);
                g_print("int32 - %d", i32);
                break;

            case DBUS_TYPE_DOUBLE:
                dbus_message_iter_get_basic(&iter, &d);
                g_print("double - %f", d);
                break;

            case DBUS_TYPE_INVALID:
                g_print("invalid dbus type");
                break;

            default:
                g_print("unhandled dbus type %d", type);
                break;
        }
        g_print("\n");

        dbus_message_iter_next (&iter);
        i++;
    }
}

static gboolean send_request(DBusConnection *dbus, const gchar *service, const gchar *object, const gchar *interface, const gchar *method, gchar **args)
{
    // returns FALSE if we were not able to successfully send the message

    //g_debug("sending message to %s %s %s.%s", service, object, interface, method);

    gboolean ret = TRUE;
    DBusMessageIter iter;
    DBusMessage *msg = dbus_message_new_method_call(service, object, interface, method);
    if (! msg)
    {
        g_printerr("Failed to create dbus message\n");
        exit(1);
    }

    dbus_message_iter_init_append(msg, &iter);
    while (args && *args)
    {
        if (!dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, args))
        {
            g_printerr("Could not append argument for dbus message\n");
            exit(1);
        }
        args++;
    }

    DBusPendingCall *pending = NULL;
    if (!dbus_connection_send_with_reply(dbus, msg, &pending, -1) || !pending)
    { 
        g_printerr("Failed to send dbus message\n");
        exit(1);
    }

    dbus_connection_flush(dbus);
    dbus_message_unref(msg);

    dbus_pending_call_block(pending);
    msg = dbus_pending_call_steal_reply(pending);
    if (!msg)
    {
        g_printerr("Missing reply message\n"); 
        exit(1); 
    }
    dbus_pending_call_unref(pending);

    if (dbus_message_get_type(msg) == DBUS_MESSAGE_TYPE_ERROR)
        ret = FALSE;
    print_dbus_message_args(msg);
    dbus_message_unref(msg);
    return ret;
}

static const gchar *find_name_attribute(const gchar **attribute_names, const gchar **attribute_values)
{
    while (*attribute_names)
    {
        if (strcmp(*attribute_names, "name") == 0)
            return *attribute_values;
        attribute_names++;
        attribute_values++;
    }
    return NULL;
}

static void introspect_open_tag(GMarkupParseContext *context, const gchar *element_name, const gchar **attribute_names, 
                                const gchar **attribute_values, gpointer user_data, GError **error)
{
    IntrospectParseState *state = (IntrospectParseState *)user_data;

    if (strcmp(element_name, "method") == 0)
    {
        const gchar *name = find_name_attribute(attribute_names, attribute_values);
        if (!name)
            return;
        if (strcasecmp(name, state->search) == 0)
        {
            state->found_interface = g_strdup(state->current_interface);
            state->found_method = g_strdup(name);
        }
    }
    else if (strcmp(element_name, "interface") == 0)
    {
        const gchar *name = find_name_attribute(attribute_names, attribute_values);
        if (!name)
            return;

        if (state->current_interface)
            g_free(state->current_interface);
        state->current_interface = g_strdup(name);
    }
}

static gboolean find_matching_method(gchar *introspection, const gchar *requested, gchar **interface, gchar **method)
{
    // tries to find a method in the introspection data that matches the requested name ignoring case
    // returns true if a method is found and the interface and method name are set
    GMarkupParser parser = {
        introspect_open_tag,    /* open tags */
        NULL,                   /* close tags */
        NULL,                   /* text */
        NULL,                   /* passthrough */
        NULL,                   /* error handler */
    };

    IntrospectParseState state;
    memset(&state, 0, sizeof(state));
    state.search = requested;

    gboolean ret = FALSE;

    GMarkupParseContext *context = g_markup_parse_context_new(&parser, 0, &state, NULL);
    GError *error = NULL;
    if (!g_markup_parse_context_parse(context, introspection, strlen(introspection), &error))
    {
        g_printerr("Could not parse introspection data: %s", error->message);
		g_error_free(error);
    }

    g_free(state.current_interface);
    if (state.found_interface)
    {
        ret = TRUE;
        *interface = state.found_interface;
        *method = state.found_method;
    }

    g_markup_parse_context_free(context);
    return ret;
}

static void send_to_powerlaunch(DBusConnection *dbus, const gchar *request, gchar **args)
{
    if (!send_request(dbus, POWERLAUNCH_SERVICE, POWERLAUNCH_OBJECT_PATH, POWERLAUNCH_INTERFACE, request, args))
        g_printerr("Could not send request %s to powerlaunch\n", request);
}

static void handle_request(DBusGConnection *gdbus, const gchar *request, gchar **args)
{
    gchar *introspect_data = NULL;
    GError *error = NULL;
    DBusConnection *dbus = dbus_g_connection_get_connection(gdbus);
    
    // we first try sending to powered
    // if that fails (e.g. the method doesn't exist), we send to powerlaunch

	DBusGProxy *powered_introspect = dbus_g_proxy_new_for_name(gdbus, POWERED_SERVICE, POWERED_OBJECT_PATH, DBUS_INTERFACE_INTROSPECTABLE);
    if (! org_freedesktop_DBus_Introspectable_introspect(powered_introspect, &introspect_data, &error))
    {
        g_printerr("Could not get introspection data for powered: %s\n", error->message);
		g_error_free(error);
        send_to_powerlaunch(dbus, request, args);
        return;
    }

    if (strcasecmp(request, "introspect") == 0)
        g_print(introspect_data);
    else
    {
        gchar *interface = NULL;
        gchar *method = NULL;
        if (find_matching_method(introspect_data, request, &interface, &method))
        {
            if (!send_request(dbus, POWERED_SERVICE, POWERED_OBJECT_PATH, interface, method, args))
                g_printerr("Error sending request to powered\n");
            g_free(interface);
            g_free(method);
        }
        else
            send_to_powerlaunch(dbus, request, args);
    }
    g_free(introspect_data);
}

int main(int argc, char *argv[])
{
    g_type_init();

    if (argc == 1)
    {
        g_printerr("Usage: %s request [args]\n", argv[0]);
        return 1;
    }

	GError *error = NULL;
    DBusGConnection *gdbus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (! gdbus) 
    {
        g_printerr("Failed to open connection to bus: %s\n", error->message);
		g_error_free(error);
		exit(1);
	}

    handle_request(gdbus, argv[1], argv+2);

    dbus_g_connection_unref(gdbus);
    return 0;
}
