/*
  Copyright (c) 2007-2008 Austin Che  

  Handles actions and events dealing with GUI widgets
*/

#include "powerlaunch.h"
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <unistd.h>

#if HILDON == 2008
#  include <hildon/hildon-banner.h>
#  include <hildon/hildon-note.h>
#elif HILDON == 2007
#  include <hildon-widgets/hildon-banner.h>
#  include <hildon-widgets/hildon-note.h>
#endif

#define TOP_WINDOW_NAME "__window__"// *** get rid of this
#define GLADE_KEY "__.GLADE.__" // choose a name not likely to be used as a widget name

typedef struct
{
    gulong click_hook_id;
    guint button_press_signal_id;
    GtkWindow *hidden;          // default hidden window used for modes that don't have top-level windows
    GHashTable *windows;        // string (window name) -> GtkWindow * (we own GtkWindow)

    GtkWidget *visible;        // the visible widget
} GUI_Info;

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static GUI_Info *gui = NULL;

/* ******************************************************************* 
   Helper functions
   ******************************************************************* */

static void grab_all_input(GtkWidget *widget)
{
    // grab pointer and keyboard
    while (1)
    {
        if (gdk_pointer_grab(widget->window, TRUE, GDK_BUTTON_PRESS_MASK|GDK_BUTTON_RELEASE_MASK|GDK_POINTER_MOTION_MASK, widget->window, NULL, GDK_CURRENT_TIME) == GDK_GRAB_SUCCESS)
        {
            if (gdk_keyboard_grab(widget->window, TRUE, GDK_CURRENT_TIME) == GDK_GRAB_SUCCESS)
                break;
            
            gdk_pointer_ungrab(GDK_CURRENT_TIME);
        }
        g_usleep(50 * 1000);
    }
}

static void ungrab_input()
{
    gdk_keyboard_ungrab(GDK_CURRENT_TIME);
    gdk_pointer_ungrab(GDK_CURRENT_TIME);
}

static gboolean widget_supports_lists(GtkWidget *widget)
{
    return GTK_IS_COMBO_BOX(widget) || GTK_IS_TREE_VIEW(widget) || GTK_IS_ICON_VIEW(widget);
}

static GtkWidget *load_widget(const gchar *name)
{
    // loads a widget by name from a glade file
    if (!name)
        return NULL;

    GtkWidget *widget = NULL;
    const gchar *widget_name;
    gchar *ns = NULL;
    gchar *glade_file;
    gboolean free_ns = FALSE;
    gchar *period = strchr(name, '.');
    if (period)
    {
        ns = g_strndup(name, (period - name));
        widget_name = period + 1;
        free_ns = TRUE;
    }
    else
    {
        // if no namespace is given, default is to use the namespace of the current mode
        ns = (gchar *)mode_get_namespace(current_mode());
        widget_name = name;
    }

    glade_file = find_file(ns, ".glade");
    if (glade_file)
    {
        GladeXML *glade = glade_xml_new(glade_file, widget_name, NULL);
        if (! glade)
            g_warning("Could not load widget '%s' in file '%s'", widget_name, glade_file);
        else
        {
            widget = GTK_WIDGET(glade_xml_get_widget(glade, widget_name));
            if (! widget)
            {
                g_warning("Could not get widget '%s.%s'", ns, widget_name);
                g_object_unref(G_OBJECT(glade));
            }
            else
                g_signal_connect_swapped(G_OBJECT(widget), "destroy", G_CALLBACK(g_object_unref), G_OBJECT(glade));
        }
        g_free(glade_file);
    }
    if (free_ns)
        g_free(ns);
    return widget;
}

static GtkWidget *get_widget(const gchar *name)
{
    // tries to find a widget by name
    if (!name)
        return NULL;

    GtkWindow *window = NULL;
    gchar *period = strchr(name, '.');
    if (!period)
    {
        window = g_hash_table_lookup(gui->windows, name);
        if (window)
            return GTK_WIDGET(window);
        else
            return GTK_WIDGET(gui->hidden); // if the window is NULL, we use a shared hidden widget
    }

    // deal with more complicated (nested) widget name specifications
    gchar *dup = g_strdup(name);      // use duplicate to make changes
    gchar *tmp = dup;
    period = strchr(tmp, '.');
    *period = '\0';
    window = g_hash_table_lookup(gui->windows, tmp);
    if (!window)
    {
        g_warning("No such window %s", tmp);
        g_free(dup);
        return NULL;
    }

    GladeXML *glade = g_object_get_data(G_OBJECT(window), GLADE_KEY);
    if (!glade)
    {
        g_warning("Unable to get glade info from window %s", tmp);
        g_free(dup);
        return NULL;
    }

    GtkWidget *widget2;
    GtkWidget *widget = GTK_WIDGET(window);
    while (widget && period)
    {
        tmp = period + 1;
        period = strchr(tmp, '.');
        if (period)
            *period = '\0';

        // first see if we have a child of the name
        widget2 = g_object_get_data(G_OBJECT(widget), tmp);
        if (widget2)
            widget = widget2;
        else
        {
            // else we see if we can look up via glade information
            glade = g_object_get_data(G_OBJECT(widget), GLADE_KEY);
            if (glade)
                widget = glade_xml_get_widget(glade, tmp);
            else
                widget = NULL;
        }
    } 

    g_free(dup);
    return widget;
}

/* ******************************************************************* 
   Callbacks
   ******************************************************************* */

static void hidden_window_event_realize(GtkWidget *widget, gpointer data)
{
    // move hidden window off the screen so it's really hidden
    gtk_window_move(GTK_WINDOW(widget), -1, -1);
}

static gboolean event_keyrelease(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    LaunchMode *mode = current_mode();

    g_debug("Mode: %s -- Key released : 0x%x (%s)", mode_get_name(mode), event->keyval, gdk_keyval_name(event->keyval));

    // make handler name based on the keyval
    gchar buf[24];
    g_snprintf(buf, sizeof(buf), "key_release_%x", event->keyval);
    mode_run_handler(mode, buf);

    return FALSE;
}

static gboolean event_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
    LaunchMode *mode = current_mode();

    g_debug("Mode: %s -- Key pressed : 0x%x (%s)", mode_get_name(mode), event->keyval, gdk_keyval_name(event->keyval));

    // make handler name based on the keyval
    gchar buf[24];
    g_snprintf(buf, sizeof(buf), "key_press_%x", event->keyval);
    mode_run_handler(mode, buf);

    // we currently always return false to let other widgets such as treeviews handle keys also
    return FALSE;
}

static gboolean event_focus_change(GtkWidget *widget, GdkEventFocus *event, gpointer data)
{
    if (event->in)
    {
        // gained focus
        if (gui->visible != widget)
            g_warning("Widget lost focus without ever receiving lost focus signal");
        gui->visible = widget;
    }
    else
    {
        // lost focus
        if (gui->visible == widget)
            gui->visible = NULL;
        else
            g_warning("Widget received lost focus event but didn't have focus!");
    }
    return FALSE;    
}

static gboolean click_signal_handler(GSignalInvocationHint *ihint, guint n_param_values, const GValue *param_values, gpointer data)
{
    // need to have mode passed in and run handler for appropriate mode here...
    GObject *object = g_value_get_object(param_values + 0);
    GtkWidget *widget = GTK_WIDGET(object);
    const gchar *name = gtk_widget_get_name(widget);
    g_debug("Received click for widget %s", name);
    launcher_run_handler(name);
    return TRUE;                // if we return FALSE, this signal hook will be disconnected and destroyed
}

/* ******************************************************************* 
   Action handlers
   ******************************************************************* */

#ifdef HILDON
static void run_action_banner(ArgumentList *args)
{
    const gchar *message = get_argument_str(args, 0);
    const gchar *icon = store_get_variable("banner_icon");

    if (!message)
        g_warning("NULL passed to show banner");
    else
    {
        // passing NULL as the first argument makes it treated as a system message 
        // and not associated with any of our windows which is what we need
        // as our windows can disappear (such as when we're reloading)
        hildon_banner_show_information_with_markup(NULL, icon, message);
    }
}

static void run_action_confirm(ArgumentList *args)
{
    const gchar *message = get_argument_str(args, 0);
    if (!message)
        warn_and_return("NULL passed to confirm dialog");

    gboolean confirm = FALSE;
    GtkWidget *dialog = hildon_note_new_confirmation(NULL, message);
    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_OK)
        confirm = TRUE;
    gtk_widget_destroy(dialog);
    if (confirm)
        launcher_run_handler(get_argument_str(args, 1));
    else if (get_num_args(args) > 2)
        launcher_run_handler(get_argument_str(args, 2));
}
#endif

static void run_action_window_new(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    const gchar *type = get_argument_str(args, 1);

    if (!name || !type)
    {
        g_warning("Empty arguments to window_new!");
        return;
    }

    g_debug("creating new window %s of type %s", name, type);
    if (g_hash_table_remove(gui->windows, name))
        g_warning("Existing window of name %s exists", name);

    if (strcmp(type, "hidden") == 0)
        return;

    GtkWindow *window = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));
    g_hash_table_insert(gui->windows, (gchar *)string_ref_dup(name), window);

    if (strcmp(type, "dialog") == 0)
        gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_DIALOG);
    else if (strcmp(type, "normal") == 0)
        ;                       // do nothing special
    else if (strcmp(type, "center") == 0)
        gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
    else if (strcmp(type, "bottom") == 0)
        gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_TOOLBAR);
    else if (strcmp(type, "top") == 0)
        gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_DOCK);
    else if (strcmp(type, "fullmodal") == 0)
    {
        // a fullscreen 'dialog' window that's always on top
        GdkScreen *screen = gdk_screen_get_default();
        gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_DIALOG);
        gtk_window_set_default_size(window, gdk_screen_get_width(screen), gdk_screen_get_height(screen));
        gtk_window_set_decorated(window, FALSE);
    }
    else
    {
        if (strcmp(type, "fullscreen") != 0)
            g_warning("Unknown layout type '%s'. Using a fullscreen window", type);
        // use this hint to get it to NOT show up in the task switcher
        gtk_window_set_type_hint(window, GDK_WINDOW_TYPE_HINT_UTILITY);
        gtk_window_fullscreen(window);
    }

    if (get_num_args(args) > 2)
        gtk_window_set_title(window, get_argument_str(args, 2));

    g_signal_connect(G_OBJECT(window), "key_press_event", G_CALLBACK(event_keypress), NULL);
    g_signal_connect(G_OBJECT(window), "key_release_event", G_CALLBACK(event_keyrelease), NULL);
    g_signal_connect(G_OBJECT(window), "focus_in_event", G_CALLBACK(event_focus_change), NULL);
    g_signal_connect(G_OBJECT(window), "focus_out_event", G_CALLBACK(event_focus_change), NULL);

    gtk_widget_set_name(GTK_WIDGET(window), name);
}

static void run_action_widget_add(ArgumentList *args)
{
    // third argument (new widget name is optional). default is to use the existing name of the widget
    const gchar *container = get_argument_str(args, 0);
    const gchar *child = get_argument_str(args, 1);

    GtkWidget *container_widget = get_widget(container);
    GtkWidget *child_widget = load_widget(child);
    if (!container_widget || !child_widget)
        g_warning("Could not find widget '%s' or '%s'", container, child);
    else
    {
        if (get_num_args(args) > 2)
            gtk_widget_set_name(child_widget, get_argument_str(args, 2));

        if (!GTK_IS_CONTAINER(container_widget))
            g_warning("Widget named '%s' is not a container", container);
        else
        {
            // save the glade xml tree 
            GladeXML *glade = glade_get_widget_tree(child_widget);
            if (glade)
                g_object_set_data(G_OBJECT(container_widget), GLADE_KEY, glade);
            g_object_set_data(G_OBJECT(container_widget), gtk_widget_get_name(child_widget), child_widget);
            gtk_container_add(GTK_CONTAINER(container_widget), child_widget);
        }
    }
}

static void run_action_widget_color(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    GtkWidget *widget = get_widget(name);
    if (!widget)
        g_warning("Could not find widget '%s'", name);
    else
    {
        int i;
        int num = get_num_args(args);
        for (i = 1; i < num; i += 2)
        {
            GdkColor color;
            const gchar *color_name = get_argument_str(args, i+1);
            if (!gdk_color_parse(color_name, &color))
                g_warning("could not parse color string '%s'", color_name);
            else
            {
                const gchar *type = get_argument_str(args, i);
                if (strcmp(type, "bg") == 0)
                    gtk_widget_modify_bg(widget, GTK_STATE_NORMAL, &color);
                else if (strcmp(type, "fg") == 0)
                    gtk_widget_modify_fg(widget, GTK_STATE_NORMAL, &color);
                else if (strcmp(type, "text") == 0)
                    gtk_widget_modify_text(widget, GTK_STATE_NORMAL, &color);
                else if (strcmp(type, "base") == 0)
                    gtk_widget_modify_base(widget, GTK_STATE_NORMAL, &color);
            }
        }
    }
}

static void run_action_widget_focus(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    GtkWidget *widget = get_widget(name);
    if (!widget)
        g_warning("Could not find widget '%s'", name);
    else if (! GTK_WIDGET_CAN_FOCUS(widget))
        g_warning("widget '%s' cannot receive focus", name);
    else
        gtk_widget_grab_focus(widget);
}

static void run_action_widget_hide(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    GtkWidget *widget = get_widget(name);
    if (!widget)
        g_warning("Could not find widget '%s'", name);
    else
    {
        gtk_widget_hide_all(widget);
        if (widget == GTK_WIDGET(gui->hidden))
            ungrab_input();
    }
}

static void run_action_widget_list(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    GtkWidget *widget = get_widget(name);
    if (!widget)
        g_warning("Could not find widget '%s'", name);
    else if (!widget_supports_lists(widget))
        g_warning("Widget '%s' does not support lists", name);
    else
    {
        int num = get_num_args(args);
        int i;

        GtkListStore *store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
        GtkTreeIter iter;
        for (i = 1; i < num-1; i += 2)
        {
            gtk_list_store_append(store, &iter);
            gtk_list_store_set(store, &iter, 0, get_argument_str(args, i), -1);
            gtk_list_store_set(store, &iter, 1, get_argument_str(args, i+1), -1);
        }

        if (GTK_IS_COMBO_BOX(widget))
        {
            GtkComboBox *combo = GTK_COMBO_BOX(widget);
            gtk_combo_box_set_model(combo, GTK_TREE_MODEL(store));

            GtkCellLayout *layout = GTK_CELL_LAYOUT(widget);

            gtk_cell_layout_clear(layout);

            GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
            gtk_cell_layout_pack_start(layout, renderer, FALSE);
            gtk_cell_layout_set_attributes(layout, renderer, "text", 1, NULL);
        }
        else if (GTK_IS_TREE_VIEW(widget))
        {
            GtkTreeView *treeview = GTK_TREE_VIEW(widget);
            gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));

            GtkCellRenderer *renderer = gtk_cell_renderer_text_new();
            GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes("Title", renderer, "text", 1, NULL);
            // if the treeview already has a column, remove it
            GtkTreeViewColumn *old = gtk_tree_view_get_column(treeview, 0);
            if (old)
                gtk_tree_view_remove_column(treeview, old);
            gtk_tree_view_append_column(treeview, column);
        }
    }
}

static void run_action_widget_set(ArgumentList *args)
{
    // sets the value of a widget
    // widget types have a default property that this action refers to
    const gchar *name = get_argument_str(args, 0);
    const gchar *value = get_argument_str(args, 1);
    GtkWidget *widget = get_widget(name);
    if (!widget)
        g_warning("widget_set: could not find widget '%s'", name);
    else if (!value)
        g_warning("widget_set: cannot set null value for widget '%s'", name);
    else
    {
        if (GTK_IS_LABEL(widget))
            gtk_label_set_label(GTK_LABEL(widget), value);
        else if (GTK_IS_BUTTON(widget))
            gtk_button_set_label(GTK_BUTTON(widget), value);
        else if (GTK_IS_ENTRY(widget))
            gtk_entry_set_text(GTK_ENTRY(widget), value);
        else if (GTK_IS_IMAGE(widget))
            gtk_image_set_from_icon_name(GTK_IMAGE(widget), value, -1);
        else if (GTK_IS_CALENDAR(widget))
        {
            GtkCalendar *calendar = GTK_CALENDAR(widget);
            struct tm *tm = NULL;
            time_t now = time(NULL);
            tm = localtime(&now);
            if (strcmp(value, "now") != 0)
            {
                // try parsing the date string, otherwise use default of 'now'
                GDate *date = g_date_new();
                g_date_set_parse(date, value);
                if (g_date_valid(date))
                    g_date_to_struct_tm(date, tm);
                g_date_free(date);
            }

            gtk_calendar_clear_marks(calendar);
            gtk_calendar_select_month(calendar, tm->tm_mon, tm->tm_year + 1900);
            gtk_calendar_select_day(calendar, tm->tm_mday);
            gtk_calendar_mark_day(calendar, tm->tm_mday);
        }
        else if (GTK_IS_TREE_VIEW(widget))
        {
            GtkTreeView *treeview = GTK_TREE_VIEW(widget);
            GtkTreeModel *model = gtk_tree_view_get_model(treeview);
            GtkTreeIter iter;
            if (! gtk_tree_model_get_iter_from_string(model, &iter, value))
                g_warning("Invalid path selection specification: '%s'", value);
            else
                gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(widget)), &iter);
        }
        else
            g_warning("Widget '%s' is an unknown widget type", name);
    }
}

static void run_action_widget_show(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    GtkWidget *widget = get_widget(name);
    if (!widget)
        g_warning("%s: could not find widget '%s'", mode_get_name(current_mode()), name);
    else
    {
        gtk_widget_show_all(widget);
        if (GTK_IS_WINDOW(widget))
        {
            GtkWindow *window = GTK_WINDOW(widget);
            if (window == gui->hidden)
                grab_all_input(GTK_WIDGET(gui->hidden));
            else
                gtk_window_present(window);
        }
    }
}

/* ******************************************************************* 
   Static functions
   ******************************************************************* */

static GtkWindow *create_hidden_window()
{
    GtkWindow *window = GTK_WINDOW(g_object_new(GTK_TYPE_WINDOW,
                                                "decorated", FALSE,
                                                "default-height", 1,
                                                "default-width", 1,
                                                "modal", TRUE,
                                                "type-hint", GDK_WINDOW_TYPE_HINT_UTILITY,
                                                NULL));
    g_signal_connect(GTK_WIDGET(window), "realize", G_CALLBACK(hidden_window_event_realize), NULL);
    g_signal_connect(G_OBJECT(window), "key_press_event", G_CALLBACK(event_keypress), NULL);
    g_signal_connect(G_OBJECT(window), "key_release_event", G_CALLBACK(event_keyrelease), NULL);
    gtk_widget_set_name(GTK_WIDGET(window), TOP_WINDOW_NAME);
    return window;
}

static gboolean data_widget_getval(ActionData *data, GValue *val)
{
    const gchar *widget_name = data->val.v_str;
    GtkWidget *widget = get_widget(widget_name);
    if (!widget)
        warn_and_return_false("Could not find widget named '%s'", widget_name);

    g_value_init(val, G_TYPE_STRING);

    if (GTK_IS_TREE_VIEW(widget))
    {
        GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));
        GtkTreeModel *model = NULL;
        GtkTreeIter iter;
        if (gtk_tree_selection_get_selected(selection, &model, &iter))
        {
            gchar *text;
            gtk_tree_model_get(model, &iter, 0, &text, -1);
            g_value_take_string(val, text);
        }
    }
    else if (GTK_IS_ENTRY(widget))
    {
        g_value_set_static_string(val, gtk_entry_get_text(GTK_ENTRY(widget)));
    }
    else
        warn_and_return_false("Cannot get value of widget '%s' (unhandled type)", widget_name);

    return TRUE;
}

/* ******************************************************************* 
   Exported Functions
   ******************************************************************* */

void widget_init()
{
    static ActionDataType _ACTION_DATA_TYPE_WIDGET = {
        data_widget_getval,
        data_string_free,
        TRUE,
    };

    g_debug("Initializing widget subsystem");

    if (gui)
    {
        g_warning("GUI system already initialized!");
        widget_uninit();
    }

    // Don't load gtk as root due to security issues
    if (getuid() == 0)
        return;

    action_data_subst_define("widget", &_ACTION_DATA_TYPE_WIDGET);
#ifdef HILDON
    action_define("banner", run_action_banner, 1, FALSE);
    action_define("confirm", run_action_confirm, 2, TRUE);    
#endif
    action_define("widget_add", run_action_widget_add, 2, TRUE);
    action_define("widget_color", run_action_widget_color, 3, TRUE);
    action_define("widget_focus", run_action_widget_focus, 1, FALSE);
    action_define("widget_hide", run_action_widget_hide, 1, FALSE);
    action_define("widget_list", run_action_widget_list, 2, TRUE);
    action_define("widget_set", run_action_widget_set, 2, FALSE);
    action_define("widget_show", run_action_widget_show, 1, FALSE);
    action_define("window_new", run_action_window_new, 2, TRUE);

    gui = g_new0(GUI_Info, 1);

    gtk_init_check(0, NULL);
    gpointer g_class = g_type_class_ref(GTK_TYPE_WIDGET); // have to first load the type class to add the signal below
    gui->button_press_signal_id = g_signal_lookup("button-press-event", GTK_TYPE_WIDGET);
    gui->click_hook_id = g_signal_add_emission_hook(gui->button_press_signal_id, 0, click_signal_handler, NULL, NULL);
    g_type_class_unref(g_class);

    gui->hidden = create_hidden_window();
    gui->windows = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)gtk_widget_destroy);
}

void widget_uninit()
{
    g_signal_remove_emission_hook(gui->button_press_signal_id, gui->click_hook_id);
    gtk_widget_destroy(GTK_WIDGET(gui->hidden));
    g_hash_table_destroy(gui->windows);
    g_free(gui);
    gui = NULL;
}
