/*
  Copyright (c) 2007 Austin Che  
*/

#include "powerlaunch.h"
#include <string.h>
#include <unistd.h>

typedef struct
{
    GHashTable *variables;      // string->string mappings (both keys and values need to be freed)
    GHashTable *str_refs;       // string->int stores strings and a reference count. when reaches 0, frees the string
    GSList *temp_strings;     // store references of temporary strings
} MemoryStore;

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static MemoryStore *store = NULL;

/* ******************************************************************* 
   Misc
   ******************************************************************* */

GList *add_or_move_to_front(GList *list, gpointer data)
{
    GList *elem = g_list_find(list, data);
    if (elem)
    {
        // if is already in the list, we move it to the front
        list = g_list_remove_link(list, elem);
        list = g_list_concat(elem, list);
    }
    else
        list = g_list_prepend(list, data);
    return list;
}

/* ******************************************************************* 
   Memory Functions
   ******************************************************************* */

static void store_print_str_refs(gpointer key, gpointer value, gpointer data)
{
    g_warning("  count %d, string %s\n", GPOINTER_TO_INT(value), (gchar *)key);
}

void store_init()
{
    store = g_new0(MemoryStore, 1);

    store->variables = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)string_unref);
    store->str_refs = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
}

void store_uninit()
{
    if (store->variables)
        g_hash_table_destroy(store->variables);

    if (store->str_refs)
    {
        // the str_refs should be empty at this point if we've been good about unref'ing everything
        if (g_hash_table_size(store->str_refs) != 0)
        {
            g_warning("Size of string store is not zero! %d strings are left:", g_hash_table_size(store->str_refs));
            g_hash_table_foreach(store->str_refs, store_print_str_refs, NULL);
        }
        else
            g_debug("Clean unref of all strings");
        g_hash_table_destroy(store->str_refs);
    }

    if (store->temp_strings)
    {
        g_warning("Non-freed temporary strings exist");
        temp_strings_free();
    }

    g_free(store);
    store = NULL;
}

static const gchar *_string_ref(const gchar *str, gboolean string_owned)
{
    g_return_val_if_fail(str != NULL, NULL);

    gpointer origkey;
    gpointer data;
    if (g_hash_table_lookup_extended(store->str_refs, str, &origkey, &data))
    {
        // already exists so increment count and return the existing ref
        // free the passed in key only if it's different from the key we have

        guint count = GPOINTER_TO_INT(data);

        if (str != origkey && string_owned)
            g_free((char *)str);

        g_hash_table_steal(store->str_refs, origkey);
        g_hash_table_insert(store->str_refs, origkey, GINT_TO_POINTER(count+1));
        return origkey;
    }
    else
    {
        // new string, initialize count at 1
        if (! string_owned)
            str = g_strdup(str);

        g_hash_table_insert(store->str_refs, (gchar *)str, GINT_TO_POINTER(1));
        return str;
    }
}

const gchar *string_ref_dup(const gchar *str)
{
    // like string_ref but it will not change the passed in string
    // this is for when the caller does not own the string but wants a string ref
    // We will automatically dup the string if necessary and return a good string ref
    return _string_ref(str, FALSE);
}

const gchar *string_ref(gchar *str)
{
    // gets a reference to a string equal to the passed in value
    // returns the string that the caller should use
    // the passed in string may be freed so the caller must not use it again
    // when not needed anymore, string_unref should be called to decrement the reference count
    // note that passed back strings should not be changed or else strange things may happen
    return _string_ref(str, TRUE);
}

void string_unref(const gchar *str)
{
    // decrement the reference count for the passed in string
    // even though the argument is declared const, this is only for calling convenience
    // the caller must not assume the string is still valid after unrefing it
    g_return_if_fail(str != NULL);

    gpointer origkey;
    gpointer data;
    if (g_hash_table_lookup_extended(store->str_refs, str, &origkey, &data))
    {
        guint count = GPOINTER_TO_INT(data);
        g_return_if_fail(origkey == str);

        if (count == 1)
        {
            // can free the string now
            g_hash_table_remove(store->str_refs, origkey);
        }
        else
        {
            g_hash_table_steal(store->str_refs, origkey);
            g_hash_table_insert(store->str_refs, origkey, GINT_TO_POINTER(count-1));
        }
    }
    else
        g_warning("Memory problem: string '%s' being unref'd too many times", str);
}

const gchar *store_temp_string(gchar *str)
{
    // str should be owned by caller and after call, we own the string
    store->temp_strings = g_slist_prepend(store->temp_strings, str);
    return str;
}

void temp_strings_free()
{
    // free all strings in the temporary string list
    if (store->temp_strings)
    {
        g_slist_foreach(store->temp_strings, (GFunc)g_free, NULL);
        g_slist_free(store->temp_strings);
        store->temp_strings = NULL;
    }
}

const gchar *store_get_variable(const gchar *name)
{
    return g_hash_table_lookup(store->variables, name);
}

void store_set_variable(const gchar *name, const gchar *val)
{
    // both name and val are owned and remained owned by caller
    // if val == NULL, unsets a variable
    if (!val)
        g_hash_table_remove(store->variables, name);
    else
        g_hash_table_insert(store->variables, (gchar *)string_ref_dup(name), (gchar *)string_ref_dup(val));
}
