/*
  Copyright (c) 2007-2008 Austin Che

  Entry point/Initialization
*/

#include "powerlaunch.h"
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <string.h>

#define MAX_ERROR_RELOADS 5

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static gboolean launcher_reload = TRUE;

static int error_reloads = 0;
static gchar *initial_mode = NULL;
static gchar *initial_parent_mode = NULL;
static gchar *setuid_user = NULL;
static GMainLoop *main_loop = NULL;

static GOptionEntry program_options[] = {
    { "mode", 'm', 0, G_OPTION_ARG_STRING, &initial_mode, "Initial mode (default is powerlaunch.main)", "<mode>" },
    { "parent", 'p', 0, G_OPTION_ARG_STRING, &initial_parent_mode, "Initial mode for parent process when forking (default is none)", "<mode>" },
    { "user", 'u', 0, G_OPTION_ARG_STRING, &setuid_user, "Fork and setuid to this user (must be started as root)", "<username>" },
    { NULL }
};

/* ******************************************************************* 
   Static Functions
   ******************************************************************* */

static const gchar *get_program_version()
{
    return PACKAGE_VERSION;
}

static void run_action_quit(ArgumentList *args)
{
    g_debug("quitting");
    g_main_loop_quit(main_loop);
}

static void run_action_reload(ArgumentList *args)
{
    // quit gracefully from the main loop (handle all pending events) before reloading
    g_debug("reloading powerlaunch");

    launcher_reload = TRUE;
    g_main_loop_quit(main_loop);
}

static void parse_args(int argc, char **argv)
{
    GOptionContext *context = g_option_context_new("- launch programs and respond to events");
    g_option_context_set_help_enabled(context, TRUE);
    g_option_context_add_main_entries(context, program_options, NULL);
    if (! g_option_context_parse(context, &argc, &argv, NULL))
        g_warning("Error parsing arguments. Use --help for help.");
    g_option_context_free(context);
}

static void my_log_critical_handler(const gchar *log_domain, GLogLevelFlags log_level, 
                                    const gchar *message, gpointer user_data)
{
    // on a 'critical' error where it's not a fatal error (and don't want to abort if possible)
    g_log_default_handler(log_domain, log_level, message, user_data);
    error_reloads++;
    g_printerr("Reloading program due to critical error");
    if (error_reloads > MAX_ERROR_RELOADS)
    {
        g_printerr("Too many reloads due to critical error. Exiting.");
        exit(1);
    }
    run_action_reload(NULL);
}

static void maybe_fork()
{
    if (getuid() > 0)
    {
        if (setuid_user)
            g_error("Cannot setuid as not running as root!");
        return;
    }
    if (!setuid_user)
        return;

    struct passwd *user = getpwnam(setuid_user);
    if (!user)
        g_error("No such user %s", setuid_user);

    pid_t child = fork();
    if (child < 0)
        g_error("Could not fork()!");
    else if (child == 0)
    {
        // child
        if (setuid(user->pw_uid) < 0)
            g_error("Failed to setuid");
        g_debug("Child: setuid to uid %d", user->pw_uid);
    }
    else
    {
        // parent
        if (!initial_parent_mode)
            exit(0);
        initial_mode = initial_parent_mode;
    }
}

/* ******************************************************************* 
   Main
   ******************************************************************* */

int main(int argc, char *argv[])
{
    g_type_init();
    parse_args(argc, argv);
    maybe_fork();

    g_set_application_name(PACKAGE_NAME);
    g_log_set_handler(NULL, G_LOG_LEVEL_CRITICAL | G_LOG_FLAG_FATAL | G_LOG_FLAG_RECURSION, my_log_critical_handler, NULL);

    main_loop = g_main_loop_new(NULL, FALSE);

    while (launcher_reload)
    {
        launcher_reload = FALSE;

        // initialize all modules

        // these are basic, required modules
        store_init();
        actions_init();
        launcher_init();

        // these are optional
        widget_init();
        power_dbus_init();
#ifdef HAVE_GCONF
        power_gconf_init();
#endif
#ifdef HAVE_HAL
        power_hal_init();
#endif

        // this reload define action won't have any effect!
        action_define("reload", run_action_reload, 0, FALSE);
        action_define("quit", run_action_quit, 0, FALSE);
        action_sys_variable_define("version", get_program_version);

        load_initial_mode(initial_mode);

        g_debug("Begin main loop");
        g_main_loop_run(main_loop);

        // uninitialize everything
#ifdef HAVE_HAL
        power_hal_uninit();
#endif
#ifdef HAVE_GCONF
        power_gconf_uninit();
#endif
        power_dbus_uninit();
        widget_uninit();

        launcher_uninit();
        actions_uninit();
        store_uninit();
    }

    g_main_loop_unref(main_loop);
    return 0;
}
