/*
  Copyright (c) 2007 Austin Che  

  Main launcher code
*/

#include "powerlaunch.h"
#include <string.h>
#include <stdlib.h>
#include <glib.h>
#include <unistd.h>

#define POWERLAUNCH_SYSDIR SYSCONFDIR "/" PACKAGE
#define DEFAULT_NAMESPACE PACKAGE
#define MAIN_MODE "main"
#define INIT_HANDLER "on_init"
#define DEFAULT_MODE PACKAGE "." MAIN_MODE
#define MAX_RECURSION_DEPTH 20

/* ******************************************************************* 
   Type definitions
   ******************************************************************* */

typedef struct _LaunchNamespace
{
    // Everything needs to be freed
    const gchar *name;                // name of namespace
    GHashTable *modes;          // string(mode name)->LaunchMode *
} LaunchNamespace;

struct _LaunchMode
{
    gboolean initialized;

    // Everything below needs to be freed
    const gchar *name;          // fully qualified mode name
    LaunchNamespace *namespace;
    GHashTable *handlers;         // strings(handler name) to ActionList *
    GList *parent_modes;        // list of LaunchMode * of parents
    GHashTable *inherited_handlers_cache;
};

// Main structure holding all the information about the launcher's setup
typedef struct _LauncherConfig
{
    LaunchMode *current;        // should always be valid
    LaunchMode *previous;

    // Things that need to be freed
    // all strings are stored using string_ref and should be unref'd when done

    GHashTable *namespaces;          // string(namespace name)->LaunchNamespace *
    GList *default_inherited_modes;        // list of LaunchMode * of default parents
    GHashTable *default_handlers_cache; // cache of default handlers that all modes share string(name)->Action List *
    GList *search_paths;       // list of paths to search for files
} LauncherConfig;

typedef void (*ModeFn)(LaunchMode *mode);

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static LauncherConfig *launcher = NULL;

/* ******************************************************************* 
   Prototypes
   ******************************************************************* */

static LaunchMode *launcher_get_mode(const gchar *name);
static LaunchNamespace *load_namespace(const char *name);
static ActionList *mode_find_handler(LaunchMode *mode, const gchar *name);

/* ******************************************************************* 
   Callbacks
   ******************************************************************* */

static const gchar *get_current_mode_name()
{
    return launcher->current->name;
}

static const gchar *get_last_mode_name()
{
    if (launcher->previous)
        return launcher->previous->name;
    else
        return "";
}

/* ******************************************************************* 
   Mode Functions
   ******************************************************************* */

static void check_mode_initialized(LaunchMode *mode)
{
    if (! mode->initialized)
    {
        // run init handler for the given mode in the context of that mode
        // so switch temporarily to that mode
        LaunchMode *tmp = launcher->current;
        mode->initialized = TRUE;
        launcher->current = mode;

        // the init handler should not be inherited
        ActionList *actions = g_hash_table_lookup(mode->handlers, INIT_HANDLER);

        actions_run(actions);

        launcher->current = tmp;
    }
}

static void mode_switch(LaunchMode *mode)
{
    g_return_if_fail(mode);

    if (launcher->current == mode)
        return;

    g_debug("switching to mode %s", mode->name);
    launcher->previous = launcher->current;
    launcher->current = mode;

    check_mode_initialized(mode);
}

static void mode_default_inherit(LaunchMode *mode)
{
    g_return_if_fail(mode);
    g_debug("Adding mode %s as default inherited mode", mode->name);
    check_mode_initialized(mode);
    launcher->default_inherited_modes = add_or_move_to_front(launcher->default_inherited_modes, mode);
}

static void mode_current_inherit(LaunchMode *mode)
{
    // adds a mode as a parent of the current mode
    g_return_if_fail(mode);
    g_debug("Adding mode %s as parent to mode %s", mode->name, launcher->current->name);
    check_mode_initialized(mode);
    launcher->current->parent_modes = add_or_move_to_front(launcher->current->parent_modes, mode);
}

static void run_action_mode_function(ArgumentList *args, ModeFn fn)
{
    // handle general actions that take one argument that is a mode name
    const gchar *name = get_argument_str(args, 0);
    if (!name)
        warn_and_return("No mode name");
        
    LaunchMode *mode = launcher_get_mode(name);
    if (!mode)
        g_warning("No such mode: %s", name);
    else
        fn(mode);
}

static void run_action_mode_switch(ArgumentList *args)
{
    return run_action_mode_function(args, mode_switch);
}

static void run_action_default_inherit(ArgumentList *args)
{
    run_action_mode_function(args, mode_default_inherit);
}

static void run_action_inherit(ArgumentList *args)
{
    run_action_mode_function(args, mode_current_inherit);
}

static ActionList *list_find_handler(GList *modes, const gchar *name)
{
    // tries to find a handler of the given name in a list of LaunchModes
    // returns the first handler found
    while (modes)
    {
        // try inherited handlers
        ActionList *actions = mode_find_handler((LaunchMode *)modes->data, name);
        if (actions)
            return actions;
        modes = g_list_next(modes);
    }
    return NULL;
}

static ActionList *mode_find_handler(LaunchMode *mode, const gchar *name)
{
    static int recursing = 0; // detect recursion

    ActionList *actions = g_hash_table_lookup(mode->handlers, name);
    if (actions)
    {
        g_debug("Found handler: %s.%s", mode->name, name);
        return actions;
    }

    if (recursing > MAX_RECURSION_DEPTH)
    {
        g_critical("mode inheritance for %s: recursion detected!", mode->name);
        return NULL;
    }

    recursing++;
    actions = list_find_handler(mode->parent_modes, name);
    
    // we only go to the default modes at the top level if we're not recursing
    // otherwise it can recurse forever
    if (recursing == 1 && !actions)
        actions = list_find_handler(launcher->default_inherited_modes, name);

    recursing--;
    return actions;
}

static LaunchMode *launcher_get_mode_full(const gchar *namespace, const gchar *mode)
{
    LaunchNamespace *ns = g_hash_table_lookup(launcher->namespaces, namespace);
    if (!ns)
        ns = load_namespace(namespace);
    if (!ns)
        return NULL;
    return g_hash_table_lookup(ns->modes, mode);
}

static LaunchMode *launcher_get_mode(const gchar *name)
{
    // check if the mode name is qualified with a namespace (i.e. contains period)
    if (!name)
        return NULL;

    gchar *period = strchr(name, '.');
    gchar *ns = NULL;
    LaunchMode *mode;
    if (period)
    {
        ns = g_strndup(name, (period - name));
        mode = launcher_get_mode_full(ns, period + 1);
        g_free(ns);
    }
    else
    {
        // if no namespace is given, default is to use the namespace of the current mode
        mode = launcher_get_mode_full(launcher->current->namespace->name, name);
    }

    if (!mode)
        g_warning("Mode %s not found", name);
    return mode;    
}

static void mode_free(LaunchMode *mode)
{
    g_hash_table_destroy(mode->handlers);
    string_unref(mode->name);
    g_list_free(mode->parent_modes);
    g_free(mode);
}

static LaunchMode *load_mode(LaunchNamespace *namespace, GKeyFile *config, const gchar *name)
{
    // read one section from the config file
    LaunchMode *mode = g_new0(LaunchMode, 1);
    mode->name = string_ref(g_strconcat(namespace->name, ".", name, NULL)); // make fully qualified name
    mode->namespace = namespace;
    mode->handlers = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)actions_free);

    g_debug("Loading config for mode %s", mode->name);

    // iterate through all the keys
    gsize numkeys;
    gchar **keys = g_key_file_get_keys(config, name, &numkeys, NULL);
    if (! keys)
    {
        g_warning("No such group %s", name); // this shouldn't happen
        return NULL;
    }

    int i;
    for (i = 0; i < numkeys; i++)
    {
        gchar *key = keys[i];
        gsize length;

        //g_debug("Adding handler for '%s' in mode '%s'", key, mode->name);

        gchar **actions_str = g_key_file_get_string_list(config, name, key, &length, NULL);
        ActionList *actions = actions_parse(actions_str, length);
        g_strfreev(actions_str);
        if (! actions)
            continue;

        g_hash_table_insert(mode->handlers, (gchar *)string_ref(key), actions);
    }

    g_free(keys);               // all key strings should have been ref'd as hashtable keys
    return mode;
}

/* ******************************************************************* 
   Namespace Functions
   ******************************************************************* */

static void namespace_free(LaunchNamespace *ns)
{
    g_hash_table_destroy(ns->modes);
    string_unref(ns->name);
    g_free(ns);
}

static LaunchNamespace *load_namespace(const gchar *name)
{
    if (!name)
        return NULL;

    if (g_hash_table_lookup(launcher->namespaces, name))
        return NULL;

    g_debug("Loading namespace: %s", name);
    gchar *conf_filename = find_file(name, ".conf");
    if (!conf_filename)
        warn_and_return_null("Namespace %s not found", name);

    GKeyFile *config = g_key_file_new();
    GError *err = NULL;
    if (! g_key_file_load_from_file(config, conf_filename, G_KEY_FILE_NONE, &err))
    {
        g_warning("Could not load %s: %s", conf_filename, err->message);
        g_error_free(err);
        g_free(conf_filename);
        g_key_file_free(config);
        return NULL;
    }

    gsize num;
    gchar **mode_names = g_key_file_get_groups(config, &num);
    if (num == 0)
        g_warning("No modes defined in %s", conf_filename);    

    LaunchNamespace *namespace = g_new0(LaunchNamespace, 1);
    namespace->name = string_ref_dup(name);
    namespace->modes = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)mode_free);

    LaunchMode *main_mode = NULL;
    int i;
    for (i = 0; i < num; i++)
    {
        const gchar *mode_name = string_ref(mode_names[i]);
        LaunchMode *mode = load_mode(namespace, config, mode_name);
        if (mode)
        {
            g_hash_table_insert(namespace->modes, (gchar *)mode_name, mode);
            if (strcmp(mode_name, MAIN_MODE) == 0)
                main_mode = mode;
        }
    }

    g_key_file_free(config);
    g_debug("Finished reading config file: %s", conf_filename);
    g_free(conf_filename);
    g_free(mode_names);         // should have stored all the strings in the modes

    g_hash_table_insert(launcher->namespaces, (gchar *)string_ref((gchar *)namespace->name), namespace);

    if (main_mode)
        check_mode_initialized(main_mode);

    return namespace;
}

static void run_action_load(ArgumentList *args)
{
    // loads a namespace if it's not already loaded
    const gchar *ns = get_argument_str(args, 0);
    if (!ns)
        g_warning("No namespace to load");
    else
        load_namespace(ns);
}

/* ******************************************************************* 
   Exported Functions
   ******************************************************************* */

gchar *find_file(const gchar *namespace, const gchar *extension)
{
    // returns found filename or NULL
    // if non-NULL is returned, caller is responsible for freeing string
    g_return_val_if_fail(namespace != NULL, NULL);

    gchar *filename = g_strconcat(namespace, extension, NULL);
    gchar *found = NULL;
    gchar *test = NULL;

    GList *dir = launcher->search_paths;
    while (dir)
    {
        test = g_build_filename((gchar *)dir->data, filename, NULL);
        if (g_file_test(test, G_FILE_TEST_IS_REGULAR))
        {
            g_debug("Found file %s", test);
            found = test;
            break;
        }
        g_free(test);
        dir = g_list_next(dir);
    }

    g_free(filename);
    return found;
}

LaunchMode *current_mode()
{
    return launcher->current;
}

const gchar *mode_get_name(LaunchMode *mode)
{
    return mode->name;
}

const gchar *mode_get_namespace(LaunchMode *mode)
{
    return mode->namespace->name;
}

gboolean mode_run_handler(LaunchMode *mode, const gchar *name)
{
    // returns TRUE if a handler was found and successfully run
    static int recursing = 0; // detect recursion

    check_mode_initialized(mode);

    ActionList *actions = mode_find_handler(mode, name);
    if (!actions)
    {
        g_debug("Could not find handler %s.%s", mode->name, name);
        return FALSE;
    }

    if (recursing > MAX_RECURSION_DEPTH)
    {
        g_critical("max recursion detected in %s!", name);
        return FALSE;
    }

    recursing++;
    actions_run(actions);
    recursing--;

    // if at top-level, we can free any strings generated during those actions
    if (recursing == 0)
        temp_strings_free();
    return TRUE;
}

gboolean launcher_run_handler(const gchar *name)
{
    // the name can be either fully or partially qualified with the namespace and mode
    // so can have 0, 1, or 2 periods
    // returns TRUE if a handler of the name existed and something was run

    if (!name)
        warn_and_return_false("NULL handler given in mode %s", launcher->current->name);

    gchar *period = strrchr(name, '.');
    if (!period)
    {
        // no periods so run handler of the given name in the current mode
        return mode_run_handler(launcher->current, name);
    }
    else
    {
        gchar *mode_name = g_strndup(name, (period - name));
        LaunchMode *mode = launcher_get_mode(mode_name); // this will handle the mode (qualified or not)
        gboolean ret = FALSE;
        if (mode)
            ret = mode_run_handler(mode, period + 1);
        g_free(mode_name);
        return ret;
    }
}

gboolean load_initial_mode(const char *initial_mode)
{
    if (initial_mode)
    {
        // check if the mode is fully specified
        if (strchr(initial_mode, '.'))
            launcher->current = launcher_get_mode(initial_mode);
        else
            launcher->current = launcher_get_mode_full(DEFAULT_NAMESPACE, initial_mode);
        if (! launcher->current)
            g_warning("Initial mode '%s' not found", initial_mode);
    }

    if (! launcher->current)
        launcher->current = launcher_get_mode(DEFAULT_MODE);

    if (! launcher->current)
        g_critical("Failure to start. Initial mode %s not found.", initial_mode);

    if (launcher->current)
        check_mode_initialized(launcher->current);
    return (launcher->current != NULL);
}

void launcher_init(const char *initial_mode)
{
    g_debug("Launcher init");

    if (launcher)
    {
        g_warning("Detected multiple launcher initializations. Uniniting first...");
        launcher_uninit();
    }

    action_define("default_inherit", run_action_default_inherit, 1, FALSE);
    action_define("inherit", run_action_inherit, 1, FALSE);
    action_define("load", run_action_load, 1, FALSE);
    action_define("mode_switch", run_action_mode_switch, 1, FALSE);
    //action_define("path_add", run_action_path_add, 1, FALSE);
    action_sys_variable_define("mode", get_current_mode_name);
    action_sys_variable_define("lastmode", get_last_mode_name);

    launcher = g_new0(LauncherConfig, 1);
    launcher->namespaces = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)namespace_free);
    launcher->search_paths = add_or_move_to_front(launcher->search_paths, (gchar *)string_ref_dup(POWERLAUNCH_SYSDIR));
    if (getuid() > 0)           // only add user-specific home directory if not running as root
        launcher->search_paths = add_or_move_to_front(launcher->search_paths, (gchar *)string_ref(g_build_filename(g_get_home_dir(), ".powerlaunch", NULL))); 
}

void launcher_uninit()
{
    if (!launcher)
        warn_and_return("Launcher is not initialized");

    g_hash_table_destroy(launcher->namespaces);
    g_list_free(launcher->default_inherited_modes);

    g_list_foreach(launcher->search_paths, (GFunc)string_unref, NULL);
    g_list_free(launcher->search_paths);

    g_free(launcher);
    launcher = NULL;
}
