/*
  Copyright (c) 2007-2008 Austin Che

  Hardware abstraction layer (libhal)
*/

#include "powerlaunch.h"

#ifdef HAVE_HAL
#include <libhal.h>
#include <string.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>

/* ******************************************************************* 
   Type definitions
   ******************************************************************* */

typedef struct
{
    LibHalContext *hal;
} PowerHalContext;

typedef struct
{
    const char *udi;
    const char *key;
} HalPropertyChange;

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static PowerHalContext *hal;

/* ******************************************************************* 
   Static functions
   ******************************************************************* */

static gboolean data_hal_getval(ActionData *data, GValue *val)
{
    if (!data->val.v_str)
        warn_and_return_false("received NULL for Hal property");

    gchar *udi = g_strdup(data->val.v_str);

    // should be of form "/org/freedesktop/Hal/devices/computer:foo.bar"
    gchar *key = strchr(udi, ':');
    if (!key)
    {
        g_warning("%s is not of the form /udi/device:property", udi);
        g_free(udi);
        return FALSE;
    }

    *key = '\0';
    key++;
    g_debug("%s -- %s", udi, key);
    if (!libhal_device_exists(hal->hal, udi, NULL))
    {
        g_warning("%s is not a valid device", udi);
        g_free(udi);
        return FALSE;
    }

    if (!libhal_device_property_exists(hal->hal, udi, key, NULL))
    {
        g_warning("Property %s does not exist on device %s", key, udi);
        g_free(udi);
        return FALSE;
    }

    switch (libhal_device_get_property_type(hal->hal, udi, key, NULL))
    {
        case LIBHAL_PROPERTY_TYPE_INT32:
            g_value_init(val, G_TYPE_INT);
            g_value_set_int(val, libhal_device_get_property_int(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_UINT64:
            g_value_init(val, G_TYPE_INT64);
            g_value_set_uint64(val, libhal_device_get_property_uint64(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_DOUBLE:
            g_value_init(val, G_TYPE_DOUBLE);
            g_value_set_double(val, libhal_device_get_property_double(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_BOOLEAN:
            g_value_init(val, G_TYPE_BOOLEAN);
            g_value_set_boolean(val, libhal_device_get_property_bool(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_STRING:
            g_value_init(val, G_TYPE_STRING);
            g_value_take_string(val, libhal_device_get_property_string(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_INVALID:
        case LIBHAL_PROPERTY_TYPE_STRLIST:
        default:
            g_warning("Cannot handle property type for property %s on device %s", key, udi);
            g_free(udi);
            return FALSE;
    }
    g_free(udi);

    return TRUE;
}

static void event_hal_property_modified(LibHalContext *ctx, const char *udi, const char *key, dbus_bool_t is_removed, dbus_bool_t is_added)
{
    HalPropertyChange change;
    change.udi = udi;
    change.key = key;

    gchar *dev = g_strdup(udi);
    gchar *prop = g_strdup(key);
    gchar *handler = g_strconcat("hal", g_strdelimit(dev, "/", '_'), "_", g_strdelimit(prop, ".", '_'), NULL);

    ArgumentList *args = g_value_array_new(1);
    GValue gval = {0};
    g_value_init(&gval, G_TYPE_STRING);
    g_value_set_string(&gval, handler);
    g_value_array_append(args, &gval);
    g_value_unset(&gval);

    switch (libhal_device_get_property_type(hal->hal, udi, key, NULL))
    {
        case LIBHAL_PROPERTY_TYPE_INT32:
            g_value_init(&gval, G_TYPE_INT);
            g_value_set_int(&gval, libhal_device_get_property_int(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_UINT64:
            g_value_init(&gval, G_TYPE_UINT64);
            g_value_set_uint(&gval, libhal_device_get_property_uint64(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_DOUBLE:
            g_value_init(&gval, G_TYPE_DOUBLE);
            g_value_set_double(&gval, libhal_device_get_property_double(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_BOOLEAN:
            g_value_init(&gval, G_TYPE_BOOLEAN);
            g_value_set_boolean(&gval, libhal_device_get_property_bool(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_STRING:
            g_value_init(&gval, G_TYPE_STRING);
            g_value_take_string(&gval, libhal_device_get_property_string(hal->hal, udi, key, NULL));
            break;

        case LIBHAL_PROPERTY_TYPE_INVALID:
        case LIBHAL_PROPERTY_TYPE_STRLIST:
        default:
            g_warning("Cannot handle property type for property %s on device %s", key, udi);
    }

    g_value_array_append(args, &gval);
    g_value_unset(&gval);

    call_stack_push(args);
    
    launcher_run_handler(handler);
    g_free(handler);
    g_free(dev);
    g_free(prop);

    call_stack_pop();
    g_value_array_free(args);
}

static void run_action_hal_add(ArgumentList *args)
{
    // watches for a property change on a device
    const gchar *udi = get_argument_str(args, 0);
    g_debug("hal_add %s", udi);
    libhal_device_add_property_watch(hal->hal, udi, NULL);
}

static void run_action_hal_remove(ArgumentList *args)
{
    // removes a property watch
    const gchar *udi = get_argument_str(args, 0);
    g_debug("hal_remove %s", udi);
    libhal_device_remove_property_watch(hal->hal, udi, NULL);
}

/* ******************************************************************* 
   Exported functions
   ******************************************************************* */

void power_hal_init()
{
    static ActionDataType _ACTION_DATA_TYPE_HAL = {
        data_hal_getval,
        data_string_free,
        TRUE,
    };

    g_debug("Initializing hal subsystem");

    if (hal)
    {
        g_warning("Hal subsystem already initialized!");
        power_hal_uninit();
    }

    DBusGConnection *gdbus = dbus_g_bus_get(DBUS_BUS_SYSTEM, NULL);
    if (!gdbus)
    {
        g_warning("Could not initialze DBus for hal subsystem");
        return;
    }

    hal = g_new0(PowerHalContext, 1);

    DBusConnection *dbus = dbus_g_connection_get_connection(gdbus);
    hal->hal = libhal_ctx_new();
    libhal_ctx_set_dbus_connection(hal->hal, dbus);
    libhal_ctx_set_device_property_modified(hal->hal, event_hal_property_modified);
    libhal_ctx_init(hal->hal, NULL);

    action_data_subst_define("hal", &_ACTION_DATA_TYPE_HAL);
    action_define("hal_add", run_action_hal_add, 1, FALSE);
    action_define("hal_remove", run_action_hal_remove, 1, FALSE);
}

void power_hal_uninit()
{
    libhal_ctx_shutdown(hal->hal, NULL);
    libhal_ctx_free(hal->hal);
    g_free(hal);
    hal = NULL;
}

#endif
