/*
  Copyright (c) 2007-2008 Austin Che

  GConf
*/

#include "powerlaunch.h"

#ifdef HAVE_GCONF
#include <gconf/gconf-client.h>

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static GConfClient *gconf = NULL;

/* ******************************************************************* 
   Static functions
   ******************************************************************* */

static void gconf_to_gvalue(GConfValue *gconf, GValue *gval)
{
    GConfValueType type = GCONF_VALUE_INVALID;
    if (gconf)
        type = gconf->type;

    switch (type)
    {
        case GCONF_VALUE_STRING:
            g_value_init(gval, G_TYPE_STRING);
            g_value_set_string(gval, gconf_value_get_string(gconf));
            break;
        case GCONF_VALUE_INT:
            g_value_init(gval, G_TYPE_INT);
            g_value_set_int (gval, gconf_value_get_int(gconf));
            break;
        case GCONF_VALUE_BOOL:
            g_value_init(gval, G_TYPE_BOOLEAN);
            g_value_set_boolean(gval, gconf_value_get_bool(gconf));
            break;
        case GCONF_VALUE_FLOAT:
            g_value_init(gval, G_TYPE_FLOAT);
            g_value_set_float(gval, gconf_value_get_float(gconf));
            break;
        default:
            g_warning("Not able to handle GConf value type %d", type);
            break;
    }
}

static gboolean data_gconf_getval(ActionData *data, GValue *val)
{
    const gchar *key = data->val.v_str;
    if (!key)
        warn_and_return_false("Cannot retrieve value for NULL gconf key");

    GError *err = NULL;
    GConfValue *gconfval = gconf_client_get(gconf, key, &err);
    if (err)
    {
        g_warning("Could not retrieve value for gconf key=%s: %s", key, err->message);
        g_error_free(err);
        return FALSE;
    }
    gconf_to_gvalue(gconfval, val);
    return TRUE;
}

static void event_gconf_handler(GConfClient* client, guint cnxn_id, GConfEntry* entry, gpointer user_data) 
{
    gchar *key = g_strdup(gconf_entry_get_key(entry));
    GConfValue *val = gconf_entry_get_value(entry);

    g_debug("gconf key changed: key=%s;val=%s", key, gconf_value_to_string(val));

    // make a handler name by converting / into _
    gchar *handler = g_strconcat("gconf", g_strdelimit(key, "/", '_'), NULL);

    ArgumentList *args = g_value_array_new(1);
    GValue gval = {0};
    g_value_init(&gval, G_TYPE_STRING);
    g_value_set_string(&gval, handler);
    g_value_array_append(args, &gval);
    g_value_unset(&gval);

    gconf_to_gvalue(val, &gval);
    g_value_array_append(args, &gval);
    g_value_unset(&gval);

    call_stack_push(args);

    launcher_run_handler(handler);
    g_free(handler);
    g_free(key);

    call_stack_pop();
    g_value_array_free(args);
}

static void run_action_gconf_unset(ArgumentList *args)
{
    const gchar *key = get_argument_str(args, 0);
    if (!key)
    {
        g_warning("gconf_unset: NULL key");
        return;
    }

    GError *err = NULL;
    gconf_client_unset(gconf, key, &err);
    if (err)
    {
        g_warning("gconf_unset error: key=%s; error=%s", key, err->message);
        g_error_free(err);
    }
}

static void run_action_gconf_set(ArgumentList *args)
{
    const gchar *key = get_argument_str(args, 0);
    if (!key)
    {
        g_warning("gconf_set: NULL key");
        return;
    }

    GError *err = NULL;
    GValue *val = get_argument(args, 1);
    switch (G_VALUE_TYPE(val))
    {
        case G_TYPE_STRING:
            gconf_client_set_string(gconf, key, g_value_get_string(val), &err);
            break;

        case G_TYPE_BOOLEAN:
            gconf_client_set_bool(gconf, key, g_value_get_boolean(val), &err);
            break;

        case G_TYPE_INT:
            gconf_client_set_int(gconf, key, g_value_get_int(val), &err);
            break;

        case G_TYPE_UINT:
            gconf_client_set_int(gconf, key, g_value_get_uint(val), &err);
            break;

        case G_TYPE_DOUBLE:
            gconf_client_set_float(gconf, key, g_value_get_double(val), &err);
            break;

        default:
            g_warning("Unhandled value type %s for gconf_set", G_VALUE_TYPE_NAME(val));
            break;
    }

    if (err)
    {
        g_warning("gconf_set error: key=%s; val=%s; error=%s", key, g_value_to_string(val), err->message);
        g_error_free(err);
    }
}

static void run_action_gconf_add(ArgumentList *args)
{
    // handles adding and being notified when a gconf key in a directory changes
    const gchar *dir = get_argument_str(args, 0);
    if (!dir)
        warn_and_return("gconf_add: directory to watch cannot be NULL");

    GError *err = NULL;
    gconf_client_add_dir(gconf, dir, GCONF_CLIENT_PRELOAD_NONE, &err);
    if (!err)
        gconf_client_notify_add(gconf, dir, event_gconf_handler, NULL, NULL, &err);

    if (err)
    {
        g_warning("gconf_add: error adding watch on %s - %s", dir, err->message);
        g_error_free(err);        
    }
}

/* ******************************************************************* 
   Exported functions
   ******************************************************************* */

void power_gconf_init()
{
    static ActionDataType _ACTION_DATA_TYPE_GCONF = {
        data_gconf_getval,
        data_string_free,
        TRUE,
    };

    g_debug("Initializing gconf subsystem");

    gconf = gconf_client_get_default();
    if (!gconf)
        warn_and_return("Failed to create GConfClient");

    action_define("gconf_add", run_action_gconf_add, 1, FALSE);
    action_define("gconf_set", run_action_gconf_set, 2, FALSE);
    action_define("gconf_unset", run_action_gconf_unset, 1, FALSE);
    action_data_subst_define("gconf", &_ACTION_DATA_TYPE_GCONF);
}

void power_gconf_uninit()
{
    if (gconf)
        g_object_unref(gconf);
    gconf = NULL;
}

#endif
