/*
  Copyright (c) 2007-2008 Austin Che

  DBus handling
*/

#include <string.h>
#include <glib.h>
#include <stdlib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "powerlaunch.h"

/* ******************************************************************* 
   Type definitions
   ******************************************************************* */

#define DBUS_CONNECTION_FROM_OBJECT(obj) ((obj)->system_bus ? dbus->system : dbus->session)

typedef struct
{
    const gchar *name; // arbitrary name assigned by user to refer to this thing

    gboolean system_bus;            // TRUE for system bus, FALSE for session bus
    const gchar *service;
    const gchar *path;
    const gchar *interface;
} PowerDBusObject;

typedef struct
{
    // store information about the system and session bus separately
    DBusConnection *system;
    DBusConnection *session;
    dbus_int32_t message_object_slot; // from dbus_message_allocate_data_slot, we use to store name of a PowerDBusObject
    DBusMessage *incoming_msg; // temporarily stores current incoming dbus message
    GHashTable *objects;        /* string (name) -> PowerDBusObject * */
    GHashTable *object_lookup_cache;
} PowerDBusInfo;

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

static PowerDBusInfo *dbus = NULL;

/* ******************************************************************* 
   Static functions
   ******************************************************************* */

static PowerDBusObject *get_object_from_string(const gchar *str)
{
    PowerDBusObject *obj = g_hash_table_lookup(dbus->objects, str);
    if (!obj)
        g_warning("No dbus object called '%s' defined", str);
    return obj;
}

static gboolean match_dbus_object(gchar *name, PowerDBusObject *obj, PowerDBusObject *match)
{
    // tries to find an object that matches the match object
    // note that if any of the fields is empty, then that field is treated as matching everything
    // the first object that matches is returned

    if (match->system_bus != obj->system_bus)
        return FALSE;

    if (match->service && strcmp(match->service, obj->service))
        return FALSE;
    if (match->path && strcmp(match->path, obj->path))
        return FALSE;
    if (match->interface && strcmp(match->interface, obj->interface))
        return FALSE;

    return TRUE;
}

static PowerDBusObject *lookup_dbus_object(gboolean system_bus, const gchar *service, const gchar *path, const gchar *interface)
{
    PowerDBusObject obj;
    obj.system_bus = system_bus;
    obj.service = service;
    obj.path = path;
    obj.interface = interface;

    // use a cache to speed lookup
    // ; is not valid as part of dbus name so use it to separate fields as key into hashtable

    return g_hash_table_find(dbus->objects, (GHRFunc)match_dbus_object, &obj);
}

static void free_dbus_object(PowerDBusObject *obj)
{
    string_unref(obj->name);
    string_unref(obj->service);
    string_unref(obj->path);
    string_unref(obj->interface);
    g_free(obj);
}

static gboolean append_args_to_dbus_message(DBusMessage *msg, ArgumentList *args, int start)
{
    int num_args = get_num_args(args);
    DBusMessageIter iter;
    dbus_message_iter_init_append(msg, &iter);
    int i;
    const gchar *str;
    gboolean b;
    guint32 ui32;
    gint32 i32;
    double d;
    gboolean error = FALSE;
    for (i = start; i < num_args; i++)
    {
        GValue *arg = get_argument(args, i);
        switch (G_VALUE_TYPE(arg))
        {
            case G_TYPE_STRING:
                str = g_value_get_string(arg);
                error = !dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &str);
                break;

            case G_TYPE_BOOLEAN:
                b = g_value_get_boolean(arg);
                error = !dbus_message_iter_append_basic(&iter, DBUS_TYPE_BOOLEAN, &b);
                break;

            case G_TYPE_INT:
                i32 = g_value_get_int(arg);
                error = !dbus_message_iter_append_basic(&iter, DBUS_TYPE_INT32, &i32);
                break;

            case G_TYPE_UINT:
                ui32 = g_value_get_uint(arg);
                error = !dbus_message_iter_append_basic(&iter, DBUS_TYPE_UINT32, &ui32);
                break;

            case G_TYPE_DOUBLE:
                d = g_value_get_double(arg);
                error = !dbus_message_iter_append_basic(&iter, DBUS_TYPE_DOUBLE, &d);
                break;

            default:
                g_warning("Unhandled value type %s in dbus send", G_VALUE_TYPE_NAME(arg));
                error = TRUE;
                break;
        }

        if (error)
        {
            g_warning("Could not append argument for dbus message");
            dbus_message_unref(msg);
            return FALSE;
        }
    }
    return TRUE;
}

static ArgumentList *dbus_message_translate_args(DBusMessage *message, const gchar *handler)
{
    ArgumentList *args = g_value_array_new(0);

    GValue val = {0};
    g_value_init(&val, G_TYPE_STRING);
    g_value_set_string(&val, handler);
    g_value_array_append(args, &val);
    g_value_unset(&val);

    DBusMessageIter iter;
    if (!dbus_message_iter_init(message, &iter))
        return args;

    int i = 0;
    int type = dbus_message_iter_get_arg_type(&iter);
    for (i = 0; type != DBUS_TYPE_INVALID; i++)
    {
        gchar *str;
        gboolean b;
        guint32 ui32;
        gint32 i32;
        double d;

        switch (type)
        {
            case DBUS_TYPE_STRING:
                dbus_message_iter_get_basic(&iter, &str);
                g_value_init(&val, G_TYPE_STRING);
                g_value_set_string(&val, str);
                break;

            case DBUS_TYPE_BOOLEAN:
                dbus_message_iter_get_basic(&iter, &b);
                g_value_init(&val, G_TYPE_BOOLEAN);
                g_value_set_boolean(&val, b);
                break;

            case DBUS_TYPE_UINT32:
                dbus_message_iter_get_basic(&iter, &ui32);
                g_value_init(&val, G_TYPE_UINT);
                g_value_set_uint(&val, ui32);
                break;

            case DBUS_TYPE_INT32:
                dbus_message_iter_get_basic(&iter, &i32);
                g_value_init(&val, G_TYPE_INT);
                g_value_set_int(&val, i32);
                break;

            case DBUS_TYPE_DOUBLE:
                dbus_message_iter_get_basic(&iter, &d);
                g_value_init(&val, G_TYPE_DOUBLE);
                g_value_set_double(&val, d);
                break;

            case DBUS_TYPE_INVALID:
                g_warning("received invalid dbus type");
                break;

            default:
                g_debug("unhandled dbus type %d", type);
                break;
        }
        g_value_array_append(args, &val);
        g_value_unset(&val);

        dbus_message_iter_next(&iter);
        type = dbus_message_iter_get_arg_type(&iter);
    }

    return args;
}

static DBusHandlerResult event_dbus_handler_run_handler(PowerDBusObject *obj, const gchar *member, DBusMessage *message)
{
    // as we know dbus handlers cannot recursively generate another dbus message, 
    // we can handle it simply here by just storing the current call in case of dbus reply

    dbus->incoming_msg = message; 
    dbus_message_set_data(message, dbus->message_object_slot, (char *)string_ref((char *)obj->name), (DBusFreeFunction)string_unref);

    // make a handler name by using object's name and member name
    gchar *handler = g_strconcat("dbus_", obj->name, "_", member, NULL);

    ArgumentList *args = dbus_message_translate_args(message, handler);
    call_stack_push(args);
    
    launcher_run_handler(handler);
    g_free(handler);

    call_stack_pop();
    if (args)
        g_value_array_free(args);

    dbus->incoming_msg = NULL;

    return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult event_dbus_handler(DBusConnection *conn, DBusMessage *message, void *userdata)
{
    int type = dbus_message_get_type(message);
    if (type == DBUS_MESSAGE_TYPE_ERROR)
    {
        g_warning("Received dbus error message: %s", dbus_message_get_error_name(message));
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }
    else if ((type != DBUS_MESSAGE_TYPE_SIGNAL) && (type != DBUS_MESSAGE_TYPE_METHOD_CALL))
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

    PowerDBusObject *obj;
    const gchar *member = dbus_message_get_member(message);

    const gchar *dest = dbus_message_get_destination(message);
    const gchar *sender = dbus_message_get_sender(message);
    const gchar *path = dbus_message_get_path(message);
    const gchar *interface = dbus_message_get_interface(message);

    g_debug("received dbus message from %s; dest: %s; path: %s; interface: %s; member %s", dest, sender, path, interface, member);

    if (type == DBUS_MESSAGE_TYPE_SIGNAL)
        obj = lookup_dbus_object(conn == dbus->system, NULL, path, interface);
    else
        obj = lookup_dbus_object(conn == dbus->system, dest, path, interface);

    if (!obj)
    {
        g_debug("DBus message does not match any defined objects. Ignoring.");
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    return event_dbus_handler_run_handler(obj, dbus_message_get_member(message), message);
}

static void event_dbus_message_reply(DBusPendingCall *pending, void *userdata)
{
    DBusMessage *original = userdata; 
    DBusMessage *reply = dbus_pending_call_steal_reply(pending);

    if (!reply)
    {
        g_warning("Missing dbus reply");
        dbus_pending_call_unref(pending);
        return;
    }

    // *** perhaps pass the error to a handler
    if (dbus_message_get_type(reply) == DBUS_MESSAGE_TYPE_ERROR)
        g_warning("dbus error (%s) in reply to %s", dbus_message_get_error_name(reply), dbus_message_get_member(original));
    else
    {
        PowerDBusObject *obj = get_object_from_string((const gchar *)dbus_message_get_data(original, dbus->message_object_slot));
        if (obj)
            event_dbus_handler_run_handler(obj, dbus_message_get_member(original), reply);
    }

    // original DBusMessage is unref'd via dbus_pending_call_set_notify free function
    dbus_message_unref(reply);
    dbus_pending_call_unref(pending);
}

static DBusConnection *power_dbus_bus_init(DBusBusType type)
{
    // we use this get_private function so that we have a connection we can safely close in uninit
    // we need this when reloading the program 
    // (we need to get off the bus and back on in order to flush match rules)
    DBusConnection *bus = dbus_bus_get_private(type, NULL);
    if (! bus)
        warn_and_return_null("Failed to open connection to dbus %s bus", type == DBUS_BUS_SYSTEM ? "system" : "session");

    dbus_connection_setup_with_g_main(bus, NULL);
    dbus_connection_set_exit_on_disconnect(bus, FALSE);
    dbus_connection_add_filter(bus, event_dbus_handler, NULL, NULL);
    return bus;
}

static void power_dbus_bus_uninit(DBusConnection *bus)
{
    if (!bus)
        return;

    dbus_connection_remove_filter(bus, event_dbus_handler, NULL);
    dbus_connection_close(bus);
    dbus_connection_unref(bus);
    return;
}

static DBusConnection *get_bus_from_string(const gchar *str)
{
    if (strcmp(str, "system") == 0)
        return dbus->system;
    else if (strcmp(str, "session") == 0)
        return dbus->session;
    else
        warn_and_return_null("Invalid bus type: %s. Only 'session' or 'system' is allowed.", str);
}

/* ******************************************************************* 
   DBus action handlers
   ******************************************************************* */

static void run_action_dbus_define(ArgumentList *args)
{
    DBusConnection *bus = get_bus_from_string(get_argument_str(args, 1));
    if (!bus)
        return;

    const gchar *objname = get_argument_str(args, 0);
    const gchar *service = get_argument_str(args, 2);
    const gchar *path = get_argument_str(args, 3);
    const gchar *interface = get_argument_str(args, 4);

    PowerDBusObject *obj = g_new0(PowerDBusObject, 1);
    obj->name = string_ref_dup(objname);
    obj->system_bus = (bus == dbus->system);
    obj->service = string_ref_dup(service);
    obj->path = string_ref_dup(path);
    obj->interface = string_ref_dup(interface);

    g_debug("dbus_define %s %s %s %s", objname, service, path, interface);

    g_hash_table_insert(dbus->objects, (char *)string_ref((char *)obj->name), obj);
}

static void run_action_dbus_add_remove_match(ArgumentList *args, gboolean add)
{
    DBusConnection *bus = get_bus_from_string(get_argument_str(args, 0));
    if (!bus)
        return;

    const gchar *rule = get_argument_str(args, 1);
    if (!rule)
        return;

    g_debug("%sing dbus match rule %s", (add ? "Add" : "Remov"), rule);
    if (add)
        dbus_bus_add_match(bus, rule, NULL);
    else
        dbus_bus_remove_match(bus, rule, NULL);
}

static void run_action_dbus_add(ArgumentList *args)
{
    run_action_dbus_add_remove_match(args, TRUE);
}

static void run_action_dbus_remove(ArgumentList *args)
{
    run_action_dbus_add_remove_match(args, FALSE);
}

static void run_action_dbus_call(ArgumentList *args)
{
    PowerDBusObject *obj = get_object_from_string(get_argument_str(args, 0));
    const gchar *method = get_argument_str(args, 1);
    if (!obj || !method)
        return;

    DBusConnection *conn = DBUS_CONNECTION_FROM_OBJECT(obj);
    if (!conn)
        return;

    DBusMessage *msg = dbus_message_new_method_call(obj->service, obj->path, obj->interface, method);
    if (!msg)
        warn_and_return("Could not create dbus method call message for method %s", method);
    dbus_message_set_data(msg, dbus->message_object_slot, (char *)obj->name, NULL);

    if (!append_args_to_dbus_message(msg, args, 2))
        return;

    g_debug("Sending DBus method call to %s %s %s %s", obj->service, obj->path, obj->interface, method);

    DBusPendingCall *pending = NULL;
    if (!dbus_connection_send_with_reply(conn, msg, &pending, -1) || !pending)
    { 
        g_warning("Failed to make dbus method call");
        dbus_message_unref(msg);
    }
    else
    {
        // we get notified when we get a reply (or a timeout message)
        dbus_pending_call_set_notify(pending, event_dbus_message_reply, msg, (DBusFreeFunction)dbus_message_unref);
    }
}

static void run_action_dbus_signal(ArgumentList *args)
{
    PowerDBusObject *obj = get_object_from_string(get_argument_str(args, 0));
    if (!obj)
        return;
    const gchar *method = get_argument_str(args, 1);

    DBusConnection *conn = DBUS_CONNECTION_FROM_OBJECT(obj);
    if (!conn)
        return;

    DBusMessage *msg = dbus_message_new_signal(obj->path, obj->interface, method);
    if (!msg)
        warn_and_return("Could not create dbus signal message");

    if (!append_args_to_dbus_message(msg, args, 2))
        return;

    if (!dbus_connection_send(conn, msg, NULL))
        g_warning("Failed to send dbus signal message");
    dbus_message_unref(msg);
}

static void run_action_dbus_reply(ArgumentList *args)
{
    if (! dbus->incoming_msg)
        warn_and_return("No dbus message to reply to!");

    DBusMessage *reply = dbus_message_new_method_return(dbus->incoming_msg);
    if (!reply)
    {
        g_warning("Could not create dbus reply message");
        return;
    }

    if (!append_args_to_dbus_message(reply, args, 0))
        return;

    // message should be sent back on the same connection that the incoming message came in from
    const gchar *name = (const gchar *)dbus_message_get_data(dbus->incoming_msg, dbus->message_object_slot);
    PowerDBusObject *obj = get_object_from_string(name);
    if (obj && !dbus_connection_send(DBUS_CONNECTION_FROM_OBJECT(obj), reply, NULL))
        g_warning("Failed to send dbus reply message");
    dbus_message_unref(reply);
}

static void run_action_dbus_request_name(ArgumentList *args)
{
    // request a name from the bus
    DBusConnection *conn = get_bus_from_string(get_argument_str(args, 0));
    const gchar *name = get_argument_str(args, 1);
    
    if (!conn || !name)
        warn_and_return("Invalid arguments to dbus_request_name: %s, %s", get_argument_str(args, 0), get_argument_str(args, 1));

    int ret = dbus_bus_request_name(conn, name, 0, NULL);
    if (ret == DBUS_REQUEST_NAME_REPLY_ALREADY_OWNER)
        g_debug("dbus_request_name: already own the dbus name %s", name);
    else if (ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
        warn_and_return("dbus_request_name: could not request the dbus name %s", name);

    g_debug("dbus_request_name: got DBus name %s", name);
}

/* ******************************************************************* 
   Exported functions
   ******************************************************************* */

void power_dbus_init()
{
    g_debug("Initializing dbus subsystem");

    //  We don't want libdbus to abort our program on errors (for example, if it can't connect to session bus)
    setenv("DBUS_FATAL_WARNINGS", "0", 1);

    if (dbus)
    {
        g_warning("DBus system already initialized!");
        power_dbus_uninit();
    }

    dbus = g_new0(PowerDBusInfo, 1);
    dbus->system = power_dbus_bus_init(DBUS_BUS_SYSTEM);
    dbus->session = power_dbus_bus_init(DBUS_BUS_SESSION);

    // we want to store a DBusConnection along with some of our messages
    dbus->message_object_slot = -1;
    if (!dbus_message_allocate_data_slot(&dbus->message_object_slot))
        g_critical("dbus_message_allocate_data_slot failed (out of memory?)");

    dbus->objects = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)free_dbus_object);

    action_define("dbus_add", run_action_dbus_add, 2, FALSE);
    action_define("dbus_call", run_action_dbus_call, 2, TRUE);
    action_define("dbus_define", run_action_dbus_define, 5, FALSE);
    action_define("dbus_remove", run_action_dbus_remove, 2, FALSE);
    action_define("dbus_reply", run_action_dbus_reply, 0, TRUE);
    action_define("dbus_request_name", run_action_dbus_request_name, 2, FALSE);
    action_define("dbus_signal", run_action_dbus_signal, 2, TRUE);
}

void power_dbus_uninit()
{
    dbus_message_free_data_slot(&dbus->message_object_slot);
    power_dbus_bus_uninit(dbus->session);
    power_dbus_bus_uninit(dbus->system);
    g_hash_table_destroy(dbus->objects);
    g_free(dbus);
    dbus = NULL;
}
