/*
  Copyright (c) 2007-2008 Austin Che  

  Handles the actions specified in the config file
*/

#include <glib.h>
#include <locale.h>
#include <libintl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>

#include "powerlaunch.h"

//#define DEBUG_PARSE_ENABLED
#ifdef DEBUG_PARSE_ENABLED
#define DEBUG_PARSE(...) g_debug(__VA_ARGS__);
#else
#define DEBUG_PARSE(...)
#endif

/* ******************************************************************* 
   Type definitions
   ******************************************************************* */

typedef struct _ActionType
{
    const gchar *name;          /* this string owned by action_types HashTable */
    ActionFn run;
    int fixed_args;
    gboolean variable_args;
} ActionType;

typedef struct _LaunchAction
{
    ActionType *type;
    GArray *args;               // arguments to the action (an array of ActionData)
} LaunchAction;

typedef struct
{
    GHashTable *action_types;   // string (name) -> ActionType *
    GHashTable *subst_types;    // string (name) -> ActionDataType *
    GHashTable *sys_vars;       // string (name) -> SysVarFn *
    GHashTable *macros;         // string (macro name)->string (handler name)
    GHashTable *timers;         // string(name)->int(timer id)
    GArray *call_stack;         // stack of ArgumentList *
} ActionInfo;

/* ******************************************************************* 
   Prototypes
   ******************************************************************* */

static gchar *parse_data_until_end(ActionData *data, gchar *str, ActionDataType *type, gchar end);

/* ******************************************************************* 
   Static globals
   ******************************************************************* */

// BUILTIN_ACTION_DATA_TYPES defined later in this file
static ActionDataType BUILTIN_ACTION_DATA_TYPES[];
#define ACTION_DATA_TYPE_STRING (&BUILTIN_ACTION_DATA_TYPES[0])
#define ACTION_DATA_TYPE_LIST (&BUILTIN_ACTION_DATA_TYPES[1])
#define ACTION_DATA_TYPE_COMMAND (&BUILTIN_ACTION_DATA_TYPES[2])
#define ACTION_DATA_TYPE_VARIABLE (&BUILTIN_ACTION_DATA_TYPES[3])
#define ACTION_DATA_TYPE_CALL_ARGS (&BUILTIN_ACTION_DATA_TYPES[4])
#define ACTION_DATA_TYPE_SYS_VARIABLE (&BUILTIN_ACTION_DATA_TYPES[5])
#define ACTION_DATA_TYPE_GETTEXT (&BUILTIN_ACTION_DATA_TYPES[6])
#define ACTION_DATA_TYPE_CONVERT_INT (&BUILTIN_ACTION_DATA_TYPES[7])
#define ACTION_DATA_TYPE_CONVERT_DOUBLE (&BUILTIN_ACTION_DATA_TYPES[8])
#define ACTION_DATA_TYPE_CONVERT_BOOLEAN (&BUILTIN_ACTION_DATA_TYPES[9])

static ActionInfo *actioninfo = NULL;

/* ******************************************************************* 
   General Action Functions
   ******************************************************************* */

static void action_run(LaunchAction *action)
{
    if (!action->type)
        g_debug("Skipping null action");
    else
    {
        // Fill in the args
        int n = 0;
        if (action->args)
            n = action->args->len;
        ArgumentList *args = g_value_array_new(n);
        args->n_values = n;
        int i;
        for (i = 0; i < n; i++)
        {
            GValue *arg = g_value_array_get_nth(args, i);
            ActionData *data = &g_array_index(action->args, ActionData, i);
            data->type->getval(data, arg);
        }

        action->type->run(args);

        // free the args list
        for (i = 0; i < n; i++)
        {
            GValue *arg = g_value_array_get_nth(args, i);
            if (G_IS_VALUE(arg))
                g_value_unset(arg);
        }

        g_value_array_free(args);
    }
}

static void data_free(ActionData *data)
{
    if (data && data->type && data->type->free)
        data->type->free(data);
    data->type = NULL;
    data->val.v_ptr = NULL;
}

static const gchar *data_to_string(ActionData *data)
{
    g_assert(data);
    if (! data->type)
        return NULL;

    GValue val = {0};
    if (! data->type->getval(data, &val))
        return NULL;

    const gchar *ret = g_value_to_string(&val);
    g_value_unset(&val);
    return ret;
}

static void gvalue_string_to_boolean(const GValue *src_value, GValue *dest_value)
{
    // GValue transform function
    const gchar *s = g_value_get_string(src_value);
    if (! s || (*s == '\0') || (strcmp(s, "0") == 0) || (strcasecmp(s, "false") == 0))
        g_value_set_boolean(dest_value, FALSE);
    else
        g_value_set_boolean(dest_value, TRUE);
}

static gboolean g_value_to_boolean(GValue *val)
{
    if (!val || !G_IS_VALUE(val))
        return FALSE;
    
    if (G_VALUE_HOLDS(val, G_TYPE_BOOLEAN))
        return g_value_get_boolean(val);

    GValue dest = {0};
    g_value_init(&dest, G_TYPE_BOOLEAN);
    if (! g_value_transform(val, &dest) )
        warn_and_return_false("Could not transform data value type %s to boolean", G_VALUE_TYPE_NAME(val));

    gboolean b = g_value_get_boolean(&dest);
    g_value_unset(&dest);
    return b;
}

static void data_add_type(ActionData *data, ActionDataType *type, gchar *str)
{
    // we will own the passed in string after the call
    if (!type)
    {
        g_free(str);
        return;
    }
    if (! data->type)
    {
        // in simple case, we just fill in the data fields;
        data->type = type;
        data->val.v_str = string_ref(str);
        return;
    }

    ActionData tmp;
    GArray *list = data->val.v_ptr;
    // if something already exists, we create a list structure
    if (data->type != ACTION_DATA_TYPE_LIST)
    {
        // make it into a list
        memcpy(&tmp, data, sizeof(tmp));

        data->type = ACTION_DATA_TYPE_LIST;
        data->val.v_ptr = list = g_array_sized_new(FALSE, TRUE, sizeof(ActionData), 2);
        g_array_append_val(list, tmp);
    }

    // we add the passed in data to the end of the list
    ActionData *last = &g_array_index(list, ActionData, list->len-1);
    if (last->type == ACTION_DATA_TYPE_STRING && type == ACTION_DATA_TYPE_STRING)
    {
        // if we have two strings in a row, we just concatenate the strings and replace the previous string
        gchar *newstr = g_strconcat(last->val.v_str, str, NULL);
        string_unref(last->val.v_str);
        g_free(str);
        last->val.v_str = string_ref(newstr);
    }
    else
    {
        // add new element to end
        tmp.type = type;
        tmp.val.v_str = string_ref(str);
        g_array_append_val(list, tmp);
    }
}

static gchar *parse_substitution(ActionData *data, gchar *str)
{
    // we treat everything of the form $foo{bar} as a substitution
    // foo is treated as a functional type that takes bar as an argument and converts it to something else
    // Arguments are optional so $foo == $foo{}
    // Variables are special cases when foo does not match known functions
    // Variables and function names can be [a-zA-Z0-9_-]+
    // $number (e.g. $1) is handled specially to refer to a call argument

    gchar *end = str;
    gboolean numeric = TRUE;

    // the end of the name is the first character invalid for a name
    while (g_ascii_isalnum(*end) || *end == '-' || *end == '_')
    {
        if (! g_ascii_isdigit(*end))
            numeric = FALSE;
        end++;
    }

    ActionDataType *type = NULL;
    gchar *name = NULL;
    if (end == str)
    {
        // ${foo} is special form for $foo that will always use the user variables even if foo is a defined function
        type = ACTION_DATA_TYPE_VARIABLE;
    }
    else
    {
        name = g_strndup(str, end - str);
        if (numeric)
        {
            DEBUG_PARSE("Found numeric substitution: $%s", name);
            data_add_type(data, ACTION_DATA_TYPE_CALL_ARGS, name);
            return end;
        }
        else
            type = g_hash_table_lookup(actioninfo->subst_types, name);
    }

    if (*end == '{')
    {
        // $foo{bar}

        // $unknown{...} is an error
        // we continue the parsing but will throw away the data with NULL type
        if (!type)
            g_warning("$%s is an undefined function", name);
        DEBUG_PARSE("Substitution function: $%s", name);
        g_free(name);
        return parse_data_until_end(data, end+1, type, '}');
    }
    else
    {
        // $foo form
        if (type)
        {
            // function with no arguments is an error
            g_warning("No arguments to $%s", name);
            g_free(name);
        }
        else
            data_add_type(data, ACTION_DATA_TYPE_VARIABLE, name);
        return end;
    }
}

static gchar *parse_data(ActionData *data, gchar *str)
{
    // parses one "block" of data from str into the data structure
    // returns the part of str left
    switch (*str)
    {
        case '$':
            return parse_substitution(data, str+1);
        case '"':
            return parse_data_until_end(data, str+1, ACTION_DATA_TYPE_STRING, '"');
        case '`':
            return parse_data_until_end(data, str+1, ACTION_DATA_TYPE_COMMAND, '`');
        default:
            // we treat as unquoted string which is terminated by a space (or other special character)
            return parse_data_until_end(data, str, ACTION_DATA_TYPE_STRING, ' ');
    }
    return str;
}

static gchar *parse_data_until_end(ActionData *data, gchar *str, ActionDataType *type, gchar end)
{
    // parses one data element using end as the terminating character of the block
    // Put it into data with the given type
    // Will return the string beginning with the char AFTER the end

    gchar *ptr = str;
    gboolean escaped = FALSE;
    while (*ptr)
    {
        if (escaped)
            escaped = FALSE; // skip over this character and go to the next
        else if (*ptr == '\\')
            escaped = TRUE;
        else
        {
            // treat as a normal character

            if (*ptr == end)
            {
                // reached the end of the block
                *ptr = '\0';
                data_add_type(data, type, g_strcompress(str));
                return ptr+1;
            }

            gchar special;
            switch (*ptr)
            {
                case '$':
                case '"':
                case '`':
                    special = *ptr;

                    // store what we have up to this point
                    *ptr = '\0';
                    data_add_type(data, type, g_strcompress(str));

                    // recursively handle this special type
                    *ptr = special;
                    ptr = parse_data(data, ptr);

                    if (!ptr)
                        return NULL; /* no more input */

                    type = ACTION_DATA_TYPE_STRING; /* remaining parts are treated as normal strings */
                    str = ptr;
                    continue;   /* we don't need to increment ptr again */

                default:
                    break;
            }
        }
        ptr++;
    }

    if (end == ' ')
        data_add_type(data, type, g_strcompress(str)); // we treat the end of string as equivalent to seeing a space
    else
        g_warning("Missing '%c' around '%s'", end, str);
    return NULL;
}

/* ******************************************************************* 
   Specific Action Data Type Functions
   ******************************************************************* */

static gboolean data_convert_to_int(ActionData *data, GValue *val)
{
    g_value_init(val, G_TYPE_INT);
    g_value_set_int(val, strtoul(data->val.v_str, NULL, 10));
    return TRUE;
}

static gboolean data_convert_to_double(ActionData *data, GValue *val)
{
    g_value_init(val, G_TYPE_DOUBLE);
    g_value_set_double(val, strtod(data->val.v_str, NULL));
    return TRUE;
}

static gboolean data_convert_to_boolean(ActionData *data, GValue *val)
{
    g_value_init(val, G_TYPE_BOOLEAN);
    const gchar *s = data->val.v_str;
    if (! s || (*s == '\0') || (strcmp(s, "0") == 0) || (strcasecmp(s, "false") == 0))
        g_value_set_boolean(val, FALSE);
    else
        g_value_set_boolean(val, TRUE);
    return TRUE;
}

static gboolean data_call_arg_getval(ActionData *data, GValue *val)
{
    // return the nth argument to current call action
    if (actioninfo->call_stack->len == 0)
        warn_and_return_false("Call stack is empty -- cannot get call arguments");

    GValueArray *elem = g_array_index(actioninfo->call_stack, GValueArray *, actioninfo->call_stack->len-1);
    if (elem == NULL)
        warn_and_return_false("No arguments on call stack");

    int n = strtoul(data->val.v_str, NULL, 10);
    if (n >= elem->n_values)
        warn_and_return_false("Cannot get call argument %d (only %d arguments)", n, elem->n_values);

    GValue *arg = g_value_array_get_nth(elem, n);
    g_value_init(val, G_VALUE_TYPE(arg));
    g_value_copy(arg, val);
    return TRUE;
}

static gboolean data_string_getval(ActionData *data, GValue *val)
{
    g_value_init(val, G_TYPE_STRING);
    g_value_set_static_string(val, data->val.v_str);
    return TRUE;
}

void data_string_free(ActionData *data)
{
    string_unref(data->val.v_str);
}

static gboolean data_list_getval(ActionData *data, GValue *val)
{
    // A list type can store any other type that takes a string
    // The v_ptr element is a GArray of ActionData
    // The first element is special containing the type that is the overall type of entire data
    // The string is created by concatenating from the string in the first element and all remaining elements
    // and then using the type in first element to process the resulting concatenated string

    GArray *list = data->val.v_ptr;
    if (!list || list->len <= 1)
        return FALSE;

    GString *newstr = g_string_sized_new(128);
    g_string_append(newstr, (&g_array_index(list, ActionData, 0))->val.v_str);

    int i;
    for (i = 1; i < list->len; i++)
    {
        const gchar *tmp = data_to_string(&g_array_index(list, ActionData, i));
        if (tmp)
            g_string_append(newstr, tmp);
    }

    ActionData tmp;
    tmp.type = (&g_array_index(list, ActionData, 0))->type;
    tmp.val.v_str = store_temp_string(g_string_free(newstr, FALSE));

    return tmp.type->getval(&tmp, val);
}

static void data_list_free(ActionData *data)
{
    GArray *list = data->val.v_ptr;
    if (list)
    {
        int i;
        for (i = 0; i < list->len; i++)
            data_free(&g_array_index(list, ActionData, i));
        g_array_free(list, TRUE);
        data->val.v_ptr = NULL;
    }
}

static gboolean data_variable_getval(ActionData *data, GValue *val)
{
    const gchar *var = data->val.v_str;

    g_value_init(val, G_TYPE_STRING);
    g_value_set_static_string(val, store_get_variable(var));
    return TRUE;
}

static gboolean data_command_getval(ActionData *data, GValue *val)
{
    const gchar *cmd = data->val.v_str;
    if (!cmd)
        warn_and_return_false("NULL command to execute");

    gchar *result = NULL;
    gchar *ptr = NULL;
    if (g_spawn_command_line_sync(cmd, &result, NULL, NULL, NULL))
    {
        if (*result != '\0')
        {
            // we remove one trailing newline if it exists
            ptr = result + strlen(result) - 1;
            if (*ptr == '\n')
                *ptr = '\0';
        }
        g_value_init(val, G_TYPE_STRING);
        g_value_set_static_string(val, store_temp_string(result));
        return TRUE;
    }

    warn_and_return_false("error executing %s", cmd);
}

static guint32 data_get_time_spec(GValue *value)
{
    // we specify times in seconds in the config file as floats (i.e. 1.5)
    // but store everything in integers using milliseconds

    const gchar *str = NULL;
    if (G_VALUE_TYPE(value) == G_TYPE_STRING)
        str = g_value_get_string(value);

    if (!str)
        return 0;

    guint32 time_in_ms = (guint)(strtod(str, NULL) * 1000);
    if (time_in_ms == 0)
        g_warning("Bad time spec: %s", str);

    return time_in_ms;
}

static gboolean data_sys_variable_getval(ActionData *data, GValue *val)
{
    const gchar *name = data->val.v_str;
    if (!name)
        return FALSE;

    SysVarFn fn = g_hash_table_lookup(actioninfo->sys_vars, name);
    if (!fn)
        warn_and_return_false("Unknown sys variable: %s", name);
    g_value_init(val, G_TYPE_STRING);
    g_value_set_static_string(val, fn(name));
    return TRUE;
}

static gboolean data_gettext_getval(ActionData *data, GValue *val)
{
    g_value_init(val, G_TYPE_STRING);
    g_value_set_static_string(val, gettext(data->val.v_str));
    return TRUE;
}

/* ******************************************************************* 
   Create the action data type table
   ******************************************************************* */

static ActionDataType BUILTIN_ACTION_DATA_TYPES[] = {
    {                           /* string */
        data_string_getval,
        data_string_free,
        FALSE,
    },

    {                           /* list */
        data_list_getval,
        data_list_free,
        TRUE,
    },

    {                           /* command */
        data_command_getval,
        data_string_free,
        TRUE,
    },

    {                           /* variable */
        data_variable_getval,
        data_string_free,
        TRUE,
    },

    {                           /* call args */
        data_call_arg_getval,
        data_string_free,
        TRUE,
    },

    {                           /* system variable */
        data_sys_variable_getval,
        data_string_free,
        TRUE,
    },

    {                           /* gettext */
        data_gettext_getval,
        data_string_free,
        FALSE,
    },

    {                           /* convert to int */
        data_convert_to_int,
        data_string_free,
        FALSE,
    },

    {                           /* convert to double */
        data_convert_to_double,
        data_string_free,
        FALSE,
    },

    {                           /* convert to boolean */
        data_convert_to_boolean,
        data_string_free,
        FALSE,
    },
};

/* ******************************************************************* 
   Action Functions
   ******************************************************************* */

static void action_free(LaunchAction *action)
{
    // uninits the given action
    action->type = NULL;
    if (action->args)
    {
        int i;
        for (i = 0; i < action->args->len; i++)
            data_free(&g_array_index(action->args, ActionData, i));
        g_array_free(action->args, TRUE);
    }
    memset(action, 0, sizeof(LaunchAction));
}

static void parse_action_args(LaunchAction *action, char *str)
{
    // parse a string into a number of action arguments
    // we expect action->type to be set already

    int num = action->type->fixed_args;
    while ((num || action->type->variable_args) && str && *str)
    {
        if (!action->args)
            action->args = g_array_sized_new(FALSE, TRUE, sizeof(ActionData), num);

        g_array_set_size(action->args, action->args->len + 1); // create new data element by expanding array
        ActionData *data = &g_array_index(action->args, ActionData, action->args->len-1);

        // skip over spaces
        while (g_ascii_isspace(*str))
            str++;
        str = parse_data(data, str);

        if (! data->type)
        {
            g_warning("Error in parsing data for action %s. Ignoring action.", action->type->name);
            action_free(action);
            return;
        }

        num--;
    }

    if (num > 0)
    {
        // didn't get the required number of arguments
        g_warning("Missing %d arguments to %s. Ignoring action.", num, action->type->name);
        action_free(action);
        return;
    }
    
    if (str && *str)
        g_warning("Extra data for action '%s' ignored: %s", action->type->name, str);
}

static void parse_action(LaunchAction *action, char *str)
{
    // parses the given string and fills in the LaunchAction
    memset(action, 0, sizeof(*action));

    gchar *action_str = g_strstrip(str); // remove whitespace

    if (*action_str == '\0')
    {
        // empty action
        action->type = NULL;
        return;
    }

    DEBUG_PARSE("parsing %s", action_str);

    // parse out the command portion
    gchar *space = strchr(action_str, ' ');
    if (space)
    {
        *space = '\0';      // separate into new string
        space = space + 1;
    }

    action->type = g_hash_table_lookup(actioninfo->action_types, action_str);
    if (!action->type)
    {
        // if the action name doesn't match anything, we treat as macro call
        // and do the lookup/binding at runtime
        if (space)
            *(space-1) = ' ';
        action->type = g_hash_table_lookup(actioninfo->action_types, "macro");
        space = action_str;
    }

    parse_action_args(action, space);
}

/* ******************************************************************* 
   General Execution Actions
   ******************************************************************* */

static gboolean event_timer_handler(gchar *name)
{
    const gchar *timer_name = string_ref(name); // get extra ref as we don't want to unref last one from hashtable
    if (g_hash_table_remove(actioninfo->timers, timer_name))
    {
        // timer exists, so run handler
        g_debug("timed handler: '%s'", timer_name);
        launcher_run_handler(timer_name);
    }
    else
        g_warning("Timer %s called even though we have no record of it??", timer_name);
    string_unref(timer_name);
    return FALSE;               // return false to cancel this timer
}

static void run_action_exec(ArgumentList *args)
{
    const gchar *command = get_argument_str(args, 0);
    if (!command)
    {
        g_warning("exec: NULL command");
        return;
    }

    g_debug("exec %s", command);
    GError *err = NULL;
    if (!g_spawn_command_line_async(command, &err))
    {
        g_warning("Execution of %s failed: %s", command, err->message);
        return;
    }
}

static void run_action_sleep(ArgumentList *args)
{
    guint32 time_in_ms = data_get_time_spec(get_argument(args, 0));
    if (time_in_ms > 0)
        g_usleep(1000 * time_in_ms);
    else
        g_warning("Not sleeping");
}

/* ******************************************************************* 
   Variable Actions
   ******************************************************************* */

static void run_action_set(ArgumentList *args)
{
    const gchar *var = get_argument_str(args, 0);
    const gchar *val = get_argument_str(args, 1);
    if (! var || ! val)
        g_warning("Invalid 'set' action");
    else
        store_set_variable(var, val);
}

static void run_action_unset(ArgumentList *args)
{
    const gchar *var = get_argument_str(args, 0);
    if (! var)
        g_warning("Invalid 'unset' action");
    else
        store_set_variable(var, NULL);
}

/* ******************************************************************* 
   Flow Control Actions
   ******************************************************************* */

static void run_action_call(ArgumentList *args)
{
    call_stack_push(args);
    launcher_run_handler(get_argument_str(args, 0));
    call_stack_pop();
}

static void run_action_define(ArgumentList *args)
{
    const gchar *macro = get_argument_str(args, 0);
    const gchar *handler = get_argument_str(args, 1);
    if (handler)
        g_hash_table_insert(actioninfo->macros, (gchar *)string_ref_dup(macro), (gchar *)string_ref_dup(handler));
    else
        g_hash_table_remove(actioninfo->macros, macro);
}

static void run_action_eval(ArgumentList *args)
{
    gchar *str = g_strdup(get_argument_str(args, 0));
    if (!str)
        return;

    gchar *astart = str;
    gchar *aend;
    gboolean escaped = FALSE;
    LaunchAction eval_action;
    gchar *act_str;
    gboolean done;

    // split using non-escaped semicolons
    while (*astart)
    {
        aend = astart;
        done = TRUE;
        while (*aend)
        {
            if (escaped)
                escaped = FALSE;  // skip over this character and go to the next
            else if (*aend == '\\')
                escaped = TRUE;
            else if (*aend == ';')
            {
                // found the end of one action
                done = FALSE;
                break; 
            }

            aend++;
        }

        *aend = '\0';

        act_str = g_strcompress(astart);

        parse_action(&eval_action, act_str);
        if (eval_action.type)
            action_run(&eval_action);
        else
            g_warning("eval parsing failed: %s", act_str);

        action_free(&eval_action);
        g_free(act_str);

        if (done)
            break;
        else
            astart = aend + 1;
    }
    g_free(str);
}

static void run_action_if(ArgumentList *args)
{
    // basic testing for true/falseness
    // false is NULL, empty string, or a string equal to "0". Everything else is true.
    gboolean condition = g_value_to_boolean(get_argument(args, 0));
    const gchar *handler = NULL;
    if (condition)
        handler = get_argument_str(args, 1);
    else
    {
        // condition is false, check if 'else' handler is given
        if (get_num_args(args) > 2)
            handler = get_argument_str(args, 2);
    }

    if (handler)
        launcher_run_handler(handler);
}

static void run_action_if_eq(ArgumentList *args)
{
    // ifeq var val1 handler1 val2 handler2 ... else-handler[optional]
    // this is basically a 'switch' statement finding first val that matches var or running the else handler
    const gchar *var = get_argument_str(args, 0);
    int i;
    for (i = 1; i < get_num_args(args)-1; i += 2)
    {
        const gchar *val = get_argument_str(args, i); 
        const gchar *handler = get_argument_str(args, i+1);
        if (var && val && strcmp(var, val) == 0)
        {
            launcher_run_handler(handler);
            return;
        }
    }

    // no match, check if else-handler provided (odd argument out)
    if (i < get_num_args(args))
        launcher_run_handler(get_argument_str(args, i));
}

static void run_action_macro(ArgumentList *args)
{
    const gchar *macro = get_argument_str(args, 0);
    gchar *handler = g_hash_table_lookup(actioninfo->macros, macro);
    if (handler)
    {
        call_stack_push(args);
        launcher_run_handler(handler);
        call_stack_pop();
    }
    else
        g_warning("Macro '%s' not found", macro);
}

/* ******************************************************************* 
   Timer Actions
   ******************************************************************* */

static void run_action_timer_set(ArgumentList *args)
{
    guint32 time_in_ms = data_get_time_spec(get_argument(args, 1));
    if (!time_in_ms)
    {
        g_warning("Bad time spec. Ignoring timer");
        return;
    }

    const gchar *handler = get_argument_str(args, 0);

    g_debug("Setting timer for '%s' in %u ms", handler, time_in_ms);
    // g_timeout_add_seconds which is said to have more efficient system power usage
    // for timeouts in seconds is only available since glib 2.14, not available on maemo

    guint id = GPOINTER_TO_INT(g_hash_table_lookup(actioninfo->timers, handler));
    if (id)
    {
        g_debug("Timer '%s' was already set. Canceling previous timer.", handler);
        g_source_remove(id);
    }

    // make own own copy of the timer's handler name
    const gchar *timer_name = string_ref_dup(handler);
    id = g_timeout_add(time_in_ms, (GSourceFunc)event_timer_handler, (gpointer)timer_name);
    g_hash_table_insert(actioninfo->timers, (gchar *)timer_name, GINT_TO_POINTER(id));
}

static void run_action_timer_cancel(ArgumentList *args)
{
    const gchar *name = get_argument_str(args, 0);
    g_debug("Canceling timer '%s'", name);
    guint id = GPOINTER_TO_INT(g_hash_table_lookup(actioninfo->timers, name));
    if (id)
    {
        g_source_remove(id);
        g_hash_table_remove(actioninfo->timers, name);
    }
    else
        g_debug("Cannot cancel timer %s as it is not running", name);
}

/* ******************************************************************* 
   Exported Functions
   ******************************************************************* */

void action_sys_variable_define(const gchar *name, SysVarFn fn)
{
    g_hash_table_insert(actioninfo->sys_vars, (gchar *)string_ref_dup(name), fn);
}

void action_data_subst_define(const gchar *name, ActionDataType *type)
{
    g_hash_table_insert(actioninfo->subst_types, (gchar *)string_ref_dup(name), type);
}

const gchar *g_value_to_string(GValue *val)
{
    // returns the given value as a string
    // the string is not owned by the caller
    // returns NULL on error

    if (!val || !G_IS_VALUE(val))
        return NULL;
    
    gchar *str;
    if (G_VALUE_HOLDS(val, G_TYPE_STRING))
        return store_temp_string(g_value_dup_string(val));

    // transform existing type to string
    GValue dest = {0};
    g_value_init(&dest, G_TYPE_STRING);
    if (! g_value_transform(val, &dest) )
        warn_and_return_false("Could not transform data value type %s to string", G_VALUE_TYPE_NAME(val));

    str = g_value_dup_string(&dest);
    g_value_unset(&dest);
    return store_temp_string(str);
}

void action_define(const gchar *name, ActionFn fn, int fixed_args, gboolean variable_args)
{
    ActionType *type = g_new0(ActionType, 1);
    type->name = string_ref_dup(name);
    type->run = fn;
    type->fixed_args = fixed_args;
    type->variable_args = variable_args;
    g_hash_table_insert(actioninfo->action_types, (gchar *)type->name, type);
}

GValue *get_argument(ArgumentList *args, int n)
{
    // return NULL if invalid argument number
    if (n >= args->n_values)
        warn_and_return_null("Could not get argument #%d (only %d arguments)", n, args->n_values);
    return g_value_array_get_nth(args, n);
}

int get_num_args(ArgumentList *args)
{
    return args->n_values;
}

ActionList *actions_parse(gchar **actions, gsize numactions)
{
    // parse a list of actions of the form "command args"
    // the passed in actions string array may be modified but remains owned by the caller

    // allocate space for all actions
    GArray *actionlist = g_array_set_size(g_array_sized_new(FALSE, TRUE, sizeof(LaunchAction), numactions), numactions);
    int i;
    for (i = 0; i < numactions; i++)
    {
        LaunchAction *action = &g_array_index(actionlist, LaunchAction, i);
        parse_action(action, actions[i]);
    }

    return actionlist;
}

void call_stack_push(ArgumentList *args)
{
    g_array_append_val(actioninfo->call_stack, args);
}

void call_stack_pop()
{
    if (actioninfo->call_stack->len == 0)
        g_critical("Cannot pop from empty call stack!");
    else
        g_array_set_size(actioninfo->call_stack, actioninfo->call_stack->len-1);
}

void actions_run(ActionList *actions)
{
    if (! actions)
        return;

    guint32 num = actions->len;
    int i;
    for (i = 0; i < num; i++)
        action_run(&g_array_index(actions, LaunchAction, i));
}

void actions_free(ActionList *actions)
{
    if (!actions)
        return;

    int i;
    guint32 num = actions->len;
    for (i = 0; i < num; i++)
        action_free(&g_array_index(actions, LaunchAction, i));

    g_array_free(actions, TRUE);
}

void actions_init()
{
    setlocale(LC_ALL, "");

    g_value_register_transform_func(G_TYPE_STRING, G_TYPE_BOOLEAN, gvalue_string_to_boolean);

    actioninfo = g_new0(ActionInfo, 1);
    actioninfo->action_types = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)g_free);
    actioninfo->subst_types = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, NULL);    
    actioninfo->sys_vars = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, NULL);    
    actioninfo->macros = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, (GDestroyNotify)string_unref);
    actioninfo->timers = g_hash_table_new_full(g_str_hash, g_str_equal, (GDestroyNotify)string_unref, NULL);
    actioninfo->call_stack = g_array_new(FALSE, TRUE, sizeof(GValueArray *));

    action_define("call", run_action_call, 1, TRUE);
    action_define("define", run_action_define, 2, FALSE);
    action_define("eval", run_action_eval, 1, FALSE);
    action_define("exec", run_action_exec, 1, FALSE);
    action_define("if", run_action_if, 2, TRUE);
    action_define("ifeq", run_action_if_eq, 3, TRUE);
    action_define("macro", run_action_macro, 1, TRUE);
    action_define("set", run_action_set, 2, FALSE);
    action_define("sleep", run_action_sleep, 1, FALSE);
    action_define("timer_set", run_action_timer_set, 2, FALSE);
    action_define("timer_cancel", run_action_timer_cancel, 1, FALSE);
    action_define("unset", run_action_unset, 1, FALSE);

    action_data_subst_define("_", ACTION_DATA_TYPE_GETTEXT);
    action_data_subst_define("sys", ACTION_DATA_TYPE_SYS_VARIABLE);
    action_data_subst_define("exec", ACTION_DATA_TYPE_COMMAND);
    action_data_subst_define("int", ACTION_DATA_TYPE_CONVERT_INT);
    action_data_subst_define("double", ACTION_DATA_TYPE_CONVERT_DOUBLE);
    action_data_subst_define("boolean", ACTION_DATA_TYPE_CONVERT_BOOLEAN);
}

void actions_uninit()
{
    if (actioninfo->call_stack->len != 0)
        g_warning("Call stack not empty! This shouldn't happen.");
    g_array_free(actioninfo->call_stack, TRUE);
    g_hash_table_destroy(actioninfo->action_types);
    g_hash_table_destroy(actioninfo->subst_types);
    g_hash_table_destroy(actioninfo->sys_vars);
    g_hash_table_destroy(actioninfo->macros);
    g_hash_table_destroy(actioninfo->timers);
    g_free(actioninfo);
    actioninfo = NULL;
}
