/*
  Copyright (c) 2007-2008 Austin Che
*/

#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "utils.h"

void write_to_file(const gchar *file, const gchar *data)
{
    // writes a NULL terminated string to file
    GError *error = NULL;
    GIOChannel *io = g_io_channel_new_file(file, "w", &error);
    if (!io)
    {
        g_warning("Failed to open %s: %s", file, error->message);
        g_error_free(error);
        return;
    }
    if (g_io_channel_write_chars(io, data, -1, NULL, &error) != G_IO_STATUS_NORMAL)
    {
        g_warning("Failed to write to %s: %s", file, error->message);
        g_error_free(error);
    }
    g_io_channel_shutdown(io, TRUE, NULL);
}

void write_to_file_int(const gchar *file, int data)
{
    char buf[16];
    g_snprintf(buf, sizeof(buf), "%d", data);
    write_to_file(file, buf);
}

gboolean send_dbus_signal(DBusGConnection *dbus, const gchar *path, const gchar *interface, const gchar *name, int first_arg_type, ...)
{
    g_return_val_if_fail(dbus, FALSE);

    gboolean ret = TRUE;
    DBusMessage *msg = dbus_message_new_signal(path, interface, name);
    if (msg == NULL) 
    {
        g_warning("Failed to create dbus message for signal %s", name); 
        return FALSE;
    }

    va_list args;
    va_start(args, first_arg_type);
    if ( ! dbus_message_append_args_valist(msg, first_arg_type, args))
    {
        g_warning("Failed to add arguments to dbus message for signal %s", name); 
        ret = FALSE;
    }
    else if (!dbus_connection_send(dbus_g_connection_get_connection(dbus), msg, NULL)) 
    { 
        g_warning("Failed to send dbus message for signal %s", name); 
        ret = FALSE;
    }

    va_end(args);
    dbus_message_unref(msg);
    return ret;
}
