#ifndef POWERED_H
#define POWERED_H

#include <glib.h>
#include <libhal.h>
#include <dbus/dbus-glib-bindings.h>

#include "power-common.h"

// These two enums are not arbitrary constants. They match what DSME expects
typedef enum {
    DISPLAY_ON = 0,
    DISPLAY_DIM = 1,
    DISPLAY_OFF = 2,
} DisplayState;

typedef enum {
    ALARM_OFF = 0,
    ALARM_VISIBLE = 1,
    ALARM_SNOOZED = 2,
} AlarmMode;

typedef struct
{
    guint key;                  /* key for entering failsafe mode */
    guint key_hold;             /* amount of time above key needs to be held */
    guint timeout; /* amount of time before automatically leaving this mode */
    gchar *led_pattern;
    GHashTable *mappings; /* int (key code) -> string (program to execute) */

    guint key_timer;            /* timer for key hold */
    guint timeout_timer;        /* timer for timeout for leaving mode */
} FailsafeMode;

typedef struct
{
    GObject parent;
    DBusGConnection *dbus;
    LibHalContext *hal;

    gboolean emulate_mce;
    GIOChannel *dsme;

    GHashTable *send_keys;  /* int (key code) -> string (name), key name to send out via dbus */
    FailsafeMode *failsafe;     /* NULL to disable failsafe mode */
    gboolean in_failsafe;       /* TRUE if we're in failsafe mode */

    GQueue *active_led_patterns; /* patterns currently being displayed */
    GHashTable *led_patterns;   /* string (name) -> LedPattern * */

    // these store the current device state
    gchar *device_mode;
    gint system_state;         /* runlevel */
    DisplayState display_state;
    char device_lock_code[12];
    gboolean system_inactivity;
    gboolean device_locked;
    gboolean screen_locked;
    gboolean keys_locked;
    gboolean led_disabled;
} Powered;

typedef struct
{
    GObjectClass parent;
} PoweredClass;

gboolean powered_request_activate_led_pattern(Powered *obj, gchar *pattern, GError **error);
gboolean powered_request_deactivate_all_led_patterns(Powered *server, GError **error);
gboolean powered_request_deactivate_led_pattern(Powered *obj, gchar *pattern, GError **error);
gboolean powered_request_define_led_pattern(Powered *server, gchar *name, gchar *pattern, GError **error);
gboolean powered_request_disable_led(Powered *obj, GError **error);
gboolean powered_request_display_blanking_pause(Powered *server, GError **error);
gboolean powered_request_display_dim(Powered *obj, GError **error);
gboolean powered_request_display_off(Powered *obj, GError **error);
gboolean powered_request_display_on(Powered *obj, GError **error);
gboolean powered_request_enable_led(Powered *obj, GError **error);
gboolean powered_request_get_device_lock(Powered *server, gboolean *lock, GError **error);
gboolean powered_request_get_device_lock_code(Powered *obj, gchar **code, GError **error);
gboolean powered_request_get_device_mode(Powered *server, gchar **mode, GError **error);
gboolean powered_request_get_display_status(Powered *server, gchar **status, GError **error);
gboolean powered_request_get_inactivity_status(Powered *server, gboolean *status, GError **error);
gboolean powered_request_get_key_lock(Powered *server, gboolean *lock, GError **error);
gboolean powered_request_get_screen_lock(Powered *server, gboolean *lock, GError **error);
gboolean powered_request_get_system_state(Powered *server, gint *state, GError **error);
gboolean powered_request_get_tk_lock(Powered *server, gchar **mode, GError **error);
gboolean powered_request_get_version(Powered *server, gchar **code, GError **error);
gboolean powered_request_lock_device(Powered *server, GError **error);
gboolean powered_request_lock_keys(Powered *obj, GError **error);
gboolean powered_request_lock_screen(Powered *obj, GError **error);
gboolean powered_request_powerup(Powered *server, GError **error);
gboolean powered_request_reboot(Powered *server, GError **error);
gboolean powered_request_set_blank_timeout(Powered *server, gchar *timeout, GError **error);
gboolean powered_request_set_device_lock_code(Powered *obj, gchar *code, GError **error);
gboolean powered_request_set_device_mode(Powered *server, gchar *mode, GError **error);
gboolean powered_request_set_dim_timeout(Powered *server, gchar *timeout, GError **error);
gboolean powered_request_set_display_brightness(Powered *server, gchar *brightness, GError **error);
gboolean powered_request_shutdown(Powered *server, GError **error);
gboolean powered_request_soft_power_off(Powered *obj, GError **error);
gboolean powered_request_soft_power_on(Powered *obj, GError **error);
gboolean powered_request_unlock_device(Powered *server, GError **error);
gboolean powered_request_unlock_keys(Powered *obj, GError **error);
gboolean powered_request_unlock_screen(Powered *obj, GError **error);

#include "mce.h"

#define powered_send_signal(server, args...) do { send_dbus_signal(server->dbus, POWERED_OBJECT_PATH, POWERED_INTERFACE, args); } while (0)

#endif
