/*
  Copyright (c) 2007-2008 Austin Che

  Power event daemon
  Waits for events and sends them out via dbus
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <linux/input.h>

#include "powered.h"
#include "utils.h"
#include "dsme.h"
#include "mce.h"
#include "led.h"

#include "powered-dbus.h"

#define POWERED_CONF "/etc/powered.conf"
#define POWERED_CONF_KEYS_GROUP "Keys"
#define POWERED_CONF_MAIN_GROUP "Main"
#define POWERED_CONF_FAILSAFE_GROUP "Failsafe"
#define POWERED_CONF_FAILSAFE_KEYS_GROUP "FailsafeKeys"
#define POWERED_CONF_LED_PATTERNS_GROUP "LEDPatterns"

#define TOUCHSCREEN_CONTROL "/sys/devices/platform/omap2_mcspi.1/spi1.0/disable_ts"
#define KEYPAD_CONTROL_N800 "/sys/devices/platform/omap2_mcspi.1/spi1.0/disable_kp"
#define KEYPAD_CONTROL_N810 "/sys/devices/platform/i2c_omap.2/i2c-0/0-0045/disable_kp"

const char *keypad_control = KEYPAD_CONTROL_N810;

G_DEFINE_TYPE(Powered, powered, G_TYPE_OBJECT);

/* ******************************************************************* 
   DBUS methods
   These return TRUE on success, FALSE otherwise
   If it returns FALSE, the error parameter must be initialized with g_set_error
   ******************************************************************* */

gboolean powered_request_get_inactivity_status(Powered *server, gboolean *status, GError **error)
{
    *status = server->system_inactivity;
    return TRUE;
}

gboolean powered_request_get_version(Powered *server, gchar **code, GError **error)
{
    *code = g_strdup(PACKAGE_VERSION);
    return TRUE;
}

gboolean powered_request_get_device_mode(Powered *server, gchar **mode, GError **error)
{
    *mode = g_strdup(server->device_mode);
    return TRUE;
}

gboolean powered_request_set_device_mode(Powered *server, gchar *mode, GError **error)
{
    if (!mode)
        return TRUE;

    g_free(server->device_mode);
    server->device_mode = g_strdup(mode);
    mce_send_signal(server, MCE_DEVICE_MODE_SIG, DBUS_TYPE_STRING, &server->device_mode, DBUS_TYPE_INVALID);
    return TRUE;
}

gboolean powered_request_get_system_state(Powered *server, gint *state, GError **error)
{
    get_system_state(server);
    *state = server->system_state;
    return TRUE;
}

gboolean powered_request_get_device_lock(Powered *server, gboolean *lock, GError **error)
{
    *lock = server->device_locked;
    return TRUE;
}

gboolean powered_request_lock_device(Powered *server, GError **error)
{
    gchar *ptr = MCE_DEVICE_LOCKED;
    mce_send_signal(server, MCE_DEVLOCK_MODE_SIG, DBUS_TYPE_STRING, &ptr, DBUS_TYPE_INVALID);
    server->device_locked = TRUE;
    return TRUE;
}

gboolean powered_request_unlock_device(Powered *server, GError **error)
{
    gchar *ptr = MCE_DEVICE_UNLOCKED;
    mce_send_signal(server, MCE_DEVLOCK_MODE_SIG, DBUS_TYPE_STRING, &ptr, DBUS_TYPE_INVALID);
    server->device_locked = FALSE;
    return TRUE;
}

gboolean powered_request_get_device_lock_code(Powered *server, gchar **code, GError **error)
{
    *code = g_strdup(get_devicelock_code(server));
    return TRUE;
}

gboolean powered_request_set_device_lock_code(Powered *server, gchar *code, GError **error)
{
    set_devicelock_code(server, code);
    return TRUE;
}

gboolean powered_request_display_blanking_pause(Powered *server, GError **error)
{
    display_blank_pause(server);
    return TRUE;
}

gboolean powered_request_get_display_status(Powered *server, gchar **status, GError **error)
{
    *status = g_strdup(get_display_state(server));
    return TRUE;
}

gboolean powered_request_set_blank_timeout(Powered *server, gchar *timeout, GError **error)
{
    set_display_blank_timeout(server, atoi(timeout));
    return TRUE;
}

gboolean powered_request_set_dim_timeout(Powered *server, gchar *timeout, GError **error)
{
    set_display_dim_timeout(server, atoi(timeout));
    return TRUE;
}

gboolean powered_request_set_display_brightness(Powered *server, gchar *brightness, GError **error)
{
    set_display_brightness(server, atoi(brightness));
    return TRUE;
}

gboolean powered_request_display_on(Powered *server, GError **error)
{
    set_display_state(server, DISPLAY_ON);
    return TRUE;
}

gboolean powered_request_display_dim(Powered *server, GError **error)
{
    set_display_state(server, DISPLAY_DIM);
    return TRUE;
}

gboolean powered_request_display_off(Powered *server, GError **error)
{
    set_display_state(server, DISPLAY_OFF);
    return TRUE;
}

gboolean powered_request_get_screen_lock(Powered *server, gboolean *lock, GError **error)
{
    *lock = server->screen_locked;
    return TRUE;
}

gboolean powered_request_lock_screen(Powered *server, GError **error)
{
    write_to_file(TOUCHSCREEN_CONTROL, "1");
    server->screen_locked = TRUE;
    return TRUE;
}

gboolean powered_request_unlock_screen(Powered *server, GError **error)
{
    write_to_file(TOUCHSCREEN_CONTROL, "0");
    server->screen_locked = FALSE;
    return TRUE;
}

gboolean powered_request_get_key_lock(Powered *server, gboolean *lock, GError **error)
{
    *lock = server->keys_locked;
    return TRUE;
}

gboolean powered_request_lock_keys(Powered *server, GError **error)
{
    write_to_file(keypad_control, "1");
    server->keys_locked = TRUE;
    return TRUE;
}

gboolean powered_request_unlock_keys(Powered *server, GError **error)
{
    write_to_file(keypad_control, "0");
    server->keys_locked = FALSE;
    return TRUE;
}

gboolean powered_request_soft_power_off(Powered *server, GError **error)
{
    powered_request_lock_keys(server, NULL);
    powered_request_lock_screen(server, NULL);
    set_display_state(server, DISPLAY_OFF);
    set_softpoweroff(server, TRUE);
    return TRUE;
}

gboolean powered_request_soft_power_on(Powered *server, GError **error)
{
    powered_request_unlock_keys(server, NULL);
    powered_request_unlock_screen(server, NULL);
    set_display_state(server, DISPLAY_ON);
    set_softpoweroff(server, FALSE);
    return TRUE;
}

gboolean powered_request_powerup(Powered *server, GError **error)
{
    g_debug("received powerup request");
    request_powerup(server);
    return TRUE;
}

gboolean powered_request_reboot(Powered *server, GError **error)
{
    g_debug("received reboot request");
    request_reboot(server);
    return TRUE;
}

gboolean powered_request_shutdown(Powered *server, GError **error)
{
    g_debug("received shutdown request");
    mce_send_signal(server, MCE_SHUTDOWN_SIG, DBUS_TYPE_INVALID);
    request_shutdown(server);
    return TRUE;
}

gboolean powered_request_define_led_pattern(Powered *server, gchar *name, gchar *pattern, GError **error)
{
    if (!pattern)
        return TRUE;

    gchar **split = g_strsplit(pattern, ";", 0);
    int num = g_strv_length(split);
    int *pat = g_new0(int, num);
    int i;
    for (i = 0; i < num; i++)
    {
        pat[i] = atoi(split[i]);
    }

    led_pattern_define(server, name, num, pat);

    g_free(pat);
    g_strfreev(split);
    return TRUE;
}

gboolean powered_request_activate_led_pattern(Powered *server, gchar *pattern, GError **error)
{
    led_pattern_activate(server, pattern);
    return TRUE;
}

gboolean powered_request_deactivate_led_pattern(Powered *server, gchar *pattern, GError **error)
{
    led_pattern_deactivate(server, pattern);
    return TRUE;
}

gboolean powered_request_deactivate_all_led_patterns(Powered *server, GError **error)
{
    led_pattern_deactivate_all(server);
    return TRUE;
}

gboolean powered_request_enable_led(Powered *server, GError **error)
{
    led_enable(server);
    return TRUE;
}

gboolean powered_request_disable_led(Powered *server, GError **error)
{
    led_disable(server);
    return TRUE;
}

/* ******************************************************************* 
   Events
   ******************************************************************* */

static gboolean failsafe_exit(Powered *server)
{
    g_debug("Exiting failsafe mode");
    server->in_failsafe = FALSE;
    led_pattern_deactivate(server, server->failsafe->led_pattern);
    server->failsafe->timeout_timer = 0;
    return FALSE;               /* to cancel timer */
}

static gboolean failsafe_enter(Powered *server)
{
    g_debug("Entering failsafe mode");
    server->in_failsafe = TRUE;
    server->failsafe->key_timer = 0;
    led_pattern_activate(server, server->failsafe->led_pattern);
    server->failsafe->timeout_timer = g_timeout_add(server->failsafe->timeout, (GSourceFunc)failsafe_exit, (gpointer)server);
    return FALSE;
}

static gboolean handle_failsafe_mode(Powered *server, struct input_event *in)
{
    if (in->value == 0)         /* ignore key releases, only process key press */
        return TRUE;

    gchar *program = g_hash_table_lookup(server->failsafe->mappings, GINT_TO_POINTER((int)in->code));
    if (program)
    {
        g_debug("powered failsafe mode: exec %s", program);
        if (!g_spawn_command_line_async(program, NULL))
            g_warning("Failed exec of %s", program);
    }

    // we exit failsafe mode on any key press
    g_source_remove(server->failsafe->timeout_timer); 
    failsafe_exit(server);
    return TRUE;
}

static gboolean hardware_keys(GIOChannel *source, GIOCondition condition, gpointer data)
{
    Powered *server = (Powered *)data;
    int fd = g_io_channel_unix_get_fd(source);
    struct input_event in;
    if (read(fd, &in, sizeof(in)) != sizeof(in))
    {
        g_warning("Failed to read input event");
        return TRUE;
    }        

    //g_debug("time %ld, %ld type %d code %d value %d", in.time.tv_sec, in.time.tv_usec, in.type, in.code, in.value);
    if (in.code == 0 || in.type == EV_SYN)
        return TRUE;                       // ignore these

    if (server->in_failsafe)
        return handle_failsafe_mode(server, &in);

    gboolean press = (in.value == 1);

    gchar *key = g_hash_table_lookup(server->send_keys, GINT_TO_POINTER((int)in.code));
    if (key)
        powered_send_signal(server, (press ? POWERED_SIGNAL_KEY_PRESS : POWERED_SIGNAL_KEY_RELEASE), DBUS_TYPE_STRING, &key, DBUS_TYPE_INVALID);

    if (server->failsafe && in.code == server->failsafe->key)
    {
        if (press)
            server->failsafe->key_timer = g_timeout_add(server->failsafe->key_hold, (GSourceFunc)failsafe_enter, (gpointer)server);
        else if (server->failsafe->key_timer) // cancel timer on key release
            g_source_remove(server->failsafe->key_timer); 
    }
    return TRUE;
}

/* ******************************************************************* 
   Powered object and initialization
   ******************************************************************* */

static guint lookup_key_code(GKeyFile *config, const gchar *key_name)
{
    return g_key_file_get_integer(config, POWERED_CONF_KEYS_GROUP, key_name, NULL);
}

static void load_config(Powered *server)
{
    GKeyFile *config = g_key_file_new();
    GError *err = NULL;
    if (! g_key_file_load_from_file(config, POWERED_CONF, G_KEY_FILE_NONE, &err))
    {
        g_warning("Could not load %s: %s", POWERED_CONF, err->message);
        g_key_file_free(config);
        g_error_free(err);
        return;
    }

    server->emulate_mce = g_key_file_get_boolean(config, POWERED_CONF_MAIN_GROUP, "emulate_mce", NULL);

    // keys to listen for and send out over dbus
    gsize num;
    gchar **keys = g_key_file_get_string_list(config, POWERED_CONF_MAIN_GROUP, "send_keys", &num, NULL);
    if (keys)
    {
        int i;
        for (i = 0; i < num; i++)
        {
            gchar *key = keys[i];
            g_hash_table_insert(server->send_keys, GINT_TO_POINTER(lookup_key_code(config, key)), key);
        }
        g_free(keys);           // reference to all the individual key strings is stored in hash table
    }

    // read LED patterns
    keys = g_key_file_get_string_list(config, POWERED_CONF_MAIN_GROUP, "led_patterns", &num, NULL);
    if (keys)
    {
        int i;
        for (i = 0; i < num; i++)
        {
            gchar *key = keys[i];
            gsize n;
            int *pattern = g_key_file_get_integer_list(config, POWERED_CONF_LED_PATTERNS_GROUP, key, &n, NULL);
            if (!pattern)
                continue;
            led_pattern_define(server, key, n, pattern);
            g_free(pattern);
        }
        g_strfreev(keys);
    }

    if (g_key_file_get_boolean(config, POWERED_CONF_FAILSAFE_GROUP, "enabled", NULL))
    {
        server->failsafe = g_new0(FailsafeMode, 1);
        gchar *failsafe_key = g_key_file_get_string(config, POWERED_CONF_FAILSAFE_GROUP, "key", NULL);
        if (failsafe_key)
        {
            server->failsafe->key = lookup_key_code(config, failsafe_key);
            g_free(failsafe_key);
        }
        server->failsafe->key_hold = g_key_file_get_integer(config, POWERED_CONF_FAILSAFE_GROUP, "key_hold", NULL);
        server->failsafe->timeout = g_key_file_get_integer(config, POWERED_CONF_FAILSAFE_GROUP, "timeout", NULL);
        server->failsafe->mappings = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);
        server->failsafe->led_pattern = g_key_file_get_string(config, POWERED_CONF_FAILSAFE_GROUP, "led_pattern", NULL);

        // failsafe keys
        keys = g_key_file_get_keys(config, POWERED_CONF_FAILSAFE_KEYS_GROUP, &num, NULL);
        if (keys)
        {
            int i;
            for (i = 0; i < num; i++)
            {
                gchar *key = keys[i];
                gchar *program = g_key_file_get_string(config, POWERED_CONF_FAILSAFE_KEYS_GROUP, key, NULL);
                g_hash_table_insert(server->failsafe->mappings, GINT_TO_POINTER(lookup_key_code(config, key)), program);
            }
            g_strfreev(keys);           // reference to all the individual key strings is stored in hash table
        }
    }

    g_key_file_free(config);
}

static void powered_class_finalize(GObject *obj)
{
    Powered *server = (Powered *)obj;

    dbus_g_connection_unref(server->dbus);

    g_queue_free(server->active_led_patterns);
    g_hash_table_destroy(server->led_patterns);
    g_hash_table_destroy(server->send_keys);
    if (server->failsafe)
    {
        g_hash_table_destroy(server->failsafe->mappings);
        g_free(server->failsafe->led_pattern);
        g_free(server->failsafe);
    }
    g_free(server->device_mode);
}

static void powered_class_init(PoweredClass *class)
{
    GObjectClass *gobj_class = G_OBJECT_CLASS(class);
    dbus_g_object_type_install_info(powered_get_type(), &dbus_glib_powered_object_info);    
    gobj_class->finalize = powered_class_finalize;
}

static void powered_init(Powered *server) 
{
    server->send_keys = g_hash_table_new_full(g_direct_hash, g_direct_equal, NULL, g_free);
    server->emulate_mce = FALSE;
    server->in_failsafe = FALSE;
    server->device_mode = g_strdup(POWERED_MODE_NORMAL);
    server->led_patterns = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    server->active_led_patterns = g_queue_new();
    load_config(server);

    GError *error = NULL;
    
    server->dbus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (! server->dbus) 
    {
        g_printerr("Failed to open connection to bus: %s\n", error->message);
        g_error_free(error);
        exit(1);
    }

    // Create a proxy object for the bus daemon "org.freedesktop.DBus"
    DBusGProxy *bus_proxy = dbus_g_proxy_new_for_name(server->dbus, DBUS_SERVICE_DBUS, DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);
    guint request_ret;
    if (!org_freedesktop_DBus_request_name(bus_proxy, POWERED_SERVICE, 0, &request_ret, &error)) 
    {
        g_warning("Unable to request dbus name: %s", error->message);
        g_error_free(error);
        exit(1);
    }
    if (request_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
    {
        g_warning("There's already someone with the name " POWERED_SERVICE);
        exit(1);
    }

    mce_init(server, bus_proxy);

    g_object_unref(bus_proxy);

    // Register DBUS path
    dbus_g_connection_register_g_object(server->dbus, POWERED_OBJECT_PATH, G_OBJECT(server));

    DBusConnection *conn = dbus_g_connection_get_connection(server->dbus);
    server->hal = libhal_ctx_new();
    libhal_ctx_set_user_data(server->hal, server);
    libhal_ctx_set_dbus_connection(server->hal, conn);
    libhal_ctx_init(server->hal, NULL);
}

int main(int argc, char **argv)
{  
    g_type_init();

    Powered *server = g_object_new(powered_get_type(), NULL);

    GMainLoop *main_loop = g_main_loop_new(NULL, FALSE);
    GIOChannel *io;

    int num;
    char **devs = libhal_find_device_by_capability(server->hal, "input", &num, NULL);
    int i;
    for (i = 0; i < num; i++)
    {
        if (libhal_device_query_capability(server->hal, devs[i], "button", NULL))
        {
            char *dev = libhal_device_get_property_string(server->hal, devs[i], "input.device", NULL);
            g_debug("Opening input device: %s", dev);
            io = g_io_channel_new_file(dev, "r", NULL);
            if (io)
                g_io_add_watch(io, G_IO_IN, hardware_keys, server);
            libhal_free_string(dev);
        }
    }
    if (devs)
        libhal_free_string_array(devs);

    if (g_file_test(KEYPAD_CONTROL_N800, G_FILE_TEST_EXISTS))
        keypad_control = KEYPAD_CONTROL_N800;

    dsme_connect(server);
    //bme_init(server);
    get_system_state(server);
    get_devicelock_code(server);

    // just in case the device mode was something other than "normal" and we were restarted, we need to notify people
    mce_send_signal(server, MCE_DEVICE_MODE_SIG, DBUS_TYPE_STRING, &server->device_mode, DBUS_TYPE_INVALID);
    
    g_debug("Begin main loop");
    g_main_loop_run(main_loop);

    dsme_disconnect(server);
    g_main_loop_unref(main_loop);

    libhal_ctx_shutdown(server->hal, NULL);
    libhal_ctx_free(server->hal);
    g_object_unref(server);

    return 0;
}
