/*
  Copyright (c) 2007-2008 Austin Che

  MCE compatibility
*/

#define _XOPEN_SOURCE // for crypt(3)
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <dbus/dbus.h>

#include "powered.h"
#include "mce.h"
#include "utils.h"
#include "led.h"
#include "dsme.h"

#define MCE_CONFIG "/etc/mce/mce.ini"

/* ******************************************************************* 
   DBUS methods
   These return TRUE on success, FALSE otherwise
   If it returns FALSE, the error parameter must be initialized with g_set_error
   ******************************************************************* */

gboolean mce_request_set_alarm_mode(Powered *server, gchar *mode, GError **error)
{
    if (strcmp(mode, MCE_ALARM_VISIBLE) == 0)
        set_alarm_mode(server, ALARM_VISIBLE);
    else if (strcmp(mode, MCE_ALARM_SNOOZED) == 0)    
        set_alarm_mode(server, ALARM_SNOOZED);
    else if (strcmp(mode, MCE_ALARM_OFF) == 0)    
        set_alarm_mode(server, ALARM_OFF);
    else
        g_warning("unknown mode %s for set_alarm_mode", mode);
    return TRUE;
}

gboolean mce_request_validate_devicelock_code(Powered *server, gchar *code, gchar *salt, gboolean *result, GError **error)
{
    gchar *encrypted = crypt(get_devicelock_code(server), salt);
    encrypted += strlen(salt); /* skip over salt portion */
    if (strcmp(encrypted, code) == 0)
        *result = TRUE;
    else
        *result = FALSE;
    return TRUE;
}

gboolean mce_request_change_devicelock_code(Powered *server, gchar *oldcode, gchar *salt, gchar *newcode, gboolean *result, GError **error)
{
    mce_request_validate_devicelock_code(server, oldcode, salt, result, error);
    if (*result)
        set_devicelock_code(server, newcode);
    return TRUE;
}

gboolean mce_request_req_trigger_powerkey_event(Powered *server, gboolean longpress, GError **error)
{
    // we ignore this, not sure what app cares about this
    g_debug("Ignoring req_trigger_powerkey_event");
    return TRUE;
}

gboolean mce_request_get_devicelock_mode(Powered *server, gchar **mode, GError **error)
{
    *mode = g_strdup(server->device_locked ? MCE_DEVICE_LOCKED : MCE_DEVICE_UNLOCKED);
    return TRUE;
}

gboolean mce_request_req_tklock_mode_change(Powered *server, gchar *mode, GError **error)
{
    // we just send out a signal indicating that the mode changed
    // we do nothing here
    mce_send_signal(server, MCE_TKLOCK_MODE_SIG, DBUS_TYPE_STRING, &mode, DBUS_TYPE_INVALID);
    return TRUE;
}

gboolean mce_request_get_tklock_mode(Powered *server, gchar **mode, GError **error)
{
    // we always return unlocked as we don't and shouldn't be keeping track of tklock mode
    *mode = g_strdup(MCE_TK_UNLOCKED);
    g_debug("In mce_request_get_tklock_mode and returning bogus information");
    return TRUE;
}

gboolean mce_request_get_version(Powered *server, gchar **version, GError **error)
{
    // get_version for real mce returns "1.6.45"
    *version = g_strdup("1.6.45");
    return TRUE;
}

/* ******************************************************************* 
   Init
   ******************************************************************* */

void mce_init(Powered *server, DBusGProxy *bus)
{
    if (!server->emulate_mce)
        return;

    GError *error = NULL;
    gsize request_ret;
    if (!org_freedesktop_DBus_request_name(bus, MCE_SERVICE, 0, &request_ret, &error)) 
    {
        g_warning("Unable to request dbus name for mce: %s", error->message);
        g_error_free(error);
        exit(1);
    }
    
    // if we're not able to get on dbus as mce, we assume mce is already present
    // and stop emulating mce
    if (request_ret != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
    {
        g_warning("Could not get on bus as mce. Assuming mce is running.");
        server->emulate_mce = FALSE;
        return;
    }

    // ** There appear to be some problems registering the same object at multiple paths
    // On finalization of the server object, an error is thrown
    // It shouldn't affect us much
    dbus_g_connection_register_g_object(server->dbus, MCE_REQUEST_PATH, G_OBJECT(server));

    // read /etc/mce/mce.ini for LED patterns
    GKeyFile *config = g_key_file_new();
    if (! g_key_file_load_from_file(config, MCE_CONFIG, G_KEY_FILE_NONE, NULL))
    {
        g_warning("Could not load %s", MCE_CONFIG);
        g_key_file_free(config);
        return;
    }

    gsize num_patterns;
    gchar **patterns = g_key_file_get_string_list(config, "LED", "LEDPatterns", &num_patterns, NULL);
    if (patterns)
    {
        int i;
        for (i = 0; i < num_patterns; i++)
        {
            gsize num;
            gint *pattern = g_key_file_get_integer_list(config, "LEDPatternSingleColour", patterns[i], &num, NULL);
            if (!pattern)
                continue;
            if (num == 6)
            {
                // map mce's pattern format to ours
                int pat[7];
                pat[0] = 1;     /* type: timer */
                pat[1] = pattern[0]; /* priority */
                pat[2] = pattern[1]; /* when to display */
                pat[3] = pattern[2]; /* timeout */
                pat[4] = pattern[5] * 17; /* brightness */
                pat[5] = pattern[3]; /* on period */
                pat[6] = pattern[4]; /* off period */
                led_pattern_define(server, patterns[i], 7, pat);
            }
            g_free(pattern);
        }
        g_strfreev(patterns);
    }

    g_key_file_free(config);
}
