/*
  Copyright (c) 2007-2008 Austin Che

  Controls device LED
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <glib.h>

#include "powered.h"
#include "utils.h"
#include "led.h"

#define LED_COVER "/sys/class/leds/cover/"

#define LED_CONTROL_DIR "/sys/class/leds/keypad/"

// scale from 0 (off) to 255 (brightest)
#define LED_BRIGHTNESS LED_CONTROL_DIR "brightness"

#define LED_DELAY_ON LED_CONTROL_DIR "delay_on"
#define LED_DELAY_OFF LED_CONTROL_DIR "delay_off"

// This file determines how LEDs are triggered. Can be one of:
// - none: which means it's either always on or always off, depending on brightness
// - timer: delay_off and delay_on specify pattern of on/off
// - heartbeat: light pulses on and off like heart
#define LED_TRIGGER LED_CONTROL_DIR "trigger"

typedef enum {
    LED_PATTERN_NONE = 0,
    LED_PATTERN_TIMER = 1,
    LED_PATTERN_HEARTBEAT = 2,
    LED_PATTERN_RGB = 3,
} LedPatternType;

typedef struct _LedActivePattern LedActivePattern;

typedef struct
{
    int delay_on;
    int delay_off;
} LedPatternTimer;

typedef struct
{
} LedPatternHeartbeat;

typedef struct
{
    gchar *red;
    gchar *green;
    gchar *blue;
} LedPatternRGB;

typedef struct
{
    LedPatternType type;
    guint priority;               /* 0 (highest priority) to whatever number you want (lowest) */
    guint when;                   /* under what conditions this pattern will be displayed */
    guint timeout;                /* how long this pattern will stay active (sec) (0 for infinity) */
    LedActivePattern *active;   /* pointer to active struct if it's active */

    guint brightness;             /* 0 to 255 */

    union 
    {
        LedPatternTimer timer;
        LedPatternHeartbeat heartbeat;
    } params;
} LedPattern;


struct _LedActivePattern
{
    Powered *server;
    LedPattern *pattern;
    guint timer;
    GTimeVal activation_time;        /* when this pattern was activated */
};

static void led_set_pattern(LedPatternType type)
{
    switch (type)
    {
        case LED_PATTERN_NONE:
            write_to_file(LED_TRIGGER, "none");
            break;
        case LED_PATTERN_TIMER:
            write_to_file(LED_TRIGGER, "timer");
            break;
        case LED_PATTERN_HEARTBEAT:
            write_to_file(LED_TRIGGER, "heartbeat");
            break;
    }    
}

static void led_activate_timer(guint delay_off, guint delay_on)
{
    write_to_file_int(LED_DELAY_OFF, delay_off);
    write_to_file_int(LED_DELAY_ON, delay_on);
}

static void led_set_brightness(guint brightness)
{
    write_to_file_int(LED_BRIGHTNESS, brightness);    
}

static void led_deactivate()
{
    led_set_brightness(0);
    led_set_pattern(LED_PATTERN_NONE);
}

static void led_show_pattern(LedPattern *pattern)
{
    if (pattern)
    {
        led_set_brightness(pattern->brightness);
        led_set_pattern(pattern->type);
        if (pattern->type == LED_PATTERN_TIMER)
            led_activate_timer(pattern->params.timer.delay_off, pattern->params.timer.delay_on);
    }
    else
        led_deactivate();
}

static void led_update(Powered *server)
{
    // get highest priority pattern that should be displayed currently
    GList *ptr = server->active_led_patterns->head;
    while (ptr)
    {
        // *** todo: check if this pattern should be active
        break;
        ptr = ptr->next;
    }

    LedPattern *pattern = NULL;
    if (ptr)
        pattern = ((LedActivePattern *)ptr->data)->pattern;

    if (pattern && !server->led_disabled)
        led_show_pattern(pattern);
    else
        led_show_pattern(NULL);
}

static gint led_pattern_compare_priority(gconstpointer a, gconstpointer b, gpointer data)
{
    if (a == b)
        return 0;

    LedActivePattern *p1 = ((LedActivePattern *)a);
    LedActivePattern *p2 = ((LedActivePattern *)b);

    if (p1->pattern->priority == p2->pattern->priority)
    {
        // the pattern that was activated last will be given higher priority
        if (p1->activation_time.tv_sec == p2->activation_time.tv_sec)
            return (p1->activation_time.tv_usec < p2->activation_time.tv_usec) ? 1 : -1;
        else
            return (p1->activation_time.tv_sec < p2->activation_time.tv_sec) ? 1 : -1;
    }
    else
        return p1->pattern->priority - p2->pattern->priority;
}

static void led_free_active_pattern(LedActivePattern *pattern)
{
    pattern->pattern->active = NULL;
    if (pattern->timer)
        g_source_remove(pattern->timer); 
    g_free(pattern);
}

static void led_remove_active_pattern(LedActivePattern *pattern)
{
    g_queue_remove(pattern->server->active_led_patterns, pattern);
    led_update(pattern->server);
    led_free_active_pattern(pattern);
}

void led_pattern_define(Powered *server, const gchar *name, int num, int *pattern)
{
    if (!name)
        return;

    // check number of arguments
    if (num < 5)
        return;

    LedPatternType type = pattern[0];
    if ((type == LED_PATTERN_NONE) || (type == LED_PATTERN_HEARTBEAT))
    {
        if (num != 5)
            return;
    }
    else if (type == LED_PATTERN_TIMER)
    {
        if (num != 7)
            return;
    }
    else
        return;

    led_pattern_deactivate(server, name);

    LedPattern *lpattern = g_new0(LedPattern, 1);
    lpattern->type = type;
    lpattern->priority = pattern[1];
    lpattern->when = pattern[2];
    lpattern->timeout = pattern[3];
    lpattern->brightness = pattern[4];
    if (type == LED_PATTERN_TIMER)
    {
        lpattern->params.timer.delay_on = pattern[5];
        lpattern->params.timer.delay_off = pattern[6];
    }

    g_hash_table_insert(server->led_patterns, g_strdup(name), lpattern);
}

void led_enable(Powered *server)
{
    server->led_disabled = FALSE;
    led_update(server);
}

void led_disable(Powered *server)
{
    server->led_disabled = TRUE;
    led_update(server);
}

void led_pattern_activate(Powered *server, const gchar *name)
{
    if (!name)
        return;

    LedPattern *pattern = g_hash_table_lookup(server->led_patterns, name);
    if (!pattern)
        return;

    if (pattern->active)
        led_remove_active_pattern(pattern->active);
    LedActivePattern *active = g_new0(LedActivePattern, 1);
    active->server = server;
    active->pattern = pattern;
    g_get_current_time(&active->activation_time);

    if (pattern->timeout)
        active->timer = g_timeout_add(pattern->timeout * 1000, (GSourceFunc)led_remove_active_pattern, (gpointer)active);

    pattern->active = active;
    g_queue_insert_sorted(server->active_led_patterns, active, led_pattern_compare_priority, NULL);
    led_update(server);
}

void led_pattern_deactivate(Powered *server, const gchar *name)
{
    if (!name)
        return;

    LedPattern *pattern = g_hash_table_lookup(server->led_patterns, name);
    if (!pattern)
        return;

    if (pattern->active)
        led_remove_active_pattern(pattern->active);
}

void led_pattern_deactivate_all(Powered *server)
{
    GList *ptr = server->active_led_patterns->head;
    while (ptr)
    {
        led_free_active_pattern(ptr->data);
        ptr = ptr->next;
    }
    g_queue_free(server->active_led_patterns);
    server->active_led_patterns = g_queue_new();
    led_update(server);
}
