/*
  Copyright (c) 2007-2008 Austin Che

  For communicating with DSME
  Most of this was reverse-engineered and is not guaranteed to be correct
*/

#include <stdio.h>
#include <glib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <dbus/dbus.h>

#include "powered.h"
#include "utils.h"
#include "dsme.h"
#include "mce.h"

#define DISPLAY_BLANK_PAUSE_TIMEOUT 60000 /* 60 seconds */

#define DSME_GET_DEVICELOCK_CODE 0x1200
#define DSME_IN_GET_DEVICELOCK_CODE 0x1201
#define DSME_SET_DEVICELOCK_CODE 0x1202
#define DSME_IN_SET_DEVICELOCK_CODE 0x1203

#define DSME_DISPLAY_BLANKING_PAUSE 0x0201
#define DSME_DISPLAY_BLANKING_ALLOW 0x0202
#define DSME_SET_DISPLAY_BLANK_TIMEOUT 0x0203
#define DSME_SET_DISPLAY_DIM_TIMEOUT 0x0204
#define DSME_SET_DISPLAY_STATE 0x0280
#define DSME_IN_DISPLAY_STATE 0x0285
#define DSME_SET_DISPLAY_BRIGHTNESS 0x0289

#define DSME_IN_SYSTEM_STATE 0x0301
#define DSME_GET_SYSTEM_STATE 0x0302
#define DSME_IN_SYSTEM_INACTIVITY 0x0303
#define DSME_IN_SAVE_UNSAVED_DATA 0x0304
#define DSME_REQUEST_POWERUP 0x0305
#define DSME_REQUEST_SHUTDOWN 0x0306
#define DSME_SET_ALARM_MODE 0x0307
#define DSME_REQUEST_REBOOT 0x0308
#define DSME_IN_SYSTEM_ACTIVITY 0x0311


// I've received the string /usr/sbin/ke-recv and /usr/bin/hildon-input-method with this message on shutdown
// these programs crash on shutdown. I think this is a message about that
// the above strings followed by \0 0x01 \0 and then 0x2c or 0x30
//#define DSME_IN_PROCESS_UNKNOWN 0x0406

// PROCESS_PING 0x0504
#define DSME_SET_SOFTPOWERON 0x0800
#define DSME_SET_SOFTPOWEROFF 0x0801

static guint display_blank_pause_timer = 0;

static void dsme_write(Powered *server, const void *buf, size_t nbytes)
{
    gsize written;
    g_io_channel_write_chars(server->dsme, buf, nbytes, &written, NULL);
    if (written != nbytes)
        g_warning("Could not write %d bytes to dsme. Only wrote %d bytes", nbytes, written);
}

static void dsme_write_simple_msg(Powered *server, guint32 type)
{
    struct
    {
        guint32 len;
        guint32 type;
    } msg;
    msg.len = sizeof(msg);
    msg.type = type;
    dsme_write(server, &msg, sizeof(msg));
}

static void dsme_write_msg_one_arg(Powered *server, guint32 type, guint32 arg)
{
    struct
    {
        guint32 len;
        guint32 type;
        guint32 data;
    } msg;
    msg.len = sizeof(msg);
    msg.type = type;
    msg.data = arg;
    dsme_write(server, &msg, sizeof(msg));
}

static int dsme_read(Powered *server, void *buf, size_t nbytes)
{
    gsize read;
    g_io_channel_read_chars(server->dsme, buf, nbytes, &read, NULL);
    if (read != nbytes)
    {
        g_warning("Could not read %d bytes from dsme, got %d bytes", nbytes, read);
        return -1;
    }
    return read;
}

static gboolean check_data_size(size_t read_bytes, size_t expected)
{
    if (read_bytes != expected)
    {
        g_warning("unexpected data size: expected %d, got %d", expected, read_bytes);
        return FALSE;
    }
    return TRUE;
}

static void parse_lock_code(Powered *server, void *buf, size_t nbytes)
{
    struct
    {
        char str[16];           /* "lock_code" */
        guint32 a1;             /* 0x01 */
        guint32 a2;             /* 0x10 */
        guint32 a3;             /* 0x01 */
        char code[12];          /* the device code */
    } *msg = buf;
    
    if (!check_data_size(nbytes, sizeof(*msg)))
        return;

    if (strcmp(msg->str, "lock_code") == 0)
        strncpy(server->device_lock_code, msg->code, sizeof(server->device_lock_code));
    else
        g_debug("Unknown dsme lock code message with string %s", msg->str);
}

static void parse_display_state(Powered *server, void *buf, size_t nbytes)
{
    if (!check_data_size(nbytes, 4))
        return;
    server->display_state = (*(int *)buf);

    gchar *name = get_display_state(server);

    if (server->display_state != DISPLAY_OFF)
    {
        // we auto unlock keys/screen on display on simply because this could cause a lot of headaches if they remained locked
        powered_request_unlock_keys(server, NULL);
        powered_request_unlock_screen(server, NULL);
    }

    powered_send_signal(server, POWERED_SIGNAL_DISPLAY_STATE, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
    mce_send_signal(server, MCE_DISPLAY_SIG, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
}

static void parse_system_state(Powered *server, void *buf, size_t nbytes)
{
    // DSME sends us the runlevel as the system state
    // 0x00 on shutdown
    // 0x02 normally
    // 0x05 on shutdown and on charger, should go to actdead
    // 0x06 on reboot
    if (!check_data_size(nbytes, 4))
        return;
    int state = (*(int *)buf);
    powered_send_signal(server, POWERED_SIGNAL_SYSTEM_STATE, DBUS_TYPE_INT32, &state, DBUS_TYPE_INVALID);
    server->system_state = state;
}

static gboolean read_from_dsme(GIOChannel *source, GIOCondition condition, gpointer data)
{
    Powered *server = (Powered *)data;
    struct
    {
        guint32 len;
        guint32 type;
    } header;

    int ret;
    if ((ret = dsme_read(server, &header, sizeof(header))) < 0)
    {
        g_warning("Could not read dsme header...Bailing");
        return TRUE;
    }

    char buf[256];
    int data_size = header.len - ret;
    if (data_size > sizeof(buf))
    {
        g_warning("Length in dsme header %d is unexpectedly large", header.len);
        g_io_channel_seek_position(source, 0, G_SEEK_END, NULL);
        return TRUE;
    }
        
    if ((ret = dsme_read(server, buf, data_size)) < 0)
    {
        g_warning("Could not read dsme data");
        return TRUE;
    }

    int i;
    gboolean b_true = TRUE;
    gboolean b_false = FALSE;
    switch (header.type)
    {
        case DSME_IN_SYSTEM_STATE:
            parse_system_state(server, buf, ret);
            break;
        case DSME_IN_GET_DEVICELOCK_CODE:
            parse_lock_code(server, buf, ret);
            break;            
        case DSME_IN_DISPLAY_STATE:
            parse_display_state(server, buf, ret);
            break;
        case DSME_IN_SET_DEVICELOCK_CODE:
            /* data payload is all 0 */
            break;

        case DSME_IN_SYSTEM_INACTIVITY:
            server->system_inactivity = TRUE;
            powered_send_signal(server, POWERED_SIGNAL_INACTIVITY, DBUS_TYPE_BOOLEAN, &b_true, DBUS_TYPE_INVALID);
            mce_send_signal(server, MCE_INACTIVITY_SIG, DBUS_TYPE_BOOLEAN, &b_true, DBUS_TYPE_INVALID);
            break;

        case DSME_IN_SYSTEM_ACTIVITY:
            server->system_inactivity = FALSE;
            powered_send_signal(server, POWERED_SIGNAL_INACTIVITY, DBUS_TYPE_BOOLEAN, &b_false, DBUS_TYPE_INVALID);
            mce_send_signal(server, MCE_INACTIVITY_SIG, DBUS_TYPE_BOOLEAN, &b_false, DBUS_TYPE_INVALID);
            break;

        case DSME_IN_SAVE_UNSAVED_DATA:
            powered_send_signal(server, POWERED_SIGNAL_SAVE_DATA, DBUS_TYPE_INVALID);
            mce_send_signal(server, MCE_DATA_SAVE_SIG, DBUS_TYPE_INVALID);
            break;

        default:
            g_debug("Unhandled message from dsme of type 0x%x with %d bytes of data", header.type, ret);

            for (i = 0; i < ret; i++)
            {
                printf("%02x ", buf[i]);
                if (i % 50 == 49)
                    printf("\n");
            }
            if (ret)
                printf("\n");

            break;
    }

    return TRUE;                // don't remove this event source
}

static gboolean display_blank_allow(Powered *server)
{
    dsme_write_simple_msg(server, DSME_DISPLAY_BLANKING_ALLOW);
    display_blank_pause_timer = 0;
    return FALSE;               // cancel timer
}

/* ******************************************************************* 
   Exported functions
   ******************************************************************* */

void dsme_connect(Powered *server)
{
    struct sockaddr_un addr;

    int dsmesock = socket(PF_UNIX, SOCK_STREAM, 0);
    if (dsmesock < 0) 
        g_error("dsme socket failed");

    memset(&addr, 0, sizeof(addr));
    addr.sun_family = AF_UNIX;
    strcpy(addr.sun_path, "/tmp/dsmesock");
  
    if (connect(dsmesock, (struct sockaddr *)&addr, sizeof(addr)) < 0)
    {
        close(dsmesock);
        g_error("dsme connect failed");
    }

    g_debug("Connected to dsme");

    server->dsme = g_io_channel_unix_new(dsmesock);

    g_io_channel_set_flags(server->dsme, G_IO_FLAG_NONBLOCK, NULL);
    g_io_channel_set_encoding(server->dsme, NULL, NULL);
    g_io_channel_set_buffered(server->dsme, FALSE);
    g_io_channel_set_close_on_unref(server->dsme, TRUE);
    g_io_add_watch(server->dsme, G_IO_IN, read_from_dsme, server);
}

void set_softpoweroff(Powered *server, gboolean off)
{
    // we override any display blank pause if softpoweroff is requested
    if (off && display_blank_pause_timer)
    {
        display_blank_allow(server);
        g_source_remove(display_blank_pause_timer);
    }

    dsme_write_simple_msg(server, off ? DSME_SET_SOFTPOWEROFF : DSME_SET_SOFTPOWERON);
}

void display_blank_pause(Powered *server)
{
    dsme_write_simple_msg(server, DSME_DISPLAY_BLANKING_PAUSE);
    if (display_blank_pause_timer)
        g_source_remove(display_blank_pause_timer); 
    display_blank_pause_timer = g_timeout_add(DISPLAY_BLANK_PAUSE_TIMEOUT, (GSourceFunc)display_blank_allow, (gpointer)server);
}

void get_system_state(Powered *server)
{
    // this should cause dsme to send us back the state
    // we don't block for it here
    dsme_write_simple_msg(server, DSME_GET_SYSTEM_STATE);
}

gchar *get_display_state(Powered *server)
{
    // these strings match mce's strings
    static gchar *DISPLAY_STATE_NAMES[] = {
        "on",   // MCE_DISPLAY_ON_STRING
        "dimmed", // MCE_DISPLAY_DIM_STRING
        "off", // MCE_DISPLAY_OFF_STRING
    };

    if (server->display_state > 2)
    {
        g_warning("Unknown display state %d", server->display_state);
        return "unknown";
    }
    else
        return DISPLAY_STATE_NAMES[server->display_state];
}

void set_display_blank_timeout(Powered *server, guint timeout)
{
    dsme_write_msg_one_arg(server, DSME_SET_DISPLAY_BLANK_TIMEOUT, timeout);
}

void set_display_dim_timeout(Powered *server, guint timeout)
{
    dsme_write_msg_one_arg(server, DSME_SET_DISPLAY_DIM_TIMEOUT, timeout);
}

void set_display_brightness(Powered *server, guint brightness)
{
    dsme_write_msg_one_arg(server, DSME_SET_DISPLAY_BRIGHTNESS, brightness);
}

void set_display_state(Powered *server, DisplayState state)
{
    // data of 0 will turn display on, 1 will dim, and 2 will turn display off
    dsme_write_msg_one_arg(server, DSME_SET_DISPLAY_STATE, state);
}

gchar *get_devicelock_code(Powered *server)
{
    struct
    {
        guint32 len;
        guint32 type;
        char data[16];
        guint32 a1;
        guint32 a2;             /* this changes, not sure what this is */
    } msg;
    msg.len = sizeof(msg);
    msg.type = DSME_GET_DEVICELOCK_CODE;
    strncpy(msg.data, "lock_code", sizeof(msg.data));
    msg.a1 = 0x01;
    msg.a2 = 0x00027ec8;

    // this should cause dsme to send us back the lock code
    // we don't block for it (in general, avoiding any type of blocking)
    // we return server->device_lock_code which was the last code we knew about which should be the current code
    // unless some other process changed it behind our back, in which case the next call to this function will be correct
    dsme_write(server, &msg, sizeof(msg));
    return server->device_lock_code;
}

void set_devicelock_code(Powered *server, const char *code)
{
    // at most only, the first 10 characters of the code are used
    struct
    {
        guint32 len;
        guint32 type;
        char data[16];
        guint32 a1;
        guint32 a2;
        guint32 a3;
        char code[12];
    } msg;
    msg.len = sizeof(msg);
    msg.type = DSME_SET_DEVICELOCK_CODE;
    strncpy(msg.data, "lock_code", sizeof(msg.data));
    msg.a1 = 0x01;
    msg.a2 = 0x10;
    msg.a3 = 0x01;
    strncpy(msg.code, code, 10);
    dsme_write(server, &msg, sizeof(msg));
    get_devicelock_code(server); // get the new code back from dsme
}

void set_alarm_mode(Powered *server, AlarmMode mode)
{
    dsme_write_msg_one_arg(server, DSME_SET_ALARM_MODE, mode);
}

void request_powerup(Powered *server)
{
    dsme_write_simple_msg(server, DSME_REQUEST_POWERUP);
}

void request_shutdown(Powered *server)
{
    dsme_write_msg_one_arg(server, DSME_REQUEST_SHUTDOWN, 0);
}

void request_reboot(Powered *server)
{
    dsme_write_simple_msg(server, DSME_REQUEST_REBOOT);
}

void dsme_disconnect(Powered *server)
{
    if (server->dsme)
        g_io_channel_shutdown(server->dsme, TRUE, NULL);
    server->dsme = NULL;
}

