/*
  TOE - Tims own texteditor.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "BaseDialogs.h"

#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>

#include <Lum/Model/String.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/Table.h>

#include <Lum/Panel.h>
#include <Lum/Space.h>
#include <Lum/Text.h>
#include <Lum/WindowGroup.h>

#include "Database.h"

#include "Data.h"
#include <iostream>
class ValueDialog : public Lum::Dialog
{
private:
  bool                  result;
  Lum::Model::ActionRef okAction;
  std::wstring          caption;
  size_t                size;

public:
  Lum::Model::StringRef value;

public:
  ValueDialog(const std::wstring& caption, size_t size, const std::wstring& value)
  : result(false),
    okAction(new Lum::Model::Action),
    caption(caption),
    size(size),
    value(new Lum::Model::String)
  {
    AttachModel(okAction);
    AttachModel(GetClosedAction());

    this->value->Set(value);
  }

  ~ValueDialog()
  {
    UnattachModel(GetClosedAction());
    UnattachModel(okAction);
  }

  virtual void PreInit()
  {
    Lum::Button      *button;
    Lum::ButtonRow   *row;
    Lum::String      *string;
    Lum::Panel       *vPanel;
    Lum::Text        *text;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,false);

    vPanel=new Lum::VPanel;
    vPanel->SetFlex(true,false);

    text=new Lum::Text;
    text->SetFlex(true,false);
    text->SetText(caption);

    vPanel->Add(text);

    vPanel->Add(new Lum::VSpace);

    string=new Lum::String;
    string->SetFlex(true,false);
    string->SetWidth(Lum::Base::Size::stdCharWidth,size);
    string->SetModel(value);

    vPanel->Add(string);

    vPanel->Add(new Lum::VSpace);

    row=new Lum::ButtonRow();
    row->SetFlex(true,false);

    button=new Lum::Button;
    button->SetFlex(true,true);
    button->SetModel(okAction);
    button->SetText(L"_OK");
    button->SetType(Lum::Button::typeCommit);
    row->Add(button);

    button=new Lum::Button;
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    button->SetText(L"_Cancel");
    button->SetType(Lum::Button::typeCancel);
    row->Add(button);

    vPanel->Add(row);

    wGroup->SetMain(vPanel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      result=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else {
      Dialog::Resync(model,msg);
    }
  }

  bool GetResult() const
  {
    return result;
  }
};

bool GetStringValue(Lum::OS::Window* parent,
                    const std::wstring& caption,
                    size_t width,
                    std::wstring& value)
{
  ValueDialog *dialog;

  dialog=new ValueDialog(caption,width,value);
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  if (!dialog->GetResult()) {
    delete dialog;
    return false;
  }

  value=dialog->value->Get();
  delete dialog;
  return true;
}

class LinkNameTypeDialog : public Lum::Dialog
{
private:
  Lum::Model::ListTableRef model;
  Lum::Model::ActionRef    okAction;
  Lum::Model::StringRef    nameModel;
  Lum::Model::StringRef    typeModel;
  bool                     success;

public:
  LinkNameTypeDialog()
  : model(new Lum::Model::ListTable(1)),
    okAction(new Lum::Model::Action),
    nameModel(new Lum::Model::String),
    typeModel(new Lum::Model::String),
    success(false)
  {
    AttachModel(okAction);
    AttachModel(GetClosedAction());
    AttachModel(nameModel);
    AttachModel(typeModel);

    nameModel->Set(L"");

    std::list<Base::TypePtr>::const_iterator type;

    type=Base::Type::typeList.begin();
    while (type!=Base::Type::typeList.end()) {
      Lum::Model::ListTable::StdEntry *entry;

      if (type==Base::Type::typeList.begin()) {
        typeModel->Set((*type)->GetName());
      }

      entry=new Lum::Model::ListTable::StdEntry(model);
      entry->SetString(1,(*type)->GetName());
      model->Append(entry);

      ++type;
    }
  }

  ~LinkNameTypeDialog()
  {
    UnattachModel(typeModel);
    UnattachModel(nameModel);
    UnattachModel(GetClosedAction());
    UnattachModel(okAction);
  }

  virtual void PreInit()
  {
    Lum::Button      *button;
    Lum::ButtonRow   *row;
    Lum::Combo       *combo;
    Lum::Label       *label;
    Lum::String      *string;
    Lum::Panel       *vPanel;
    Lum::Text        *text;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,false);

    vPanel=new Lum::VPanel;
    vPanel->SetFlex(true,false);

    text=new Lum::Text;
    text->SetFlex(true,false);
    text->SetText(L"Create link entry...");

    vPanel->Add(text);

    vPanel->Add(new Lum::VSpace);

    label=new Lum::Label();
    label->SetFlex(true,false);

    string=new Lum::String();
    string->SetFlex(true,false);
    string->SetModel(nameModel);
    label->AddLabel(new Lum::Text(L"Name:"),string);

    combo=new Lum::TextCombo();
    combo->SetFlex(true,false);
    if (!combo->SetModel(typeModel)) {
      std::cout << "Cannot set model!" << std::endl;
    }
    combo->SetTableModel(model);
    label->AddLabel(new Lum::Text(L"Type:"),combo);

    vPanel->Add(label);

    vPanel->Add(new Lum::VSpace);

    row=new Lum::ButtonRow();
    row->SetFlex(true,false);

    button=new Lum::Button;
    button->SetFlex(true,true);
    button->SetModel(okAction);
    button->SetText(L"_OK");
    button->SetType(Lum::Button::typeCommit);
    row->Add(button);

    button=new Lum::Button;
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    button->SetText(L"_Cancel");
    button->SetType(Lum::Button::typeCancel);
    row->Add(button);

    vPanel->Add(row);

    wGroup->SetMain(vPanel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      success=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else if (model==nameModel || model==typeModel) {
      if (!nameModel->IsNull() && !typeModel->IsNull() && !nameModel->Empty() && !typeModel->Empty()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  bool ClosedSuccess()
  {
    return success;
  }

  void GetResult(std::wstring& name, std::wstring& type) const
  {
    name=nameModel->Get();
    type=typeModel->Get();
    std::cout << Lum::Base::WStringToString(name) << " " << Lum::Base::WStringToString(type) << std::endl;
  }
};

bool GetLinkNameType(Lum::OS::Window* parent,
                     std::wstring& linkName,
                     std::wstring& linkType)
{
  LinkNameTypeDialog *dialog;

  dialog=new LinkNameTypeDialog();
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  if (dialog->ClosedSuccess()) {
    dialog->GetResult(linkName,linkType);

    delete dialog;
    return true;
  }
  else {
    delete dialog;
    return false;
  }
}

class DataEntry : public Lum::Model::ListTable::Entry
{
public:
  Base::DataPtr data;

public:
  DataEntry(Lum::Model::ListTable* table,Base::DataPtr data)
  : Lum::Model::ListTable::Entry(table),data(data)
  {
    // no code
  }

  std::wstring GetString(size_t column) const
  {
    return data->GetDisplayText();
  }

  bool IsGreater(const Entry* other, size_t column) const
  {
    const DataEntry *tmp=dynamic_cast<const DataEntry*>(other);

    return data->IsGreater(tmp->data);
  }
};


class DataLinkDialog : public Lum::Dialog
{
private:
  Lum::Model::ListTableRef           model;
  Lum::Model::ActionRef              okAction;
  Lum::Model::StringRef              nameModel;
  Lum::Model::SizeTRef               typeModel;
  Lum::Model::ListTableRef           dataModel;
  Lum::Model::SingleLineSelectionRef selectionModel;
  bool                               success;

public:
  DataLinkDialog()
  : model(new Lum::Model::ListTable(1)),
    okAction(new Lum::Model::Action),
    nameModel(new Lum::Model::String),
    typeModel(new Lum::Model::SizeT),
    dataModel(new Lum::Model::ListTable(1)),
    selectionModel(new Lum::Model::SingleLineSelection()),
    success(false)
  {
    AttachModel(okAction);
    AttachModel(GetClosedAction());
    AttachModel(nameModel);
    AttachModel(typeModel);
    AttachModel(selectionModel);

    nameModel->Set(L"");

    std::list<Base::TypePtr>::const_iterator type;
    type=Base::Type::typeList.begin();
    while (type!=Base::Type::typeList.end()) {
      Lum::Model::ListTable::StdEntry *entry;

      entry=new Lum::Model::ListTable::StdEntry(model);
      entry->SetString(1,(*type)->GetName());
      model->Append(entry);

      ++type;
    }

    typeModel->Set(1);
  }

  ~DataLinkDialog()
  {
    UnattachModel(selectionModel);
    UnattachModel(typeModel);
    UnattachModel(nameModel);
    UnattachModel(GetClosedAction());
    UnattachModel(okAction);
  }

  virtual void PreInit()
  {
    Lum::Button      *button;
    Lum::ButtonRow   *row;
    Lum::Combo       *combo;
    Lum::Label       *label;
    Lum::Table       *dataTable;
    Lum::String      *string;
    Lum::Panel       *vPanel;
    Lum::Text        *text;
    Lum::WindowGroup *wGroup;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);

    vPanel=new Lum::VPanel;
    vPanel->SetFlex(true,true);

    text=new Lum::Text;
    text->SetFlex(true,false);
    text->SetText(L"Select dataset to link to...");

    vPanel->Add(text);

    vPanel->Add(new Lum::VSpace);

    label=new Lum::Label();
    label->SetFlex(true,false);

    string=new Lum::String();
    string->SetFlex(true,false);
    string->SetModel(nameModel);
    label->AddLabel(new Lum::Text(L"Link name:"),string);

    combo=new Lum::IndexCombo();
    combo->SetFlex(true,false);
    combo->SetModel(typeModel);
    combo->SetTableModel(model);
    label->AddLabel(new Lum::Text(L"Type:"),combo);

    vPanel->Add(label);

    vPanel->Add(new Lum::VSpace);

    dataTable=new Lum::Table();
    dataTable->SetFlex(true,true);
    dataTable->SetModel(dataModel);
    dataTable->SetSelection(selectionModel);
    vPanel->Add(dataTable);

    vPanel->Add(new Lum::VSpace);

    row=new Lum::ButtonRow();
    row->SetFlex(true,false);

    button=new Lum::Button;
    button->SetFlex(true,true);
    button->SetModel(okAction);
    button->SetText(L"_OK");
    button->SetType(Lum::Button::typeCommit);
    row->Add(button);

    button=new Lum::Button;
    button->SetFlex(true,true);
    button->SetModel(GetClosedAction());
    button->SetText(L"_Cancel");
    button->SetType(Lum::Button::typeCancel);
    row->Add(button);

    vPanel->Add(row);

    wGroup->SetMain(vPanel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      success=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else if (model==nameModel || model==typeModel || model==selectionModel) {
      if (!nameModel->IsNull() && !typeModel->IsNull() && !selectionModel->IsNull() &&
          !nameModel->Empty() && selectionModel->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }

      if (model==typeModel) {
        dataModel->Off();
        dataModel->Clear();
        if (!model->IsNull()) {
          ::Base::DataConstIterator iter;
          ::Base::TypePtr           type;

          type=::Base::Type::GetTypeByIndex(typeModel->Get()-1);

          iter=database->database.begin();
          while (iter!=database->database.end()) {
            if ((*iter)->GetType()==type) {
              dataModel->Append(new DataEntry(dataModel,(*iter)));
            }
            ++iter;
          }
        }
        dataModel->Sort(1);
        dataModel->On();
      }
    }
    else {
      Dialog::Resync(model,msg);
    }
  }

  bool ClosedSuccess()
  {
    return success;
  }

  void GetResult(std::wstring& name, ::Base::DataPtr& data) const
  {
    name=nameModel->Get();
    data=dynamic_cast<DataEntry*>(dataModel->GetEntry(selectionModel->GetLine()))->data;
  }
};


bool GetDataLink(Lum::OS::Window* parent,
                 std::wstring& linkName,
                 Base::DataPtr& data)
{
  DataLinkDialog *dialog;

  dialog=new DataLinkDialog();
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  if (dialog->ClosedSuccess()) {
   dialog->GetResult(linkName,data);

    delete dialog;
    return true;
  }
  else {
    delete dialog;
    return false;
  }
}
