/*
 * This file is part of phonelink
 *
 * Copyright (C) 2008 FLL.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <string.h>
#include <locale.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include <config.h>

#include "phonelink.h"

#ifdef GNOKII_GCONF
#include "gnokii-gconf.h"
#endif

#include "contactinfos.h"
#include "client-contactinfos-dbus.h"
#include "marshall.h"

#include "dbussignal.h"

#if HILDON == 1
#include <hildon/hildon.h>
#else
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-banner.h>
#endif

#include <libosso.h>
#include <dbus/dbus-glib.h>
#include <bt-dbus.h>

#include "gnokii.h"

#define _(String) String

struct UICStruct {
	GtkListStore *storeNumber;
	GtkWidget *searchEntry;
	GtkWidget *treeNumber;
	GtkListStore *storeSearch;
	GtkWidget *treeSearch;
	GtkWidget *contactInfosCheckBox;
	GtkWidget *sendSMSButton;
	GtkWidget *smsLabel;
	gchar *smsText;
	osso_context_t *osso_context;
	gboolean contactInfosAvailable;
	gboolean dial;
	gboolean gsm;
	gboolean transformPlus;
	gboolean ignoreCI;
	gboolean getCalls;
	gboolean getSms;
	gulong sigId;
	gboolean phoneOpened;
	gboolean pollingPhone;
	gboolean noConfig;
	gint callDetected;
};
typedef struct UICStruct UIComponents;

static struct gn_statemachine * state;

/* Description des colonnes*/
enum
{
   NAME_COLUMN,
   NUMBER_COLUMN,
   N_COLUMNS
};

static void dial (UIComponents *uic);
static void sendSMSToTree (UIComponents *uic);
void dialNumberFromSearch(UIComponents * uic, gboolean gsm);
void dialNumber(UIComponents * uic, gchar *number);
static void sendSMS (gchar *number, gchar *text);

static void  quit_event( UIComponents *uic, guint callback_action, GtkWidget *menu_item	) {
	GConfClient *gconf_client = NULL;

	gconf_client = gconf_client_get_default ();
	gconf_client_set_bool(gconf_client, GCONF_TRANSFORM_PLUS_KEY, uic->transformPlus, NULL);
	gconf_client_set_bool(gconf_client, GCONF_IGNORE_CONTACTINFOS_KEY, uic->ignoreCI, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NOTIFY_CALL_KEY, uic->getCalls, NULL);
	gconf_client_set_bool(gconf_client, GCONF_NOTIFY_SMS_KEY, uic->getSms, NULL);
	gconf_client_suggest_sync (gconf_client, NULL);
	g_object_unref (gconf_client);

	if (uic->phoneOpened) {
		uic->phoneOpened = !(uic->phoneOpened);
		gn_lib_phone_close(state);
	}

        gn_lib_phoneprofile_free(&state);
        gn_lib_library_free();

	gtk_main_quit ();
	osso_deinitialize(uic->osso_context);
	g_free(uic);
					
}

/* callback to poll phone and continue as long as there is polling to do ... */
gboolean pollPhone_cb(UIComponents *uic) {
	
	if (uic->phoneOpened && uic->pollingPhone) {
		uic->pollingPhone = FALSE;
		gn_sm_loop(1, state);
		uic->pollingPhone = TRUE;
		return TRUE;
	}
	return FALSE;
}


gboolean sendDBusIncomingCall_cb(gn_call_info *call_info) {

	sendDBusIncomingCallSignal(call_info->number, call_info->name); 
	return FALSE;
}

gboolean sendDBusIncomingSms_cb(gn_sms *message) {

	sendDBusIncomingSmsSignal(message->remote.number, message->user_data[0].u.text); 
	return FALSE;
}


/* callback to check availability of contactinfos service */
gboolean checkAvailability_cb(UIComponents *uic) {
	
	DBusGConnection *connection;
	GError *error = NULL;
	DBusGProxy *proxy;
	gboolean availability = FALSE;
	gboolean ret;

	error = NULL;
	connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
	if (connection == NULL) 	{
		g_printerr ("Failed to open connection to bus: %s\n", error->message);
		g_error_free (error);
		return FALSE;
	}

	/* Create a proxy object for the "bus driver" (name "org.freedesktop.DBus") */

	proxy = dbus_g_proxy_new_for_name (connection,
				     CONTACTINFOS_DBUS_NAME,
				     CONTACTINFOS_DBUS_PATH,
				     CONTACTINFOS_DBUS_INTERFACE);


	ret = com_nokia_contactinfos_is_available (proxy, &availability, &error);

	if (ret) {
		gtk_widget_set_sensitive(uic->contactInfosCheckBox, availability);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(uic->contactInfosCheckBox), availability);
	}

	g_object_unref (proxy);
	
	return FALSE;

}

void oneResultSignal_cb(
				DBusGProxy *proxy, 
				gchar *name, 
				gchar *company, 
				guint typeInfo, 
				gchar *type, 
				gchar *value, 
				guint elementId, 
				guint valueId, 
				gint retCode, 
				UIComponents *uic) {
	GtkTreeIter iter;
    gchar *text;
					
	if (retCode == CI_NOT_FOUND) {
        text = g_strdup_printf(_("ContactInfos found nothing for %s"), (gchar *) gtk_entry_get_text(GTK_ENTRY(uic->searchEntry)));
		hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(uic->searchEntry)), GTK_STOCK_DIALOG_INFO, text);
        g_free(text);
		dbus_g_proxy_disconnect_signal(proxy, CONTACTINFOS_DBUS_RESULTS_SIGNAL,
			G_CALLBACK(oneResultSignal_cb), uic);

		g_object_unref (proxy);
		return;
	}
					
	if (retCode >= CI_NO_ERROR) {
		gtk_list_store_append (uic->storeNumber, &iter);
		gtk_list_store_set (uic->storeNumber, &iter, 
					NAME_COLUMN, name, 
					NUMBER_COLUMN, value, 
					-1);
	}
	
	if (retCode < CI_NO_ERROR || (elementId == CI_LAST_ELEMENT_ID && valueId==CI_LAST_VALUE_ID)) {
		dbus_g_proxy_disconnect_signal(proxy, CONTACTINFOS_DBUS_RESULTS_SIGNAL,
			G_CALLBACK(oneResultSignal_cb), uic);

		g_object_unref (proxy);
	}
	
	if (elementId == CI_LAST_ELEMENT_ID && valueId==CI_LAST_VALUE_ID) {
		if (uic->dial) {
			dial(uic);
		} else {
			sendSMSToTree(uic);
		}
	}

}

void on_menu_quit(GtkMenuItem * menuitem, UIComponents *uic) {
	quit_event( uic, 0, NULL);
}

void on_menu_about(GtkMenuItem * menuitem, UIComponents *uic) {
	gchar **authors = g_strsplit("Fred Lef\xc3\xa9v\xc3\xa8re-Laoide\n", "\n", -1);
	GdkPixbuf *logo = gdk_pixbuf_new_from_file(ICONFILE, NULL);
	gchar *comment = g_strdup_printf("Gnokii Compile : %x\nGnokii Runtime : %x", LIBGNOKII_VERSION,gn_lib_version());
	gtk_show_about_dialog(GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(menuitem))),
		"version", VERSION,
		"website", "https://garage.maemo.org/projects/phonelink/",
		"website-label", _("Garage Page @ Maemo"),
		"authors", authors,
		"logo", logo,
		"comments", comment,
		NULL);
	g_strfreev(authors);
	/*
	gdk_pixbuf_free(logo);
	*/
	g_free(comment);
}

void on_menu_transform(GtkCheckMenuItem *checkMenuitem, UIComponents *uic) {
	uic->transformPlus = !(uic->transformPlus);
}

void on_menu_ignoreCI(GtkCheckMenuItem *checkMenuitem, UIComponents *uic) {
	uic->ignoreCI = !(uic->ignoreCI);
}

gboolean useContactInfos(UIComponents *uic) {
	gboolean useCI = TRUE;
	gchar *c;
	gint len, i;
	
	if (uic->ignoreCI) {
		useCI = FALSE;
		c = gtk_entry_get_text(GTK_ENTRY(uic->searchEntry));
		if (c) {
			len = strlen(c);
			for (i=0; i<len; i++) {
				if (g_ascii_isalpha(c[i])) {
					useCI = TRUE;
					break;
				}
			}
		}
	}
	if (useCI) {
		useCI = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(uic->contactInfosCheckBox));
	}
	return useCI;
}

gn_error hangup(int lowlevel_id) {
	gn_call_info	callinfo;
	gn_data		data;
	
	memset(&callinfo, 0, sizeof(callinfo));
	callinfo.call_id = lowlevel_id;

	gn_data_clear(&data);
	data.call_info = &callinfo;

	return gn_sm_functions(GN_OP_CancelCall, &data, state);
}

gn_error answer(int lowlevel_id) {
	gn_call_info	callinfo;
	gn_data		data;
	
	memset(&callinfo, 0, sizeof(callinfo));
	callinfo.call_id = lowlevel_id;

	gn_data_clear(&data);
	data.call_info = &callinfo;

	return gn_sm_functions(GN_OP_AnswerCall, &data, state);
}


void notify_sms(gn_sms *message, struct gn_statemachine *state, UIComponents *uic) {
	/* in this callback function you can't use those libgnokii functions that send a packet */
	gchar *descr;
	GtkWidget *note;
	gint resultCode;
	
	g_printf("\n===== ===== Incoming SMS ===== =====\n");
	g_printf("===== from %s =====\n", message->remote.number);

	g_idle_add ((GSourceFunc)sendDBusIncomingSms_cb, message);

	descr = g_strdup_printf(_("SMS received from %s :\n%s"), message->remote.number, message->user_data[0].u.text);
	note = hildon_note_new_confirmation_add_buttons(NULL, descr, _("Reply"), GTK_RESPONSE_ACCEPT, _("Forget it"), GTK_RESPONSE_REJECT, NULL);
	g_free(descr);
	gtk_widget_show_all (note);

	resultCode = gtk_dialog_run (GTK_DIALOG (note));
	
	switch (resultCode) {
	case GTK_RESPONSE_ACCEPT:
		gtk_entry_set_text(GTK_ENTRY(uic->searchEntry), message->remote.number);
		if (!uic->ignoreCI) {
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(uic->contactInfosCheckBox), FALSE);
		}
		gtk_container_set_focus_child(GTK_CONTAINER(gtk_widget_get_toplevel(uic->searchEntry)), uic->searchEntry);
		break;
	default:
		break;
	}
	gtk_widget_destroy (note);

}

void notify_call(gn_call_status call_status, gn_call_info *call_info, struct gn_statemachine *state, UIComponents *uic) {
	gchar *descr;
	GtkWidget *note;
	gint resultCode;
	
	/* in this callback function you can't use those libgnokii functions that send a packet */
	
	if (call_status != GN_CALL_Incoming) {
		return;
	}
		
	if (uic->callDetected == call_info->call_id) {
		return;
	}

	g_printf("\n===== ===== Incoming Call ===== =====\n");
	
	uic->callDetected = call_info->call_id;
	
	g_idle_add ((GSourceFunc)sendDBusIncomingCall_cb, call_info);
	
	descr = g_strdup_printf(_("Incoming Call\nnumber: %s\nname: %s"), call_info->number, call_info->name);
	note = hildon_note_new_confirmation_add_buttons(NULL, descr, _("Answer"), GTK_RESPONSE_ACCEPT, _("Hang Up"), GTK_RESPONSE_REJECT, _("Ignore"), GTK_RESPONSE_CANCEL, NULL);
	g_free(descr);
	gtk_widget_show_all (note);

	resultCode = gtk_dialog_run (GTK_DIALOG (note));
	
	switch (resultCode) {
	case GTK_RESPONSE_ACCEPT:
		answer(call_info->call_id);
		break;
	case GTK_RESPONSE_REJECT:
		hangup(call_info->call_id);
		break;
	default:
		break;
	}
	gtk_widget_destroy (note);

	uic->callDetected = -1;
	
}

gboolean getCalls(GtkWidget *topLevel, UIComponents *uic) {
	gn_data		data;
	gn_error	error = GN_ERR_NONE;
    gchar       *text;
	
	if (uic->getCalls) {
		if (!uic->phoneOpened) {
			error = gn_lib_phone_open(state);
		}
		if (error == GN_ERR_NONE) {
			/* Set up so that we get notified of incoming calls, if supported by the driver */
			uic->phoneOpened = TRUE;
			gn_data_clear(&data);
			data.call_notification = notify_call;
			data.callback_data = uic;
			error = gn_sm_functions(GN_OP_SetCallNotification, &data, state);
		}
		if (error != GN_ERR_NONE) {
                text = g_strdup_printf(_("This driver doesn't support call signaling.\n(%s)"), gn_error_print(error));
			hildon_banner_show_information(topLevel, GTK_STOCK_DIALOG_INFO, text);
            g_free(text);
			if (uic->phoneOpened) {
				uic->phoneOpened = !(uic->phoneOpened);
				gn_lib_phone_close(state);
			}
			return FALSE;
		} else {
			if (!uic->pollingPhone) {
				uic->pollingPhone = TRUE;
				g_idle_add ((GSourceFunc)pollPhone_cb, uic);
			}
			hildon_banner_show_information(topLevel, GTK_STOCK_DIALOG_INFO, _("Waiting for a call ..."));
		}
	} else {
		hildon_banner_show_information(topLevel, GTK_STOCK_DIALOG_INFO, _("Stopped waiting for a call ..."));
		gn_data_clear(&data);
		data.call_notification = NULL;
		gn_sm_functions(GN_OP_SetCallNotification, &data, state);
		if (!uic->getSms) {
			uic->pollingPhone = FALSE;
			if (uic->phoneOpened) {
				uic->phoneOpened = !(uic->phoneOpened);
				gn_lib_phone_close(state);
			}
		}
	}
	return TRUE;
}

void on_menu_getCalls(GtkCheckMenuItem *checkMenuitem, UIComponents *uic) {
	uic->getCalls = !(uic->getCalls);
	if (getCalls(GTK_WIDGET(gtk_widget_get_toplevel(GTK_WIDGET(checkMenuitem))), uic) == FALSE) {
		uic->getCalls = !(uic->getCalls);
		gtk_check_menu_item_set_active(checkMenuitem, FALSE);
	}
}

gboolean getSms(GtkWidget *topLevel, UIComponents *uic) {
	gn_data		data;
	gn_error	error = GN_ERR_NONE;
    gchar       *text;
	
	if (uic->getSms) {
		if (!uic->phoneOpened) {
			error = gn_lib_phone_open(state);
		}
		/* Set up so that we get notified of incoming sms, if supported by the driver */
		if (error == GN_ERR_NONE) {
			uic->phoneOpened = TRUE;
			gn_data_clear(&data);
			data.on_sms = notify_sms;
			data.callback_data = uic;
			error = gn_sm_functions(GN_OP_OnSMS, &data, state);
		}
		if (error != GN_ERR_NONE) {
            text = g_strdup_printf(_("This driver doesn't support sms signaling.\n(%s)"), gn_error_print(error));
			hildon_banner_show_information(topLevel, GTK_STOCK_DIALOG_INFO, text);
            g_free(text);
			if (uic->phoneOpened) {
				uic->phoneOpened = !(uic->phoneOpened);
				gn_lib_phone_close(state);
			}
			return FALSE;
		} else {
			if (!uic->pollingPhone) {
				uic->pollingPhone = TRUE;
				g_idle_add ((GSourceFunc)pollPhone_cb, uic);
			}
			hildon_banner_show_information(topLevel, GTK_STOCK_DIALOG_INFO, _("Waiting for SMS ..."));
		}
	} else {
		hildon_banner_show_information(topLevel, GTK_STOCK_DIALOG_INFO, _("Stopped waiting for SMS ..."));
		gn_data_clear(&data);
		data.on_sms = NULL;
		gn_sm_functions(GN_OP_OnSMS, &data, state);
		if (!uic->getCalls) {
			uic->pollingPhone = FALSE;
			if (uic->phoneOpened) {
				uic->phoneOpened = !(uic->phoneOpened);
				gn_lib_phone_close(state);
			}
		}
	}
	return TRUE;
}

void on_menu_getSms(GtkCheckMenuItem *checkMenuitem, UIComponents *uic) {
	uic->getSms = !(uic->getSms);
	if (getSms(GTK_WIDGET(gtk_widget_get_toplevel(GTK_WIDGET(checkMenuitem))), uic) == FALSE) {
		uic->getSms = !(uic->getSms);
		gtk_check_menu_item_set_active(checkMenuitem, FALSE);
	}
}

GtkMenu *create_menus(UIComponents *uic) {
	GtkWidget *menuBar;
	GtkWidget *menuItem;
	GtkWidget *image;
	
	menuBar = gtk_menu_new ();
	
	menuItem = gtk_check_menu_item_new_with_label (_("Transform + into 00"));
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(menuItem), uic->transformPlus);
	g_signal_connect (G_OBJECT (menuItem), "toggled", G_CALLBACK (on_menu_transform), (gpointer) uic);
	gtk_menu_append(menuBar, menuItem);
	
	menuItem = gtk_check_menu_item_new_with_label (_("Ignore ContactInfos when number"));
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(menuItem), uic->ignoreCI);
	g_signal_connect (G_OBJECT (menuItem), "toggled", G_CALLBACK (on_menu_ignoreCI), (gpointer) uic);
	gtk_menu_append(menuBar, menuItem);
	
	gtk_menu_append(menuBar, gtk_separator_menu_item_new());
	
	menuItem = gtk_check_menu_item_new_with_label (_("Notify me on incoming call"));
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(menuItem), uic->getCalls);
	g_signal_connect (G_OBJECT (menuItem), "toggled", G_CALLBACK (on_menu_getCalls), (gpointer) uic);
	gtk_widget_set_sensitive(menuItem, !uic->noConfig);
	gtk_menu_append(menuBar, menuItem);
	
	menuItem = gtk_check_menu_item_new_with_label (_("Notify me on SMS reception"));
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(menuItem), uic->getSms);
	g_signal_connect (G_OBJECT (menuItem), "toggled", G_CALLBACK (on_menu_getSms), (gpointer) uic);
	gtk_widget_set_sensitive(menuItem, !uic->noConfig);
	gtk_menu_append(menuBar, menuItem);
	
	gtk_menu_append(menuBar, gtk_separator_menu_item_new());
	
	menuItem = gtk_image_menu_item_new_with_label (_("About ..."));
	image = gtk_image_new_from_stock (GTK_STOCK_ABOUT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuItem), image);
	g_signal_connect (G_OBJECT (menuItem), "activate", G_CALLBACK (on_menu_about), (gpointer) uic);
	gtk_menu_append(menuBar, menuItem);
	
	gtk_menu_append(menuBar, gtk_separator_menu_item_new());
	
	menuItem = gtk_image_menu_item_new_with_label (_("Quit"));
	image = gtk_image_new_from_stock (GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuItem), image);
	g_signal_connect (G_OBJECT (menuItem), "activate", G_CALLBACK (on_menu_quit), (gpointer) uic);
	gtk_menu_append(menuBar, menuItem);
	
	return GTK_MENU(menuBar);
}



void getDBusNumbers(UIComponents *uic, gboolean dial, gboolean gsm) {
	DBusGConnection *connection;
	GError *error = NULL;
	DBusGProxy *proxy;
	gboolean availability;
	gboolean multipleSel=FALSE;
	guint typeInfo=CI_ALL_PHONE_NUMBER;;
	
	uic->dial = dial;
	uic->gsm = gsm;
	
	gtk_list_store_clear(uic->storeNumber);

	connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
	if (connection == NULL) {
		g_printerr ("Failed to open connection to bus: %s\n", error->message);
		g_error_free (error);
		return;
	}

	/* Create a proxy object for the "bus driver" (name "org.freedesktop.DBus") */
	proxy = dbus_g_proxy_new_for_name (connection,
				     CONTACTINFOS_DBUS_NAME,
				     CONTACTINFOS_DBUS_PATH,
				     CONTACTINFOS_DBUS_INTERFACE);


	dbus_g_object_register_marshaller(
		dbus_glib_marshal_contactinfos_VOID__STRING_STRING_UINT_STRING_STRING_UINT_UINT_INT,
		G_TYPE_NONE,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_UINT,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_UINT,
		G_TYPE_UINT,
		G_TYPE_INT,
		G_TYPE_INVALID);

	dbus_g_proxy_add_signal(proxy,
		CONTACTINFOS_DBUS_RESULTS_SIGNAL,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_UINT,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_UINT,
		G_TYPE_UINT,
		G_TYPE_INT,
		G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, CONTACTINFOS_DBUS_RESULTS_SIGNAL,
		G_CALLBACK(oneResultSignal_cb), uic, NULL);
	
	multipleSel = !dial;

	/* Start DBus Call */
	com_nokia_contactinfos_get_contact_infos (
								proxy, 
								typeInfo, 
								gtk_entry_get_text(GTK_ENTRY(uic->searchEntry)), 
								TRUE, 
								TRUE, 
								multipleSel, 
								0, 
								&availability, 
								&error);
/*
	if (error) {
		res  = g_strdup_printf("Error\n  Code = %d\n  Message = %s\n", error->code, error->message);
		gtk_text_buffer_insert_at_cursor (appData->buffer, res, -1);
		g_free(res);
	} else {
		if (availability) {
			res  = g_strdup_printf("callDbusSignal_cb No Error\nWaiting for an answer ...\n");
			gtk_text_buffer_insert_at_cursor (appData->buffer, res, -1);
			g_free(res);
		} else {
			res  = g_strdup_printf("callDbusSignal_cb No Error\nForget it ...\n");
			gtk_text_buffer_insert_at_cursor (appData->buffer, res, -1);
			g_free(res);
		}
	}
*/
}

void dialDBus_cb(GtkButton *widget, UIComponents *uic) {

	if (useContactInfos(uic)) {
		getDBusNumbers(uic, TRUE, TRUE);
	} else {
		/* dial the number from the searchEntry */
		dialNumberFromSearch(uic, TRUE);
	}
	
}

/* another callback */
static gboolean delete_event( GtkWidget *widget, GdkEvent *event, gpointer data ) {
	quit_event(data, 0, NULL);
	return TRUE;
}

static gchar * normalizeNumber( gchar *number, gboolean transformPlus) {

	if (!number) {
		return number;
	}

	gchar * normalizedNumber = (gchar *) g_malloc(strlen(number)+1);
	guint i = 0;
	guint j = 0;
	g_strstrip(number);
	if (transformPlus && number[0] == '+') {
		normalizedNumber[0] = '0';
		normalizedNumber[1] = '0';
		j = 2;
		i = 1;
	}

	for (; number[i]; i++) {
		if (g_ascii_isdigit(number[i]) || number[i] == '+') {
			normalizedNumber[j++] = number[i];
		}
	}
	normalizedNumber[j] = '\0';
	return normalizedNumber;
}


static gchar * normalizeNumber4Sip( gchar *number, gboolean transformPlus ) {

	gchar * num = normalizeNumber(number, transformPlus);
	gchar * res = g_strconcat("sip:", num, NULL); /*, "@freephonie.net" */
	g_free(num);
	return res;
}
	
static void dialSip_cb (GtkButton *widget, UIComponents *uic) {
	
	if (useContactInfos(uic)) {
		getDBusNumbers(uic, TRUE, FALSE);
	} else {
		/* dial the number from the searchEntry */
		dialNumberFromSearch(uic, FALSE);
	}
	
}

static guint dialSip(gchar * number, gchar ** message, UIComponents *uic) {
	
/* initialise the connection */
	osso_rpc_t retval;
	osso_return_t ret;
	ret = osso_rpc_run(uic->osso_context,
			"com.nokia.gizmo",
			"/com/nokia/gizmo",
			"com.nokia.gizmo",
			"callto", 
			&retval,
			DBUS_TYPE_STRING, number,
			DBUS_TYPE_INVALID);
	osso_rpc_free_val(&retval);
	if (ret !=  OSSO_OK) {
		*message=g_strdup_printf("Can't dial Sip phone through DBus : %d", ret);
		return 1;
	}
	return 0;
}

static guint dialGsm(gchar *number, gchar **message, gboolean phoneOpened) {

	guint ret = 0;

	gn_data gdata;
        gn_error error;
        gn_call_info call_info;
        int call_id;
				
	/* initialise the connection */
	if (!phoneOpened && (error = gn_lib_phone_open(state)) != GN_ERR_NONE) {
		g_printf("Telephone interface init failed: %s Quitting.\n", gn_error_print(error));
		*message=g_strdup_printf("Telephone interface init failed: %s Quitting.", gn_error_print(error));
		return 1;
	}

	memset(&call_info, 0, sizeof(call_info));
	snprintf(call_info.number, sizeof(call_info.number), "%s", number);
	call_info.type = GN_CALL_Voice;
	call_info.send_number = GN_CALL_Default;
	gn_data_clear(&gdata);
	gdata.call_info = &call_info;
        if ((error = gn_call_dial(&call_id, &gdata, state)) != GN_ERR_NONE) {
		g_printf("Can't dial the phone : %s", gn_error_print(error));
		*message=g_strdup_printf("Can't dial the phone : %s", gn_error_print(error));
		ret = 2;
	} else {
		g_printf("Dialled call, id: %d (lowlevel id: %d)\n", call_id, call_info.call_id);
	}
	 /* tear down the connection */
	if (!phoneOpened) {
		gn_lib_phone_close(state);
	}
	return ret;
																			 
}

void dialNumberFromSearch(UIComponents * uic, gboolean gsm) {
	gchar *number;
	
	uic->gsm = gsm;
	number = (gchar *) gtk_entry_get_text(GTK_ENTRY(uic->searchEntry));
	dialNumber(uic, number);
}

void dialNumber(UIComponents * uic, gchar *number) {
	gchar *normalizedNumber;
	gchar *sipNormalizedNumber;
	gchar *text;
	
	if (number && number[0] != '\0') {
		normalizedNumber = normalizeNumber(number, uic->transformPlus);
		if (normalizedNumber == NULL || normalizedNumber[0] == '\0') {
			text = g_strdup_printf("Number %s is not Valid !\n", number);
			hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(uic->searchEntry)), GTK_STOCK_DIALOG_ERROR, text);
			g_free(text);
			g_free(normalizedNumber);
			return;
		}
		text = g_strdup_printf("Dialing %s ...", normalizedNumber);
		hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(uic->searchEntry)), GTK_STOCK_CONNECT, text);
		g_free(text);
		text=NULL;
		guint ret = 0;
		if (uic->gsm == TRUE) {
			ret = dialGsm(normalizedNumber, &text, uic->phoneOpened);
		} else {
			sipNormalizedNumber = normalizeNumber4Sip(normalizedNumber, uic->transformPlus);
			ret = dialSip(sipNormalizedNumber, &text, uic);
			g_free(sipNormalizedNumber);
		}
		if (ret != 0) {
			hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(uic->searchEntry)), GTK_STOCK_DIALOG_ERROR, text);
		}
		g_free(text);
		g_free(normalizedNumber);
	}
}

static void dial (UIComponents * uic) {
	
	gchar *number;
	GtkTreeIter iter;
	GtkTreeModel *model;

	/* get the data */
	model = GTK_TREE_MODEL(uic->storeNumber);
	if (gtk_tree_model_get_iter_first(model, &iter)) {
		gtk_tree_model_get(model, &iter, NUMBER_COLUMN, &number, -1);
		dialNumber(uic, number);
		g_free(number);
	}
}

static gchar * getSMSText(UIComponents * uic) {
	
	if (!uic->smsText || strlen(uic->smsText) == 0) {
		return NULL;
	}
	return uic->smsText;
}

void smsSetCount(UIComponents * uic, gint count) {
	gchar *lib;
	lib = g_strdup_printf("SMS : %d car.", count);
	gtk_label_set_text(GTK_LABEL(uic->smsLabel), lib);
	g_free(lib);
}

/* Called when something is typed into the sms zone */
void smsBufferChanged_cb(GtkTextBuffer *textBuffer, UIComponents * uic) {
	/* Enable the SMS Button */
	GtkTextIter start, end;
	int count = 0;
	
	gtk_text_buffer_set_modified (textBuffer, FALSE);
	gtk_text_buffer_get_bounds(textBuffer, &start, &end);
	uic->smsText = gtk_text_buffer_get_text(textBuffer, &start, &end, TRUE);

	if (!uic->noConfig) {
		if (!uic->smsText || (count = strlen(uic->smsText)) == 0) {
			gtk_widget_set_sensitive(uic->sendSMSButton, FALSE);
		} else {
			gtk_widget_set_sensitive(uic->sendSMSButton, TRUE);
		}
	}
	smsSetCount(uic, count);
	
}

static gchar * cleanNumber(gchar * number) {
	guint i, j;

	if (number) {
		for (i = j = 0; number[i]; i++) {
			if (number[i] != ' ') {
				number[j++] = number[i];
			}
		}
		number[j] = '\0';
	}
	return number;
}

/*
static void getPhoneInfos_cb(GtkButton *widget, gpointer data) {
	gchar *text;
	text = getPhoneInfos(gtk_widget_get_toplevel(GTK_WIDGET(widget)));
	if (text) {
		g_print("%s\n", text);
		g_free(text);
	}
}
*/

static void showMessageSendSMS(GtkWidget *widget, gchar *number) {
	gchar *text;
	text = g_strdup_printf("Sending SMS to %s ...", number);
	hildon_banner_show_information(GTK_WIDGET(gtk_widget_get_toplevel(widget)), GTK_STOCK_DND, text);
	g_free(text);
}

static void sendSMS_cb (GtkButton *widget, UIComponents *uic) {

	gchar *normalizedNumber;
	gchar *text;
	gn_error error;

	if (useContactInfos(uic)) {
		getDBusNumbers(uic, FALSE, TRUE);
	} else {
		/* dial the number from the searchEntry */
		normalizedNumber=normalizeNumber((gchar *) gtk_entry_get_text(GTK_ENTRY(uic->searchEntry)), uic->transformPlus);
		text = getSMSText(uic);
		if (text) {
			showMessageSendSMS(GTK_WIDGET(widget), normalizedNumber);
			/* initialise the connection */
			if (!uic->phoneOpened && (error = gn_lib_phone_open(state)) != GN_ERR_NONE) {
				g_print("Telephone interface init failed: %s Quitting.\n", gn_error_print(error));
				return;
			}
			
			sendSMS(normalizedNumber, text);
			g_free(text);

			/* tear down the connection */
			if (!uic->phoneOpened) {
				gn_lib_phone_close(state);
			}
		}
		g_free(normalizedNumber);
	}

}

static void sendSMS (gchar *number, gchar *text) {
	gn_sms sms;
	gn_error error;
	gn_data data;

	gn_data_clear(&data);
	
	/* The maximum length of an uncompressed concatenated short message is
	 * 	   255 * 153 = 39015 default alphabet characters */
	int input_len, i, curpos = 0;

	g_printf("sendSMS(number = %s, text = %s)\n", number, text);
	
	input_len = GN_SMS_MAX_LENGTH;

	/* The memory is zeroed here */
	gn_sms_default_submit(&sms);

	memset(&sms.remote.number, 0, sizeof(sms.remote.number));
	strncpy(sms.remote.number, number, sizeof(sms.remote.number) - 1);
	if (sms.remote.number[0] == '+') {
		sms.remote.type = GN_GSM_NUMBER_International;
	} else {
		sms.remote.type = GN_GSM_NUMBER_Unknown;
	}
	
	data.message_center = calloc(1, sizeof(gn_sms_message_center));
	data.message_center->id = 1;
	if (gn_sm_functions(GN_OP_GetSMSCenter, &data, state) == GN_ERR_NONE) {
		strcpy(sms.smsc.number, data.message_center->smsc.number);
		sms.smsc.type = data.message_center->smsc.type;
	} else {
		g_print("Cannot read the SMSC number from your phone. If the sms send will fail, please use --smsc option explicitely giving the number.\n");
	}
	free(data.message_center);

	if (!sms.smsc.type) sms.smsc.type = GN_GSM_NUMBER_Unknown;

	i = strlen(text);
	strncpy(sms.user_data[curpos].u.text, text, input_len);
	sms.user_data[curpos].u.text[i] = '\0';
	sms.user_data[curpos].length = i;
	if (sms.user_data[curpos].length < 1) {
		g_print("Empty message. Quitting.\n");
		g_free(number);
		return ;
	}
	sms.user_data[curpos].type = GN_SMS_DATA_Text;
	if (!gn_char_def_alphabet(sms.user_data[curpos].u.text)) {
		sms.dcs.u.general.alphabet = GN_SMS_DCS_UCS2;
	}
	sms.user_data[++curpos].type = GN_SMS_DATA_None;
	data.sms = &sms;

	/* Send the message. */
	error = gn_sms_send(&data, state);

	if (error == GN_ERR_NONE) {
		g_print("Send succeeded!\n");
	} else {
		g_print("SMS Send failed (%s)\n", gn_error_print(error));
	}

}

gboolean sendSMSToIter (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, UIComponents *uic) {
	gchar *text;
	gchar *number;
	gchar *normalizedNumber;
	
	text=getSMSText(uic);
			
	gtk_tree_model_get(model, iter, NUMBER_COLUMN, &number, -1);
	normalizedNumber = normalizeNumber(number, uic->transformPlus);
	showMessageSendSMS(GTK_WIDGET(uic->searchEntry), normalizedNumber);
	sendSMS (normalizedNumber, text);
	g_free(normalizedNumber);
	g_free(text);
	return FALSE;
}

static void sendSMSToTree (UIComponents *uic) {
	gn_error error;
	
	/* initialise the connection */
	if (!uic->phoneOpened && (error = gn_lib_phone_open(state)) != GN_ERR_NONE) {
		g_print("Telephone interface init failed: %s Quitting.\n", gn_error_print(error));
		return;
	}
	
	gtk_tree_model_foreach (GTK_TREE_MODEL(uic->storeNumber), (GtkTreeModelForeachFunc)sendSMSToIter, (gpointer)uic);

	/* tear down the connection */
	if (!uic->phoneOpened) {
		gn_lib_phone_close(state);
	}
}

gboolean clearSMSTextView_cb(GtkTextView *smsTextView, GdkEventFocus *event, UIComponents *uic) {
	GtkTextBuffer *buffer;
	buffer = gtk_text_view_get_buffer(smsTextView);
	gtk_text_buffer_set_text(buffer, "", -1);
	gtk_text_buffer_set_modified (buffer, FALSE);
	g_signal_handler_disconnect(G_OBJECT(smsTextView), uic->sigId);
	return FALSE;
}


/**
 * Create the menu items needed for the drop down menu.
 */
/*
void menu_init(HildonWindow *window) {
    /* Create needed handles. */
/*
    GtkMenu *menu;
    GtkWidget *submenu;
    GtkWidget *menu_item;
    printf("%s()\n", __PRETTY_FUNCTION__);

    /* Get the menu of our view. */
/*
    menu = GTK_MENU(gtk_menu_new());

    /* Create the menu items. */

    /* The "Config" submenu. */
/*
    gtk_menu_append(menu, menu_item
            = gtk_menu_item_new_with_label("configuration"));


    /* We need to show menu items. */
/*
    gtk_widget_show_all(GTK_WIDGET(menu));

    hildon_window_set_menu(HILDON_WINDOW(window), menu);

    /* Connect the "Route" signals. */
/*
    g_signal_connect(G_OBJECT(menu_item), "activate",
                      G_CALLBACK(getPhoneInfos_cb), NULL);

}
*/

int main( int argc, char *argv[] ) {
	    /* GtkWidget is the storage type for widgets */
	GtkWidget *button;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkWidget *table;
	GtkWidget *scrolled_window;
	GtkWidget *smsTextView;
	GtkTextBuffer *buffer;
	GConfClient *gconf_client = NULL;
	gboolean gnokiiFile = FALSE;
    gchar *text;
	
#ifdef GNOKII_GCONF
	gchar ** gnokiiCfg;
#endif	
	
	HildonProgram *app;
	HildonWindow *appview;
	
	UIComponents *uic = g_new0(UIComponents, 1);

	/* This is called in all GTK applications. Arguments are parsed
	 * from the command line and are returned to the application. */
	gtk_init (&argc, &argv);

	/* Create a new window */
        app = HILDON_PROGRAM( hildon_program_get_instance() );
        g_set_application_name(_("PhoneLink" ));
	
        /* Create HildonAppView and set it to HildonApp */
        appview = HILDON_WINDOW( hildon_window_new());
        hildon_program_add_window( app, appview );

	g_type_init ();

	/* Read GConf Info */
	gconf_client = gconf_client_get_default ();
	uic->transformPlus = gconf_client_get_bool(gconf_client, GCONF_TRANSFORM_PLUS_KEY, NULL);
	uic->ignoreCI = gconf_client_get_bool(gconf_client, GCONF_IGNORE_CONTACTINFOS_KEY, NULL);
	uic->getCalls= gconf_client_get_bool(gconf_client, GCONF_NOTIFY_CALL_KEY, NULL);
	uic->getSms= gconf_client_get_bool(gconf_client, GCONF_NOTIFY_SMS_KEY, NULL);

/* Gnokii : read the config */
	gn_error err;
#ifdef GNOKII_GCONF
	if (!isGnokiiRcFileUsed(gconf_client, NULL)) {
		gnokiiCfg = getGnokiiCfg(gconf_client, NULL);
		if (gnokiiCfg) {
			gnokiiFile = TRUE;
			state = g_new0(struct gn_statemachine,1);
			if ((err = gn_cfg_memory_read ((const char **)gnokiiCfg)) != GN_ERR_NONE) {
				text = g_strdup_printf(_("Error reading gnokii configuration %s\n%s\n%s\n%s\n%s"), gn_error_print(err), gnokiiCfg[0], gnokiiCfg[1], gnokiiCfg[2], gnokiiCfg[3]);
                g_printf(text);
				hildon_banner_show_information(GTK_WIDGET(appview), GTK_STOCK_DIALOG_ERROR, text);
                g_free(text);
				gnokiiFile = FALSE;
			} else if ((err = gn_cfg_phone_load(NULL, state)) != GN_ERR_NONE) {
                text = g_strdup_printf(_("gn_cfg_phone_load %s"), gn_error_print(err));
				g_printf(text);
				hildon_banner_show_information(GTK_WIDGET(appview), GTK_STOCK_DIALOG_ERROR, text);
                g_free(text);
				gnokiiFile = FALSE;
			}
		} else {
			g_printf(_("Gnokii-GConf not configured\n"));
			hildon_banner_show_information(GTK_WIDGET(appview), GTK_STOCK_DIALOG_ERROR, _("Gnokii-GConf not configured"));
		}
	}
#endif
	if (!gnokiiFile) {
		if ((err = gn_lib_phoneprofile_load(NULL, &state)) != GN_ERR_NONE) {
			g_printf("%s\n", gn_error_print(err));
			hildon_banner_show_information(GTK_WIDGET(appview), GTK_STOCK_DIALOG_ERROR,  gn_error_print(err));
			uic->noConfig = TRUE;
		}
	}

	g_object_unref (gconf_client);
	
	g_print("avant osso init\n");

	/* Initialize maemo application */
	uic->osso_context = osso_initialize(APPLICATION_DBUS_SERVICE, VERSION, FALSE, NULL);

	g_printf("apres osso_init %x\n", uic->osso_context);
	            
	/* Check that initialization was ok */
	if (uic->osso_context == NULL) {
		 hildon_banner_show_information(GTK_WIDGET(appview), GTK_STOCK_DIALOG_ERROR, "osso_initialize error\n");
		return OSSO_ERROR;
	}
	
	
	/* Here we just set a handler for delete_event that immediately
	*      * exits GTK. */
	g_signal_connect (G_OBJECT (appview), "delete_event",
		      G_CALLBACK (delete_event), (gpointer) uic);

	// Menus
	hildon_window_set_menu (appview, create_menus (uic));

	// la boite gnrale
	table = gtk_table_new(9, 5, TRUE);
	gtk_container_add (GTK_CONTAINER (appview), table);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	
	/* Add a field & 2 buttons */
	uic->searchEntry = gtk_entry_new();
	gtk_table_attach_defaults( GTK_TABLE(table), uic->searchEntry, 0, 3, 0, 1);

	/* Buttons */
	button = gtk_button_new_with_label ("Dial");
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	g_signal_connect (G_OBJECT (button), "clicked",
	                  G_CALLBACK (dialDBus_cb), (gpointer) uic);
	gtk_widget_set_sensitive(button, !uic->noConfig);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 3, 5, 0, 1);
	gtk_widget_grab_default(button);
	
	uic->contactInfosCheckBox = gtk_check_button_new_with_label ("Use ContactInfos");
	gtk_widget_set_sensitive(uic->contactInfosCheckBox, FALSE);
	gtk_table_attach_defaults( GTK_TABLE(table), uic->contactInfosCheckBox, 3, 5, 1, 2);

	uic->sendSMSButton = gtk_button_new_with_label ("Send Sms");
	g_signal_connect (G_OBJECT (uic->sendSMSButton), "clicked",
			G_CALLBACK (sendSMS_cb), (gpointer) uic);
	gtk_widget_set_sensitive(uic->sendSMSButton, FALSE);
	gtk_table_attach_defaults(GTK_TABLE(table), uic->sendSMSButton, 3, 4, 2, 3);

	button = gtk_button_new_with_label ("Dial Sip");
	g_signal_connect (G_OBJECT (button), "clicked",
	                  G_CALLBACK (dialSip_cb), (gpointer) uic);
	gtk_table_attach_defaults(GTK_TABLE(table), button, 4, 5, 2, 3);
			
	// la liste des numros choisis
	uic->storeNumber = gtk_list_store_new (N_COLUMNS, G_TYPE_STRING, G_TYPE_STRING);
	
	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
		
	uic->treeNumber = gtk_tree_view_new_with_model (GTK_TREE_MODEL (uic->storeNumber));
	
	GTK_WIDGET_UNSET_FLAGS(uic->treeNumber, GTK_CAN_FOCUS);
	gtk_container_add(GTK_CONTAINER(scrolled_window), uic->treeNumber);
	gtk_table_attach_defaults( GTK_TABLE(table), scrolled_window, 0, 3, 1, 4);
	
        renderer = gtk_cell_renderer_text_new ();
        column = gtk_tree_view_column_new_with_attributes ("Number", renderer, "text", NUMBER_COLUMN, NULL);
        gtk_tree_view_append_column (GTK_TREE_VIEW (uic->treeNumber), column);
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Name", renderer, "text", NAME_COLUMN, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (uic->treeNumber), column);
	
	/* textEdit for SMSes */
	uic->smsLabel = gtk_label_new (NULL);
	smsSetCount(uic, 0);
	gtk_table_attach_defaults( GTK_TABLE(table), uic->smsLabel, 3, 5, 3, 4);
	
	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);

	smsTextView = gtk_text_view_new();
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(smsTextView), GTK_WRAP_WORD_CHAR);
	
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(smsTextView));
	gtk_text_buffer_set_text(buffer, _("Enter SMS Text here"), -1);
	gtk_text_buffer_set_modified (buffer, FALSE);
	g_signal_connect (G_OBJECT (buffer), "modified-changed", G_CALLBACK (smsBufferChanged_cb), (gpointer) uic);

	uic->sigId = g_signal_connect (G_OBJECT (smsTextView), "focus-in-event", G_CALLBACK (clearSMSTextView_cb), (gpointer) uic);

	gtk_container_add(GTK_CONTAINER(scrolled_window), smsTextView);

	gtk_table_attach_defaults( GTK_TABLE(table), scrolled_window, 0, 5, 4, 9);
	
	gtk_widget_show_all (GTK_WIDGET(appview));
	
	/* check availability and show or hide the Buttons */
	g_idle_add ((GSourceFunc)checkAvailability_cb, uic);

	uic->callDetected = -1;

	if (!uic->noConfig) {
		if (uic->getSms) {
			getSms(GTK_WIDGET(appview), uic);
		}
		
		if (uic->getCalls) {
			getCalls(GTK_WIDGET(appview), uic);
		}
	}
	
	/* Rest in gtk_main and wait for the fun to begin! */
	gtk_main ();

	return 0;
}

