/*
  PackageView - Show debian package details
  Copyright (C) 2008  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <fstream>
#include <iostream>

#include <Lum/OS/Main.h>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Selection.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Main.h>

#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/String.h>
#include <Lum/Table.h>
#include <Lum/Text.h>

#include "Package.h"
#include "PackageDetails.h"
#include "PackageList.h"

#include "config.h"

std::wstring        ossoVersion;

static Lum::AppInfo info;

class PackageFormatProvider : public Lum::TableView::FormatProvider
{
public:
  Lum::TableView::Alignment GetAlignment(size_t column, size_t row) const
  {
    switch (column) {
    case 1:
      return Lum::TableView::left;
      break;
    case 2:
      return Lum::TableView::right;
      break;
    default:
      assert(false);
      break;
    }
  }
};

class PackageTable : public Lum::Model::StdTable<PackageEntry>
{
public:
  class PackageDataProvider : public Lum::Model::StdTable<PackageEntry>::DataProvider
  {
    std::wstring GetString(const PackageTable::Iterator& iter, size_t column) const
    {
      return iter->GetString(column);
    }

    Lum::Object* GetObject(const PackageTable::Iterator& iter, size_t column) const
    {
      return iter->GetObject(column);
    }
  };

  class PackageComparator : public Lum::Model::StdTable<PackageEntry>::Comparator
  {
  public:
    bool operator()(const PackageEntry& a, const PackageEntry& b, size_t column, bool down) const
    {
      return a.GetPackage()->GetName()<b.GetPackage()->GetName();
    }
  };

public:
  PackageTable()
  : Lum::Model::StdTable<PackageEntry>(new PackageDataProvider(),
                                       new PackageComparator())
  {
    // no code
  }
};

typedef Lum::Base::Reference<PackageTable> PackageTableRef;

class MainDialog : public Lum::Dialog
{
private:
  std::list<Package>                 packages;
  Lum::Model::ActionRef              clearFilterAction;
  Lum::Model::ActionRef              showPackageAction;
  Lum::Model::ActionRef              aboutAction;
  std::wstring                       lastSearch;
  Lum::Model::StringRef              search;
  Lum::Model::ActionRef              searchTimerAction;
  PackageTableRef                    currentPackages;
  Lum::Model::SingleLineSelectionRef selection;
  PackageListTask                    packageListTask;

public:
  MainDialog()
  : clearFilterAction(new Lum::Model::Action()),
    showPackageAction(new Lum::Model::Action()),
    aboutAction(new Lum::Model::Action()),
    search(new Lum::Model::String(L"")),
    searchTimerAction(new Lum::Model::Action()),
    currentPackages(new PackageTable()),
    selection(new Lum::Model::SingleLineSelection()),
    packageListTask(packages)
  {
    Observe(packageListTask.GetFinishedAction());

    Observe(GetOpenedAction());
    Observe(GetClosedAction());

    Observe(search);
    Observe(searchTimerAction);

    Observe(selection);

    Observe(clearFilterAction);
    Observe(showPackageAction);
    Observe(aboutAction);

    DetectOssoVersion();
  }

  virtual void PreInit()
  {
    Lum::Panel            *vpanel,*hpanel;
    Lum::Table            *table;
    Lum::Model::HeaderRef headerModel;

    vpanel=Lum::VPanel::Create(true,true);

    hpanel=Lum::HPanel::Create(true,false);
    hpanel->Add(new Lum::Text(L"Filter:"));
    hpanel->AddSpace();
    hpanel->Add(Lum::String::Create(search,true,false));
    hpanel->Add(Lum::Button::Create(L"_Clear",clearFilterAction,false,true));

    vpanel->Add(hpanel);
    vpanel->AddSpace();

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Package",Lum::Base::Size::stdCharWidth,28,true);
    headerModel->AddColumn(L"Version",Lum::Base::Size::stdCharWidth,12,false);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetMinHeight(Lum::Base::Size::stdCharHeight,5);
    table->SetHeight(Lum::Base::Size::stdCharHeight,10);
    table->SetModel(currentPackages);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(showPackageAction);
    table->SetShowHeader(true);
    table->SetFormatProvider(new PackageFormatProvider());
    table->GetTableView()->SetAutoFitColumns(true);
    table->GetTableView()->SetAutoHSize(true);

    vpanel->Add(table);

    SetMain(vpanel);

    Lum::MenuDesc *menu=new Lum::MenuDesc();

    menu
      ->AddMenuItemSub(_ld(menuProject))
        ->AddMenuItemAction(_ld(menuProjectQuit),Lum::OS::qualifierControl,L"q",GetClosedAction());
    menu
      ->AddMenuItemSub(_ld(menuHelp))
        ->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->AddMenuItemAction(_ld(menuHelpAbout),aboutAction);

    SetMenu(menu);

    packageListTask.SetParent(this);
    packageListTask.SetCaption(L"Reading package list...");

    Dialog::PreInit();
  }

  void DetectOssoVersion()
  {
    std::ifstream file;

    file.open("/etc/osso_software_version",std::ios::in);

    if (file) {
      std::string version;

      file >> version;
      file.close();

      if (version.find("2006")!=std::string::npos) {
        ossoVersion=L"OS 2006";
      }
      else if (version.find("2007")!=std::string::npos) {
        ossoVersion=L"OS 2007";
      }
      else if (version.find("2008")!=std::string::npos) {
        ossoVersion=L"OS 2008";
      }
    }

    if (ossoVersion.empty()) {
      ossoVersion=L"OS ????";
    }
  }

  void UpdateEntries()
  {
    currentPackages->Disable();
    currentPackages->Off();
    currentPackages->Clear();

    std::wstring search=this->search->Get();

    for (std::list<Package>::iterator iter=packages.begin();
         iter!=packages.end();
         ++iter) {
      if (search.empty() || iter->GetName().find(search)!=std::wstring::npos) {
        currentPackages->Append(PackageEntry(&(*iter)));
      }
    }

    currentPackages->Sort(1,true);
    currentPackages->On();
    currentPackages->Enable();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      currentPackages->Disable();
      currentPackages->Off();
      packageListTask.Start();
    }
    else if (model==clearFilterAction && clearFilterAction->IsFinished()) {
      lastSearch==L"";
      search->Set(L"");
      UpdateEntries();
    }
    else if (model==showPackageAction && showPackageAction->IsFinished()) {
      Package*       package=currentPackages->GetEntry(selection->GetLine()).GetPackage();
      PackageDetails *dialog;

      dialog=new PackageDetails(*package);
      dialog->SetParent(this);
      dialog->SetTitle(L"Details...");

      if (dialog->Open()) {
        dialog->EventLoop();
        dialog->Close();
      }

      delete dialog;
    }
    else if (model==aboutAction && aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      /*if (configurationChanged && !SaveConfig()) {
        std::cerr << "Error while storing configuration!" << std::endl;
      }*/
    }
    else if (model==packageListTask.GetFinishedAction() && packageListTask.GetFinishedAction()->IsFinished()) {
      UpdateEntries();
    }
    else if (model==search) {
      Lum::OS::display->RemoveTimer(searchTimerAction);
      Lum::OS::display->AddTimer(1,500000,searchTimerAction);
    }
    else if (model==searchTimerAction && searchTimerAction->IsFinished()) {
      if (search->Get()!=lastSearch) {
        UpdateEntries();
        lastSearch=search->Get();
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainDialog>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Show installed applications..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2008, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainDialog>::Prepare();
  }
};


LUM_MAIN(Main,Lum::Base::StringToWString(PACKAGE_NAME))
